/* ------------------------------------------------------------------------- */
/*
 *  tcp.h
 *
 *  Copyright (c) 2004 - 2008, clown. All rights reserved.
 *
 *  Redistribution and use in source and binary forms, with or without
 *  modification, are permitted provided that the following conditions
 *  are met:
 *
 *    - Redistributions of source code must retain the above copyright
 *      notice, this list of conditions and the following disclaimer.
 *    - Redistributions in binary form must reproduce the above copyright
 *      notice, this list of conditions and the following disclaimer in the
 *      documentation and/or other materials provided with the distribution.
 *    - No names of its contributors may be used to endorse or promote
 *      products derived from this software without specific prior written
 *      permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 *  TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 *  PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 *  LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 *  NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 *  SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  Last-modified: Wed 16 Jul 2008 18:50:03 JST
 */
/* ------------------------------------------------------------------------- */
#ifndef CLX_TCP_H
#define CLX_TCP_H

#include <string>
#include <stdexcept>
#include "socket.h"
#include "sockaddress.h"
#include "sockstream.h"
#include "sockmanager.h"
#include "timer.h"

namespace clx {
	namespace tcp {
		/* ----------------------------------------------------------------- */
		//  basic_socket
		/* ----------------------------------------------------------------- */
		template <
			int Family,
			class CharT = char,
			class Traits = std::char_traits<CharT>
		>
		class basic_socket : public basic_rawsocket<SOCK_STREAM, Family> {
		public:
			typedef basic_rawsocket<SOCK_STREAM, Family> rawsocket;
			typedef basic_sockaddress<Family, IPPROTO_TCP> sockaddress;
			typedef CharT char_type;
			typedef typename std::basic_string<CharT, Traits> string_type;
			
			// constructor and destructor
			basic_socket() : rawsocket(), addr_() {}
			
			basic_socket(const basic_socket& cp) : addr_(cp.addr_) {
				this->assign(cp);
			}
			
			basic_socket& operator=(const basic_socket& cp) {
				addr_ = cp.addr_;
				this->assign(cp);
			}
			
			explicit basic_socket(socket_int s, const sockaddress& addr) :
				rawsocket(s), addr_(addr) {}
			
			explicit basic_socket(const char_type* host, int port) :
				rawsocket(), addr_(host, port) { this->connect(); }
			
			explicit basic_socket(const string_type& host, int port) :
				rawsocket(), addr_(host, port) { this->connect(); }
			
			explicit basic_socket(const char_type* host, const char_type* service) :
				rawsocket(), addr_(host, service) { this->connect(); }
			
			explicit basic_socket(const string_type& host, const string_type& service) :
				rawsocket(), addr_(host, service) { this->connect(); }
			
			virtual ~basic_socket() {}
			
			// operator
			basic_socket& connect(const char_type* host, int port) {
				addr_.assign(host, port);
				return this->connect();
			}
			
			basic_socket& connect(const string_type& host, int port) {
				return this->connect(host.c_str(), port);
			}
			
			basic_socket& connect(const char_type* host, const char_type* service) {
				addr_.assign(host, service);
				return this->connect();
			}
			
			basic_socket& connect(const string_type& host, const string_type& service) {
				return this->connect(host.c_str(), service.c_str());
			}
			
			int send(const char_type* dat, int n) {
				return ::send(this->socket(), dat, n, 0);
			}
			
			int send(const string_type& dat) {
				return this->send(dat.c_str(), static_cast<int>(dat.size()));
			}
			
			int recv(char_type* buf, int n) {
				return ::recv(this->socket(), buf, n, 0);
			}
			
			const sockaddress& address() const { return addr_; }
			
		private:
			sockaddress addr_;
			
			basic_socket& connect() {
				if (this->open() < 0) throw clx::socket_error("socket");
				if (::connect(this->socket(), (struct sockaddr*)addr_.data(), addr_.size()) == -1) {
					throw clx::socket_error("connect");
				}
				return *this;
			}
		};
		
		/* ----------------------------------------------------------------- */
		//  basic_acceptor
		/* ----------------------------------------------------------------- */
		template <
			int Family,
			class CharT = char,
			class Traits = std::char_traits<CharT>
		>
		class basic_acceptor : public basic_rawsocket<SOCK_STREAM, Family> {
		public:
			typedef basic_rawsocket<SOCK_STREAM, Family> rawsocket;
			typedef basic_sockaddress<Family, IPPROTO_TCP> sockaddress;
			typedef CharT char_type;
			typedef typename std::basic_string<CharT, Traits> string_type;
			
			static const int default_nlisten = 100;
			
			// constructor and destructor
			basic_acceptor() : rawsocket(), nlisten_(default_nlisten) {}
			
			explicit basic_acceptor(int port, int n = default_nlisten) :
				rawsocket(), addr_(port), nlisten_(default_nlisten) { this->open(); }
			
			explicit basic_acceptor(const char_type* service, int n = default_nlisten) :
				rawsocket(), addr_(service), nlisten_(default_nlisten) { this->open(); }
			
			explicit basic_acceptor(const string_type& service, int n = default_nlisten) :
				rawsocket(), addr_(service), nlisten_(default_nlisten) { this->open(); }
			
			virtual ~basic_acceptor() {}
			
			// operator
			basic_acceptor& open(int port, int n = default_nlisten) {
				addr_.assign(port);
				nlisten_ = n;
				return this->open();
			}
			
			basic_acceptor& open(const char_type* service, int n = default_nlisten) {
				addr_.assign(service);
				nlisten_ = n;
				return this->open();
			}
			
			basic_acceptor& open(const string_type& service, int n = default_nlisten) {
				return this->open(service.c_str(), n);
			}
			
			basic_socket<Family> accept(int retry = 10) {
				socket_int s = -1;
				sockaddress addr;
				typename sockaddress::inet_type tmp;
				socklen_type len = sizeof(tmp);
				
				for (int i = 0; i < retry; i++) {
					s = ::accept(this->socket(), (struct sockaddr*)&tmp, &len);
					if (s == -1) clx::sleep(1.0);
					else break;
				}
				addr = tmp;
				return basic_socket<Family>(s, addr);
			}
		
		private:
			sockaddress addr_;
			int nlisten_;
			
			basic_acceptor& open() {
				if (rawsocket::open() < 0) throw socket_error("socket");
				if (::bind(this->socket(), (struct sockaddr*)addr_.data(), addr_.size()) == -1) {
					throw socket_error("bind");
				}
				if (::listen(this->socket(), nlisten_) == -1) throw socket_error("listen");
				return *this;
			}
		};
		
		typedef basic_rawsocket<SOCK_STREAM, AF_INET> rawsocket;
		typedef basic_socket<AF_INET> socket;
		typedef basic_acceptor<AF_INET> acceptor;
		typedef basic_sockaddress<AF_INET, IPPROTO_TCP> sockaddress;
		typedef basic_sockstream<socket> sockstream;
		typedef basic_sockmanager<SOCK_STREAM, AF_INET> sockmanager;
	};
}

#endif // CLX_TCP_H
