/* ------------------------------------------------------------------------- */
/*
 *  vstream.h
 *
 *  Copyright (c) 2004 - 2009, clown. All rights reserved.
 *
 *  Redistribution and use in source and binary forms, with or without
 *  modification, are permitted provided that the following conditions
 *  are met:
 *
 *    - Redistributions of source code must retain the above copyright
 *      notice, this list of conditions and the following disclaimer.
 *    - Redistributions in binary form must reproduce the above copyright
 *      notice, this list of conditions and the following disclaimer in the
 *      documentation and/or other materials provided with the distribution.
 *    - No names of its contributors may be used to endorse or promote
 *      products derived from this software without specific prior written
 *      permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 *  TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 *  PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 *  LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 *  NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 *  SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  Last-modified: Thu 23 Apr 2009 07:47:00 JST
 */
/* ------------------------------------------------------------------------- */
#ifndef CLX_VSTREAM_H
#define CLX_VSTREAM_H

#include <istream>
#include <streambuf>
#include <string>
#include "shared_ptr.h"

namespace clx {
	/* --------------------------------------------------------------------- */
	//  basic_ivstreambuf
	/* --------------------------------------------------------------------- */
	template <
		class CharT,
		class Traits = std::char_traits<CharT>
	>
	class basic_ivstreambuf : public std::basic_streambuf<CharT, Traits> {
	public:
		typedef size_t size_type;
		typedef CharT char_type;
		typedef typename Traits::pos_type pos_type;
		typedef typename Traits::off_type off_type;
		
		template <class Container>
		basic_ivstreambuf(const Container& data) :
			std::basic_streambuf<CharT, Traits>(),
			p_(new data_container<Container>(data)) {
			this->setg(p_->begin(), p_->begin(), p_->end());
		}
		
		template <class Type>
		basic_ivstreambuf(const Type* data, size_type n) :
			std::basic_streambuf<CharT, Traits>(),
			p_(new data_array<Type>(data, n)) {
			this->setg(p_->begin(), p_->begin(), p_->end());
		}
		
		virtual ~basic_ivstreambuf() throw() {}
		
	protected:
		virtual pos_type seekoff(off_type off, std::ios_base::seekdir way,
			std::ios_base::openmode which = std::ios_base::in) {
			if (which & std::ios_base::in) {
				pos_type last = pos_type(p_->end() - p_->begin());
				pos_type pos = 0;
				if (way == std::ios_base::cur) pos = pos_type(this->gptr() - p_->begin());
				else if (way == std::ios_base::end) pos = last;
				pos += pos_type(off);
				if (pos < 0 || pos > last) return pos_type(-1);
				this->setg(p_->begin(), p_->begin() + pos, p_->end());
				return pos;
			}
			return pos_type(-1);
		}
		
		virtual pos_type seekpos(pos_type pos, std::ios_base::openmode which = std::ios_base::in) {
			if (which & std::ios_base::in) {
				pos_type last = pos_type(p_->end() - p_->begin());
				if (pos < 0 || pos > last) return pos_type(-1);
				this->setg(p_->begin(), p_->begin() + pos, p_->end());
				return pos;
			}
			return pos_type(-1);
		}
		
		virtual std::streamsize showmanyc() {
			return std::streamsize(-1);
		}
		
	private:
		/* ----------------------------------------------------------------- */
		//  base_container
		/* ----------------------------------------------------------------- */
		class base_container {
		public:
			typedef CharT char_type;
			typedef CharT* iterator;
			
			base_container() {}
			virtual ~base_container() throw() {}
			
			virtual iterator begin() = 0;
			virtual iterator end() = 0;
		};
		
		/* ----------------------------------------------------------------- */
		//  data_container
		/* ----------------------------------------------------------------- */
		template <class T>
		class data_container : public base_container {
		public:
			typedef typename base_container::iterator iterator;
			
			data_container(const T& v) : v_(const_cast<T&>(v)) {}
			virtual ~data_container() throw() {}
			
			virtual iterator begin() { return reinterpret_cast<iterator>(&(*v_.begin())); }
			virtual iterator end() { return reinterpret_cast<iterator>(&(*v_.end())); }
			
		private:
			T& v_;
		};
		
		/* ----------------------------------------------------------------- */
		//  data_array
		/* ----------------------------------------------------------------- */
		template <class T>
		class data_array : public base_container {
		public:
			typedef typename base_container::iterator iterator;
			
			data_array(const T* v, size_type n) : v_(const_cast<T*>(v)), size_(n) {}
			virtual ~data_array() throw() {}
			
			virtual iterator begin() { return reinterpret_cast<iterator>(v_); }
			virtual iterator end() { return reinterpret_cast<iterator>(v_ + size_); }
			
		private:
			T* v_;
			size_type size_;
		};
		
		typedef shared_ptr<base_container> container_ptr;
		
		container_ptr p_;
	};
	
	/* --------------------------------------------------------------------- */
	//  basic_ivstream
	/* --------------------------------------------------------------------- */
	template <
		class CharT,
		class Traits = std::char_traits<CharT>
	>
	class basic_ivstream : public std::basic_istream<CharT, Traits> {
	public:
		typedef basic_ivstreambuf<CharT, Traits> streambuf_type;
		typedef typename streambuf_type::size_type size_type;
		
		template <class Container>
		basic_ivstream(const Container& data) :
			std::basic_istream<CharT, Traits>(0), buf_(data) {
			this->rdbuf(&buf_);
		}
		
		template <class Type>
		basic_ivstream(const Type* data, size_type n) :
			std::basic_istream<CharT, Traits>(0), buf_(data, n) {
			this->rdbuf(&buf_);
		}
		
		virtual ~basic_ivstream() throw() { this->rdbuf(0); }
		
	private:
		streambuf_type buf_;
	};
	
	typedef basic_ivstream<char> ivstream;
}

#endif // CLX_VSTREAM_H
