package jp.ac.kyoto_u.jfftw3;

import java.nio.ByteBuffer;
import java.nio.DoubleBuffer;
import java.nio.ByteOrder;
import java.util.Set;

/**
 * 各種プランの基底となる抽象クラスです。
 * @author Katsutoshi Itoyama (itoyama[at]kuis.kyoto-u.ac.jp)
 * @version 2009.10.04
 */
abstract class Plan {

  static {
    System.loadLibrary("jfftw3");
    planSize = sizeofPlan();
  }

  private static final int planSize;

  final ByteBuffer srcBbuff;
  final ByteBuffer dstBbuff;
  private final DoubleBuffer srcDbuff;
  private final DoubleBuffer dstDbuff;
  final byte[] planHandler;

  /**
   * 各種プラン（サブクラス）で使用される入出力バッファの確保を行います。
   * @param srcSize 入力配列のサイズ
   * @param dstSize 出力配列のサイズ
   */
  Plan(final int srcSize,
       final int dstSize) {
    if (srcSize <= 0)
      throw new IllegalArgumentException
        (String.format("srcSize must be positive: %d", srcSize));
    if (dstSize <= 0)
      throw new IllegalArgumentException
        (String.format("dstSize must be positive: %d", dstSize));

    srcBbuff = fftwMalloc(srcSize).order(ByteOrder.nativeOrder());
    dstBbuff = fftwMalloc(dstSize).order(ByteOrder.nativeOrder());
    srcDbuff = srcBbuff.asDoubleBuffer();
    dstDbuff = dstBbuff.asDoubleBuffer();
    planHandler = new byte[planSize];
  }

  public DoubleBuffer src() {
    return srcDbuff;
  }

  public DoubleBuffer dst() {
    return dstDbuff;
  }

  /** このプランをファイナライズします。 */
  @Override protected void finalize() throws Throwable {
    try {
      super.finalize();
    } finally {
      destroyPlan();
    }
  }

  private static final native
  ByteBuffer fftwMalloc(int size);

  private static final native
  int sizeofPlan();

  /** 作成したプランに基づいて変換を行います。 */
  public final native
  void execute();

  private final native
  void destroyPlan();

}
