/*
 * Copyright 2009 Project CodeCluster
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *     http://www.apache.org/licenses/LICENSE-2.0
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KI ND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.codecluster.filter;

import javax.servlet.Filter;
import javax.servlet.FilterConfig;
import javax.servlet.FilterChain;
import javax.servlet.ServletRequest;
import javax.servlet.ServletResponse;
import javax.servlet.ServletException;
import java.io.IOException;
import java.util.logging.LogManager;
import java.util.logging.Logger;

/**
 * java.util.logging の設定を読み直すサーブレットフィルタです。<br>
 * 主に開発中に VM の再起動を行わずに頻繁に設定変更する場合に使用できます。<br>
 * VM によって事前に読み込まれた設定を上書きするので注意して利用してください。<br>
 * <br>
 * 以下パラメータでプロパティ名を指定できます。<br>
 * {@code <param-name>properties</param-name>}<br>
 * {@code <param-value>C2logging.properties</param-value>}<br>
 * 指定がない場合には "C2logging.properties" を読み込みます。<br>
 * <br>
 * 以下パラメータで読込みタイミングをフィルタ読込み時に変更できます。<br>
 * 無指定（デフォルト）は毎リクエスト読込みです。<br>
 * {@code <param-name>trigger</param-name>}<br>
 * {@code <param-value>init</param-value>}<br>
 * 
 */
public class LoggingConfigurationFilter implements Filter {
	private static Logger logger = Logger.getLogger(LoggingConfigurationFilter.class.getName());
	private static final String DEFAULT_PROPERTIES = "C2logging.properties";
	private String prop;
	private boolean isInit = false;

	/* (non-Javadoc)
	 * @see javax.servlet.Filter#init(javax.servlet.FilterConfig)
	 */
	public void init(FilterConfig config) throws ServletException {
		prop = config.getInitParameter("properties");
		if (prop == null) {
			prop = DEFAULT_PROPERTIES;
		}

		String trigger = config.getInitParameter("trigger");
		if (trigger != null && trigger.equalsIgnoreCase("init")) {
			isInit = true;
		}

		if (isInit) {
			readConfig();
		}
		logger.info("init....");
	}

	/* (non-Javadoc)
	 * @see javax.servlet.Filter#doFilter(javax.servlet.ServletRequest, javax.servlet.ServletResponse, javax.servlet.FilterChain)
	 */
	public void doFilter(ServletRequest request, ServletResponse response,
			FilterChain chain) throws ServletException, IOException {

		if (!isInit) {
			readConfig();
		}
		chain.doFilter(request, response);
	}

	/* (non-Javadoc)
	 * @see javax.servlet.Filter#destroy()
	 */
	public void destroy() {
		prop = null;
		logger.info("destroy....");
	}

	private void readConfig() {
		try {
			LogManager.getLogManager().readConfiguration(Thread.currentThread().getContextClassLoader().getResourceAsStream(prop));
		} catch (SecurityException e) {
			e.printStackTrace();
		} catch (IOException e) {
			e.printStackTrace();
		}
		logger.info("readConfiguration java.util.logging properties = " + Thread.currentThread().getContextClassLoader().getResource(prop).toString());
	}
}
