/*
 * Copyright 2009 Project CodeCluster
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *     http://www.apache.org/licenses/LICENSE-2.0
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KI ND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.codecluster.filter;

import javax.servlet.Filter;
import javax.servlet.FilterConfig;
import javax.servlet.FilterChain;
import javax.servlet.ServletRequest;
import javax.servlet.ServletResponse;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpServletResponseWrapper;

import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.util.logging.Level;
import java.util.logging.Logger;

/**
 * SSLアクセラレータ環境下などでリダイレクトURLを生成する際に https が http となっていしまう問題を解決するサーブレットフィルタです。<br>
 * アクセラレータやApacheでSSL(HTTPS)であることを示すHTTPヘッダを付与しておくことでスキーマを決定します。<br>
 * <br>
 * 以下のパラメータでSSL判定を行うHTTPヘッダを指定します。デフォルトは指定されておらず、書き換えを行いません。<br>
 * HTTPヘッダが存在すれば SSL とみなし、リダイレクト絶対URLを生成するときに http:// を https:// に書き換えます。<br>
 * {@code <param-name>secure-header</param-name>}<br>
 * {@code <param-value>X-HTTPS</param-value>}<br>
 * <br>
 * sendRedirect() 呼び出し時点で http://, https:// ではじまる URL が指定されている場合には書き換えは行いません。<br> 
 * @see SendRedirectFilterEx
 */
public class SendRedirectFilter implements Filter {
	private static final Logger logger = Logger.getLogger(SendRedirectFilter.class.getName());
	private String secureHeader = null;

	/* (non-Javadoc)
	 * @see javax.servlet.Filter#init(javax.servlet.FilterConfig)
	 */
	public void init(FilterConfig config) throws ServletException {
		secureHeader = config.getInitParameter("secure-header");
		if (logger.isLoggable(Level.FINE)) {
    		logger.fine("init: secureHeader = " + secureHeader);
		}
	}

	/* (non-Javadoc)
	 * @see javax.servlet.Filter#doFilter(javax.servlet.ServletRequest, javax.servlet.ServletResponse, javax.servlet.FilterChain)
	 */
	public void doFilter(ServletRequest request, ServletResponse response,
			FilterChain chain) throws ServletException, IOException {

		Response res = new Response((HttpServletResponse)response, (HttpServletRequest)request, secureHeader);
		chain.doFilter(request, res);
	}

	/* (non-Javadoc)
	 * @see javax.servlet.Filter#destroy()
	 */
	public void destroy() {
		secureHeader = null;
	}
	
	/**
	 * sendRedirect() を独自のルールで変更できるようにした HttpServletResponseWrapper クラスです。 
	 */
	@SuppressWarnings("deprecation")
	public class Response extends HttpServletResponseWrapper {
		private HttpServletRequest request = null;
		private String secureHeader = null;

		public Response(HttpServletResponse response, HttpServletRequest request, String secureHeader) {
			super(response);
			this.request = request;
			this.secureHeader = secureHeader;
		}

		/**
		 * http://, https:// で始まらないアドレスが指定された場合、SSL判定用のヘッダがあるかを検査し、
		 * 必要があればリダイレクトプロトコルを書き換えてリダイレクト先を決定します。<br>
		 * 
		 * @see javax.servlet.http.HttpServletResponseWrapper#sendRedirect(java.lang.String)
		 */
		@Override
		public void sendRedirect(String location) throws IOException {
			// そのままリダイレクト
			if (secureHeader == null
					|| location.startsWith("http://")
					|| location.startsWith("https://")) {
				if (logger.isLoggable(Level.FINE)) {
		    		logger.fine("sendRedirect: not rewrite: " + location);
				}
				super.sendRedirect(location);
				return;
			}
			
			String header = request.getHeader(secureHeader);
			if (header == null) {
				if (logger.isLoggable(Level.FINE)) {
		    		logger.fine("sendRedirect: no secureHeder: " + location);
				}
				super.sendRedirect(location);
				return;
			}
			
			// リダイレクト先を https 用に再構築
			String newLocation = location;
			try {
				// 相対アドレスの調整
				URI uri = (new URI(request.getRequestURL().toString())).resolve(location);
				// URL 再構築
				int port = request.getServerPort();
				if (port == 443 || port == 80) {
					// デフォルトポート番号
					port = -1;
				}
				URL url = new URL("https", request.getServerName(), port, uri.toURL().getFile());
				newLocation = url.toString();
				if (logger.isLoggable(Level.FINE)) {
		    		logger.fine("sendRedirect: " + location + " -> " + newLocation);
				}
			} catch (MalformedURLException e) {
				// もとのままとする
				if (logger.isLoggable(Level.FINE)) {
		    		logger.fine(e.toString());
		    	}
			} catch (URISyntaxException e) {
				// もとのままとする
				if (logger.isLoggable(Level.FINE)) {
		    		logger.fine(e.toString());
		    	}
			}

			super.sendRedirect(newLocation);
		}
	}
}
