/*
 * Copyright 2009 Project CodeCluster
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *     http://www.apache.org/licenses/LICENSE-2.0
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KI ND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.codecluster.filter;

import javax.servlet.Filter;
import javax.servlet.FilterConfig;
import javax.servlet.FilterChain;
import javax.servlet.ServletRequest;
import javax.servlet.ServletResponse;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpServletResponseWrapper;

import java.io.IOException;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.codecluster.C2Constants;
import org.codecluster.http.AutoRedirectRuleImpl;
import org.codecluster.http.RedirectRule;
import org.codecluster.http.SimpleRedirectRuleImpl;
import org.codecluster.util.C2Properties;
import org.codecluster.util.C2PropertiesManager;

/**
 * {@link HttpServletResponse#sendRedirect(String)} の解決ルールを独自実装したクラスへ切り替えるフィルタです。<br>
 * C2config.xml に {@link RedirectRule} インタフェースを実装したクラスを指定します。<br>
 * <br>
 * @see SendRedirectFilter
 * @see RedirectRule
 * @see SimpleRedirectRuleImpl
 * @see AutoRedirectRuleImpl
 */
public class SendRedirectFilterEx implements Filter {
	private static final Logger logger = Logger.getLogger(SendRedirectFilterEx.class.getName());

	/* (non-Javadoc)
	 * @see javax.servlet.Filter#init(javax.servlet.FilterConfig)
	 */
	public void init(FilterConfig config) throws ServletException {
	}

	/* (non-Javadoc)
	 * @see javax.servlet.Filter#doFilter(javax.servlet.ServletRequest, javax.servlet.ServletResponse, javax.servlet.FilterChain)
	 */
	public void doFilter(ServletRequest request, ServletResponse response,
			FilterChain chain) throws ServletException, IOException {

		Response res = new Response((HttpServletResponse)response, (HttpServletRequest)request);
		chain.doFilter(request, res);
	}

	/* (non-Javadoc)
	 * @see javax.servlet.Filter#destroy()
	 */
	public void destroy() {
	}
	
	/**
	 * sendRedirect() を独自のルールで変更できるようにした HttpServletResponseWrapper クラスです。
	 */
	@SuppressWarnings("deprecation")
	public class Response extends HttpServletResponseWrapper implements C2Constants {
		private HttpServletRequest request = null;
		
		public Response(HttpServletResponse response, HttpServletRequest request) {
			super(response);
			this.request = request;
		}

		/**
		 * {@link RedirectRule} インタフェースを実装したリダイレクトルールを呼び出します。<br>
		 * <br>
		 * http://, https:// で始まらないアドレスが指定された場合、独自のリダイレクト先アドレス生成ロジックを呼び出して、
		 * リダイレクト先を決定します。<br>
		 * デフォルトで {@link SimpleRedirectRuleImpl} と {@link AutoRedirectRuleImpl} が提供されます。
		 * 
		 * @see javax.servlet.http.HttpServletResponseWrapper#sendRedirect(java.lang.String)
		 */
		@Override
		public void sendRedirect(String location) throws IOException {
			String orig = location;

			// そのままリダイレクト
			if (location.startsWith("http://") || location.startsWith("https://")) {
				if (logger.isLoggable(Level.FINE)) {
		    		logger.fine("redirect: through: " + location);
				}
				super.sendRedirect(location);
				return;
			}

			C2Properties prop = C2PropertiesManager.getC2PropertiesFromXML(DEFAULT_CONFIG_XML);
			String ruleClassName = prop.getProperty(CONF_REDIRECT_RULE_CLASS);
			if (ruleClassName != null) {
				Class<?> ruleClazz;
				try {
					ruleClazz = Class.forName(ruleClassName);
					Object obj = ruleClazz.newInstance();
					if (obj instanceof RedirectRule) {
						location = ((RedirectRule)obj).redirect(location, this.request);
					}
				} catch (ClassNotFoundException e) {
					e.printStackTrace();
				} catch (InstantiationException e) {
					e.printStackTrace();
				} catch (IllegalAccessException e) {
					e.printStackTrace();
				}
				
				if (logger.isLoggable(Level.FINE)) {
		    		logger.fine("redirect: rewrite: " + orig + " => " + location + " by " + ruleClassName);
				}
			} else {
				if (logger.isLoggable(Level.FINE)) {
		    		logger.fine("redirect: through: " + location);
				}
			}

			super.sendRedirect(location);
		}
	}

}
