/*
 * Copyright 2009 Project CodeCluster
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *     http://www.apache.org/licenses/LICENSE-2.0
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KI ND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.codecluster.filter;

import javax.servlet.Filter;
import javax.servlet.FilterConfig;
import javax.servlet.FilterChain;
import javax.servlet.ServletRequest;
import javax.servlet.ServletResponse;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;
import java.io.IOException;
import java.util.logging.Level;
import java.util.logging.Logger;

/**
 * HTTP レスポンスヘッダに HttpSession のセッションIDを設定するサーブレットフィルタです。<br>
 * <br>
 * 以下のパラメータで出力するヘッダを指定できます。<br>
 * {@code <param-name>header</param-name>}<br>
 * {@code <param-value>X-HTTP-SESSION-ID</param-value>}<br>
 * ヘッダ指定がないか、有効な HttpSession がない場合には出力しません。<br>
 * <br>
 * 主に負荷分散装置、リバースプロキシなどに向けて情報を伝達する必要がある場合に使用します。
 * 
 */
public class SetSessionHeaderFilter implements Filter {
	private static Logger logger = Logger.getLogger(SetSessionHeaderFilter.class.getName());
	private String header = null;

	/* (non-Javadoc)
	 * @see javax.servlet.Filter#init(javax.servlet.FilterConfig)
	 */
	public void init(FilterConfig config) throws ServletException {
		header = config.getInitParameter("header");
		if (header != null) {
			header = header.trim();
			if (header.equals("")) {
				header = null;
			}
		}
		logger.fine("init filter. extended session id header: " + header);
	}

	/* (non-Javadoc)
	 * @see javax.servlet.Filter#doFilter(javax.servlet.ServletRequest, javax.servlet.ServletResponse, javax.servlet.FilterChain)
	 */
	public void doFilter(ServletRequest request, ServletResponse response,
			FilterChain chain) throws ServletException, IOException {

		chain.doFilter(request, response);

		// 追加ヘッダの指定があり、有効なセッション中である場合、追加ヘッダにセッション情報を設定する
		if (header != null) {
			HttpSession s = ((HttpServletRequest)request).getSession(false);
			if (s != null) {
				if (logger.isLoggable(Level.FINE)) {
					logger.fine("set session id: " + header + ", " + s.getId());
				}
				((HttpServletResponse)response).setHeader(header, s.getId());
			} else {
				if (logger.isLoggable(Level.FINE)) {
					logger.fine("response session: null");
				}
			}
		}
	}

	/* (non-Javadoc)
	 * @see javax.servlet.Filter#destroy()
	 */
	public void destroy() {
		header = null;
	}
}
