/*
 * Copyright 2009 Project CodeCluster
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *     http://www.apache.org/licenses/LICENSE-2.0
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KI ND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.codecluster.session;

import javax.servlet.http.HttpSession;

import org.codecluster.C2Constants;
import org.codecluster.util.C2Properties;
import org.codecluster.util.C2PropertiesManager;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Enumeration;
import java.util.Iterator;

/**
 * HttpSession 内に格納される仮想的なセッション情報(vsession)を操作するためのクラスです。<br>
 * <br>
 * {@link C2SessionManager} によりインスタンス化されることを想定されて設計されており、
 * コンストラクタにより HttpSession と関連づけられます。<br>
 * <br>
 * 仮想セッションは HttpSession(jsessionid) + リクエストパラメータ(デフォルトでは vsid) で識別され、
 * 同一 HttpSession 内で複数のセッションを管理することを目的としています。<br>
 * <br>
 * このクラス自体が HttpSession に格納されるわけでなく、HttpSession 上のオブジェクトを操作するための、
 * ツール群であることに注意してください。<br>
 * <br>
 * 仮想セッションの有効期限を設定することができますが、有効期限については既存セッションを取得するコンストラクタと、
 * メソッド {@link #isExpires() isExpires()} によってのみチェックされます。<br>
 * setAttribute(), getAttribute() などの仮想セッション情報操作中には有効期限チェックが行われないことに注意してください。<br>
 * <br>
 * 有効期限の初期値はプロパティ XML ファイル中に記述されたプロパティ  "c2session-timeout" で指定した時間（単位分）となり、
 * "-1" を指定すると無期限となります。<br>
 * 
 */
public class C2Session {
	/**
	 * この仮想セッションが作られた日時を格納するためのキー。規定値({@value})
	 */
	protected static final String KEY_CREATION_TIME = "$CreationTime$";
	/**
	 * この仮想セッションの有効日時を格納するためのキー。規定値({@value})
	 */
	protected static final String KEY_EXPIRES_TIME = "$ExpiresTime$";
	/**
	 * この仮想セッションの最終取得日時を格納するためのキー。規定値({@value})
	 */
	protected static final String KEY_LASTACCESSED_TIME = "$LastAccessedTime$";
	
	/**
	 * この仮想セッションが格納されている HttpSession
	 */
	protected HttpSession session;
	/**
	 * この仮想セッションが格納されている HttpSession キーの prefix
	 */
	protected String attributePrefix;
	/**
	 * この仮想セッションのセッションIDを受け取ったリクエストパラメータ名
	 */
	protected String sessionKey;
	/**
	 * この仮想セッションのセッションID
	 */
	protected String sessionId;
	
	/**
	 * 仮想セッションを作成します。<br>
	 * 指定された仮想セッションIDの仮想セッションが存在せず、create が true の場合は、
	 * 新しい仮想セッションを作成して関連付け、false の場合は、NoSessionException をスローします。<br>
	 * 仮想セッションの有効期限の初期値はプロパティ XML ファイル中に記述されたプロパティ  "c2session-timeout" で指定した時間（単位分）となり、
	 * "-1" を指定すると無期限となります。正しい数値を指定していない場合には無期限とします。<br>
	 * <br>
	 * 仮想セッションが存在する場合には、新規作成せずに既存仮想セッションを関連付けます。<br>
	 * 既存仮想セッションの有効期限が切れていた場合は、例外 ExpiresSessionException をスローします。<br>
	 * <br>
	 * 
	 * @param session 関連付ける HttpSession
	 * @param key 仮想セッションIDのリクエストパラメータ名
	 * @param id 仮想セッションID
	 * @param create 指定された仮想セッションがないときに新規セッションを作成する場合は true。例外を返す場合は false
	 * @throws NoSessionException 既存セッションが存在しない場合にスローされます。
	 * @throws ExpiresSessionException 既存セッションが有効期限切れの場合にスローされます。
	 * @throws IllegalArgumentException パラメータが無効な場合にスローされます。
	 */
	public C2Session(HttpSession session, String key, String id, boolean create) throws NoSessionException, ExpiresSessionException {
		if (session == null) throw new IllegalArgumentException("session is null");
		if (key == null) throw new IllegalArgumentException("key is null");
		if (id == null)	throw new IllegalArgumentException("id is null");

		this.session = session;
		this.sessionKey = key;
		this.sessionId = id;
		this.attributePrefix = C2Session.class.getName() + '_' + id + '_';

		if (session.getAttribute(attributePrefix + KEY_CREATION_TIME) == null) {
			if (create == true) {
				// プロパティから有効期限（分）の値を取得して設定する
				C2Properties prop = C2PropertiesManager.getC2PropertiesFromXML(C2Constants.DEFAULT_CONFIG_XML);
				String sesionTimeout = prop.getProperty(C2Constants.CONF_C2SESSION_TIMEOUT, "-1"); // default unlimited
				long timeout = -1;
				try {
					timeout = Long.parseLong(sesionTimeout);
				}
				// 値が正しく取得できない場合には無期限とする
				catch (NumberFormatException e) {
					// unlimited
				}

				setCreationTime();
				if (timeout < 0) {
					setExpiresTime(-1);
				} else {
					setMaxAge(timeout * 60);
				}
			} else {
				throw new NoSessionException(NoSessionException.NO_VIRTUAL_SESSION, "invalid session: jsession=" + session.getId() + ", vsession=" + sessionId);
			}
		}

		if (isExpires()) {
			throw new ExpiresSessionException("expires session: jsession=" + session.getId() + ", vsession=" + sessionId);
		}
		
		setLastAccessedTime();
	}

	/**
	 * この仮想セッションが格納されている HttpSession を返します。
	 * @return 格納されている HttpSession
	 */
	public HttpSession getHttpSession() {
		return session;
	}

	/**
	 * この仮想セッションが格納されている HttpSession のセッションIDを返します。<br>
	 * getHttpSession().getId() と等価です。
	 * @return この仮想セッションが格納されている HttpSession のセッションID
	 */
	public String getHttpSessionId() {
		return session.getId();
	}
	
	/**
	 * この仮想セッションのセッションIDを受け取ったリクエストパラメータ名を返します。
	 * @return この仮想セッションのセッションIDを受け取ったリクエストパラメータ名
	 */
	public String getKey() {
		return sessionKey;
	}

	/**
	 * この仮想セッションのセッションIDを返します。
	 * @return この仮想セッションの仮想セッションID
	 */
	public String getId() {
		return sessionId;
	}
	
	/**
	 * この仮想セッションに指定した名前でオブジェクトを保存します。オブジェクトはシリアライズ可能でなくてはなりません。
	 * @param key オブジェクトの名前を指定する文字列
	 * @param value 保存されるオブジェクト
	 * @throws IllegalArgumentException key が null か、value がシリアライズ可能オブジェクトでない場合にスローされます。
	 * @throws IllegalStateException セッションが無効である場合にスローされます。
	 */
	public void setAttribute(String key, Object value) /* throws NotSerializableException */ {
		// パラメータエラー
		if (key == null) {
			throw new IllegalArgumentException("key is null");
		}
		// null 以外でシリアライズできないものはセッションに格納させない
		if (value != null && !(value instanceof Serializable)) {
			//throw new NotSerializableException(value.getClass().getName());
			throw new IllegalArgumentException("Not Serializable: " + value.getClass().getName());
		}
		validateSession();

		String realKey = getRealKey(key);
		session.setAttribute(realKey, value);
	}

	/**
	 * この仮想セッションから指定した名前のオブジェクトを取り出します。オブジェクトがない場合には null を返します。
	 * @param key オブジェクトの名前を指定する文字列
	 * @return 指定された名前のオブジェクト
	 * @throws IllegalArgumentException key が null の場合にスローされます。
	 * @throws IllegalStateException セッションが無効である場合にスローされます。
	 */
	public Object getAttribute(String key) {
		if (key == null) {
			throw new IllegalArgumentException();
		}
		validateSession();
		String realKey = getRealKey(key);
		return session.getAttribute(realKey);
	}

	/**
	 * この仮想セッション内のオブジェクトの名前(キー)の Iterator を返します。
	 * @return この仮想セッション内のオブジェクトの名前(キー)の Iterator
	 * @throws IllegalStateException セッションが無効である場合にスローされます。
	 */
	@SuppressWarnings("unchecked")
	public Iterator<String> getAttributeNames() {
		validateSession();
		Enumeration<String> enu = session.getAttributeNames();
		ArrayList<String> ar = new ArrayList<String>();
		while (enu.hasMoreElements()) {
			String k = enu.nextElement();
			if (k.startsWith(attributePrefix)) {
				// attributePrefix を取り除いて返却
				ar.add(k.substring(attributePrefix.length()));
			}
		}
		return ar.iterator();
	}

	/**
	 * この仮想セッション内のオブジェクトの HttpSession 中に格納されている実際のキーの Iterator を返します。
	 * @return キーの Iterator<String>
	 * @throws IllegalStateException セッションが無効である場合にスローされます。
	 */
	@SuppressWarnings("unchecked")
	protected Iterator<String> getAttributeRealNames() {
		validateSession();
		Enumeration<String> enu = session.getAttributeNames();
		ArrayList<String> ar = new ArrayList<String>();
		while (enu.hasMoreElements()) {
			String k = enu.nextElement();
			if (k.startsWith(attributePrefix)) {
				ar.add(k);
			}
		}
		return ar.iterator();
	}
	
	/**
	 * この仮想セッションから指定した名前のオブジェクトを削除します。
	 * @param key オブジェクトの名前を指定する文字列
	 * @throws IllegalStateException セッションが無効である場合にスローされます。
	 */
	public void removeAttribute(String key) {
		if (key == null) {
			throw new IllegalArgumentException();
		}
		validateSession();
		String realKey = getRealKey(key);
		session.removeAttribute(realKey);
	}

	/**
	 * この仮想セッションからすべてのオブジェクトを削除します。
	 * @throws IllegalStateException セッションが無効である場合にスローされます。
	 */
	public void removeAllAttribute() {
		// 作られた日時を保存して、全削除を実施後、復元する
		long ctime = getCreationTime();
		invalidate();
		setCreationTime(ctime);
	}
	
	/**
	 * この仮想セッションを無効にします。
	 * @throws IllegalStateException セッションが無効である場合にスローされます。
	 */
	public void invalidate() {
		// remove all
		Iterator<String> it = getAttributeRealNames();
		while (it.hasNext()) {
			session.removeAttribute(it.next());
		}
	}

	/**
	 * この仮想セッションが作成された時刻を、GMT 1970年 1 月 1 日 0 時からのミリ秒単位で返します。
	 * @return GMT 1970年 1 月 1 日 からのミリ秒単位で表した、このセッションが作成された時刻を示す long
	 * @throws IllegalStateException セッションが無効である場合にスローされます。
	 */
	public long getCreationTime() {
		return ((Long)getAttribute(KEY_CREATION_TIME)).longValue();
	}

	/**
	 * この仮想セッションが作成された時刻を、GMT 1970年 1 月 1 日 0 時からのミリ秒単位で設定します。
	 * @param creation GMT 1970年 1 月 1 日 からのミリ秒単位で表した、このセッションが作成された時刻を示す long
	 * @throws IllegalStateException セッションが無効である場合にスローされます。
	 */
	protected void setCreationTime(long creation) {
		session.setAttribute(attributePrefix + KEY_CREATION_TIME, new Long(creation));
	}

	/**
	 * この仮想セッションが作成された時刻を、現在時刻で設定します。
	 * @throws IllegalStateException セッションが無効である場合にスローされます。
	 */
	protected void setCreationTime() {
		setCreationTime(System.currentTimeMillis());
	}

	/**
	 * この仮想セッションの有効期限を、GMT 1970年 1 月 1 日 0 時からのミリ秒単位で返します。<br>
	 * <br>
	 * 有効期限が設定されていない場合は -1 を返します。<br>
	 * @return GMT 1970年 1 月 1 日 からのミリ秒単位で表した、このセッションの有効期限を示す long
	 * @throws IllegalStateException セッションが無効である場合にスローされます。
	 */
	public long getExpiresTime() {
		return ((Long)getAttribute(KEY_EXPIRES_TIME)).longValue();
	}

	/**
	 * この仮想セッションの有効期限を、GMT 1970年 1 月 1 日 0 時からのミリ秒単位で設定します。<br>
	 * 有効期限を設定しない場合は -1 を設定します。<br>
	 * <br>
	 * 有効期限はコンストラクタ C2Session() か、isExpires() でのみ検査されます。<br>
	 * setAttribute(), getAttribute() などのでは検査されません。<br>
	 * @param exp GMT 1970年 1 月 1 日 からのミリ秒単位で表した、このセッションの有効期限を示す long
	 * @throws IllegalStateException セッションが無効である場合にスローされます。
	 */
	public void setExpiresTime(long exp) {
		// 0 以下は -1 に補正
		if (exp < 0) {
			exp = -1;
		}
		setAttribute(KEY_EXPIRES_TIME, new Long(exp));
	}
	/**
	 * この仮想セッションの有効期限を、セッション作成時刻からの経過秒数で設定します。
	 * <br>
	 * 有効期限はコンストラクタ C2Session() か、isExpires() でのみ検査されます。<br>
	 * setAttribute(), getAttribute() などのでは検査されません。<br>
	 * @param sec このセッションが作成された時刻からの秒数を示す long
	 * @throws IllegalStateException セッションが無効である場合にスローされます。
	 */
	public void setMaxAge(long sec) {
		if (sec < 0) {
			sec = 0;
		}
		long exp = getCreationTime() + sec * 1000;
		setAttribute(KEY_EXPIRES_TIME, new Long(exp));
	}

	/**
	 * この仮想セッションが有効期限に達しているかを検査します。<br>
	 * @return 有効期限切れの場合 true を返す
	 * @throws IllegalStateException セッションが無効である場合にスローされます。
	 */
	public boolean isExpires() {
		long exp = getExpiresTime();
		if (exp == -1) {
			return false;
		}
		long now = System.currentTimeMillis();
		return now > exp;
	}


	/**
	 * この仮想セッションの最終取得時刻を、GMT 1970年 1 月 1 日 0 時からのミリ秒単位で設定します。<br>
	 * 現在の実装では HTTPリクエスト毎にコンストラクタする想定で、コンストラクタでのみ情報が更新されます。<br>
	 */
	protected void setLastAccessedTime() {
		// validateSession();
		long last = System.currentTimeMillis();
		session.setAttribute(attributePrefix + KEY_LASTACCESSED_TIME, new Long(last));
	}
	
	/**
	 * この仮想セッションの最終取得時刻を、GMT 1970年 1 月 1 日 0 時からのミリ秒単位で返します。<br>
	 * 現在の実装では HTTPリクエスト毎にコンストラクタする想定で、コンストラクタでのみ情報が更新されます。<br>
	 * <br>
	 * @return GMT 1970年 1 月 1 日 からのミリ秒単位で表した、このセッションの最終取得時刻を示す long
	 * @throws IllegalStateException セッションが無効である場合にスローされます。
	 */
	public long getLastAccessedTime() {
		return ((Long)getAttribute(KEY_LASTACCESSED_TIME)).longValue();
	}
	
	/**
	 * この仮想セッションにオブジェクトを保存、取得するときの HttpSession キーを生成します。
	 * @param virtualKey オブジェクトの名前を指定する文字列
	 * @return HttpSession 内でのオブジェクトを示す文字列
	 */
	protected String getRealKey(String virtualKey) {
		String realKey = attributePrefix + virtualKey;
		return realKey;
	}
	
	/**
	 * この仮想セッション内の全てのオブジェクトを toString() によって文字列化して返します。
	 * @return 仮想セッション内の全てのオブジェクトのテキスト文字列
	 */
	public String toString() {
		try {
			validateSession();
		}
		catch(IllegalStateException e) {
			return "invalid session: jsession=" + session.getId() + ", vsession=" + sessionId;
		}

		StringBuffer sb = new StringBuffer();
		sb.append("session dump: jsession=" + session.getId() + ", vsession=" + sessionId + "\n\t{\n");
		Iterator<String> it = getAttributeNames();

		ArrayList<String> ar = new ArrayList<String>(); 
		while (it.hasNext()) {
			ar.add(it.next());
		}
		Collections.sort(ar);

		for (String key: ar) {
			Object value = getAttribute(key);
			if (value == null) {
				sb.append("\t\t" + key + " is null\n");
			} else {
				sb.append("\t\t" + key + "=[" + value.toString() + "]\n");
			}
		}
		sb.append("\t}");
		return sb.toString();
	}

	/**
	 * この仮想セッションが有効かを検査し、無効であれば IllegalStateException をスローします。<br>
	 * <br>
	 * 有効期限は検査しないことに注意。<br>
	 * @throws IllegalStateException セッションが無効である場合にスローされます。
	 */
	private void validateSession() {
		// CREATION_TIME の存在が仮想セッションの有効を示す
		if (session.getAttribute(attributePrefix + KEY_CREATION_TIME) == null) {
			throw new IllegalStateException("invalid session: jsession=" + session.getId() + ", vsession=" + sessionId);
		}
	}

}
