/*
 * Copyright 2009 Project CodeCluster
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *     http://www.apache.org/licenses/LICENSE-2.0
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KI ND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.codecluster.struts12;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import java.lang.reflect.Method;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.apache.struts.action.ActionForm;
import org.apache.struts.action.ActionForward;
import org.apache.struts.action.ActionMapping;
import org.apache.struts.actions.DispatchAction;

import org.codecluster.struts12.annotations.Action;

/**
 * Struts 1.2 の DispatchAction を拡張した Action クラスです。<br>
 * <br>
 * アノテーションで実行メソッドを指定することができるように拡張したクラスです。<br>
 * 
 */
public class C2DispatchAction extends DispatchAction {
	private static final Logger logger = Logger.getLogger(C2DispatchAction.class.getName());
	
	private static String DEFAULT_METHOD_NAME = "defaultAction";
	private static String DEFAULT_MAPPING_NAME = "init";
	
	private static ThreadLocal<HttpServletRequest> threadRequest = new ThreadLocal<HttpServletRequest>();
	private static ThreadLocal<HttpServletResponse> threadResponse = new ThreadLocal<HttpServletResponse>();

	/**
	 * 本スレッドで実行中の HttpServletRequest を取得する
	 * @return HttpServletRequest
	 */
	public static HttpServletRequest getRequest() {
		return threadRequest.get();
	}
	/**
	 * 本スレッドで実行中の HttpServletResponse を取得する
	 * @return HttpServletResponse
	 */
	public static HttpServletResponse getResponse() {
		return threadResponse.get();
	}

	/* (non-Javadoc)
	 * @see org.apache.struts.actions.DispatchAction#dispatchMethod(org.apache.struts.action.ActionMapping, org.apache.struts.action.ActionForm, javax.servlet.http.HttpServletRequest, javax.servlet.http.HttpServletResponse, java.lang.String)
	 */
	@Override
	protected ActionForward dispatchMethod(ActionMapping mapping, ActionForm form,
			HttpServletRequest request, HttpServletResponse response,
			String name) throws Exception {

		ActionForward forward = null;
		ActionForward ret = null;

		// スレッドローカルエリアにセッション、リクエスト、レスポンスを保存
		threadRequest.set(request);
		threadResponse.set(response);

		try {
			// execute メソッドの前処理
			if ((ret = preDispatch(mapping, form, request, response, name)) != null) {
				return ret;
			}
			
			// execute の実行
			if ((ret = super.dispatchMethod(mapping, form, request, response, name)) != null) {
				forward = ret;
			}
		
			// execute メソッドの後処理
			if ((ret = postDispatch(mapping, form, request, response, name)) != null) {
				forward = ret;
			}
		}
		finally {
			threadRequest.remove();
			threadResponse.remove();
		}

		return forward;
	}

	/**
     * Action アノテーションをチェックして実行すべきメソッド名を返します。<br>
     * <br>
     * Action アノテーションで指定されたパラメータ名がリクエスト内にある場合、
     * 該当メソッドを実行対象としますので、同一フォーム内の複数 submit ボタンの振り分けに利用できます。<br>
     * <br>
     * アノテーションとリクエスト内のパラメータが複数組マッチする場合の実行メソッドは未定義です。
     * そのようなフォームは作らないでください。<br>
     * <br>
     * 実行すべきメソッドが見つからない場合には defaultAction() が実行され、
     * デフォルトの実装は、"init" にフォワードするよう記述されています。<br>
     * 
	 * @see org.apache.struts.actions.DispatchAction#getMethodName(org.apache.struts.action.ActionMapping, org.apache.struts.action.ActionForm, javax.servlet.http.HttpServletRequest, javax.servlet.http.HttpServletResponse, java.lang.String)
	 */
	@Override
    protected String getMethodName(ActionMapping mapping, ActionForm form,
            HttpServletRequest request, HttpServletResponse response,
            String parameter) throws Exception {

        Method methods[] = getClass().getMethods();
        
        for (Method method: methods) {
        	Action anoAction = method.getAnnotation(org.codecluster.struts12.annotations.Action.class);
        	if (anoAction != null) {
        		String key = anoAction.value();
        		if ((request.getParameter(key) != null)
        				|| (request.getParameter(key + ".x") != null)
        				|| (request.getParameter(key + ".index") != null)) {

        			if (logger.isLoggable(Level.FINE)) {
            			logger.fine("dispatch to method '" + method.getName() + "()' in class '"
            					+ getClass().getName() + "' or super by annotation '" + key + "'.");
        			}
        			return method.getName();
        		}
        	}
        }

		if (logger.isLoggable(Level.FINE)) {
			logger.fine("not found annotation or method, so execute method '" + DEFAULT_METHOD_NAME + "()'.");
		}

        return DEFAULT_METHOD_NAME;
    }
	
    /* (non-Javadoc)
	 * @see org.apache.struts.actions.DispatchAction#getParameter(org.apache.struts.action.ActionMapping, org.apache.struts.action.ActionForm, javax.servlet.http.HttpServletRequest, javax.servlet.http.HttpServletResponse)
	 */
	@Override
	protected String getParameter(ActionMapping mapping, ActionForm form,
			HttpServletRequest request, HttpServletResponse response)
			throws Exception {
		// parameter not used.
		return "";
	}
	
	/**
     * Action 処理の呼び出し前に実行される処理を記述します。<br>
     * <br>
     * デフォルトでは何も処理しません。オーバーライドして記述してください。<br>
     * 戻り値に null 以外の ActionForward を返すと Action を呼び出さずに ActionForward に従った処理に移ります。<br>
     * <br>
	 * @return ActionForward
	 * @throws Exception
	 * @see org.apache.struts.actions.DispatchAction#dispatchMethod(org.apache.struts.action.ActionMapping, org.apache.struts.action.ActionForm, javax.servlet.http.HttpServletRequest, javax.servlet.http.HttpServletResponse, java.lang.String)
	 */
	protected ActionForward preDispatch(ActionMapping mapping, ActionForm form,
			HttpServletRequest request, HttpServletResponse response,
			String name) throws Exception {

		// 事前共通機能の追加

		return null;
	}

	/**
     * Action 処理の呼び出し後に実行される処理を記述します。<br>
     * <br>
     * デフォルトでは何も処理しません。オーバーライドして記述してください。<br>
     * 戻り値に null 以外の ActionForward を返すと Action の結果を無視して本メソッドで指定した
     * ActionForward に従った処理に移ります。<br>
     * <br>
	 * @return ActionForward
	 * @throws Exception
	 * @see org.apache.struts.actions.DispatchAction#dispatchMethod(org.apache.struts.action.ActionMapping, org.apache.struts.action.ActionForm, javax.servlet.http.HttpServletRequest, javax.servlet.http.HttpServletResponse, java.lang.String)
	 */
	protected ActionForward postDispatch(ActionMapping mapping, ActionForm form,
			HttpServletRequest request, HttpServletResponse response, String name) throws Exception {

		// 事後共通機能の追加

		return null;
	}
	
	/**
	 * 実行すべき Action が見つからない場合のデフォルト処理です。<br>
	 * <br>
	 * mapping.findForward("init") を返却します。
	 * @return mapping.findForward("init") を返却します。
	 * @throws Exception
	 */ 
	public ActionForward defaultAction(ActionMapping mapping, ActionForm form,
			HttpServletRequest request, HttpServletResponse response) throws Exception  {

		return mapping.findForward(DEFAULT_MAPPING_NAME);
	}
}
