/*
 * Copyright 2009 Project CodeCluster
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *     http://www.apache.org/licenses/LICENSE-2.0
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KI ND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.codecluster.util;

import java.util.Date;
import java.util.HashMap;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.codecluster.C2Constants;
import org.codecluster.util.C2Properties;

/**
 * キャッシュ機能付きプロパティファイル管理クラスです。<br>
 * <br>
 * プロパティ XML ファイル中に記述されたプロパティ "cache" で指定した時間（単位秒）は
 * プロパティの再読み込みを行いません。"-1" を指定すると完全に再読み込みは行わなくなります。<br>
 * ただし、reload フラグを true で、呼び出した場合にはキャッシュ時間にかかわらず
 * 強制的に再読み込みを行い C2Properties を返します。<br>
 * 
 */
public class C2PropertiesManager implements C2Constants {
	private static final Logger logger = Logger.getLogger(C2PropertiesManager.class.getName());

	/**
	 * プロパティ格納マップ
	 */
	private static HashMap<String, C2Properties> propMap = new HashMap<String, C2Properties>();

	/**
	 * プロパティのキャッシュ期限
	 */
	private static HashMap<String, Long> expiresTimeMap = new HashMap<String, Long>();

	/**
	 */
	private C2PropertiesManager() {
	}

	/**
	 * プロパティ XML の読み込みをキャッシュ時間を考慮して行います。<br>
	 * プロパティファイルは ContextClassLoader の getResource() によって識別され、getResourceAsStream() によって読み込まれます。
	 * @param xmlfilename プロパティ XML ファイル名
	 * @param reload true でプロパティファイルを強制再読み込みします
	 * @return C2Properties
	 */
	public static C2Properties getC2PropertiesFromXML(String xmlfilename, boolean reload) {
		return loadXML(xmlfilename, reload);
	}
	
	/**
	 * プロパティ XML の読み込みをキャッシュ時間を考慮して行います。
	 * プロパティファイルは ContextClassLoader の getResource() によって識別され、getResourceAsStream() によって読み込まれます。
	 * @param xmlfilename プロパティ XML ファイル名
	 * @return C2Properties
	 */
	public static C2Properties getC2PropertiesFromXML(String xmlfilename) {
		return loadXML(xmlfilename, false);
	}

	/**
	 * プロパティ XML の読み込みをキャッシュ時間を考慮して行います。
	 * @param xmlfilename プロパティ XML ファイル名
	 * @param reload true でプロパティファイルを強制再読み込みします
	 * @return C2Properties
	 */
	private static C2Properties loadXML(String xmlfilename, boolean reload) {
		String key = Thread.currentThread().getContextClassLoader().getResource(xmlfilename).toString();

		C2Properties prop = null;
		boolean loaded = false;

		synchronized (propMap) {
			prop = propMap.get(key);

			if (reload == true) {
				prop = null;
			}

			if (prop != null) {
				long expiresTime;
				Long l = expiresTimeMap.get(key);
				if (l == null) {
					expiresTime = 0;
				} else {
					expiresTime = l.longValue();
				}

				if (expiresTime >= 0 && expiresTime < System.currentTimeMillis()) {
					prop = null;
				}
			}
			
			if (prop == null) {
				prop = new C2Properties();
				try {
					prop.loadFromXML(xmlfilename);
					loaded = true;
				}
				catch (Exception e)
				{
					// ignore error
				}

				propMap.put(key, prop);
			} else {
				prop.setCached(true);
			}
		}

		if (loaded) {
			// Cache TTL
			String cacheTTL = prop.getProperty(CONF_CACHE_SECONDS, "-1"); // default unlimited

			long ttl = Long.parseLong(cacheTTL);
			if (ttl < 0) {
				ttl = -1;
			} else {
				ttl = System.currentTimeMillis() + ttl * 1000;
			}
			expiresTimeMap.put(key, new Long(ttl));

			if (logger.isLoggable(Level.FINE)) {
				Date expire = new Date(ttl);
				logger.fine("loaded properites:"
						+ "\n file=" + key
						+ "\n expires=" + expire.toString());
			}
		}

		return prop;
	}
}
