package jp.sf.codememo;

import java.io.File;

import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.Path;
import org.eclipse.jdt.core.ICodeAssist;
import org.eclipse.jdt.core.ICompilationUnit;
import org.eclipse.jdt.core.IJavaElement;
import org.eclipse.jdt.core.IMember;
import org.eclipse.jdt.core.IPackageFragment;
import org.eclipse.jdt.core.IType;
import org.eclipse.jdt.core.JavaModelException;
import org.eclipse.jdt.ui.JavaElementLabels;
import org.eclipse.jdt.ui.JavaUI;
import org.eclipse.jface.text.ITextSelection;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.ui.IEditorPart;
import org.eclipse.ui.IWorkbenchPart;

/**
 * 
 * @author bskuroneko
 */
public class JavaElementResolver implements IElementResolver {

    private static final long MEMBER_LABEL_FLAG = JavaElementLabels.M_PARAMETER_TYPES;
    
    public Object findSelectedElement(IWorkbenchPart part, ISelection selection) {
        IJavaElement found = null;
        try {
            if (part instanceof IEditorPart && selection instanceof ITextSelection) {
                IJavaElement javaElement = JavaUI.getEditorInputJavaElement(((IEditorPart)part).getEditorInput());
                if (javaElement instanceof ICodeAssist) {
                    if (javaElement instanceof ICompilationUnit) {
                        ICompilationUnit cunit= (ICompilationUnit)javaElement;
                        if (cunit.isWorkingCopy()) {
                            cunit.reconcile(
                                    ICompilationUnit.NO_AST,
                                    false /* don't force problem detection */,
                                    null /* use primary owner */,
                                    null /* no progress monitor */);
                        }
                    }
                    IJavaElement[] elements= ((ICodeAssist)javaElement).codeSelect(((ITextSelection)selection).getOffset(), ((ITextSelection)selection).getLength());
                    if (elements != null && elements.length > 0) {
                        found = elements[0];
                    }
                }
            } else if (selection instanceof IStructuredSelection) {
                IStructuredSelection ss = (IStructuredSelection) selection;
                if (ss.size() == 1) {
                    Object element = ss.getFirstElement();
                    found = findJavaElement(element);
                }
            }
            
        } catch (JavaModelException e) {
            found = null;
        }

        if (isSupportedElement(found)) {
            return found;
        }
        return null;
    }

    private IJavaElement findJavaElement(Object element) {
        if (element == null) {
            return null;
        }

        IJavaElement je = null;
        if (element instanceof IAdaptable) {
            je = (IJavaElement)((IAdaptable)element).getAdapter(IJavaElement.class);
        }
        return je;
    }

    private boolean isSupportedElement(Object element) {
        if (!(element instanceof IJavaElement)) {
            return false;
        }
        IJavaElement je = (IJavaElement) element;
        switch (je.getElementType()) {
        case IJavaElement.PACKAGE_FRAGMENT:
            return !((IPackageFragment)je).isDefaultPackage();
        case IJavaElement.TYPE:
        case IJavaElement.METHOD:
        case IJavaElement.FIELD:
            return true;
        }
        
        return false;
    }

    public File getRegistryPath(IPath basePath, Object element) {
        if (!isSupportedElement(element)) {
            return null;
        }
        IJavaElement je = (IJavaElement) element;
        IPackageFragment packageFragment = null;
        IType type = null;
        IMember member = null;
        switch (je.getElementType()) {
        case IJavaElement.PACKAGE_FRAGMENT:
            packageFragment = (IPackageFragment)je;
            break;
        case IJavaElement.TYPE:
            type = (IType) je;
            packageFragment = type.getPackageFragment();
            break;
        case IJavaElement.METHOD:
        case IJavaElement.FIELD:
            member = (IMember)je;
            type = member.getDeclaringType();
            packageFragment = type.getPackageFragment();
            break;
        default:
            throw new IllegalArgumentException();
        }
        
        File packageDir = getPackageDirectory(basePath, packageFragment);
        String name = null;
        if (type == null) {
            name = "package";
        }
        else {
            name = type.getTypeQualifiedName('$');
        }
        
        if (member != null) {
            name += "." + JavaElementLabels.getTextLabel(member, MEMBER_LABEL_FLAG);
        }
        return new File(packageDir, name + ".txt");
    }

    private File getPackageDirectory(IPath basePath, IPackageFragment fragment) {
        String filepath = "java/" + fragment.getElementName().replace('.', '/');
        return basePath.append(filepath).toFile();
    }

    public String getLabel(Object element) {
        if (!isSupportedElement(element)) {
            return null;
        }
        File path = getRegistryPath(Path.EMPTY, element);
        String filePath = path.getPath().replace('\\', '/').replace('/', '.');
        return filePath.substring("java/".length(), filePath.length() - 4);
    }

}
