/* ---------------------------------------------------------------------
%   Copyright (C) 2007 Association for the COINS Compiler Infrastructure
%       (Read COPYING for detailed information.)
--------------------------------------------------------------------- */
package coins.ir;

import coins.sym.FlowAnalSym;
import coins.sym.Sym;

//========================================//
//                           Added on Nov. 2004.

/** IR0
<PRE>
 * Simplified Intermediate Representation interface (IR0).
 *
 * IR0 is the simplified interface of IR (Intermediate Representation).
 * Simple compiler can be built by using IR0, HIR0, and Sym0.
 * Advanced methods to make complicated compilers will be find in
 * the interface IR which extends IR0, HIR which extends HIR0, and
 * Sym which extends Sym0.
 *
 *  Intermediate representation of programs consists of
 *  following kinds of information:
 *
 *    High level intermediate representation (HIR)
 *      Abstract representation of the logical structure of
 *      source programs.
 *
 *    Low level intermediate representation (LIR)
 *      Abstract representation of computer operations corresponding
 *      to the source program.
 *
 *    Symbol table
 *      Inforamtion of symbols used in the source program and
 *      symbols generated by the compiler such as registers, temporals,
 *      internal labels.
 *
 *    Flow information
 *      Control flow, data flow information, loop information,
 *      and parallelization information.
 *
 *  These components are linked with each other so that various
 *  information can be get quickly for optimization and
 *  parallelization of the program.
 *
 *  The IR interface declares methods that are common between HIR
 *  and LIR. By using them, we can make modules that are applicable
 *  both to HIR and LIR which implement IR interface.
 *    HIR class implements HIR interface and IR interface.
 *    LIR class implements LIR interface and IR interface.
 *
 *  In HIR, operation is represented by a tree where each of its
 *  subtree represents some logical construct of source program
 *  mostly language-independent and machine-independent mannar.
 *
 *  In LIR, operation is represented by a tree each node of which
 *  either represents abstract machine instruction or control
 *  information. The abstract machine instructions may represent
 *  information specific to a target machine but they take common
 *  format so that LIR processing routines can be made common to
 *  various target machines.
 *
 *  Each subtree has a root node to which other nodes are connected
 *  directly or indirectly. A node having one or more children is
 *  called nonleaf. A node having no child is called leaf.
 *  Each subtree is represented by its root node.
 *  A child of the root node (node connected directly to the root
 *  node) in a subtree is a source operand of the root node and the
 *  root node itself represents the result of the expression
 *  represented by the subtree.
 *  A subtree can be changed to a sequence of nodes by traversing it
 *  in depth-first order from left to right.
 *
 *  An LIR subtree representing a machine operation is sometimes
 *  called an instruction.  Operands of the LIR instruction may be
 *  an abstract register or a symbol (such as variable or constant
 *  or label) or other instruction subtree.
 *
 *  Each node of HIR or LIR has index number to identify it if it
 *  represents an executable operation. The index number may be
 *  used in procedures analyzing or transforming the program
 *  represented either in HIR or LIR for optimization and
 *  parallelization.
 *  The analysis/transformation procedures can be coded to be
 *  applicable both to HIR and LIR by using methods described
 *  in this interface.
 *
 *  Each node of HIR is attached with HIR type that represents the
 *  type of expression represented by the node. The HIR type may be
 *  base type defined in this compiler system, or a type
 *  defined in the input program or class introduced by input
 *  language or defined by the input program. If the node is such
 *  one as having no value (neither r-value nore l-value), then null
 *  is attached as its data type.
 *  Each node of LIR is attached with basic type such as
 *  I8, I32, F32, F64, etc. representing the type of the value or
 *  address represented by the node. Aggregate data type or class
 *  are decomposed into basic types when HIR is transformed into LIR
 *  by decomposing aggregates into its elements.
 *
 *  Creation of HIR tree can be done by using HIR factory methods
 *  (as shown in HIR.java).
 *  Creation of LIR instruction sequence can be done by using
 *  LIR factory methods.
</PRE>
**/
public interface
IR0 extends Cloneable
{

//==== Methods to set/get informatin to/from IR node ====//

/** getIndex
 *  Get the index number assigned to "this" node.
 *  (The index number is used to refer the node in data flow analysis
 *   and debug-print.)
 *  @return the index number assigned to "this" node.
 *      If the node has no index number, return 0.
**/
int
getIndex();

/** getOperator
 *  Get operation code of "this" node.
 *  @return the operation code (value of operator field) of
 *      "this" node.
**/
int
getOperator();

//---- Methods to link child to IR node ----//

/** getChildCount
 *  Get the number of children that may be attached to this node.
 *  Abbreviated child (null child) is also counted.
 *  @return the number of children of this node.
**/
public int
getChildCount();

/** getChild
 *  @param pNumber the child number of the child to be returned.
 *  @return the specified child of this node.
**/
public IR
getChild( int pNumber );

/** setChild
 *  Set pHir as pNumber-th child of this node.
 *  @param pNumber Child number
 *  @param pIR Child node to be connected.
**/
public void
setChild( int pNumber, IR pIr );

} // IR0 interface

