/**
 * Celcel を作成する。
 * @param table Celcel化対象テーブル。もしくはCelcel化対象テーブルのコンテナ。
 * @param fixRows 固定表示行数。
 * @param fixColumns 固定表示列数。
 */
function Celcel(table, fixRows, fixColumns) {

	//// ローカル変数定義
	var handler = this;
	var container;
	var htmlElement;

	var header;
	var rowHeader;
	var columnHeader;
	var detail;

	var headerDiv;
	var rowHeaderDiv;
	var columnHeaderDiv;
	var detailDiv;

	var detail;

	var menu;

	//// プロパティ定義

	/**
	 * HTML実体要素オブジェクト。
	 */
	this.htmlElement = null;

	/**
	 * 選択モード。
	 */
	this.selectionMode = "cell";

	/**
	 * 自動選択
	 */
	this.autoSelection = false;

	/**
	 * 選択状態を格納するための hidden タグ。
	 */
	this.selectionHolder = null;

	/**
	 * 行・列選択時に呼び出されるリクエスト。
	 */
	this.requestOnselected = null;

	/**
	 * クリックに呼び出されるリクエスト。
	 */
	this.requestOnclicked = null;


	// メソッド定義

	/**
	 * 選択された範囲の内容をクリアする。
	 */
	this.clear = function() {
		clear0(header);
		clear0(rowHeader);
		clear0(columnHeader);
		clear0(detail);

		function clear0(table) {
			if(!table.selection) {
				return;
			}

			for(var i=0; i<table.selection.length; i++) {
				if(!table.selection[i].readOnly) {
					table.selection[i].value = "";
				}
			}
		}
	}

	/**
	 * 選択された範囲の内容をクリップボードにコピーする。
	 */
	this.copy = function() {
		if(!window.clipboardData) {
			return;
		}

		var arr = new Array();
		var index = 0;

		index = createText(header, columnHeader, arr, index);
		createText(rowHeader, detail, arr, index);

		window.clipboardData.setData("Text", arr.join(""));


		function createText(leftTable, rightTable, arr, index) {
			var maxRowIndex = -1;
			var minRowIndex = Number.POSITIVE_INFINITY;

			if(leftTable.selection) {
				maxRowIndex = Math.max(leftTable.selectionMaxRowIndex, maxRowIndex);
				minRowIndex = Math.min(leftTable.selectionMinRowIndex, minRowIndex);
			}
			if(rightTable.selection) {
				maxRowIndex = Math.max(rightTable.selectionMaxRowIndex, maxRowIndex);
				minRowIndex = Math.min(rightTable.selectionMinRowIndex, minRowIndex);
			}

			if(maxRowIndex==-1 || minRowIndex==Number.POSITIVE_INFINITY) {
				return index;
			}

			for(var i=minRowIndex; i<=maxRowIndex; i++) {
				if(leftTable.selection) {
					index = createTextByInput(leftTable, i, arr, index);
				}
				if(rightTable.selection) {
					index = createTextByInput(rightTable, i, arr, index);
				}
				index--;
				arr[index++] = "\r\n";
			}

			function createTextByInput(table, i, arr, index) {
				var minColumnIndex = table.selectionMinColumnIndex;
				var maxColumnIndex = table.selectionMaxColumnIndex;

				for(var j=minColumnIndex; j<=maxColumnIndex; j++) {
					input = getInput(table.rows[i].cells[j]);
					var value = "";
					if(input) {
						value = input.value;
					}
					arr[index++] = value;
					arr[index++] = "\t";
				}

				return index;
			}

			return index;
		}
	}

	/**
	 * テーブル全体の内容をクリップボードにコピーする。
	 */
	this.copyAll = function() {
		if(!window.clipboardData) {
			return;
		}
		var arr = new Array();
		var index = 0;

		index = copyAllTable(header, columnHeader, arr, index);
		copyAllTable(rowHeader, detail, arr, index);

		window.clipboardData.setData("Text", arr.join(""));

		function copyAllTable(leftTable, rightTable, arr, index) {
			var leftTableRowLength = leftTable.rows.length;
			var rightTableRowLength = rightTable.rows.length;
			var rowLength = Math.max(leftTableRowLength, rightTableRowLength);
			for(var i=0; i < rowLength; i++) {
				if(leftTableRowLength > i) { 
					index = createTextByColumn(leftTable, i, arr, index);
				}
				if(rightTableRowLength > i ) {
					index = createTextByColumn(rightTable, i, arr, index);
				}
				index--;
				arr[index++] = "\r\n";
			}
			return index;
		}

		function createTextByColumn(table, i, arr, index) {
			for(var j = 0; j < table.rows[i].cells.length; j++) {
				var cell = table.rows[i].cells[j];
				input = getInput(cell);
				var value = "";
				if(input) {
					value = input.value;
				}
				else {
					value = cell.innerText;
				}
				arr[index++] = value;
				arr[index++] = "\t";
			}
			return index;
		}
	}

	/**
	 * 選択された範囲の内容の切り取り（クリップボードへのコピーとクリア）を行なう。
	 */
	this.cut = function() {
		handler.copy();
		handler.clear();
	}

	/**
	 * 現在のクリップボードの内容を貼り付ける。
	 */
	this.paste = function() {
		if(!window.clipboardData) {
			return;
		}

		var text = window.clipboardData.getData("Text");
		if(!text || text.length==0) {
			return;
		}

		var table = handler.selectionStartingTable;
		var rowIndex = table.selectionMinRowIndex;

		var rowText = text.split("\r\n");
		for(var i = 0; i<rowText.length; i++) {
			if(rowText[i].length==0) {
				break;
			}
			if(i+rowIndex >= table.rows.length) {
				if(table.lowerTable) {
					table = table.lowerTable;
					rowIndex = 0 - i;
				}
				else {
					break;
				}
			}
			var row = table.rows[i+rowIndex];
			var columnIndex = handler.selectionStartingTable.selectionMinColumnIndex;

			var columnText = rowText[i].split("\t");
			for(var j=0; j<columnText.length; j++) {
				if(j+columnIndex >= row.cells.length) {
					if(table.rightTable) {
						row = table.rightTable.rows[i+rowIndex];
						columnIndex = 0 - j;
					}
					else {
						break;
					}
				}

				var cell = row.cells[j+columnIndex];
				var input = getInput(cell);
				if(input && !input.readOnly) {
					input.value = columnText[j];
				}
			}
		}
	}

	/**
	 * 選択範囲が存在するかどうかを調べる。
	 * @return 選択範囲が存在する場合は true。
	 */
	this.hasSelection = function() {
		return header.selection || rowHeader.selection || 
				columnHeader.selection || detail.selection;
	}

	/**
	 * 表の幅を設定する。
	 * @param width 表の幅。
	 */
	this.setWidth = function(width) {
		width = parseInt(width);
		width = width - rowHeaderDiv.scrollWidth - getScrollMargin(detailDiv);
		if(width <= 0) {
			return;
		}
		columnHeaderDiv.style.width = width;
		detailDiv.style.width = width + getScrollMargin(detailDiv);
		htmlElement.style.width = rowHeaderDiv.scrollWidth + parseInt(detailDiv.style.width);
	}

	/**
	 * 表の高さを設定する。
	 * @param height 表の高さ。
	 */
	this.setHeight = function(height) {
		height = parseInt(height);
		height = height - columnHeaderDiv.scrollHeight - getScrollMargin(detailDiv);
		if(height <= 0) {
			return;
		}
		rowHeaderDiv.style.height = height;
		detailDiv.style.height = height + getScrollMargin(detailDiv);
		htmlElement.style.height = columnHeaderDiv.scrollHeight + parseInt(detailDiv.style.height) ;
	}

	this.setClientWidth = function(width) {
		handler.setWidth(width);
		return true;
	}

	this.setClientHeight = function(height) {
		handler.setHeight(height);
		return true;
	}

	this.pack = function() {
		try {
			handler.setWidth(htmlElement.parentNode.clientWidth - IWidget.getMarginWidth(htmlElement));
			handler.setHeight(htmlElement.offsetHeight - IWidget.getMarginHeight(htmlElement));
		}
		catch(e) {
		}
	}

	/**
	 * 表のサイズを設定する。
	 * @param width 表の幅。
	 * @param height 表の高さ。
	 */
	this.setSize = function(width, height) {
		handler.setWidth(width);
		handler.setHeight(height);
	}

	/**
	 * スクロールバー表示をするかしないかを設定する。
	 * @param scroll スクロールバー表示をする場合は true を指定する。
	 */
	this.setScroll = function(scroll) {
		if(scroll) {
			if(detailDiv.style.overflow!="scroll") {
				detailDiv.style.overflow = "scroll";
				htmlElement.parentNode.onresize = null;
			}
			else {
				return;
			}
		}
		else {
			detailDiv.style.overflow = "hidden";
			detailDiv.style.width = parseInt(detailDiv.style.width) - Celcel.prototype.SCROLL_MARGIN
			detailDiv.style.height = parseInt(detailDiv.style.height) - Celcel.prototype.SCROLL_MARGIN
		}
		handler.setSize(parseInt(detailDiv.style.width), parseInt(detailDiv.style.height));
	}

	/**
	 * コンテクストメニューへ項目追加を行なう。
	 * @param text 項目名称。
	 * @param func メニュークリック時に実行するファンクション。
	 */
	this.addMenu = function(text, func) {
		menu.addItem(text, func);
	}

	/**
	 * コンテクストメニューを表示する。
	 * @param event メニュー表示の起因となったイベント。
	 */
	this.showMenu = function(event) {
		menu.setEnabled("copyAll", true);
		var hasSelection = handler.hasSelection();
		menu.setEnabled("copy", hasSelection);
		menu.setEnabled("cut", hasSelection);
		menu.setEnabled("paste", hasSelection);
		menu.setEnabled("clear", hasSelection);
		menu.show(event);
	}


	/**
	 * セルの input 部分でキー入力したときに呼び出すための
	 * イベントハンドラ。
	 */
	Celcel.input_onkeydown = function(event) {

		event = IWidget.editEvent(event);
		var input = event.target;
		var table = getTable(input);

		//IDebug.print("keyCode:" + event.keyCode);

		if(event.keyCode==16 || // shift
				event.keyCode==17 || event.keyCode==18 ) { // ctrl, alt	
			event.returnValue = true;
			return event.returnValue;
		}


		if(event.keyCode==46) { // delete
			if(table.selection) {
				table.handler.clear();
				event.returnValue = false;
				table.isCellEditMode = false;
				return event.returnValue;
			}
		}

		if(!table.isCellEditMode) {
			if(event.ctrlKey && event.keyCode==86) { // Ctrl+V
				table.handler.paste();
				event.returnValue = false;
				return event.returnValue;
			}
			else if(event.ctrlKey && event.keyCode==67) { // Ctrl+C
				table.handler.copy();
				event.returnValue = false;
				return event.returnValue;
			}
			else if(event.ctrlKey && event.keyCode==88) { // Ctrl+X
				table.handler.cut();
				event.returnValue = false;
				return event.returnValue;
			}
			else if(event.keyCode==8 && input && input.readOnly) { // BS
				event.returnValue = false;
				return event.returnValue;
			}
		}

		var srcInput = input;
		if(!event.shiftKey || event.keyCode==9) {
			clearAllSelections(table);
		}

		if(!table.isCellEditMode) {
			if(event.keyCode==37) { // left
				var item = getPrevInput(table, input, !event.shiftKey);
				if(item) {
					input = item;
				}
				setFocus(input);
				updateSelection(event, getTable(input), input);
				event.returnValue = false;
			}
			else if(event.keyCode==39) { // right
				var item = getNextInput(table, input, !event.shiftKey);
				if(item) {
					input = item;
				}
				setFocus(input);
				updateSelection(event, getTable(input), input);
				event.returnValue = false;
			}
			else {
				event.returnValue = true;
				if(event.keyCode==113) { // F2
					if(input.setSelectionRange) { // for Moz
						input.setSelectionRange(input.value.length, input.value.length);
					}
					else { // for IE
						event.keyCode = 34;
					}
				}
			}
		}

		if(event.keyCode==27) { // ESC
			input.select();
		}
		else if(event.keyCode==40 || event.keyCode==13) { // down or enter
			input = getLowerInput(table, input);
			setFocus(input);
			updateSelection(event, getTable(input), input);
			event.returnValue = false;
		}
		else if(event.keyCode==38) { // up
			input = getUpperInput(table, input);
			setFocus(input);
			updateSelection(event, getTable(input), input);
			event.returnValue = false;
		}

		if(event.keyCode!=9 && srcInput==input) {
			updateSelection(event, getTable(input), input);
		}

		// editMode を設定する。
		if(
			(event.keyCode==37 && !table.isCellEditMode) || // LEFT
			(event.keyCode==39  && !table.isCellEditMode) || // RIGHT
				event.keyCode==9 || event.keyCode==27 || // TAB, ESC
				event.keyCode==40 || event.keyCode==13 || // DOWN, ENTER
				event.keyCode==38) { // UP
			table.isCellEditMode = false; // editMode を false にする。
		}
		else { // それ以外のキーは editMode を true にする。
			if(input && !input.readOnly) {
				table.isCellEditMode = true;
			}
			else {
				table.isCellEditMode = false;
			}
		}

		return event.returnValue;

		/**
		 * より上に位置する input を取得する。
		 */
		function getUpperInput(table, input) {
			var cell = getCell(input);
			var rowIndex = cell.parentNode.rowIndex - 1;
			var prevInput = getLastInputInColumn(table, cell.cellIndex, rowIndex);
			if(prevInput) {
				return prevInput;
			}

			if(table.upperTable) {
				prevInput = getLastInputInColumn(
						table.upperTable, cell.cellIndex, table.upperTable.rows.length-1);
				if(prevInput) {
					return prevInput;
				}
			}

			return input;
		}

		/**
		 * より下に位置する input を取得する。
		 */
		function getLowerInput(table, input) {
			var cell = getCell(input);
			var rowIndex = cell.parentNode.rowIndex + 1;
			var nextInput = getFirstInputInColumn(table, cell.cellIndex, rowIndex);

			if(nextInput) {
				return nextInput;
			}

			if(table.lowerTable) {
				nextInput = getFirstInputInColumn(table.lowerTable, cell.cellIndex, 0);
				if(nextInput) {
					return nextInput;
				}
			}

			return input;
		}

		/**
		 * より前に位置する input を取得する。
		 */
		function getPrevInput(table, input, movePrevRow) {
			var cell = getCell(input);
			var prevInput = getLastInputInRow(cell.parentNode, cell.cellIndex - 1);

			if(prevInput) {
				return prevInput;
			}

			if(isAvailableTable(table.leftTable)) {
				var leftRow = table.leftTable.rows[cell.parentNode.rowIndex];
				prevInput = getLastInputInRow(leftRow, leftRow.cells.length - 1);
				if(prevInput) {
					return prevInput;
				}
			}

			if(!movePrevRow) {
				return null;
			}
			var row = getPrevRow(input);
			if(row && isAvailableTable(table.rightTable)) {
				table = table.rightTable;
				row = table.rows[row.rowIndex];
			}
			else if(!row && isAvailableTable(table.crossTable)) {
				table = table.crossTable;
				row = table.rows[table.rows.length-1];
			}

			if(!row) {
				return null;
			}
			input = getLastInputInRow(row, row.cells.length - 1);
			if(!input && table && isAvailableTable(table.leftTable)) {
				input = getLastInputInRow(table.leftTable.rows[row.rowIndex], 
											table.leftTable.rows[row.rowIndex].cells.length - 1);
			}
			return input;


			function getPrevRow(input) {
				var row = getRow(input).previousSibling;
				return getPrevNode(row); // for Moz

				function getPrevNode(node) {
					if(node && node.nodeType && node.nodeType!=1) {
						node = node.previousSibling;
					}
					return node;
				}
			}
		}




		/**
		 * より後に位置する input を取得する。
		 */
		function getNextInput(table, input, moveNextRow) {
			var cell = getCell(input);
			var nextInput = getFirstInputInRow(cell.parentNode, cell.cellIndex + 1);

			if(nextInput) {
				return nextInput;
			}

			if(isAvailableTable(table.rightTable)) {
				nextInput = getFirstInputInRow(
								table.rightTable.rows[cell.parentNode.rowIndex], 0);
				if(nextInput) {
					return nextInput;
				}
			}

			if(!moveNextRow) {
				return null;
			}

			var row = getNextRow(input);
			if(row && isAvailableTable(table.leftTable)) {
				table = table.leftTable;
				row = table.rows[row.rowIndex];
			}
			else if(!row && isAvailableTable(table.crossTable)) {
				table = table.crossTable;
				row = table.rows[0];
			}

			if(!row) {
				return null;
			}

			input = getFirstInputInRow(row, 0);
			if(!input && table && isAvailableTable(table.rightTable)) {
				input = getFirstInputInRow(table.rightTable.rows[row.rowIndex], 0);
			}
			return input;


			function getNextRow(input) {
				var row = getRow(input).nextSibling;
				return getNextNode(row); // for Moz

				function getNextNode(node) {
					if(node && node.nodeType && node.nodeType!=1) {
						node = node.nextSibling;
					}
					return node;
				}
			}
		}

	}


	/**
	 * セルの input 部分にカーソルがセットされたときに呼び出すための
	 * イベントハンドラ。
	 */
	Celcel.input_onfocus = function(event) {
		event = IWidget.editEvent(event);
		var input = event.target;

		input.select();

		if(input.readOnly && input.createTextRange) {
			var range = input.createTextRange();
			if(range) {
				range.move("character", 0);
				range.select();
			}
		}
	}

	/**
	 * セルの input 部分がクリックされたときに呼び出すための
	 * イベントハンドラ。
	 */
	Celcel.input_onclick = function(event) {
		var handler = selectCell(event);
//		if(handler.onclick) {
//			handler.onclick(event);
//		}
		event = IWidget.editEvent(event);
		var input = event.target;
		click(input);
	}

	/**
	 * セルの input 部分または td, th 部分上にマウスが載った場合に
	 * 呼び出すためのイベントハンドラ。
	 */
	Celcel.cell_onmouseover = function(event) {
		event = IWidget.editEvent(event);
		var table = getTable(event.target);
		if(table.handler.autoSelection) {
			selectCell(event);
		}
	}

	/**
	 * ヘッダがクリックされたときに呼び出すためのイベントハンドラ。
	 */
	Celcel.header_onclick = function(event) {
		event = IWidget.editEvent(event);
		var table = getTable(event.target);

		if(!event.ctrlKey && !event.shiftKey) {
			clearAllSelections(table);
		}

		selectAll(table);
		selectAll(table.rightTable);
		selectAll(table.lowerTable);
		selectAll(table.crossTable);

		function selectAll(table) {
			if(updateCellSelection(getFirstInput(table))) {
				updateCellSelection(getLastInput(table));
			}
		}

		function getFirstInput(table) {
			if(table.firstInput) {
				return table.firstInput;
			}
			for(var i=0; i < table.rows.length; i++) {
				var input = getFirstInputInRow(table.rows[i], 0);
				if(input) {
					table.firstInput = input;
					return input;
				}
			}
			return null;
		}

		function getLastInput(table) {
			if(table.lastInput) {
				return table.lastInput;
			}
			for(var i=table.rows.length-1; i >= 0; i--) {
				var input = getLastInputInRow(table.rows[i], table.rows[i].cells.length-1);
				if(input) {
					table.lastInput = input;
					return input;
				}
			}
			return null;
		}

	}

	/**
	 * 行ヘッダ部分がクリックされたときに呼び出すためのイベントハンドラ。
	 */
	Celcel.rowHeader_onclick = function(event) {
		event = IWidget.editEvent(event);
		var input = event.target;
		var table = getTable(event.target);

		if(table.handler.selectionMode=="column") {
			return;
		}

		if(!event.ctrlKey && !event.shiftKey) {
			clearAllSelections(table);
		}

		updateRowSelection(event, table, input);

		click(input);
	}


	/**
	 * 列ヘッダ部分がクリックされたときに呼び出すためのイベントハンドラ。
	 */
	Celcel.columnHeader_onclick = function(event) {
		event = IWidget.editEvent(event);
		var input = event.target;
		var table = getTable(input);

		if(table.handler.selectionMode=="row") {
			return;
		}

		if(!event.ctrlKey && !event.shiftKey) {
			clearAllSelections(table);
		}

		updateColumnSelection(event, table, input);

		click(input);
	}

	function click(cell) {
		if(handler.onclicked) {
			handler.onclicked(cell);
		}
		if(handler.requestOnclicked) {
			new CongaRequest().send(handler.requestOnclicked, 
					handler.selectionHolder.name + "=" + handler.selectionHolder.value);
		}
	}


	//// イベント定義

	/**
	 * 行・列選択時に呼び出されるイベントハンドラ。
	 */
	this.onselected = null;

	/**
	 * クリックに呼び出されるイベントハンドラ。
	 */
	this.onclicked = null;


	//// ローカルメソッド定義

	/**
	 * Celcel を初期化する。
	 * @param table Celcel化対象テーブル。もしくは明細テーブル。
	 * @param fixRows 固定表示行数。
	 * @param fixColumns 固定表示列数。
	 */
	function init(table, fixRows, fixColumns) {

		if(table.tagName=="TABLE") {
			container = table.parentNode;
			htmlElement = createHtmlElement(handler);
			container.appendChild(htmlElement);
		}
		else {
			container = table;
			htmlElement = createHtmlElement(handler, container);
			var tables = htmlElement.getElementsByTagName("table");
			if(tables.length==1) {
				table = tables.item(0);
			}
			else {
				for(var i=0; i<tables.length; i++) {
					var t = tables.item(i);
					if(t.id.match("-columnHeader$")) {
						columnHeader = t;
					}
					else if(t.id.match("-rowHeader$")) {
						rowHeader = t;
					}
					else if(t.id.match("-header$")) {
						header = t;
					}
					else {
						table = t;
					}
				}
			}
		}
		handler.htmlElement = htmlElement;

		headerDiv = createDiv(handler, htmlElement, header, "hidden");
		rowHeaderDiv = createDiv(handler, htmlElement, rowHeader, "hidden");
		columnHeaderDiv = createDiv(handler, htmlElement, columnHeader, "hidden");
		detailDiv = createDiv(handler, htmlElement, table, "scroll");

		header = headerDiv.firstChild;
		rowHeader = rowHeaderDiv.firstChild;
		columnHeader  = columnHeaderDiv.firstChild;
		detail = detailDiv.firstChild;

		menu = createContextMenu(htmlElement);

		if(!Celcel.prototype.SCROLL_MARGIN) {
			Celcel.prototype.SCROLL_MARGIN = 16;
		}

		if(!Celcel.prototype.BORDER_WIDTH) {
			Celcel.prototype.BORDER_WIDTH = 4;
		}

		if(!fixRows) {
			fixRows = 0;
		}
		if(!fixColumns) {
			fixColumns = 0;
		}

		fixRows = parseInt(fixRows);
		fixColumns = parseInt(fixColumns);

		// table 分割
		var divide = false;
		if((fixRows > 0 && columnHeader.getElementsByTagName("col").length==0) || 
				(fixColumns > 0 && rowHeader.getElementsByTagName("col").length==0)) {
			divide = true;
		}

		// table 分割
		if(divide) {
			var colList = table.getElementsByTagName("col");
			addColGroup(header, colList, 0, fixColumns);
			addColGroup(rowHeader, colList, 0, fixColumns);
			addColGroup(columnHeader, colList, fixColumns, colList.length);
			removeColGroup(colList, fixColumns);
			divideTable(fixRows, fixColumns, header, rowHeader, columnHeader, detail);
		}
		else if(!hasEventHandler()) {
			initEvents(header, Celcel.header_onclick);
			initEvents(rowHeader, Celcel.rowHeader_onclick);
			initEvents(columnHeader, Celcel.columnHeader_onclick);
			initEvents(detail, null);
		}


		// ヘッダ設定
	//	initColumBorder(header, rowHeader, columnHeaderDiv);
	//	initColumBorder(rowHeader, header, detailDiv);
		initColumBorder(columnHeader, detail, null);
		initColumBorder(detail, columnHeader, null);

		// 位置設定
		headerDiv.style.position = "absolute";
		rowHeaderDiv.style.position = "absolute";
		columnHeaderDiv.style.position = "absolute";
		detailDiv.style.position = "absolute";

		headerDiv.style.top = 0;
		headerDiv.style.left = 0;

		columnHeaderDiv.style.top = 0;

		rowHeaderDiv.style.top = headerDiv.scrollHeight;
		detailDiv.style.top = columnHeaderDiv.scrollHeight;

		columnHeaderDiv.style.left = headerDiv.scrollWidth;
		detailDiv.style.left = rowHeaderDiv.scrollWidth;

		detailDiv.onscroll = function() {
			columnHeaderDiv.scrollLeft = detailDiv.scrollLeft;
			rowHeaderDiv.scrollTop = detailDiv.scrollTop;
		}
		rowHeaderDiv.onscroll = function() {
			detailDiv.scrollTop = rowHeaderDiv.scrollTop;
		}
		columnHeaderDiv.onscroll = function() {
			detailDiv.scrollLeft = columnHeaderDiv.scrollLeft;
		}

		// 相互参照作成
		header.rightTable = columnHeader;
		header.lowerTable = rowHeader;
		header.crossTable = detail;

		rowHeader.upperTable = header;
		rowHeader.rightTable = detail;
		rowHeader.crossTable = columnHeader;

		columnHeader.leftTable = header;
		columnHeader.lowerTable = detail;
		columnHeader.crossTable = rowHeader;

		detail.leftTable = rowHeader;
		detail.upperTable = columnHeader;
		detail.crossTable = header;

		header.menu = menu;
		rowHeader.menu = menu;
		columnHeader.menu = menu;
		detail.menu = menu;

		handler.setSize(detailDiv.scrollWidth + 
						rowHeaderDiv.scrollWidth + getScrollMargin(detailDiv) + 3, 
						detailDiv.scrollHeight + 
						columnHeaderDiv.scrollHeight + getScrollMargin(detailDiv) + 4);

		handler.selectionHolder = IWidget.getValueHolder(htmlElement, "table");

		detailDiv.style.overflow = "auto";
		htmlElement.style.width = "100%";
		htmlElement.parentNode.onresize = function(){
			if(handler.packed) {
				handler.packed = false;
				return;
			}
			handler.pack();
			handler.packed = true;
		}

		/**
		 * colgroup を追加する。
		 */
		function addColGroup(table, colList, startCol, colCount) {
			var colgroup = document.createElement("colgroup");
			table.appendChild(colgroup);

			for(var i=startCol; i < colCount; i++) {
				var col = document.createElement("col");
				col.style.width = colList.item(i).style.width;
				colgroup.appendChild(col);
			}
		}

		/**
		 * colgroup を削除する。
		 */
		function removeColGroup(colList, startCol) {
			for(var i=startCol-1; i >= 0; i--) {
				var parent = colList.item(i).parentNode;
				parent.removeChild(colList.item(i));
			}
		}

		/**
		 * テーブル分割を行なう。
		 */
		function divideTable(fixRows, fixColumns, header, rowHeader, columnHeader, table) {

			var rowLength = table.rows.length;
			var rowIndex = 0;
			for(var i=0; i < rowLength; i++) {
				var headerTr = null;
				var columnHeaderTr = null;
				var rowHeaderTr = null;

				var row = table.rows[rowIndex];
				var rowHeight = row.scrollHeight;

				if(i < fixRows) {
					headerTr = createTr(header, rowHeight);
					columnHeaderTr = createTr(columnHeader, rowHeight);
				}
				else {
					if(fixColumns > 0) {
						rowHeaderTr = createTr(rowHeader, rowHeight);
					}
					rowIndex++;
				}

				var colLength = row.cells.length;
				var columnInex = 0;
				for(var j=0; j < colLength; j++) {
					var cell = row.cells[columnInex];
					if(j < fixColumns) {
						if(headerTr) {
							moveChild(row, headerTr, cell);
							
							if(!getInput(cell)) {
								if(i==0) {
									if(j==0) {
										cell.onclick = Celcel.header_onclick;
									}
									else {
										cell.onclick = Celcel.columnHeader_onclick;
									}
								}
								else {
									if(j==0) {
										cell.onclick = Celcel.rowHeader_onclick;
									}
								}
							}
						}
						else if(rowHeaderTr) {
							moveChild(row, rowHeaderTr, cell);
							if(j==0 && !getInput(cell)) {
								cell.onclick = Celcel.rowHeader_onclick;
							}
						}
						else {
							// detail にそのまま残す。
							columnInex++;
						}
					}
					else {
						if(columnHeaderTr) {
							moveChild(row, columnHeaderTr, cell);
							if(i==0 && !getInput(cell)) {
								cell.onclick = Celcel.columnHeader_onclick;
							}
						}
						else {
							// detail にそのまま残す。
							columnInex++;
						}
					}
					setEventHandlers(cell);
				}
				if(i < fixRows) {
					table.getElementsByTagName("tbody").item(0).removeChild(row);
				}
			}

			/**
			 * セルを別テーブルに移動する。
			 */
			function moveChild(src, dst, child) {
				src.removeChild(child);
				dst.appendChild(child);
			}

			/**
			 * tr 要素を作成する。
			 */
			function createTr(table, rowHeight) {
				var tr = document.createElement("tr");
				tr.style.height = rowHeight;
				table.getElementsByTagName("tbody").item(0).appendChild(tr);
				return tr;
			}

		}

		/**
		 * 列境界線を初期化する。
		 */
		function initColumBorder(table, friend, rightDiv) {
			var HELF_BORDER_WIDTH = Celcel.prototype.BORDER_WIDTH/2;

			table.columnBorder = new Array();

			var border = table.border;
			var colRight = parseInt(border);

			var colList = table.getElementsByTagName("col");
			for(var i = 0; i < colList.length; i++) {
				var col = colList.item(i);
				col.leftPos = colRight;

				col.rightCols = new Array();
				for(var j = i+1; j< colList.length; j++) {
					col.rightCols[j-i-1] = colList.item(j);
				}

				colRight = col.leftPos + col.scrollWidth;

				var b = new IBorder(table.parentNode, "vertical", 
						table.scrollHeight, Celcel.prototype.BORDER_WIDTH);
				table.columnBorder[i] = b;
				b.col = col;
				b.index = i;
				if(rightDiv && (i + 1 == colList.length)) {
					b.rightDiv = rightDiv;
					col.rightDiv = rightDiv;
				}
				b.setPosition(colRight-HELF_BORDER_WIDTH, 0);

				col.border = b.htmlElement;
				col.colRight = colRight;

				b.onmove = function(left, top) {
					onmove0(this, left, top);
					friend.columnBorder[this.index].col.border.style.left = left;
					onmove0(friend.columnBorder[this.index], left, top);
				}
			}

			function onmove0(border, left, top) {
				left = left + Celcel.prototype.BORDER_WIDTH/2;

				var currentWidth = border.col.scrollWidth;
				var newWidth = left - border.col.leftPos;
				if(newWidth < 0) {
					border.col.border.style.left = border.col.colRight;
					return;
				}

				border.col.style.width = newWidth;

				// 列幅が縮まらなかった場合の補正
				if(border.col.scrollWidth > newWidth) {
					newWidth = border.col.scrollWidth;
					border.col.style.width = newWidth;
					left = border.col.leftPos + newWidth;
					border.col.border.style.left = left;
				}

				border.col.colRight = left;
				var diff = border.col.scrollWidth - currentWidth;
				for(var i=0; i<border.col.rightCols.length; i++) {
					border.col.rightCols[i].leftPos =
						border.col.rightCols[i].leftPos + diff;
					border.col.rightCols[i].border.style.left =
						parseInt(border.col.rightCols[i].border.style.left) + diff;
					if(border.col.rightCols[i].rightDiv) {
						border.col.rightCols[i].rightDiv.style.left = 
								border.col.rightCols[i].border.style.left;
					}
				}

				if(border.rightDiv) {
					border.rightDiv.style.left = left;
				}
			}
		}

		/**
		 * 各セルに対してイベントハンドラ設定を行なう。
		 */
		function initEvents(table, onclickHandler) {
			var rowLength = table.rows.length;
			for(var i=0; i < rowLength; i++) {
				var row = table.rows[i];

				var colLength = row.cells.length;
				for(var j=0; j < colLength; j++) {
					var cell = row.cells[j];
					if(onclickHandler) {
						cell.onclick = onclickHandler;
					}
					setEventHandlers(cell);
				}
			}
		}

		/**
		 * イベントハンドラが設定済みかどうかを調べる。
		 * @return イベントハンドラが設定済みの場合は true。
		 */
		function hasEventHandler() {
			if(detail && detail.rows.length > 0 && detail.rows[0].cells.length > 0) {
				return detail.rows[0].cells[0].onclick!=null;
			}
			return false;
		}

	}

	/**
	 * Celcel の HTML要素を作成する。
	 * @param handler Celcelインスタンス。
	 * @return HTML要素。
	 */
	function createHtmlElement(handler, container) {
		var htmlElement;
		if(container) {
			htmlElement = container;
		}
		else {
			htmlElement = document.createElement("div");
		}
		htmlElement.className = "celcel";
		htmlElement.style.position = "relative";
		htmlElement.handler = handler;
		return htmlElement;
	}

	/**
	 * Celcel 用のコンテキストメニューを作成する。
	 */
	function createContextMenu(htmlElement) {
		var menu = new IMenu();
		menu.addTarget(htmlElement);
		menu.addItem("copyAll", function(){handler.copyAll();});
		menu.addItem("copy", function(){handler.copy();});
		menu.addItem("cut", function(){handler.cut();});
		menu.addItem("paste", function(){handler.paste();});
		menu.addItem("clear", function(){handler.clear();});
		return menu;
	}

	/**
	 * テーブル格納用の div を作成する。
	 */
	function createDiv(handler, htmlElement, table, overflow) {
		var div = document.createElement("div");
		div.className = "table-div";
		htmlElement.appendChild(div);
		div.style.display = "inline";
		div.style.overflow = overflow;

		if(table) {
			table.parentNode.removeChild(table);
		}
		else {
			table = document.createElement("table");
			table.appendChild(document.createElement("tbody"));
		}

		table.style.borderCollapse = "collapse";
		table.style.position = "relative";
	    table.style.tableLayout = "fixed";

		div.appendChild(table);
		table.border = "1";
		table.handler = handler;

		return div;
	}

	/**
	 * スクロールバーの幅を取得する。
	 */
	function getScrollMargin(detailDiv) {
		if(detailDiv.style.overflow=="scroll") {
			return Celcel.prototype.SCROLL_MARGIN;
		}
		else {
			return 0;
		}
	}

	/**
	 * イベントハンドラの設定を行なう。
	 */
	function setEventHandlers(cell) {
		var input = getInput(cell);
		if(input) {
			input.onkeydown = Celcel.input_onkeydown;
			input.onfocus = Celcel.input_onfocus;
			input.onclick = Celcel.input_onclick;
			input.onmouseover = Celcel.cell_onmouseover;
		}
		cell.onmouseover = Celcel.cell_onmouseover;
	}

	/**
	 * 指定された input に focus をセットする。
	 */
	function setFocus(input) {
		if(input && input.focus) {
			input.focus();
		}
	}

	/**
	 * セルの選択を行なう。
	 */
	function selectCell(event) {
		event = IWidget.editEvent(event);
		var input = event.target;
		var table = getTable(input);

		if(!event.ctrlKey && !event.shiftKey) {
			clearAllSelections(table);
		}
		updateSelection(event, table, input);
		return table.handler;
	}

	/**
	 * 選択状態を更新する。
	 */
	function updateSelection(event, table, input) {
		if(table.handler.selectionMode=="row") {
			updateRowSelection(event, table, input);
		}
		else if(table.handler.selectionMode=="column") {
			updateColumnSelection(event, table, input);
		}
		else { // (table.handler.selectionMode=="cell")
			updateCellSelection(input);
		}
	}

	/**
	 * 行選択状態を更新する。
	 */
	function updateRowSelection(event, table, input) {
		var row;
		if(input.nodeName=="INPUT") {
			row = getRow(input);
		}
		else {
			row = input.parentNode;
		}
		if(isAvailableTable(table.leftTable)) {
			selectRow(table.leftTable, table.leftTable.rows[row.rowIndex]);
		}
		selectRow(table, row);
		if(isAvailableTable(table.rightTable)) {
			selectRow(table.rightTable, table.rightTable.rows[row.rowIndex]);
		}

		if(rowHeader.rows.length>0) {
			select(rowHeader.rows[row.rowIndex].cells[0], false);
		}
		else {
			handler.selectionHolder.value = row.rowIndex.toString();
			execute(row.cells[0]);
		}

		function selectRow(table, row) {
			if(updateCellSelection(getFirstInputInRow(row, 0))) {
				updateCellSelection(getLastInputInRow(row, row.cells.length-1));
			}
		}
	}

	/**
	 * 列選択状態を更新する。
	 */
	function updateColumnSelection(event, table, input) {
		var cell;
		if(input.nodeName=="INPUT") {
			cell = getCell(input);
		}
		else {
			cell = input;
		}
		var index = cell.cellIndex;
		if(isAvailableTable(table.upperTable)) {
			selectColumn(table.upperTable, index);
		}
		selectColumn(table, index);
		if(isAvailableTable(table.lowerTable)) {
			selectColumn(table.lowerTable, index);
		}

		function selectColumn(table, index) {
			if(updateCellSelection(getFirstInputInColumn(table, index, 0))) {
				updateCellSelection(getLastInputInColumn(table, index, table.rows.length-1));
			}
			else if(table.rows.length > 0 && table.rows[0].cells.length > index) {
				select(table.rows[0].cells[index], true);
			}
		}
	}

	function select(headerCell, isColumnSelection) {
		if(!headerCell || !headerCell.id) {
			return;
		}

		var index = headerCell.id.indexOf("-");
		if(index==-1) {
			return;
		}

		if(isColumnSelection) {
			handler.selectionHolder.value = "," + headerCell.id.substring(index+1);
		}
		else {
			handler.selectionHolder.value = headerCell.id.substring(index+1);
		}

		execute(headerCell);
	}

	function execute(cell) {
		if(handler.onselected) {
			handler.onselected(cell);
		}
		if(handler.requestOnselected) {
			new CongaRequest().send(handler.requestOnselected, 
					handler.selectionHolder.name + "=" + handler.selectionHolder.value);
		}
	}

	/**
	 * 全てのテーブルの選択状態をクリアする。
	 */
	function clearAllSelections(table) {
		table.handler.selectionStartingTable = null;
		clearSelection(table);
		if(table.upperTable) {
			clearSelection(table.upperTable);
		}
		if(table.lowerTable) {
			clearSelection(table.lowerTable);
		}
		if(table.leftTable) {
			clearSelection(table.leftTable);
		}
		if(table.rightTable) {
			clearSelection(table.rightTable);
		}
		if(table.crossTable) {
			clearSelection(table.crossTable);
		}
	}

	/**
	 * セルの選択状態を更新する。
	 */
	function updateCellSelection(input) {
		if(!input) {
			return false;
		}

		var table = getTable(input);
		var rowIndex = getRow(input).rowIndex;
		var columnIndex = getCell(input).cellIndex;

		updateSelectionForEndingTable(table, rowIndex, columnIndex);
		updateSelectionForOtherTable(table, rowIndex, columnIndex);

//		handler.selectionHolder.value = たぶんカーソル移動が遅くなるので今はやらない


		return true;

		/**
		 * 他テーブルの選択状態を更新する。
		 */
		function updateSelectionForOtherTable(table, rowIndex, columnIndex) {

			if(table.handler.selectionStartingTable==table) { // 全てクリア
				clearSelection(table.leftTable);
				clearSelection(table.rightTable);
				clearSelection(table.lowerTable);
				clearSelection(table.upperTable);
				clearSelection(table.crossTable);
			}
			else if(table.handler.selectionStartingTable==table.leftTable) {
				clearSelection(table.lowerTable);
				clearSelection(table.upperTable);
				clearSelection(table.crossTable);
				updateSelectionForEndingTable(table.leftTable, 
					rowIndex, table.leftTable.rows[0].cells.length - 1);
			}
			else if(table.handler.selectionStartingTable==table.rightTable) {
				clearSelection(table.lowerTable);
				clearSelection(table.upperTable);
				clearSelection(table.crossTable);
				updateSelectionForEndingTable(table.rightTable, rowIndex, 0);
			}
			else if(table.handler.selectionStartingTable==table.upperTable) {
				clearSelection(table.leftTable);
				clearSelection(table.rightTable);
				clearSelection(table.crossTable);
				updateSelectionForEndingTable(table.upperTable, 
					table.upperTable.rows.length - 1, columnIndex);
			}
			else if(table.handler.selectionStartingTable==table.lowerTable) {
				clearSelection(table.leftTable);
				clearSelection(table.rightTable);
				clearSelection(table.crossTable);
				updateSelectionForEndingTable(table.lowerTable, 0, columnIndex);
			}
			else if(table.handler.selectionStartingTable==table.crossTable) {
				var columnIndexForCross;
				var rowIndexForCross;
				if(table.leftTable) {
					columnIndexForCross = table.leftTable.rows[0].cells.length - 1;
					updateSelectionForEndingTable(table.leftTable, rowIndex, columnIndexForCross);
				}
				if(table.rightTable) {
					columnIndexForCross = 0;
					updateSelectionForEndingTable(table.rightTable, rowIndex, columnIndexForCross);
				}
				if(table.upperTable) {
					rowIndexForCross = table.upperTable.rows.length - 1;
					updateSelectionForEndingTable(table.upperTable, rowIndexForCross, columnIndex);
				}
				if(table.lowerTable) {
					rowIndexForCross = 0;
					updateSelectionForEndingTable(table.lowerTable, rowIndexForCross, columnIndex);
				}
				updateSelectionForEndingTable(table.crossTable, rowIndexForCross, columnIndexForCross);
			}
		}
	}

	/**
	 * 選択終了ポイントのテーブルの選択状態を更新する。
	 */
	function updateSelectionForEndingTable(table, rowIndex, columnIndex) {

		if(!table.selection) {
			initSelectionStartIndex(table, rowIndex, columnIndex);

			table.selectionMinRowIndex = table.selectionStartRowIndex;
			table.selectionMaxRowIndex = table.selectionStartRowIndex;

			table.selectionMinColumnIndex = table.selectionStartColumnIndex
			table.selectionMaxColumnIndex = table.selectionStartColumnIndex

			if(!table.handler.selectionStartingTable) {
				table.handler.selectionStartingTable = table;
			}
		}
		else {
			if(rowIndex == table.selectionMinRowIndex || rowIndex == table.selectionMaxRowIndex) {
				if(columnIndex < table.selectionMinColumnIndex) {
					addSelection(table, 
						table.selectionMinRowIndex, table.selectionMaxRowIndex,
						columnIndex, table.selectionMinColumnIndex - 1
					);
					table.selectionMinColumnIndex = columnIndex;
					return;
				}
				else if(columnIndex > table.selectionMaxColumnIndex) {
					addSelection(table, 
						table.selectionMinRowIndex, table.selectionMaxRowIndex,
						table.selectionMinColumnIndex + 1, columnIndex
					);
					table.selectionMaxColumnIndex = columnIndex;
					return;
				}
			}

			if(columnIndex == table.selectionMinColumnIndex || columnIndex == table.selectionMaxColumnIndex) {
				if(rowIndex < table.selectionMinRowIndex) {
					addSelection(table, 
						rowIndex, table.selectionMinRowIndex - 1,
						table.selectionMinColumnIndex, table.selectionMaxColumnIndex
					);
					table.selectionMinRowIndex = rowIndex;
					return;
				}
				else if(rowIndex > table.selectionMaxRowIndex) {
					addSelection(table, 
						table.selectionMaxRowIndex + 1, rowIndex,
						table.selectionMinColumnIndex, table.selectionMaxColumnIndex
					);
					table.selectionMaxRowIndex = rowIndex;
					return;
				}
			}
		}


		var maxRowIndex = Math.max(table.selectionStartRowIndex, rowIndex);
		var minRowIndex = Math.min(table.selectionStartRowIndex, rowIndex);

		var maxColumnIndex = Math.max(table.selectionStartColumnIndex, columnIndex);
		var minColumnIndex = Math.min(table.selectionStartColumnIndex, columnIndex);

		clearSelection(table);
		table.selection = new Array();

		addSelection(table, minRowIndex, maxRowIndex, minColumnIndex, maxColumnIndex);

		table.selectionMinRowIndex = minRowIndex;
		table.selectionMinColumnIndex = minColumnIndex;
		table.selectionMaxRowIndex = maxRowIndex;
		table.selectionMaxColumnIndex = maxColumnIndex;


		function initSelectionStartIndex(table, rowIndex, columnIndex) {

			if(!table.handler.selectionStartingTable || 
					table.handler.selectionStartingTable==table) {
				table.selectionStartRowIndex = rowIndex;
				table.selectionStartColumnIndex = columnIndex;
				return;
			}

			if(table.handler.selectionStartingTable==table.crossTable) {
				if(table.lowerTable) {
					table.selectionStartRowIndex = table.rows.length - 1;
				}
				else {
					table.selectionStartRowIndex = 0;
				}

				if(table.rightTable) {
					table.selectionStartColumnIndex = table.rows[0].cells.length - 1;
				}
				else {
					table.selectionStartColumnIndex = 0;
				}
			}
			else if(table.handler.selectionStartingTable==table.lowerTable) {
				table.selectionStartRowIndex = table.rows.length - 1;
				table.selectionStartColumnIndex = table.lowerTable.selectionStartColumnIndex;
			}
			else if(table.handler.selectionStartingTable==table.upperTable) {
				table.selectionStartRowIndex = 0;
				table.selectionStartColumnIndex = table.upperTable.selectionStartColumnIndex;
			}
			else if(table.handler.selectionStartingTable==table.rightTable) {
				table.selectionStartRowIndex = table.rightTable.selectionStartRowIndex;
				table.selectionStartColumnIndex = table.rows[0].cells.length - 1;
			}
			else if(table.handler.selectionStartingTable==table.leftTable) {
				table.selectionStartRowIndex = table.leftTable.selectionStartRowIndex;
				table.selectionStartColumnIndex = 0;
			}
		}
	}

	/**
	 * 指定されたテーブルの選択状態をクリアする。
	 */
	function clearSelection(table) {
		if(!table) {
			return;
		}

		if(table.selection) {
			for(var i=0; i<table.selection.length; i++) {
				var input = table.selection[i];
				getCell(input).style.backgroundColor = "white";
				input.style.backgroundColor = "white";
			}
		}
		table.selection = null;
	}

	/**
	 * 選択範囲を追加する。
	 */
	function addSelection(table, startRowIndex, endRowIndex, startColumnIndex, endColumnIndex) {
		if(!table.selection) {
			table.selection = new Array();
		}

		for(var i=startRowIndex; i<=endRowIndex; i++) {
			for(var j=startColumnIndex; j<=endColumnIndex; j++) {
				var cell = table.rows[i].cells[j];
				var input = getInput(cell);
				if(input) {
					input.style.backgroundColor = "#ccccff";
					table.selection.push(input);
					cell.style.backgroundColor = "#ccccff";
				}
			}
		}
	}

	/**
	 * 列の中で最後の input を取得する。
	 */
	function getLastInputInColumn(table, columnIndex, rowIndex) {
		for(var i=rowIndex; i >= 0; i--) {
			var input = getInput(table.rows[i].cells[columnIndex]);
			if(input) {
				return input;
			}
		}
		return null;
	}

	/**
	 * 行の中で最後の input を取得する。
	 */
	function getLastInputInRow(row, index) {
		for(var i=index; i >= 0; i--) {
			var input = getInput(row.cells[i]);
			if(input) {
				return input;
			}
		}
		return null;
	}

	
	/**
	 * 列の中で最初の input を取得する。
	 */
	function getFirstInputInColumn(table, columnIndex, rowIndex) {
		for(var i=rowIndex; i < table.rows.length; i++) {
			var input = getInput(table.rows[i].cells[columnIndex]);
			if(input) {
				return input;
			}
		}
		return null;
	}

	/**
	 * 行の中で最初の input を取得する。
	 */
	function getFirstInputInRow(row, index) {
		for(var i=index; i < row.cells.length; i++) {
			var input = getInput(row.cells[i]);
			if(input) {
				return input;
			}
		}
		return null;
	}

	/**
	 * 指定された input をもつ行を取得する。
	 */
	function getRow(input) {
		return getCell(input).parentNode;
	}

	/**
	 * 指定された input をもつセルを取得する。
	 */
	function getCell(input) {
		return input.parentNode;
	}

	/**
	 * 指定されたセルの中の input を取得する。
	 */
	function getInput(cell) {
		if(!cell) {
			return null;
		}

		var children = cell.childNodes;

		for(var j=0; j < children.length; j++) {
			var child = children.item(j);
			if(child.nodeName!="INPUT") {
				continue;
			}
			return child;
		}
		return null;
	}

	/**
	 * 指定されたオブジェクトの親要素から table を検索して返す。
	 */
	function getTable(obj) {
		while(obj) {
			if(obj.nodeName=="TABLE") {
				return obj;
			}
			obj = obj.parentNode;
		}
	}

	/**
	 * 指定されたテーブルが利用可能かどうかを調べる。
	 */
	function isAvailableTable(table) {
		return table && table.rows.length > 0;
	}


	//// インスタンスの初期化
	init(table, fixRows, fixColumns, header, rowHeader, columnHeader);
}


