/**
 * CongaResponse を作成する。
 * 
 * CongaRequest から呼び出され，XML の解釈を行なう。
 * 
 *  解析できる XML の形式:
 *    タグは2種類。ひとつの c (container) タグの中に 複数の e (element)タグ
 *    または c タグを記述する。
 *    e タグの中に e タグや c タグを記述することはできない。
 *    例：
 *      <c>
 *        <e>...</e>
 *        <e>...</e>
 *        <c>
 *          <e>...</e>
 *          <e>...</e>
 *        </c>
 *      </c>
 *
 *    最上位の c タグは parse 関数が解釈し，下層の e, c タグの内容解釈は handler 関数が解釈する。
 *    parse も handler も利用者により設定可能なので
 *    タグの属性は任意設定可能だが，標準的には以下のものがある。
 *
 *    最上位の c タグの属性 : 
 *       is-error エラーが存在するかどうかを示す。
 *       message エラーメッセージ
 *
 *    下層の e, c タグの属性
 *       type 受信データのタイプ。handler 関数のセレクタとなる。
 *       id 値を設定する対象となるオブジェクトの id。
 *       is-error エラーデータかどうかを示す。
 * 
 *  CongaResponse.parse メソッドでは，上記のタグ内容を handler が容易に利用できるように
 *  以下のプロパティとして展開する。
 *    id - 下層の e, c タグの id 属性で指定された値。
 *    type - 下層の e, c タグの type 属性で指定された値。
 *    value - 下層の e, c タグ内のテキスト値。
 *    errore - 下層の e, c タグの is-error 属性に "true" が設定されている場合は true，
 *             それ以外の場合は false。
 *    errorc - 最上位の c タグの is-error 属性に "true" が設定されている場合は true，
 *             それ以外の場合は false。
 *    etag - 下層の e, c タグノードオブジェクト。
 *
 *     handler type 毎の handler 関数の連想配列
 *        デフォルト設定:
 *          CongaResponse.handler["window"] = CongaResponse.setValue2Window;
 *          CongaResponse.handler["iwindow"] = CongaResponse.setValue2IWindow;
 *          CongaResponse.handler["celcel"] = CongaResponse.setValue2Celcel;
 *          CongaResponse.handler["tree"] = CongaResponse.setValue2ITree;
 *          CongaResponse.handler["tree-node"] = CongaResponse.setValue2ITreeNode;
 *          CongaResponse.handler["split-pane"] = CongaResponse.setValue2SplitPane;
 *          CongaResponse.handler["outer"] = CongaResponse.setValue4Outer;
 *          CongaResponse.handler["self"] = CongaResponse.setValue2Self;
 *          CongaResponse.handler["alert"] = CongaResponse.setValue4Alert;
 *          CongaResponse.handler["confirm"] = CongaResponse.setValue4Confirm;
 *          CongaResponse.handler["method"] = CongaResponse.setValue4Method;
 *          CongaResponse.handler["script"] = CongaResponse.setValue4Script;
 *          CongaResponse.handler["message"] = CongaResponse.showMessage;
 *          CongaResponse.handler["focus"] = CongaResponse.setFocus;
 *          CongaResponse.handler["label"] = CongaResponse.setLabel;
 *          CongaResponse.handler["default"] = CongaResponse.setValue4Default;
 *
 */
function CongaResponse() {

	var congaResponse = this;

	/**
	 * 下層の e, c タグの id 属性で指定された値。
	 */
	this.id;

	/**
	 * 下層の e, c タグの type 属性で指定された値。
	 */
	this.type;

	/**
	 * 下層の e, c タグ内のテキスト値。
	 */
	this.value;

	/**
	 * 下層の e, c タグの is-error 属性に "true" が設定されている場合は true，
	 * それ以外の場合は false。
	 */
	this.errore;

	/**
	 * 最上位の c タグの is-error 属性に "true" が設定されている場合は true，
	 * それ以外の場合は false。
	 */
	this.errorc;

	/**
	 * 下層の e, c タグノードオブジェクト。
	 */
	this.etag;

	/**
	 * XML ドキュメントの解析を行なう。
	 * @param cTagList c タグのリスト。
	 */
	this.parse = function(doc) {
		if(!doc) {
			doc = document.getElementsByTagName("xml").item(0);
		}

		var cTagList = doc.getElementsByTagName("c");
		for(var i=0; i<cTagList.length; i++) {
			var cTag = cTagList.item(i);
			parseCTag(cTag);
		}

		function parseCTag(cTag) {
			var message = null;
			var errorc = cTag.getAttribute("is-error")=="true";
			var list = cTag.childNodes;
			var length = list.length;
			var defaultHandler = CongaResponse.handler["default"];

			for(var i=0; i<length; i++) {
				var item = list.item(i);
				if(item.tagName!="e" && item.tagName!="c") {
					continue;
				}
				var id = item.getAttribute("id");
				var type = item.getAttribute("type");
				var msg = item.getAttribute("message");

				if(msg && !message) {
					message = msg;
				}

				var handler = CongaResponse.handler[type];
				if(!handler) {
					handler = defaultHandler;
				}

				var value = null;
				if(item.firstChild) {
					value = item.firstChild.nodeValue;
				}

				setup(id, type, value, item.getAttribute("is-error")=="true", errorc, item);
				handler(congaResponse);
			}

			if(!message) {
				message = cTag.getAttribute("message");
			}

			if(message) {
				var handler = CongaResponse.handler["message"];
				if(!handler) {
					handler = CongaResponse.showMessage;
				}
				setup(null, "message", message, false, errorc, null);
				handler(congaResponse);
			}

			function setup(id, type, value, errore, errorc, etag) {
				congaResponse.id = id;
				congaResponse.type = type;
				congaResponse.value = value;
				congaResponse.errore = errore;
				congaResponse.errorc = errorc;
				congaResponse.etag = etag;
			}
		}
	}




	/**
	 * type="window" で呼び出される handler。
	 * 新しいウインドウを開きその中に値を出力する。
	 */
	CongaResponse.setValue2Window = function(response) {
		var option = response.etag.getAttribute("option");
		if(!option) {
			option = "";
		}
		if(!response.id) {
			response.id = "window";
		}
		var w = window.open("about:blank", response.id, option)
		w.document.write(response.value);
		w.document.close();
	}

	/**
	 * type="iwindow" で呼び出される handler。
	 */
	CongaResponse.setValue2IWindow = function(response) {
		var win;
		var elem;
		if(response.id) {
			elem = document.getElementById(response.id);
		}

		if(elem && elem.handler) {
			win = elem.handler;
		}
		else {
			var modal = response.etag.getAttribute("modal")=="true";
			var parent = null;
			var parentId = response.etag.getAttribute("parent-id");
			if(parentId) {
				parent = document.getElementById(parentId);
			}
			win = new IWindow(null, null, false, parent, modal);

			if(response.id) {
				win.htmlElement.id = response.id;
			}
		}

		var title = response.etag.getAttribute("title");
		if(title) {
			win.setTitle(title);
		}

		var content = null;
		var contentId = response.etag.getAttribute("content-id");
		var contentByUrl = response.etag.getAttribute("content-by-url");
		if(contentId) {
			content = document.getElementById(contentId);
		}
		else if(contentByUrl) {
			content = contentByUrl;
		}
		else if(response.value) {
			content = response.value;
		}

		if(content) {
			if(contentByUrl) {
				win.setContentByURL(content);
				if(!title) {
					win.setTitle(content);
				}
			}
			else {
				win.setContent(content);
			}
		}

		if(response.etag.getAttribute("title-bar")=="false") {
			win.hideTitleBar();
		}
		if(response.etag.getAttribute("resizable")=="false") {
			win.setResizable(false);
		}

		var icon = response.etag.getAttribute("icon");
		if(icon) {
			win.setIcon(icon);
		}

		var ecb = response.etag.getAttribute("enable-close-button");
		if(ecb=="false") {
			win.setEnableCloseButton(false);
		}

		var emb = response.etag.getAttribute("enable-minimize-button");
		if(emb=="false") {
			win.setEnableMinimizeButton(false);
		}

		var scroll = response.etag.getAttribute("scroll");
		if(scroll!=null) {
			win.setScroll(scroll);
		}

		var bounds = response.etag.getAttribute("bounds");
		if(bounds) {
			eval("win.setBounds("+bounds+")");
		}

		var position = response.etag.getAttribute("position");
		if(position) {
			eval("win.setPosition("+position+")");
		}

		var size = response.etag.getAttribute("size");
		if(size) {
			eval("win.setSize("+size+")");
		}

		if(!bounds && !size) {
			win.pack();
		}

		if(!bounds && !position) {
			win.show(window.event);
		}
		else {
			win.show();
		}
	}

	/**
	 * type="tree" で呼び出される handler。
	 * 指定された値を id 引数で指定されたタグの innerHTML プロパティとして設定し，ITree に変換する。
	 */
	CongaResponse.setValue2ITree = function(response) {
		var elem = document.getElementById(response.id);
		if(elem && !response.errore) {
			if(response.value) {
				elem = updateHTMLElement(elem, response.id, response.value);
				var itree = new ITree(elem);
				itree.requestOnselected = response.etag.getAttribute("request-onselected");
				itree.requestOnopening = response.etag.getAttribute("request-onopening");
				itree.requestOnclosing = response.etag.getAttribute("request-onclosing");
				itree.requestOnloading = response.etag.getAttribute("request-onloading");
			}
			else {
				elem.innerHTML = "";
			}
		}
	}

	/**
	 * type="tree-node" で呼び出される handler。
	 * 指定された値を id 引数で指定されたタグの中の tree の中の
	 * node-id 属性で指定された id をもつノードを更新する。
	 */
	CongaResponse.setValue2ITreeNode = function(response) {
		var elem = document.getElementById(response.id);
		if(elem && !response.errore) {
			var tree = elem;
			var nodeId = response.etag.getAttribute("node-id");
			if(response.value && tree && tree.handler && nodeId) {
				var node = document.getElementById(nodeId);
				tree.handler.updateNode(node, response.value);
			}
		}
	}

	/**
	 * type="celcel" で呼び出される handler。
	 * 指定された値を id 引数で指定されたタグの innerHTML プロパティとして設定し，Celcel に変換する。
	 */
	CongaResponse.setValue2Celcel = function(response) {
		var elem = document.getElementById(response.id);
		if(elem && !response.errore) {
			if(response.value) {
				var oWidth = elem.offsetWidth;
				var oHeight = elem.offsetHeight;

				elem = updateHTMLElement(elem, response.id, response.value);

				var nWidth = elem.offsetWidth;
				var nHeight = elem.offsetHeight;

				var fixRows = response.etag.getAttribute("fix-rows");
				var fixCols = response.etag.getAttribute("fix-cols");

				var celcel = new Celcel(elem, fixRows, fixCols);

				var scroll = response.etag.getAttribute("scroll");
				if(scroll!=null) {
					celcel.setScroll(scroll!="false");
				}
				celcel.requestOnselected = response.etag.getAttribute("request-onselected");
				celcel.requestOnclicked = response.etag.getAttribute("request-onclicked");
				celcel.selectionMode = response.etag.getAttribute("selection-mode");
				celcel.autoSelection = response.etag.getAttribute("auto-selection")=="true";

				var height = response.etag.getAttribute("height");
				if(height) {
					celcel.setHeight(height);
				}
				else if(elem.offsetHeight < oHeight || oHeight < nHeight){
					celcel.setHeight(oHeight);
				}
				if(elem.offsetWidth < oWidth || oWidth < nWidth ) {
					celcel.setWidth(oWidth);
				}
			}
			else {
				elem.innerHTML = "";
			}
		}
	}

	/**
	 * type="split-pane" で呼び出される handler。
	 * id 引数で指定された HTML 要素で ISplitPane を呼び出す。
	 */
	CongaResponse.setValue2SplitPane = function(response) {
		var elem = document.getElementById(response.id);
		if(elem) {
			var orientation = response.etag.getAttribute("orientation");
			if(!orientation) {
				orientation = "v";
			}
			var split = new ISplitPane(elem, orientation);
			var dividerLocation = response.etag.getAttribute("divider-location");
			if(dividerLocation) {
				split.setDividerLocation(parseInt(dividerLocation));
			}
		}
	}

	/**
	 * type="outer" で呼び出される handler。
	 * id 引数で指定された HTML 要素を書き換える。
	 */
	CongaResponse.setValue4Outer = function(response) {
		var elem = document.getElementById(response.id);
		if(elem) {
			updateHTMLElement(elem, response.id, response.value);
		}
	}

	/**
	 * type="self" で呼び出される handler。
	 * 現在ウインドウに表示されている内容を指定された値で上書きする。
	 */
	CongaResponse.setValue2Self = function(response) {
		if(!response.errore) {
			document.clear();
			document.write(response.value);
			document.close();
		}
	}

	/**
	 * type="script" で呼び出される handler。
	 * 指定された値を javascript として実行する。
	 */
	CongaResponse.setValue4Script = function(response) {
		var command = response.etag.getAttribute("value");
		if(command) {
			eval(command);
		}
	}

	/**
	 * type="method" で呼び出される handler。
	 * value 属性で指定された値を id 引数で指定されたオブジェクトのメソッドとして実行する。
	 */
	CongaResponse.setValue4Method = function(response) {
		var elem = document.getElementById(response.id);
		var command = response.etag.getAttribute("value");
		if(elem && command) {
			if(elem.handler) {
				eval("elem.handler." + command);
			}
			else {
				eval("elem." + command);
			}
		}
	}

	/**
	 * type="alert" で呼び出される handler。
	 * value 属性で指定された文字列を alert 表示する。
	 * また，request 属性が指定されている場合，その属性の値を URL とみなして，リクエスト送信を行なう。
	 */
	CongaResponse.setValue4Alert = function(response) {
		var text = response.etag.getAttribute("value");
		var req = response.etag.getAttribute("request");
		var method = response.etag.getAttribute("method");
		if(!method) {
			method = "POST";
		}

		window.alert(text);
		if(req){
			new CongaResponse(method).send(req, "");
		}
	}

	/**
	 * type="confirm" で呼び出される handler。
	 * value 属性で指定された文字列を confirm 表示する。
	 * また，request-if-ok 属性が指定され，OKボタンが押された場合，
	 * その属性の値を URL とみなして，リクエスト送信を行なう。
	 * また，request-if-cancel 属性が指定され，Cancelボタンが押された場合，
	 * その属性の値を URL とみなして，リクエスト送信を行なう。
	 */
	CongaResponse.setValue4Confirm = function(response) {
		var text = response.etag.getAttribute("value");
		var reqIfYes = response.etag.getAttribute("request-if-ok");
		var reqIfNo = response.etag.getAttribute("request-if-cancel");
		var method = response.etag.getAttribute("method");
		if(!method) {
			method = "POST";
		}

		if(window.confirm(text)) {
			new CongaResponse(method).send(reqIfYes, "");
		}
		else if(reqIfNo){
			new CongaResponse(method).send(reqIfNo, "");
		}
	}

	/**
	 * type 指定がなかった場合，あるいは，指定されている type に該当する handler  が登録されて
	 * いなかった場合に呼び出される。
	 * id 引数で指定された id をもつオブジェクトの value 属性または，innerHTML 属性として設定を行なう。
	 */
	CongaResponse.setValue4Default = function(response) {

		var elem = document.getElementById(response.id);

		if(!elem) {
			return;
		}

		if(response.errore) {
			if(elem.value!=null) {
				elem.style.backgroundColor = "pink";
			}
		}
		else {
			elem.style.backgroundColor = "white";
			if(elem.value!=null) {
				if(response.value==null) {
					elem.value = "";
				}
				else {
					elem.value = response.value;
				}
			}
			else if(elem.innerHTML!=null) {
				if(response.value==null) {
					elem.innerHTML = "";
				}
				else {
					elem.innerHTML = response.value;
				}
			}
		}

		var disabled = response.etag.getAttribute("disabled");
		if(disabled) {
			if(disabled=="true") {
				elem.disabled = true;
			}
			else {
				elem.disabled = false;
			}
		}
	}

	/**
	 * エラーメッセージ表示を行なう。
	 */
	CongaResponse.showMessage = function(response) {
		window.alert(response.value);
	}

	/**
	 * フォーカス表示を行なう。
	 */
	CongaResponse.setFocus = function(response) {
		var elem = document.getElementById(response.id);
		if(elem && elem.focus) {
			setTimeout(function(){elem.focus();},200);
		}
	}

	/**
	 * ラベル設定を行なう。
	 */
	CongaResponse.setLabel = function(response) {
		var elem = document.getElementById(response.id);
		var value = response.etag.getAttribute("value");

		if(elem && elem.innerHTML) {
			if(value==null) {
				elem.innerHTML = "";
			}
			else {
				elem.innerHTML = value;
			}
		}
	}


	//// ローカルメソッド定義

	/**
	 * インスタンスを初期化する。
	 */
	function init() {
		if(!CongaResponse.handler) {
			CongaResponse.handler = new Array();
			CongaResponse.handler["default"] = CongaResponse.setValue4Default;
			CongaResponse.handler["window"] = CongaResponse.setValue2Window;
			CongaResponse.handler["iwindow"] = CongaResponse.setValue2IWindow;
			CongaResponse.handler["celcel"] = CongaResponse.setValue2Celcel;
			CongaResponse.handler["tree"] = CongaResponse.setValue2ITree;
			CongaResponse.handler["tree-node"] = CongaResponse.setValue2ITreeNode;
			CongaResponse.handler["itree"] = CongaResponse.setValue2ITree;
			CongaResponse.handler["itree-node"] = CongaResponse.setValue2ITreeNode;
			CongaResponse.handler["split-pane"] = CongaResponse.setValue2SplitPane;
			CongaResponse.handler["outer"] = CongaResponse.setValue4Outer;
			CongaResponse.handler["self"] = CongaResponse.setValue2Self;
			CongaResponse.handler["alert"] = CongaResponse.setValue4Alert;
			CongaResponse.handler["confirm"] = CongaResponse.setValue4Confirm;
			CongaResponse.handler["method"] = CongaResponse.setValue4Method;
			CongaResponse.handler["script"] = CongaResponse.setValue4Script;
			CongaResponse.handler["message"] = CongaResponse.showMessage;
			CongaResponse.handler["focus"] = CongaResponse.setFocus;
			CongaResponse.handler["label"] = CongaResponse.setLabel;
		}
	}

	/**
	 * HTML 要素の更新を行なう。
	 * @param id 更新する HTML 要素のID。
	 * @param value 更新する HTML 文字列。
	 */
	function updateHTMLElement(elem, id, value) {
		if(elem.outerHTML) {
			elem.outerHTML = value;
			elem = document.getElementById(id);
		}
		else {
			var dummy = document.createElement("div");
			dummy.innerHTML = value;
			var newElem = dummy.getElementsByTagName(elem.tagName).item(0);
			elem.parentNode.replaceChild(newElem, elem);
			elem = newElem;
		}

		if(elem.parentNode && elem.parentNode.handler
				&& elem.parentNode.handler.updateChild) {
			elem.parentNode.handler.updateChild(elem);
		}
		return elem;
	}

	//// インスタンスの初期化
	init();

}
new CongaResponse();


