/*
 * Conga DHTML Scripts
 *
 * Copyright (C) 2005 TEAM NGA
 *
 * このフォルダ中に存在するソースコードを使用する際には，以下の使用許諾
 * に従う必要があります。
 *
 *
 * [使用許諾]
 *
 *   以下では，「ソースコード」，「ソースコードから生成されたドキュメン
 * ト」，「ソースコードをコンパイルして作成されたバイナリファイル」の三
 * 者を「ライブラリ」と呼びます。たとえ，ソースコードが単体で実行可能な
 * ものである場合でも，ここでは「ライブラリ」と呼びます。
 *   この使用許諾の対象となる「使用」とは，「ライブラリ」の複製・配布・
 * 変更，「ライブラリ」を使ったアプリケーションの開発，「ライブラリ」の
 * 実行等，「ライブラリ」に関する一切の活動のことを表します。
 *   この使用許諾によって許諾を受ける者を「使用者」呼びます。
 *
 * (1)
 *   「ライブラリ」には一切の保証がありません。使用者もしくは使用者から
 *   「ライブラリ」を配布された第三者による「ライブラリ」の使用，または
 *   「ライブラリ」を使用して作成されたアプリケーション，システム等の使
 *   用により発生したいかなる損害に対しても著作権者は一切責任を負いませ
 *   ん。これらの損害に対してはすべて使用者が責任を負うものとします。
 *
 * (2)
 *   他の使用者と著作権者が「ライブラリ」を使用することを，使用者が妨げ
 *   てはなりません。
 *
 * (3)
 *   使用者は「ライブラリ」の複製・変更・配布を自由に行うことができます。
 *                                                                 以上
 */


/**
 * Celcel を作成する。
 * @param table Celcel化対象テーブル。もしくはCelcel化対象テーブルのコンテナ。
 * @param fixRows 固定表示行数。
 * @param fixColumns 固定表示列数。
 */
function Celcel(table, fixRows, fixColumns) {

	//// ローカル変数定義
	var handler = this;
	var container;
	var htmlElement;

	var header;
	var rowHeader;
	var columnHeader;
	var detail;

	var headerDiv;
	var rowHeaderDiv;
	var columnHeaderDiv;
	var detailDiv;

	var detail;

	var menu;

	//// プロパティ定義

	/**
	 * HTML実体要素オブジェクト。
	 */
	this.htmlElement = null;

	/**
	 * 選択モード。
	 */
	this.selectionMode = "cell";

	/**
	 * 自動選択
	 */
	this.autoSelection = false;

	/**
	 * 選択状態を格納するための hidden タグ。
	 */
	this.selectionHolder = null;

	/**
	 * 行・列選択時に呼び出されるリクエスト。
	 */
	this.requestOnselected = null;

	/**
	 * クリックに呼び出されるリクエスト。
	 */
	this.requestOnclicked = null;


	// メソッド定義

	/**
	 * 選択された範囲の内容をクリアする。
	 */
	this.clear = function() {
		clear0(header);
		clear0(rowHeader);
		clear0(columnHeader);
		clear0(detail);

		function clear0(table) {
			if(!table.selection) {
				return;
			}

			for(var i=0; i<table.selection.length; i++) {
				if(!table.selection[i].readOnly) {
					table.selection[i].value = "";
				}
			}
		}
	}

	/**
	 * 選択された範囲の内容をクリップボードにコピーする。
	 */
	this.copy = function() {
		if(!window.clipboardData) {
			return;
		}

		var arr = new Array();
		var index = 0;

		index = createText(header, columnHeader, arr, index);
		createText(rowHeader, detail, arr, index);

		window.clipboardData.setData("Text", arr.join(""));


		function createText(leftTable, rightTable, arr, index) {
			var maxRowIndex = -1;
			var minRowIndex = Number.POSITIVE_INFINITY;

			if(leftTable.selection) {
				maxRowIndex = Math.max(leftTable.selectionMaxRowIndex, maxRowIndex);
				minRowIndex = Math.min(leftTable.selectionMinRowIndex, minRowIndex);
			}
			if(rightTable.selection) {
				maxRowIndex = Math.max(rightTable.selectionMaxRowIndex, maxRowIndex);
				minRowIndex = Math.min(rightTable.selectionMinRowIndex, minRowIndex);
			}

			if(maxRowIndex==-1 || minRowIndex==Number.POSITIVE_INFINITY) {
				return index;
			}

			for(var i=minRowIndex; i<=maxRowIndex; i++) {
				if(leftTable.selection) {
					index = createTextByInput(leftTable, i, arr, index);
				}
				if(rightTable.selection) {
					index = createTextByInput(rightTable, i, arr, index);
				}
				index--;
				arr[index++] = "\r\n";
			}

			function createTextByInput(table, i, arr, index) {
				var minColumnIndex = table.selectionMinColumnIndex;
				var maxColumnIndex = table.selectionMaxColumnIndex;

				for(var j=minColumnIndex; j<=maxColumnIndex; j++) {
					input = getInput(table.rows[i].cells[j]);
					var value = "";
					if(input) {
						value = input.value;
					}
					arr[index++] = value;
					arr[index++] = "\t";
				}

				return index;
			}

			return index;
		}
	}

	/**
	 * テーブル全体の内容をクリップボードにコピーする。
	 */
	this.copyAll = function() {
		if(!window.clipboardData) {
			return;
		}
		var arr = new Array();
		var index = 0;

		index = copyAllTable(header, columnHeader, arr, index);
		copyAllTable(rowHeader, detail, arr, index);

		window.clipboardData.setData("Text", arr.join(""));

		function copyAllTable(leftTable, rightTable, arr, index) {
			var leftTableRowLength = leftTable.rows.length;
			var rightTableRowLength = rightTable.rows.length;
			var rowLength = Math.max(leftTableRowLength, rightTableRowLength);
			for(var i=0; i < rowLength; i++) {
				if(leftTableRowLength > i) { 
					index = createTextByColumn(leftTable, i, arr, index);
				}
				if(rightTableRowLength > i ) {
					index = createTextByColumn(rightTable, i, arr, index);
				}
				index--;
				arr[index++] = "\r\n";
			}
			return index;
		}

		function createTextByColumn(table, i, arr, index) {
			for(var j = 0; j < table.rows[i].cells.length; j++) {
				var cell = table.rows[i].cells[j];
				input = getInput(cell);
				var value = "";
				if(input) {
					value = input.value;
				}
				else {
					value = cell.innerText;
				}
				arr[index++] = value;
				arr[index++] = "\t";
			}
			return index;
		}
	}

	/**
	 * 選択された範囲の内容の切り取り（クリップボードへのコピーとクリア）を行なう。
	 */
	this.cut = function() {
		handler.copy();
		handler.clear();
	}

	/**
	 * 現在のクリップボードの内容を貼り付ける。
	 */
	this.paste = function() {
		if(!window.clipboardData) {
			return;
		}

		var text = window.clipboardData.getData("Text");
		if(!text || text.length==0) {
			return;
		}

		var table = handler.selectionStartingTable;
		var rowIndex = table.selectionMinRowIndex;

		var rowText = text.split("\r\n");
		for(var i = 0; i<rowText.length; i++) {
			if(rowText[i].length==0) {
				break;
			}
			if(i+rowIndex >= table.rows.length) {
				if(table.lowerTable) {
					table = table.lowerTable;
					rowIndex = 0 - i;
				}
				else {
					break;
				}
			}
			var row = table.rows[i+rowIndex];
			var columnIndex = handler.selectionStartingTable.selectionMinColumnIndex;

			var columnText = rowText[i].split("\t");
			for(var j=0; j<columnText.length; j++) {
				if(j+columnIndex >= row.cells.length) {
					if(table.rightTable) {
						row = table.rightTable.rows[i+rowIndex];
						columnIndex = 0 - j;
					}
					else {
						break;
					}
				}

				var cell = row.cells[j+columnIndex];
				var input = getInput(cell);
				if(input && !input.readOnly) {
					input.value = columnText[j];
				}
			}
		}
	}

	/**
	 * 選択範囲が存在するかどうかを調べる。
	 * @return 選択範囲が存在する場合は true。
	 */
	this.hasSelection = function() {
		return header.selection || rowHeader.selection || 
				columnHeader.selection || detail.selection;
	}

	/**
	 * 表の幅を設定する。
	 * @param width 表の幅。
	 */
	this.setWidth = function(width) {
		width = parseInt(width);
		width = width - rowHeaderDiv.scrollWidth - getScrollMargin(detailDiv);
		if(width <= 0) {
			return;
		}
		columnHeaderDiv.style.width = width;
		detailDiv.style.width = width + getScrollMargin(detailDiv);
		htmlElement.style.width = rowHeaderDiv.scrollWidth + parseInt(detailDiv.style.width);
	}

	/**
	 * 表の高さを設定する。
	 * @param height 表の高さ。
	 */
	this.setHeight = function(height) {
		height = parseInt(height);
		height = height - columnHeaderDiv.scrollHeight - getScrollMargin(detailDiv);
		if(height <= 0) {
			return;
		}
		rowHeaderDiv.style.height = height;
		detailDiv.style.height = height + getScrollMargin(detailDiv);
		htmlElement.style.height = columnHeaderDiv.scrollHeight + parseInt(detailDiv.style.height) ;
	}

	this.setClientWidth = function(width) {
		handler.setWidth(width);
		return true;
	}

	this.setClientHeight = function(height) {
		handler.setHeight(height);
		return true;
	}

	this.pack = function() {
		try {
			handler.setWidth(htmlElement.parentNode.clientWidth - IWidget.getMarginWidth(htmlElement));
			handler.setHeight(htmlElement.offsetHeight - IWidget.getMarginHeight(htmlElement));
		}
		catch(e) {
		}
	}

	/**
	 * 表のサイズを設定する。
	 * @param width 表の幅。
	 * @param height 表の高さ。
	 */
	this.setSize = function(width, height) {
		handler.setWidth(width);
		handler.setHeight(height);
	}

	/**
	 * スクロールバー表示をするかしないかを設定する。
	 * @param scroll スクロールバー表示をする場合は true を指定する。
	 */
	this.setScroll = function(scroll) {
		if(scroll) {
			if(detailDiv.style.overflow!="scroll") {
				detailDiv.style.overflow = "scroll";
				htmlElement.parentNode.onresize = null;
			}
			else {
				return;
			}
		}
		else {
			detailDiv.style.overflow = "hidden";
			detailDiv.style.width = parseInt(detailDiv.style.width) - Celcel.prototype.SCROLL_MARGIN
			detailDiv.style.height = parseInt(detailDiv.style.height) - Celcel.prototype.SCROLL_MARGIN
		}
		handler.setSize(parseInt(detailDiv.style.width), parseInt(detailDiv.style.height));
	}

	/**
	 * コンテクストメニューへ項目追加を行なう。
	 * @param text 項目名称。
	 * @param func メニュークリック時に実行するファンクション。
	 */
	this.addMenu = function(text, func) {
		menu.addItem(text, func);
	}

	/**
	 * コンテクストメニューを表示する。
	 * @param event メニュー表示の起因となったイベント。
	 */
	this.showMenu = function(event) {
		menu.setEnabled("copyAll", true);
		var hasSelection = handler.hasSelection();
		menu.setEnabled("copy", hasSelection);
		menu.setEnabled("cut", hasSelection);
		menu.setEnabled("paste", hasSelection);
		menu.setEnabled("clear", hasSelection);
		menu.show(event);
	}


	/**
	 * セルの input 部分でキー入力したときに呼び出すための
	 * イベントハンドラ。
	 */
	Celcel.input_onkeydown = function(event) {

		event = IWidget.editEvent(event);
		var input = event.target;
		var table = getTable(input);

		//IDebug.print("keyCode:" + event.keyCode);

		if(event.keyCode==16 || // shift
				event.keyCode==17 || event.keyCode==18 ) { // ctrl, alt	
			event.returnValue = true;
			return event.returnValue;
		}


		if(event.keyCode==46) { // delete
			if(table.selection) {
				table.handler.clear();
				event.returnValue = false;
				table.isCellEditMode = false;
				return event.returnValue;
			}
		}

		if(!table.isCellEditMode) {
			if(event.ctrlKey && event.keyCode==86) { // Ctrl+V
				table.handler.paste();
				event.returnValue = false;
				return event.returnValue;
			}
			else if(event.ctrlKey && event.keyCode==67) { // Ctrl+C
				table.handler.copy();
				event.returnValue = false;
				return event.returnValue;
			}
			else if(event.ctrlKey && event.keyCode==88) { // Ctrl+X
				table.handler.cut();
				event.returnValue = false;
				return event.returnValue;
			}
			else if(event.keyCode==8 && input && input.readOnly) { // BS
				event.returnValue = false;
				return event.returnValue;
			}
		}

		var srcInput = input;
		if(!event.shiftKey || event.keyCode==9) {
			clearAllSelections(table);
		}

		if(!table.isCellEditMode) {
			if(event.keyCode==37) { // left
				var item = getPrevInput(table, input, !event.shiftKey);
				if(item) {
					input = item;
				}
				setFocus(input);
				updateSelection(event, getTable(input), input);
				event.returnValue = false;
			}
			else if(event.keyCode==39) { // right
				var item = getNextInput(table, input, !event.shiftKey);
				if(item) {
					input = item;
				}
				setFocus(input);
				updateSelection(event, getTable(input), input);
				event.returnValue = false;
			}
			else {
				event.returnValue = true;
				if(event.keyCode==113) { // F2
					if(input.setSelectionRange) { // for Moz
						input.setSelectionRange(input.value.length, input.value.length);
					}
					else { // for IE
						event.keyCode = 34;
					}
				}
			}
		}

		if(event.keyCode==27) { // ESC
			input.select();
		}
		else if(event.keyCode==40 || event.keyCode==13) { // down or enter
			input = getLowerInput(table, input);
			setFocus(input);
			updateSelection(event, getTable(input), input);
			event.returnValue = false;
		}
		else if(event.keyCode==38) { // up
			input = getUpperInput(table, input);
			setFocus(input);
			updateSelection(event, getTable(input), input);
			event.returnValue = false;
		}

		if(event.keyCode!=9 && srcInput==input) {
			updateSelection(event, getTable(input), input);
		}

		// editMode を設定する。
		if(
			(event.keyCode==37 && !table.isCellEditMode) || // LEFT
			(event.keyCode==39  && !table.isCellEditMode) || // RIGHT
				event.keyCode==9 || event.keyCode==27 || // TAB, ESC
				event.keyCode==40 || event.keyCode==13 || // DOWN, ENTER
				event.keyCode==38) { // UP
			table.isCellEditMode = false; // editMode を false にする。
		}
		else { // それ以外のキーは editMode を true にする。
			if(input && !input.readOnly) {
				table.isCellEditMode = true;
			}
			else {
				table.isCellEditMode = false;
			}
		}

		return event.returnValue;

		/**
		 * より上に位置する input を取得する。
		 */
		function getUpperInput(table, input) {
			var cell = getCell(input);
			var rowIndex = cell.parentNode.rowIndex - 1;
			var prevInput = getLastInputInColumn(table, cell.cellIndex, rowIndex);
			if(prevInput) {
				return prevInput;
			}

			if(table.upperTable) {
				prevInput = getLastInputInColumn(
						table.upperTable, cell.cellIndex, table.upperTable.rows.length-1);
				if(prevInput) {
					return prevInput;
				}
			}

			return input;
		}

		/**
		 * より下に位置する input を取得する。
		 */
		function getLowerInput(table, input) {
			var cell = getCell(input);
			var rowIndex = cell.parentNode.rowIndex + 1;
			var nextInput = getFirstInputInColumn(table, cell.cellIndex, rowIndex);

			if(nextInput) {
				return nextInput;
			}

			if(table.lowerTable) {
				nextInput = getFirstInputInColumn(table.lowerTable, cell.cellIndex, 0);
				if(nextInput) {
					return nextInput;
				}
			}

			return input;
		}

		/**
		 * より前に位置する input を取得する。
		 */
		function getPrevInput(table, input, movePrevRow) {
			var cell = getCell(input);
			var prevInput = getLastInputInRow(cell.parentNode, cell.cellIndex - 1);

			if(prevInput) {
				return prevInput;
			}

			if(isAvailableTable(table.leftTable)) {
				var leftRow = table.leftTable.rows[cell.parentNode.rowIndex];
				prevInput = getLastInputInRow(leftRow, leftRow.cells.length - 1);
				if(prevInput) {
					return prevInput;
				}
			}

			if(!movePrevRow) {
				return null;
			}
			var row = getPrevRow(input);
			if(row && isAvailableTable(table.rightTable)) {
				table = table.rightTable;
				row = table.rows[row.rowIndex];
			}
			else if(!row && isAvailableTable(table.crossTable)) {
				table = table.crossTable;
				row = table.rows[table.rows.length-1];
			}

			if(!row) {
				return null;
			}
			input = getLastInputInRow(row, row.cells.length - 1);
			if(!input && table && isAvailableTable(table.leftTable)) {
				input = getLastInputInRow(table.leftTable.rows[row.rowIndex], 
											table.leftTable.rows[row.rowIndex].cells.length - 1);
			}
			return input;


			function getPrevRow(input) {
				var row = getRow(input).previousSibling;
				return getPrevNode(row); // for Moz

				function getPrevNode(node) {
					if(node && node.nodeType && node.nodeType!=1) {
						node = node.previousSibling;
					}
					return node;
				}
			}
		}




		/**
		 * より後に位置する input を取得する。
		 */
		function getNextInput(table, input, moveNextRow) {
			var cell = getCell(input);
			var nextInput = getFirstInputInRow(cell.parentNode, cell.cellIndex + 1);

			if(nextInput) {
				return nextInput;
			}

			if(isAvailableTable(table.rightTable)) {
				nextInput = getFirstInputInRow(
								table.rightTable.rows[cell.parentNode.rowIndex], 0);
				if(nextInput) {
					return nextInput;
				}
			}

			if(!moveNextRow) {
				return null;
			}

			var row = getNextRow(input);
			if(row && isAvailableTable(table.leftTable)) {
				table = table.leftTable;
				row = table.rows[row.rowIndex];
			}
			else if(!row && isAvailableTable(table.crossTable)) {
				table = table.crossTable;
				row = table.rows[0];
			}

			if(!row) {
				return null;
			}

			input = getFirstInputInRow(row, 0);
			if(!input && table && isAvailableTable(table.rightTable)) {
				input = getFirstInputInRow(table.rightTable.rows[row.rowIndex], 0);
			}
			return input;


			function getNextRow(input) {
				var row = getRow(input).nextSibling;
				return getNextNode(row); // for Moz

				function getNextNode(node) {
					if(node && node.nodeType && node.nodeType!=1) {
						node = node.nextSibling;
					}
					return node;
				}
			}
		}

	}


	/**
	 * セルの input 部分にカーソルがセットされたときに呼び出すための
	 * イベントハンドラ。
	 */
	Celcel.input_onfocus = function(event) {
		event = IWidget.editEvent(event);
		var input = event.target;

		input.select();

		if(input.readOnly && input.createTextRange) {
			var range = input.createTextRange();
			if(range) {
				range.move("character", 0);
				range.select();
			}
		}
	}

	/**
	 * セルの input 部分がクリックされたときに呼び出すための
	 * イベントハンドラ。
	 */
	Celcel.input_onclick = function(event) {
		var handler = selectCell(event);
//		if(handler.onclick) {
//			handler.onclick(event);
//		}
		event = IWidget.editEvent(event);
		var input = event.target;
		click(input);
	}

	/**
	 * セルの input 部分または td, th 部分上にマウスが載った場合に
	 * 呼び出すためのイベントハンドラ。
	 */
	Celcel.cell_onmouseover = function(event) {
		event = IWidget.editEvent(event);
		var table = getTable(event.target);
		if(table.handler.autoSelection) {
			selectCell(event);
		}
	}

	/**
	 * ヘッダがクリックされたときに呼び出すためのイベントハンドラ。
	 */
	Celcel.header_onclick = function(event) {
		event = IWidget.editEvent(event);
		var table = getTable(event.target);

		if(!event.ctrlKey && !event.shiftKey) {
			clearAllSelections(table);
		}

		selectAll(table);
		selectAll(table.rightTable);
		selectAll(table.lowerTable);
		selectAll(table.crossTable);

		function selectAll(table) {
			if(updateCellSelection(getFirstInput(table))) {
				updateCellSelection(getLastInput(table));
			}
		}

		function getFirstInput(table) {
			if(table.firstInput) {
				return table.firstInput;
			}
			for(var i=0; i < table.rows.length; i++) {
				var input = getFirstInputInRow(table.rows[i], 0);
				if(input) {
					table.firstInput = input;
					return input;
				}
			}
			return null;
		}

		function getLastInput(table) {
			if(table.lastInput) {
				return table.lastInput;
			}
			for(var i=table.rows.length-1; i >= 0; i--) {
				var input = getLastInputInRow(table.rows[i], table.rows[i].cells.length-1);
				if(input) {
					table.lastInput = input;
					return input;
				}
			}
			return null;
		}

	}

	/**
	 * 行ヘッダ部分がクリックされたときに呼び出すためのイベントハンドラ。
	 */
	Celcel.rowHeader_onclick = function(event) {
		event = IWidget.editEvent(event);
		var input = event.target;
		var table = getTable(event.target);

		if(table.handler.selectionMode=="column") {
			return;
		}

		if(!event.ctrlKey && !event.shiftKey) {
			clearAllSelections(table);
		}

		updateRowSelection(event, table, input);

		click(input);
	}


	/**
	 * 列ヘッダ部分がクリックされたときに呼び出すためのイベントハンドラ。
	 */
	Celcel.columnHeader_onclick = function(event) {
		event = IWidget.editEvent(event);
		var input = event.target;
		var table = getTable(input);

		if(table.handler.selectionMode=="row") {
			return;
		}

		if(!event.ctrlKey && !event.shiftKey) {
			clearAllSelections(table);
		}

		updateColumnSelection(event, table, input);

		click(input);
	}

	function click(cell) {
		if(handler.onclicked) {
			handler.onclicked(cell);
		}
		if(handler.requestOnclicked) {
			new CongaRequest().send(handler.requestOnclicked, 
					handler.selectionHolder.name + "=" + handler.selectionHolder.value);
		}
	}


	//// イベント定義

	/**
	 * 行・列選択時に呼び出されるイベントハンドラ。
	 */
	this.onselected = null;

	/**
	 * クリックに呼び出されるイベントハンドラ。
	 */
	this.onclicked = null;


	//// ローカルメソッド定義

	/**
	 * Celcel を初期化する。
	 * @param table Celcel化対象テーブル。もしくは明細テーブル。
	 * @param fixRows 固定表示行数。
	 * @param fixColumns 固定表示列数。
	 */
	function init(table, fixRows, fixColumns) {

		if(table.tagName=="TABLE") {
			container = table.parentNode;
			htmlElement = createHtmlElement(handler);
			container.appendChild(htmlElement);
		}
		else {
			container = table;
			htmlElement = createHtmlElement(handler, container);
			var tables = htmlElement.getElementsByTagName("table");
			if(tables.length==1) {
				table = tables.item(0);
			}
			else {
				for(var i=0; i<tables.length; i++) {
					var t = tables.item(i);
					if(t.id.match("-columnHeader$")) {
						columnHeader = t;
					}
					else if(t.id.match("-rowHeader$")) {
						rowHeader = t;
					}
					else if(t.id.match("-header$")) {
						header = t;
					}
					else {
						table = t;
					}
				}
			}
		}
		handler.htmlElement = htmlElement;

		headerDiv = createDiv(handler, htmlElement, header, "hidden");
		rowHeaderDiv = createDiv(handler, htmlElement, rowHeader, "hidden");
		columnHeaderDiv = createDiv(handler, htmlElement, columnHeader, "hidden");
		detailDiv = createDiv(handler, htmlElement, table, "scroll");

		header = headerDiv.firstChild;
		rowHeader = rowHeaderDiv.firstChild;
		columnHeader  = columnHeaderDiv.firstChild;
		detail = detailDiv.firstChild;

		menu = createContextMenu(htmlElement);

		if(!Celcel.prototype.SCROLL_MARGIN) {
			Celcel.prototype.SCROLL_MARGIN = 16;
		}

		if(!Celcel.prototype.BORDER_WIDTH) {
			Celcel.prototype.BORDER_WIDTH = 4;
		}

		if(!fixRows) {
			fixRows = 0;
		}
		if(!fixColumns) {
			fixColumns = 0;
		}

		fixRows = parseInt(fixRows);
		fixColumns = parseInt(fixColumns);

		// table 分割
		var divide = false;
		if((fixRows > 0 && columnHeader.getElementsByTagName("col").length==0) || 
				(fixColumns > 0 && rowHeader.getElementsByTagName("col").length==0)) {
			divide = true;
		}

		// table 分割
		if(divide) {
			var colList = table.getElementsByTagName("col");
			addColGroup(header, colList, 0, fixColumns);
			addColGroup(rowHeader, colList, 0, fixColumns);
			addColGroup(columnHeader, colList, fixColumns, colList.length);
			removeColGroup(colList, fixColumns);
			divideTable(fixRows, fixColumns, header, rowHeader, columnHeader, detail);
		}
		else if(!hasEventHandler()) {
			initEvents(header, Celcel.header_onclick);
			initEvents(rowHeader, Celcel.rowHeader_onclick);
			initEvents(columnHeader, Celcel.columnHeader_onclick);
			initEvents(detail, null);
		}


		// ヘッダ設定
	//	initColumBorder(header, rowHeader, columnHeaderDiv);
	//	initColumBorder(rowHeader, header, detailDiv);
		initColumBorder(columnHeader, detail, null);
		initColumBorder(detail, columnHeader, null);

		// 位置設定
		headerDiv.style.position = "absolute";
		rowHeaderDiv.style.position = "absolute";
		columnHeaderDiv.style.position = "absolute";
		detailDiv.style.position = "absolute";

		headerDiv.style.top = 0;
		headerDiv.style.left = 0;

		columnHeaderDiv.style.top = 0;

		rowHeaderDiv.style.top = headerDiv.scrollHeight;
		detailDiv.style.top = columnHeaderDiv.scrollHeight;

		columnHeaderDiv.style.left = headerDiv.scrollWidth;
		detailDiv.style.left = rowHeaderDiv.scrollWidth;

		detailDiv.onscroll = function() {
			columnHeaderDiv.scrollLeft = detailDiv.scrollLeft;
			rowHeaderDiv.scrollTop = detailDiv.scrollTop;
		}
		rowHeaderDiv.onscroll = function() {
			detailDiv.scrollTop = rowHeaderDiv.scrollTop;
		}
		columnHeaderDiv.onscroll = function() {
			detailDiv.scrollLeft = columnHeaderDiv.scrollLeft;
		}

		// 相互参照作成
		header.rightTable = columnHeader;
		header.lowerTable = rowHeader;
		header.crossTable = detail;

		rowHeader.upperTable = header;
		rowHeader.rightTable = detail;
		rowHeader.crossTable = columnHeader;

		columnHeader.leftTable = header;
		columnHeader.lowerTable = detail;
		columnHeader.crossTable = rowHeader;

		detail.leftTable = rowHeader;
		detail.upperTable = columnHeader;
		detail.crossTable = header;

		header.menu = menu;
		rowHeader.menu = menu;
		columnHeader.menu = menu;
		detail.menu = menu;

		handler.setSize(detailDiv.scrollWidth + 
						rowHeaderDiv.scrollWidth + getScrollMargin(detailDiv) + 3, 
						detailDiv.scrollHeight + 
						columnHeaderDiv.scrollHeight + getScrollMargin(detailDiv) + 4);

		handler.selectionHolder = IWidget.getValueHolder(htmlElement, "table");

		detailDiv.style.overflow = "auto";
		htmlElement.style.width = "100%";
		htmlElement.parentNode.onresize = function(){
			if(handler.packed) {
				handler.packed = false;
				return;
			}
			handler.pack();
			handler.packed = true;
		}

		/**
		 * colgroup を追加する。
		 */
		function addColGroup(table, colList, startCol, colCount) {
			var colgroup = document.createElement("colgroup");
			table.appendChild(colgroup);

			for(var i=startCol; i < colCount; i++) {
				var col = document.createElement("col");
				col.style.width = colList.item(i).style.width;
				colgroup.appendChild(col);
			}
		}

		/**
		 * colgroup を削除する。
		 */
		function removeColGroup(colList, startCol) {
			for(var i=startCol-1; i >= 0; i--) {
				var parent = colList.item(i).parentNode;
				parent.removeChild(colList.item(i));
			}
		}

		/**
		 * テーブル分割を行なう。
		 */
		function divideTable(fixRows, fixColumns, header, rowHeader, columnHeader, table) {

			var rowLength = table.rows.length;
			var rowIndex = 0;
			for(var i=0; i < rowLength; i++) {
				var headerTr = null;
				var columnHeaderTr = null;
				var rowHeaderTr = null;

				var row = table.rows[rowIndex];
				var rowHeight = row.scrollHeight;

				if(i < fixRows) {
					headerTr = createTr(header, rowHeight);
					columnHeaderTr = createTr(columnHeader, rowHeight);
				}
				else {
					if(fixColumns > 0) {
						rowHeaderTr = createTr(rowHeader, rowHeight);
					}
					rowIndex++;
				}

				var colLength = row.cells.length;
				var columnInex = 0;
				for(var j=0; j < colLength; j++) {
					var cell = row.cells[columnInex];
					if(j < fixColumns) {
						if(headerTr) {
							moveChild(row, headerTr, cell);
							
							if(!getInput(cell)) {
								if(i==0) {
									if(j==0) {
										cell.onclick = Celcel.header_onclick;
									}
									else {
										cell.onclick = Celcel.columnHeader_onclick;
									}
								}
								else {
									if(j==0) {
										cell.onclick = Celcel.rowHeader_onclick;
									}
								}
							}
						}
						else if(rowHeaderTr) {
							moveChild(row, rowHeaderTr, cell);
							if(j==0 && !getInput(cell)) {
								cell.onclick = Celcel.rowHeader_onclick;
							}
						}
						else {
							// detail にそのまま残す。
							columnInex++;
						}
					}
					else {
						if(columnHeaderTr) {
							moveChild(row, columnHeaderTr, cell);
							if(i==0 && !getInput(cell)) {
								cell.onclick = Celcel.columnHeader_onclick;
							}
						}
						else {
							// detail にそのまま残す。
							columnInex++;
						}
					}
					setEventHandlers(cell);
				}
				if(i < fixRows) {
					table.getElementsByTagName("tbody").item(0).removeChild(row);
				}
			}

			/**
			 * セルを別テーブルに移動する。
			 */
			function moveChild(src, dst, child) {
				src.removeChild(child);
				dst.appendChild(child);
			}

			/**
			 * tr 要素を作成する。
			 */
			function createTr(table, rowHeight) {
				var tr = document.createElement("tr");
				tr.style.height = rowHeight;
				table.getElementsByTagName("tbody").item(0).appendChild(tr);
				return tr;
			}

		}

		/**
		 * 列境界線を初期化する。
		 */
		function initColumBorder(table, friend, rightDiv) {
			var HELF_BORDER_WIDTH = Celcel.prototype.BORDER_WIDTH/2;

			table.columnBorder = new Array();

			var border = table.border;
			var colRight = parseInt(border);

			var colList = table.getElementsByTagName("col");
			for(var i = 0; i < colList.length; i++) {
				var col = colList.item(i);
				col.leftPos = colRight;

				col.rightCols = new Array();
				for(var j = i+1; j< colList.length; j++) {
					col.rightCols[j-i-1] = colList.item(j);
				}

				colRight = col.leftPos + col.scrollWidth;

				var b = new IBorder(table.parentNode, "vertical", 
						table.scrollHeight, Celcel.prototype.BORDER_WIDTH);
				table.columnBorder[i] = b;
				b.col = col;
				b.index = i;
				if(rightDiv && (i + 1 == colList.length)) {
					b.rightDiv = rightDiv;
					col.rightDiv = rightDiv;
				}
				b.setPosition(colRight-HELF_BORDER_WIDTH, 0);

				col.border = b.htmlElement;
				col.colRight = colRight;

				b.onmove = function(left, top) {
					onmove0(this, left, top);
					friend.columnBorder[this.index].col.border.style.left = left;
					onmove0(friend.columnBorder[this.index], left, top);
				}
			}

			function onmove0(border, left, top) {
				left = left + Celcel.prototype.BORDER_WIDTH/2;

				var currentWidth = border.col.scrollWidth;
				var newWidth = left - border.col.leftPos;
				if(newWidth < 0) {
					border.col.border.style.left = border.col.colRight;
					return;
				}

				border.col.style.width = newWidth;

				// 列幅が縮まらなかった場合の補正
				if(border.col.scrollWidth > newWidth) {
					newWidth = border.col.scrollWidth;
					border.col.style.width = newWidth;
					left = border.col.leftPos + newWidth;
					border.col.border.style.left = left;
				}

				border.col.colRight = left;
				var diff = border.col.scrollWidth - currentWidth;
				for(var i=0; i<border.col.rightCols.length; i++) {
					border.col.rightCols[i].leftPos =
						border.col.rightCols[i].leftPos + diff;
					border.col.rightCols[i].border.style.left =
						parseInt(border.col.rightCols[i].border.style.left) + diff;
					if(border.col.rightCols[i].rightDiv) {
						border.col.rightCols[i].rightDiv.style.left = 
								border.col.rightCols[i].border.style.left;
					}
				}

				if(border.rightDiv) {
					border.rightDiv.style.left = left;
				}
			}
		}

		/**
		 * 各セルに対してイベントハンドラ設定を行なう。
		 */
		function initEvents(table, onclickHandler) {
			var rowLength = table.rows.length;
			for(var i=0; i < rowLength; i++) {
				var row = table.rows[i];

				var colLength = row.cells.length;
				for(var j=0; j < colLength; j++) {
					var cell = row.cells[j];
					if(onclickHandler) {
						cell.onclick = onclickHandler;
					}
					setEventHandlers(cell);
				}
			}
		}

		/**
		 * イベントハンドラが設定済みかどうかを調べる。
		 * @return イベントハンドラが設定済みの場合は true。
		 */
		function hasEventHandler() {
			if(detail && detail.rows.length > 0 && detail.rows[0].cells.length > 0) {
				return detail.rows[0].cells[0].onclick!=null;
			}
			return false;
		}

	}

	/**
	 * Celcel の HTML要素を作成する。
	 * @param handler Celcelインスタンス。
	 * @return HTML要素。
	 */
	function createHtmlElement(handler, container) {
		var htmlElement;
		if(container) {
			htmlElement = container;
		}
		else {
			htmlElement = document.createElement("div");
		}
		htmlElement.className = "celcel";
		htmlElement.style.position = "relative";
		htmlElement.handler = handler;
		return htmlElement;
	}

	/**
	 * Celcel 用のコンテキストメニューを作成する。
	 */
	function createContextMenu(htmlElement) {
		var menu = new IMenu();
		menu.addTarget(htmlElement);
		menu.addItem("copyAll", function(){handler.copyAll();});
		menu.addItem("copy", function(){handler.copy();});
		menu.addItem("cut", function(){handler.cut();});
		menu.addItem("paste", function(){handler.paste();});
		menu.addItem("clear", function(){handler.clear();});
		return menu;
	}

	/**
	 * テーブル格納用の div を作成する。
	 */
	function createDiv(handler, htmlElement, table, overflow) {
		var div = document.createElement("div");
		div.className = "table-div";
		htmlElement.appendChild(div);
		div.style.display = "inline";
		div.style.overflow = overflow;

		if(table) {
			table.parentNode.removeChild(table);
		}
		else {
			table = document.createElement("table");
			table.appendChild(document.createElement("tbody"));
		}

		table.style.borderCollapse = "collapse";
		table.style.position = "relative";
	    table.style.tableLayout = "fixed";

		div.appendChild(table);
		table.border = "1";
		table.handler = handler;

		return div;
	}

	/**
	 * スクロールバーの幅を取得する。
	 */
	function getScrollMargin(detailDiv) {
		if(detailDiv.style.overflow=="scroll") {
			return Celcel.prototype.SCROLL_MARGIN;
		}
		else {
			return 0;
		}
	}

	/**
	 * イベントハンドラの設定を行なう。
	 */
	function setEventHandlers(cell) {
		var input = getInput(cell);
		if(input) {
			input.onkeydown = Celcel.input_onkeydown;
			input.onfocus = Celcel.input_onfocus;
			input.onclick = Celcel.input_onclick;
			input.onmouseover = Celcel.cell_onmouseover;
		}
		cell.onmouseover = Celcel.cell_onmouseover;
	}

	/**
	 * 指定された input に focus をセットする。
	 */
	function setFocus(input) {
		if(input && input.focus) {
			input.focus();
		}
	}

	/**
	 * セルの選択を行なう。
	 */
	function selectCell(event) {
		event = IWidget.editEvent(event);
		var input = event.target;
		var table = getTable(input);

		if(!event.ctrlKey && !event.shiftKey) {
			clearAllSelections(table);
		}
		updateSelection(event, table, input);
		return table.handler;
	}

	/**
	 * 選択状態を更新する。
	 */
	function updateSelection(event, table, input) {
		if(table.handler.selectionMode=="row") {
			updateRowSelection(event, table, input);
		}
		else if(table.handler.selectionMode=="column") {
			updateColumnSelection(event, table, input);
		}
		else { // (table.handler.selectionMode=="cell")
			updateCellSelection(input);
		}
	}

	/**
	 * 行選択状態を更新する。
	 */
	function updateRowSelection(event, table, input) {
		var row;
		if(input.nodeName=="INPUT") {
			row = getRow(input);
		}
		else {
			row = input.parentNode;
		}
		if(isAvailableTable(table.leftTable)) {
			selectRow(table.leftTable, table.leftTable.rows[row.rowIndex]);
		}
		selectRow(table, row);
		if(isAvailableTable(table.rightTable)) {
			selectRow(table.rightTable, table.rightTable.rows[row.rowIndex]);
		}

		if(rowHeader.rows.length>0) {
			select(rowHeader.rows[row.rowIndex].cells[0], false);
		}
		else {
			handler.selectionHolder.value = row.rowIndex.toString();
			execute(row.cells[0]);
		}

		function selectRow(table, row) {
			if(updateCellSelection(getFirstInputInRow(row, 0))) {
				updateCellSelection(getLastInputInRow(row, row.cells.length-1));
			}
		}
	}

	/**
	 * 列選択状態を更新する。
	 */
	function updateColumnSelection(event, table, input) {
		var cell;
		if(input.nodeName=="INPUT") {
			cell = getCell(input);
		}
		else {
			cell = input;
		}
		var index = cell.cellIndex;
		if(isAvailableTable(table.upperTable)) {
			selectColumn(table.upperTable, index);
		}
		selectColumn(table, index);
		if(isAvailableTable(table.lowerTable)) {
			selectColumn(table.lowerTable, index);
		}

		function selectColumn(table, index) {
			if(updateCellSelection(getFirstInputInColumn(table, index, 0))) {
				updateCellSelection(getLastInputInColumn(table, index, table.rows.length-1));
			}
			else if(table.rows.length > 0 && table.rows[0].cells.length > index) {
				select(table.rows[0].cells[index], true);
			}
		}
	}

	function select(headerCell, isColumnSelection) {
		if(!headerCell || !headerCell.id) {
			return;
		}

		var index = headerCell.id.indexOf("-");
		if(index==-1) {
			return;
		}

		if(isColumnSelection) {
			handler.selectionHolder.value = "," + headerCell.id.substring(index+1);
		}
		else {
			handler.selectionHolder.value = headerCell.id.substring(index+1);
		}

		execute(headerCell);
	}

	function execute(cell) {
		if(handler.onselected) {
			handler.onselected(cell);
		}
		if(handler.requestOnselected) {
			new CongaRequest().send(handler.requestOnselected, 
					handler.selectionHolder.name + "=" + handler.selectionHolder.value);
		}
	}

	/**
	 * 全てのテーブルの選択状態をクリアする。
	 */
	function clearAllSelections(table) {
		table.handler.selectionStartingTable = null;
		clearSelection(table);
		if(table.upperTable) {
			clearSelection(table.upperTable);
		}
		if(table.lowerTable) {
			clearSelection(table.lowerTable);
		}
		if(table.leftTable) {
			clearSelection(table.leftTable);
		}
		if(table.rightTable) {
			clearSelection(table.rightTable);
		}
		if(table.crossTable) {
			clearSelection(table.crossTable);
		}
	}

	/**
	 * セルの選択状態を更新する。
	 */
	function updateCellSelection(input) {
		if(!input) {
			return false;
		}

		var table = getTable(input);
		var rowIndex = getRow(input).rowIndex;
		var columnIndex = getCell(input).cellIndex;

		updateSelectionForEndingTable(table, rowIndex, columnIndex);
		updateSelectionForOtherTable(table, rowIndex, columnIndex);

//		handler.selectionHolder.value = たぶんカーソル移動が遅くなるので今はやらない


		return true;

		/**
		 * 他テーブルの選択状態を更新する。
		 */
		function updateSelectionForOtherTable(table, rowIndex, columnIndex) {

			if(table.handler.selectionStartingTable==table) { // 全てクリア
				clearSelection(table.leftTable);
				clearSelection(table.rightTable);
				clearSelection(table.lowerTable);
				clearSelection(table.upperTable);
				clearSelection(table.crossTable);
			}
			else if(table.handler.selectionStartingTable==table.leftTable) {
				clearSelection(table.lowerTable);
				clearSelection(table.upperTable);
				clearSelection(table.crossTable);
				updateSelectionForEndingTable(table.leftTable, 
					rowIndex, table.leftTable.rows[0].cells.length - 1);
			}
			else if(table.handler.selectionStartingTable==table.rightTable) {
				clearSelection(table.lowerTable);
				clearSelection(table.upperTable);
				clearSelection(table.crossTable);
				updateSelectionForEndingTable(table.rightTable, rowIndex, 0);
			}
			else if(table.handler.selectionStartingTable==table.upperTable) {
				clearSelection(table.leftTable);
				clearSelection(table.rightTable);
				clearSelection(table.crossTable);
				updateSelectionForEndingTable(table.upperTable, 
					table.upperTable.rows.length - 1, columnIndex);
			}
			else if(table.handler.selectionStartingTable==table.lowerTable) {
				clearSelection(table.leftTable);
				clearSelection(table.rightTable);
				clearSelection(table.crossTable);
				updateSelectionForEndingTable(table.lowerTable, 0, columnIndex);
			}
			else if(table.handler.selectionStartingTable==table.crossTable) {
				var columnIndexForCross;
				var rowIndexForCross;
				if(table.leftTable) {
					columnIndexForCross = table.leftTable.rows[0].cells.length - 1;
					updateSelectionForEndingTable(table.leftTable, rowIndex, columnIndexForCross);
				}
				if(table.rightTable) {
					columnIndexForCross = 0;
					updateSelectionForEndingTable(table.rightTable, rowIndex, columnIndexForCross);
				}
				if(table.upperTable) {
					rowIndexForCross = table.upperTable.rows.length - 1;
					updateSelectionForEndingTable(table.upperTable, rowIndexForCross, columnIndex);
				}
				if(table.lowerTable) {
					rowIndexForCross = 0;
					updateSelectionForEndingTable(table.lowerTable, rowIndexForCross, columnIndex);
				}
				updateSelectionForEndingTable(table.crossTable, rowIndexForCross, columnIndexForCross);
			}
		}
	}

	/**
	 * 選択終了ポイントのテーブルの選択状態を更新する。
	 */
	function updateSelectionForEndingTable(table, rowIndex, columnIndex) {

		if(!table.selection) {
			initSelectionStartIndex(table, rowIndex, columnIndex);

			table.selectionMinRowIndex = table.selectionStartRowIndex;
			table.selectionMaxRowIndex = table.selectionStartRowIndex;

			table.selectionMinColumnIndex = table.selectionStartColumnIndex
			table.selectionMaxColumnIndex = table.selectionStartColumnIndex

			if(!table.handler.selectionStartingTable) {
				table.handler.selectionStartingTable = table;
			}
		}
		else {
			if(rowIndex == table.selectionMinRowIndex || rowIndex == table.selectionMaxRowIndex) {
				if(columnIndex < table.selectionMinColumnIndex) {
					addSelection(table, 
						table.selectionMinRowIndex, table.selectionMaxRowIndex,
						columnIndex, table.selectionMinColumnIndex - 1
					);
					table.selectionMinColumnIndex = columnIndex;
					return;
				}
				else if(columnIndex > table.selectionMaxColumnIndex) {
					addSelection(table, 
						table.selectionMinRowIndex, table.selectionMaxRowIndex,
						table.selectionMinColumnIndex + 1, columnIndex
					);
					table.selectionMaxColumnIndex = columnIndex;
					return;
				}
			}

			if(columnIndex == table.selectionMinColumnIndex || columnIndex == table.selectionMaxColumnIndex) {
				if(rowIndex < table.selectionMinRowIndex) {
					addSelection(table, 
						rowIndex, table.selectionMinRowIndex - 1,
						table.selectionMinColumnIndex, table.selectionMaxColumnIndex
					);
					table.selectionMinRowIndex = rowIndex;
					return;
				}
				else if(rowIndex > table.selectionMaxRowIndex) {
					addSelection(table, 
						table.selectionMaxRowIndex + 1, rowIndex,
						table.selectionMinColumnIndex, table.selectionMaxColumnIndex
					);
					table.selectionMaxRowIndex = rowIndex;
					return;
				}
			}
		}


		var maxRowIndex = Math.max(table.selectionStartRowIndex, rowIndex);
		var minRowIndex = Math.min(table.selectionStartRowIndex, rowIndex);

		var maxColumnIndex = Math.max(table.selectionStartColumnIndex, columnIndex);
		var minColumnIndex = Math.min(table.selectionStartColumnIndex, columnIndex);

		clearSelection(table);
		table.selection = new Array();

		addSelection(table, minRowIndex, maxRowIndex, minColumnIndex, maxColumnIndex);

		table.selectionMinRowIndex = minRowIndex;
		table.selectionMinColumnIndex = minColumnIndex;
		table.selectionMaxRowIndex = maxRowIndex;
		table.selectionMaxColumnIndex = maxColumnIndex;


		function initSelectionStartIndex(table, rowIndex, columnIndex) {

			if(!table.handler.selectionStartingTable || 
					table.handler.selectionStartingTable==table) {
				table.selectionStartRowIndex = rowIndex;
				table.selectionStartColumnIndex = columnIndex;
				return;
			}

			if(table.handler.selectionStartingTable==table.crossTable) {
				if(table.lowerTable) {
					table.selectionStartRowIndex = table.rows.length - 1;
				}
				else {
					table.selectionStartRowIndex = 0;
				}

				if(table.rightTable) {
					table.selectionStartColumnIndex = table.rows[0].cells.length - 1;
				}
				else {
					table.selectionStartColumnIndex = 0;
				}
			}
			else if(table.handler.selectionStartingTable==table.lowerTable) {
				table.selectionStartRowIndex = table.rows.length - 1;
				table.selectionStartColumnIndex = table.lowerTable.selectionStartColumnIndex;
			}
			else if(table.handler.selectionStartingTable==table.upperTable) {
				table.selectionStartRowIndex = 0;
				table.selectionStartColumnIndex = table.upperTable.selectionStartColumnIndex;
			}
			else if(table.handler.selectionStartingTable==table.rightTable) {
				table.selectionStartRowIndex = table.rightTable.selectionStartRowIndex;
				table.selectionStartColumnIndex = table.rows[0].cells.length - 1;
			}
			else if(table.handler.selectionStartingTable==table.leftTable) {
				table.selectionStartRowIndex = table.leftTable.selectionStartRowIndex;
				table.selectionStartColumnIndex = 0;
			}
		}
	}

	/**
	 * 指定されたテーブルの選択状態をクリアする。
	 */
	function clearSelection(table) {
		if(!table) {
			return;
		}

		if(table.selection) {
			for(var i=0; i<table.selection.length; i++) {
				var input = table.selection[i];
				getCell(input).style.backgroundColor = "white";
				input.style.backgroundColor = "white";
			}
		}
		table.selection = null;
	}

	/**
	 * 選択範囲を追加する。
	 */
	function addSelection(table, startRowIndex, endRowIndex, startColumnIndex, endColumnIndex) {
		if(!table.selection) {
			table.selection = new Array();
		}

		for(var i=startRowIndex; i<=endRowIndex; i++) {
			for(var j=startColumnIndex; j<=endColumnIndex; j++) {
				var cell = table.rows[i].cells[j];
				var input = getInput(cell);
				if(input) {
					input.style.backgroundColor = "#ccccff";
					table.selection.push(input);
					cell.style.backgroundColor = "#ccccff";
				}
			}
		}
	}

	/**
	 * 列の中で最後の input を取得する。
	 */
	function getLastInputInColumn(table, columnIndex, rowIndex) {
		for(var i=rowIndex; i >= 0; i--) {
			var input = getInput(table.rows[i].cells[columnIndex]);
			if(input) {
				return input;
			}
		}
		return null;
	}

	/**
	 * 行の中で最後の input を取得する。
	 */
	function getLastInputInRow(row, index) {
		for(var i=index; i >= 0; i--) {
			var input = getInput(row.cells[i]);
			if(input) {
				return input;
			}
		}
		return null;
	}

	
	/**
	 * 列の中で最初の input を取得する。
	 */
	function getFirstInputInColumn(table, columnIndex, rowIndex) {
		for(var i=rowIndex; i < table.rows.length; i++) {
			var input = getInput(table.rows[i].cells[columnIndex]);
			if(input) {
				return input;
			}
		}
		return null;
	}

	/**
	 * 行の中で最初の input を取得する。
	 */
	function getFirstInputInRow(row, index) {
		for(var i=index; i < row.cells.length; i++) {
			var input = getInput(row.cells[i]);
			if(input) {
				return input;
			}
		}
		return null;
	}

	/**
	 * 指定された input をもつ行を取得する。
	 */
	function getRow(input) {
		return getCell(input).parentNode;
	}

	/**
	 * 指定された input をもつセルを取得する。
	 */
	function getCell(input) {
		return input.parentNode;
	}

	/**
	 * 指定されたセルの中の input を取得する。
	 */
	function getInput(cell) {
		if(!cell) {
			return null;
		}

		var children = cell.childNodes;

		for(var j=0; j < children.length; j++) {
			var child = children.item(j);
			if(child.nodeName!="INPUT") {
				continue;
			}
			return child;
		}
		return null;
	}

	/**
	 * 指定されたオブジェクトの親要素から table を検索して返す。
	 */
	function getTable(obj) {
		while(obj) {
			if(obj.nodeName=="TABLE") {
				return obj;
			}
			obj = obj.parentNode;
		}
	}

	/**
	 * 指定されたテーブルが利用可能かどうかを調べる。
	 */
	function isAvailableTable(table) {
		return table && table.rows.length > 0;
	}


	//// インスタンスの初期化
	init(table, fixRows, fixColumns, header, rowHeader, columnHeader);
}


/*
 *
 * CongaRequest オブジェクト
 *
 *   XmlHttpRequest を使ってリクエスト送信を行い，
 *   受信したXMLデータを handleResponse メソッドによって解釈し，
 *   ドキュメントの更新を行なう。
 *   handleResponse 関数は，new CongaRequest().parse() メソッドを呼び出す。
 *
 *  コンストラクタ：
 *     CongaRequest
 *      @param method リクエスト時の method（POST または GET）を指定する。
 *                    指定がない場合は POST が設定される。
 *      @param requestForm リクエストデータ構築のための form を指定する。
 *                         指定がない場合は document.forms[0] が設定される。
 *
 *  メソッド:
 *     send(url, requestData) リクエスト送信を行なう。
 *      @param url         リクエスト送信先 URL。指定必須。
 *      @param requestData リクエスト対象データ。リクエスト対象データ文字列
 *                         を返す function かリクエスト対象データ文字列のど
 *                         ちらかが指定できる。
 *                         この引数を指定しなかった場合，コンストラクタの 
 *                         requestForm 引数で指定した form の内容が送信対象
 *                         データとして使用される。
 *
 *  プロパティ ( [ ]内はデフォルト値 ) :
 *     method リクエスト時のメソッド [コンストラクタ引数 method で指定された値]
 *
 *     contentType リクエスト時の ContentType [application/x-www-form-urlencoded;charset=UTF-8]
 *
 *     async 非同期送信するかどうか [false]
 *
 *     requestForm [コンストラクタ引数 requestForm で指定された値]
 *
 */


/**
 * CongaRequest を作成する。
 *
 * @param method リクエスト時の method（POST または GET）を指定する。
 *               指定がない場合は POST が設定される。
 * @param requestForm リクエストデータ構築のための form を指定する。
 *                    指定がない場合は document.forms[0] が設定される。
 */
function CongaRequest(method, requestForm) {
	//// ローカル変数定義
	var handler = this;


	//// プロパティ定義

	/**
	 * リクエスト時の method。（[POST] / GET）
	 */
	this.method = (method)? method : "POST";

	/**
	 * リクエストデータ構築のための form。
	 */
	this.requestForm = (requestForm)? requestForm : document.forms[0];

	/**
	 * ContentType ["application/x-www-form-urlencoded;charset=UTF-8"]
	 */
	this.contentType = "application/x-www-form-urlencoded;charset=UTF-8";

	/**
	 * 非同期リクエストを行なうかどうか [false]
	 */
	this.async = false;


	//// public メソッド定義

	/**
	 * リクエスト送信を実行する。
	 * @param url リクエスト送信先 URL。指定必須。
	 * @param requestData リクエスト対象データ。リクエスト対象データ文字列
	 *                    を返す function かリクエスト対象データ文字列のど
	 *                    ちらかが指定できる。
	 *                    この引数を指定しなかった場合，コンストラクタの 
	 *                    requestForm 引数で指定した form の内容が送信対象
	 *                    データとして使用される。
	 * @param func 送受信完了後に実行するファンクション。
	 * @return 通信が正常終了した場合は true。それ以外は false。
	 *         非同期モードで通信した場合も false。
	 */
	this.send = function (url, requestData, func) {
		if(!requestData) {
			requestData = null;
		}

		var data = null;
		if(requestData instanceof Function) {
			data = requestData();
		}
		else if(requestData) {
			data = requestData;
		}
		else if(handler.requestForm) {
			data = getFormData(handler.requestForm);
		}

		data = "_nga.async=" + ((handler.async)?"true":"false") + "&" + data;

		var cover = null;
		if(!handler.async) {
			cover = IWidget.createDisplayCover();
		}

		var xmlreq = createXMLHttpRequest();
		var handleResponse = handler.handleResponse;

		if(handler.async) {
			xmlreq.onreadystatechange =function () {
				if(xmlreq.readyState == 4 ){
					handleResponse(xmlreq, func);
				}
			}
		}

		xmlreq.open(handler.method, url, handler.async);
		xmlreq.setRequestHeader('Content-Type',handler.contentType);
		xmlreq.send(data);

		if(!handler.async) {
			if(cover.parentNode) {
				cover.parentNode.removeChild(cover);
			}
			return handleResponse(xmlreq, func);
		}

		return false;

		/**
		 * XmlHttpRequest オブジェクトを作成する。
		 * @return XmlHttpRequest オブジェクト。
		 */
		function createXMLHttpRequest() {
			if(window.ActiveXObject) {
				try {
					return new ActiveXObject("Msxml2.XMLHTTP");
				}
				catch(e) {
					try {
						return new ActiveXObject("Microsoft.XMLHTTP");
					}
					catch (e2) {
						return null;
					}
				}
			}
			else if(window.XMLHttpRequest) {
				var xmlhttp = new XMLHttpRequest();
				return xmlhttp;

			}
			else {
				return null;
			}
		}

		/**
		 * 引数で指定された form オブジェクトから送信対象データ文字列を作成する。
		 * @param form 送信対象 form。
		 * @return 送信対象データ文字列。
		 */
		function getFormData(form) {
			var arr = new Array();
			var index = 0;
			var length = form.elements.length;
			for(var i = 0; i < length; i++) {
				var elem = form.elements[i];
				if(!elem.name) {
					continue;
				}

				if(elem.checked!=undefined) {
					if(elem.type=="radio" && !elem.checked) {
						continue;
					}
				}

				if(index > 0) {
					arr[index++] = "&";
				}
				arr[index++] = elem.name;
				arr[index++] = "=";
				arr[index++] = encodeURIComponent(elem.value);
			}
			return arr.join("");
		}
	}


	/**
	 * 受信データを表示する。
	 * @param xmlreq XMLHttpRequest オブジェクト。
	 * @param func 送受信完了後に実行するファンクション。
	 * @return 通信が正常終了した場合は true。それ以外は false。
	 */
	this.handleResponse = function(xmlreq, func) {
		if(xmlreq.status!=200 || xmlreq.getResponseHeader("Content-Type").indexOf("text/xml")!=0) {
			w = window.open("", "", "width=600,height=400,top=0,left=0,resizable=yes,scrollbars=yes,status=no");
			w.document.write(xmlreq.responseText);
			w.document.close();
			return false;
		}

		new CongaResponse().parse(xmlreq.responseXML);

		if(func) {
			func();
		}
		return true;
	}

}
/**
 * CongaResponse を作成する。
 * 
 * CongaRequest から呼び出され，XML の解釈を行なう。
 * 
 *  解析できる XML の形式:
 *    タグは2種類。ひとつの c (container) タグの中に 複数の e (element)タグ
 *    または c タグを記述する。
 *    e タグの中に e タグや c タグを記述することはできない。
 *    例：
 *      <c>
 *        <e>...</e>
 *        <e>...</e>
 *        <c>
 *          <e>...</e>
 *          <e>...</e>
 *        </c>
 *      </c>
 *
 *    最上位の c タグは parse 関数が解釈し，下層の e, c タグの内容解釈は handler 関数が解釈する。
 *    parse も handler も利用者により設定可能なので
 *    タグの属性は任意設定可能だが，標準的には以下のものがある。
 *
 *    最上位の c タグの属性 : 
 *       is-error エラーが存在するかどうかを示す。
 *       message エラーメッセージ
 *
 *    下層の e, c タグの属性
 *       type 受信データのタイプ。handler 関数のセレクタとなる。
 *       id 値を設定する対象となるオブジェクトの id。
 *       is-error エラーデータかどうかを示す。
 * 
 *  CongaResponse.parse メソッドでは，上記のタグ内容を handler が容易に利用できるように
 *  以下のプロパティとして展開する。
 *    id - 下層の e, c タグの id 属性で指定された値。
 *    type - 下層の e, c タグの type 属性で指定された値。
 *    value - 下層の e, c タグ内のテキスト値。
 *    errore - 下層の e, c タグの is-error 属性に "true" が設定されている場合は true，
 *             それ以外の場合は false。
 *    errorc - 最上位の c タグの is-error 属性に "true" が設定されている場合は true，
 *             それ以外の場合は false。
 *    etag - 下層の e, c タグノードオブジェクト。
 *
 *     handler type 毎の handler 関数の連想配列
 *        デフォルト設定:
 *          CongaResponse.handler["window"] = CongaResponse.setValue2Window;
 *          CongaResponse.handler["iwindow"] = CongaResponse.setValue2IWindow;
 *          CongaResponse.handler["celcel"] = CongaResponse.setValue2Celcel;
 *          CongaResponse.handler["tree"] = CongaResponse.setValue2ITree;
 *          CongaResponse.handler["tree-node"] = CongaResponse.setValue2ITreeNode;
 *          CongaResponse.handler["split-pane"] = CongaResponse.setValue2SplitPane;
 *          CongaResponse.handler["outer"] = CongaResponse.setValue4Outer;
 *          CongaResponse.handler["self"] = CongaResponse.setValue2Self;
 *          CongaResponse.handler["alert"] = CongaResponse.setValue4Alert;
 *          CongaResponse.handler["confirm"] = CongaResponse.setValue4Confirm;
 *          CongaResponse.handler["method"] = CongaResponse.setValue4Method;
 *          CongaResponse.handler["script"] = CongaResponse.setValue4Script;
 *          CongaResponse.handler["message"] = CongaResponse.showMessage;
 *          CongaResponse.handler["focus"] = CongaResponse.setFocus;
 *          CongaResponse.handler["label"] = CongaResponse.setLabel;
 *          CongaResponse.handler["default"] = CongaResponse.setValue4Default;
 *
 */
function CongaResponse() {

	var congaResponse = this;

	/**
	 * 下層の e, c タグの id 属性で指定された値。
	 */
	this.id;

	/**
	 * 下層の e, c タグの type 属性で指定された値。
	 */
	this.type;

	/**
	 * 下層の e, c タグ内のテキスト値。
	 */
	this.value;

	/**
	 * 下層の e, c タグの is-error 属性に "true" が設定されている場合は true，
	 * それ以外の場合は false。
	 */
	this.errore;

	/**
	 * 最上位の c タグの is-error 属性に "true" が設定されている場合は true，
	 * それ以外の場合は false。
	 */
	this.errorc;

	/**
	 * 下層の e, c タグノードオブジェクト。
	 */
	this.etag;

	/**
	 * XML ドキュメントの解析を行なう。
	 * @param cTagList c タグのリスト。
	 */
	this.parse = function(doc) {
		if(!doc) {
			doc = document.getElementsByTagName("xml").item(0);
		}

		var cTagList = doc.getElementsByTagName("c");
		for(var i=0; i<cTagList.length; i++) {
			var cTag = cTagList.item(i);
			parseCTag(cTag);
		}

		function parseCTag(cTag) {
			var message = null;
			var errorc = cTag.getAttribute("is-error")=="true";
			var list = cTag.childNodes;
			var length = list.length;
			var defaultHandler = CongaResponse.handler["default"];

			for(var i=0; i<length; i++) {
				var item = list.item(i);
				if(item.tagName!="e" && item.tagName!="c") {
					continue;
				}
				var id = item.getAttribute("id");
				var type = item.getAttribute("type");
				var msg = item.getAttribute("message");

				if(msg && !message) {
					message = msg;
				}

				var handler = CongaResponse.handler[type];
				if(!handler) {
					handler = defaultHandler;
				}

				var value = null;
				if(item.firstChild) {
					value = item.firstChild.nodeValue;
				}

				setup(id, type, value, item.getAttribute("is-error")=="true", errorc, item);
				handler(congaResponse);
			}

			if(!message) {
				message = cTag.getAttribute("message");
			}

			if(message) {
				var handler = CongaResponse.handler["message"];
				if(!handler) {
					handler = CongaResponse.showMessage;
				}
				setup(null, "message", message, false, errorc, null);
				handler(congaResponse);
			}

			function setup(id, type, value, errore, errorc, etag) {
				congaResponse.id = id;
				congaResponse.type = type;
				congaResponse.value = value;
				congaResponse.errore = errore;
				congaResponse.errorc = errorc;
				congaResponse.etag = etag;
			}
		}
	}




	/**
	 * type="window" で呼び出される handler。
	 * 新しいウインドウを開きその中に値を出力する。
	 */
	CongaResponse.setValue2Window = function(response) {
		var option = response.etag.getAttribute("option");
		if(!option) {
			option = "";
		}
		if(!response.id) {
			response.id = "window";
		}
		var w = window.open("about:blank", response.id, option)
		w.document.write(response.value);
		w.document.close();
	}

	/**
	 * type="iwindow" で呼び出される handler。
	 */
	CongaResponse.setValue2IWindow = function(response) {
		var win;
		var elem;
		if(response.id) {
			elem = document.getElementById(response.id);
		}

		if(elem && elem.handler) {
			win = elem.handler;
		}
		else {
			var modal = response.etag.getAttribute("modal")=="true";
			var parent = null;
			var parentId = response.etag.getAttribute("parent-id");
			if(parentId) {
				parent = document.getElementById(parentId);
			}
			win = new IWindow(null, null, false, parent, modal);

			if(response.id) {
				win.htmlElement.id = response.id;
			}
		}

		var title = response.etag.getAttribute("title");
		if(title) {
			win.setTitle(title);
		}

		var content = null;
		var contentId = response.etag.getAttribute("content-id");
		var contentByUrl = response.etag.getAttribute("content-by-url");
		if(contentId) {
			content = document.getElementById(contentId);
		}
		else if(contentByUrl) {
			content = contentByUrl;
		}
		else if(response.value) {
			content = response.value;
		}

		if(content) {
			if(contentByUrl) {
				win.setContentByURL(content);
				if(!title) {
					win.setTitle(content);
				}
			}
			else {
				win.setContent(content);
			}
		}

		if(response.etag.getAttribute("title-bar")=="false") {
			win.hideTitleBar();
		}
		if(response.etag.getAttribute("resizable")=="false") {
			win.setResizable(false);
		}

		var icon = response.etag.getAttribute("icon");
		if(icon) {
			win.setIcon(icon);
		}

		var ecb = response.etag.getAttribute("enable-close-button");
		if(ecb=="false") {
			win.setEnableCloseButton(false);
		}

		var emb = response.etag.getAttribute("enable-minimize-button");
		if(emb=="false") {
			win.setEnableMinimizeButton(false);
		}

		var scroll = response.etag.getAttribute("scroll");
		if(scroll!=null) {
			win.setScroll(scroll);
		}

		var bounds = response.etag.getAttribute("bounds");
		if(bounds) {
			eval("win.setBounds("+bounds+")");
		}

		var position = response.etag.getAttribute("position");
		if(position) {
			eval("win.setPosition("+position+")");
		}

		var size = response.etag.getAttribute("size");
		if(size) {
			eval("win.setSize("+size+")");
		}

		if(!bounds && !size) {
			win.pack();
		}

		if(!bounds && !position) {
			win.show(window.event);
		}
		else {
			win.show();
		}
	}

	/**
	 * type="tree" で呼び出される handler。
	 * 指定された値を id 引数で指定されたタグの innerHTML プロパティとして設定し，ITree に変換する。
	 */
	CongaResponse.setValue2ITree = function(response) {
		var elem = document.getElementById(response.id);
		if(elem && !response.errore) {
			if(response.value) {
				elem = updateHTMLElement(elem, response.id, response.value);
				var itree = new ITree(elem);
				itree.requestOnselected = response.etag.getAttribute("request-onselected");
				itree.requestOnopening = response.etag.getAttribute("request-onopening");
				itree.requestOnclosing = response.etag.getAttribute("request-onclosing");
				itree.requestOnloading = response.etag.getAttribute("request-onloading");
			}
			else {
				elem.innerHTML = "";
			}
		}
	}

	/**
	 * type="tree-node" で呼び出される handler。
	 * 指定された値を id 引数で指定されたタグの中の tree の中の
	 * node-id 属性で指定された id をもつノードを更新する。
	 */
	CongaResponse.setValue2ITreeNode = function(response) {
		var elem = document.getElementById(response.id);
		if(elem && !response.errore) {
			var tree = elem;
			var nodeId = response.etag.getAttribute("node-id");
			if(response.value && tree && tree.handler && nodeId) {
				var node = document.getElementById(nodeId);
				tree.handler.updateNode(node, response.value);
			}
		}
	}

	/**
	 * type="celcel" で呼び出される handler。
	 * 指定された値を id 引数で指定されたタグの innerHTML プロパティとして設定し，Celcel に変換する。
	 */
	CongaResponse.setValue2Celcel = function(response) {
		var elem = document.getElementById(response.id);
		if(elem && !response.errore) {
			if(response.value) {
				var oWidth = elem.offsetWidth;
				var oHeight = elem.offsetHeight;

				elem = updateHTMLElement(elem, response.id, response.value);

				var nWidth = elem.offsetWidth;
				var nHeight = elem.offsetHeight;

				var fixRows = response.etag.getAttribute("fix-rows");
				var fixCols = response.etag.getAttribute("fix-cols");

				var celcel = new Celcel(elem, fixRows, fixCols);

				var scroll = response.etag.getAttribute("scroll");
				if(scroll!=null) {
					celcel.setScroll(scroll!="false");
				}
				celcel.requestOnselected = response.etag.getAttribute("request-onselected");
				celcel.requestOnclicked = response.etag.getAttribute("request-onclicked");
				celcel.selectionMode = response.etag.getAttribute("selection-mode");
				celcel.autoSelection = response.etag.getAttribute("auto-selection")=="true";

				var height = response.etag.getAttribute("height");
				if(height) {
					celcel.setHeight(height);
				}
				else if(elem.offsetHeight < oHeight || oHeight < nHeight){
					celcel.setHeight(oHeight);
				}
				if(elem.offsetWidth < oWidth || oWidth < nWidth ) {
					celcel.setWidth(oWidth);
				}
			}
			else {
				elem.innerHTML = "";
			}
		}
	}

	/**
	 * type="split-pane" で呼び出される handler。
	 * id 引数で指定された HTML 要素で ISplitPane を呼び出す。
	 */
	CongaResponse.setValue2SplitPane = function(response) {
		var elem = document.getElementById(response.id);
		if(elem) {
			var orientation = response.etag.getAttribute("orientation");
			if(!orientation) {
				orientation = "v";
			}
			var split = new ISplitPane(elem, orientation);
			var dividerLocation = response.etag.getAttribute("divider-location");
			if(dividerLocation) {
				split.setDividerLocation(parseInt(dividerLocation));
			}
		}
	}

	/**
	 * type="outer" で呼び出される handler。
	 * id 引数で指定された HTML 要素を書き換える。
	 */
	CongaResponse.setValue4Outer = function(response) {
		var elem = document.getElementById(response.id);
		if(elem) {
			updateHTMLElement(elem, response.id, response.value);
		}
	}

	/**
	 * type="self" で呼び出される handler。
	 * 現在ウインドウに表示されている内容を指定された値で上書きする。
	 */
	CongaResponse.setValue2Self = function(response) {
		if(!response.errore) {
			document.clear();
			document.write(response.value);
			document.close();
		}
	}

	/**
	 * type="script" で呼び出される handler。
	 * 指定された値を javascript として実行する。
	 */
	CongaResponse.setValue4Script = function(response) {
		var command = response.etag.getAttribute("value");
		if(command) {
			eval(command);
		}
	}

	/**
	 * type="method" で呼び出される handler。
	 * value 属性で指定された値を id 引数で指定されたオブジェクトのメソッドとして実行する。
	 */
	CongaResponse.setValue4Method = function(response) {
		var elem = document.getElementById(response.id);
		var command = response.etag.getAttribute("value");
		if(elem && command) {
			if(elem.handler) {
				eval("elem.handler." + command);
			}
			else {
				eval("elem." + command);
			}
		}
	}

	/**
	 * type="alert" で呼び出される handler。
	 * value 属性で指定された文字列を alert 表示する。
	 * また，request 属性が指定されている場合，その属性の値を URL とみなして，リクエスト送信を行なう。
	 */
	CongaResponse.setValue4Alert = function(response) {
		var text = response.etag.getAttribute("value");
		var req = response.etag.getAttribute("request");
		var method = response.etag.getAttribute("method");
		if(!method) {
			method = "POST";
		}

		window.alert(text);
		if(req){
			new CongaResponse(method).send(req, "");
		}
	}

	/**
	 * type="confirm" で呼び出される handler。
	 * value 属性で指定された文字列を confirm 表示する。
	 * また，request-if-ok 属性が指定され，OKボタンが押された場合，
	 * その属性の値を URL とみなして，リクエスト送信を行なう。
	 * また，request-if-cancel 属性が指定され，Cancelボタンが押された場合，
	 * その属性の値を URL とみなして，リクエスト送信を行なう。
	 */
	CongaResponse.setValue4Confirm = function(response) {
		var text = response.etag.getAttribute("value");
		var reqIfYes = response.etag.getAttribute("request-if-ok");
		var reqIfNo = response.etag.getAttribute("request-if-cancel");
		var method = response.etag.getAttribute("method");
		if(!method) {
			method = "POST";
		}

		if(window.confirm(text)) {
			new CongaResponse(method).send(reqIfYes, "");
		}
		else if(reqIfNo){
			new CongaResponse(method).send(reqIfNo, "");
		}
	}

	/**
	 * type 指定がなかった場合，あるいは，指定されている type に該当する handler  が登録されて
	 * いなかった場合に呼び出される。
	 * id 引数で指定された id をもつオブジェクトの value 属性または，innerHTML 属性として設定を行なう。
	 */
	CongaResponse.setValue4Default = function(response) {

		var elem = document.getElementById(response.id);

		if(!elem) {
			return;
		}

		if(response.errore) {
			if(elem.value!=null) {
				elem.style.backgroundColor = "pink";
			}
		}
		else {
			elem.style.backgroundColor = "white";
			if(elem.value!=null) {
				if(response.value==null) {
					elem.value = "";
				}
				else {
					elem.value = response.value;
				}
			}
			else if(elem.innerHTML!=null) {
				if(response.value==null) {
					elem.innerHTML = "";
				}
				else {
					elem.innerHTML = response.value;
				}
			}
		}

		var disabled = response.etag.getAttribute("disabled");
		if(disabled) {
			if(disabled=="true") {
				elem.disabled = true;
			}
			else {
				elem.disabled = false;
			}
		}
	}

	/**
	 * エラーメッセージ表示を行なう。
	 */
	CongaResponse.showMessage = function(response) {
		window.alert(response.value);
	}

	/**
	 * フォーカス表示を行なう。
	 */
	CongaResponse.setFocus = function(response) {
		var elem = document.getElementById(response.id);
		if(elem && elem.focus) {
			setTimeout(function(){elem.focus();},200);
		}
	}

	/**
	 * ラベル設定を行なう。
	 */
	CongaResponse.setLabel = function(response) {
		var elem = document.getElementById(response.id);
		var value = response.etag.getAttribute("value");

		if(elem && elem.innerHTML) {
			if(value==null) {
				elem.innerHTML = "";
			}
			else {
				elem.innerHTML = value;
			}
		}
	}


	//// ローカルメソッド定義

	/**
	 * インスタンスを初期化する。
	 */
	function init() {
		if(!CongaResponse.handler) {
			CongaResponse.handler = new Array();
			CongaResponse.handler["default"] = CongaResponse.setValue4Default;
			CongaResponse.handler["window"] = CongaResponse.setValue2Window;
			CongaResponse.handler["iwindow"] = CongaResponse.setValue2IWindow;
			CongaResponse.handler["celcel"] = CongaResponse.setValue2Celcel;
			CongaResponse.handler["tree"] = CongaResponse.setValue2ITree;
			CongaResponse.handler["tree-node"] = CongaResponse.setValue2ITreeNode;
			CongaResponse.handler["itree"] = CongaResponse.setValue2ITree;
			CongaResponse.handler["itree-node"] = CongaResponse.setValue2ITreeNode;
			CongaResponse.handler["split-pane"] = CongaResponse.setValue2SplitPane;
			CongaResponse.handler["outer"] = CongaResponse.setValue4Outer;
			CongaResponse.handler["self"] = CongaResponse.setValue2Self;
			CongaResponse.handler["alert"] = CongaResponse.setValue4Alert;
			CongaResponse.handler["confirm"] = CongaResponse.setValue4Confirm;
			CongaResponse.handler["method"] = CongaResponse.setValue4Method;
			CongaResponse.handler["script"] = CongaResponse.setValue4Script;
			CongaResponse.handler["message"] = CongaResponse.showMessage;
			CongaResponse.handler["focus"] = CongaResponse.setFocus;
			CongaResponse.handler["label"] = CongaResponse.setLabel;
		}
	}

	/**
	 * HTML 要素の更新を行なう。
	 * @param id 更新する HTML 要素のID。
	 * @param value 更新する HTML 文字列。
	 */
	function updateHTMLElement(elem, id, value) {
		if(elem.outerHTML) {
			elem.outerHTML = value;
			elem = document.getElementById(id);
		}
		else {
			var dummy = document.createElement("div");
			dummy.innerHTML = value;
			var newElem = dummy.getElementsByTagName(elem.tagName).item(0);
			elem.parentNode.replaceChild(newElem, elem);
			elem = newElem;
		}

		if(elem.parentNode && elem.parentNode.handler
				&& elem.parentNode.handler.updateChild) {
			elem.parentNode.handler.updateChild(elem);
		}
		return elem;
	}

	//// インスタンスの初期化
	init();

}
new CongaResponse();


/**
 * IAssistant。
 */
function IAssistant() {

	//// public メソッド定義

	/**
	 * 入力補助用選択リストの表示を行なう。
	 * @param selector 選択リスト。
	 * @param input 選択結果を表示する入力項目。
	 * @param requestId 選択リストの作成を行なうためのリクエストID。
	 */
	IAssistant.show = function(selector, input, requestId) {
		selector = IWidget.getHtmlElement(selector);
		input = IWidget.getHtmlElement(input);

		if(!selector && !requestId) {
			return;
		}

		if(requestId) {
			send(requestId, input, 
					function(){
						showSelector(IWidget.getHtmlElement(selector.id), input);
					}
				);
		}
		else {
			showSelector(selector, input);
		}
	}

	/**
	 * 入力補助用選択リストに表示されている内容の絞込みを行なう。
	 * @param selector 選択リスト。
	 * @param input 選択結果を表示する入力項目。
	 * @param requestId 選択リストの絞込みを行なうためのリクエストID。
	 */
	IAssistant.narrowDown = function(selector, input, requestId) {
		selector = IWidget.getHtmlElement(selector);
		input = IWidget.getHtmlElement(input);

		if(!selector && !requestId) {
			return;
		}

		if(requestId) {
			send(requestId, input, 
					function(){
						showSelector(IWidget.getHtmlElement(selector.id), input);
					}
				);
		}
		else {
			narrowDownItem();
		}

		function narrowDownItem() {
			var value = getValue(input);
			var length = selector.rows.length;
			for(var i=0; i<length; i++) {
				var item = selector.rows[i].cells[0];
				if(check(value, item.innerHTML)) {
					selector.rows[i].style.display = 'block';
				}
				else {
					selector.rows[i].style.display = 'none';
				}
			}
			selector.style.display='block';

			function check(text1, text2) {
				if(!text1) {
					return true;
				}
				var length = text1.length;
				if(length > text2.length) {
					return false;
				}

				for(var i=0; i<length; i++) {
					if(text1.charAt(i)!=text2.charAt(i)) {
						return false;
					}
				}
				return true;
			}
		}
	}

	/**
	 * 入力補助用選択リストを非表示にする。
	 * @param input 選択結果を表示する入力項目。
	 */
	IAssistant.hide = function(input) {
		input = IWidget.getHtmlElement(input);
		var selector = input.iassistSelector;
		if(!selector) {
			return;
		}
		selector.style.display='none';
	}

	/**
	 * 入力補助用選択リストの非表示を予約する。
	 * @param input 選択結果を表示する入力項目。
	 */
	IAssistant.hideLator = function(input) {
		input = IWidget.getHtmlElement(input);
		var selector = input.iassistSelector;
		if(!selector) {
			return;
		}
		selector._hideLator = setTimeout(
			function() {
				if(selector._hideLator) {
					selector.style.display='none';
				}
			},
			1000
		);
	}


	/**
	 * 特定の項目をフォーカスするときに呼び出す。
	 * @param item フォーカスする項目。
	 */
	IAssistant.focus = function(item) {
		item = IWidget.getHtmlElement(item);
		item.style.backgroundColor='blue';
		item.style.color='white';
		var selector = getTable(item);
		if(selector._hideLator) {
			clearTimeout(selector._hideLator);
			selector._hideLator = null;
		}
	}

	/**
	 * フォーカスされている項目のフォーカスを外すときに呼び出す。
	 * @param item フォーカスを外す項目。
	 */
	IAssistant.blur = function(item) {
		item = IWidget.getHtmlElement(item);
		item.style.backgroundColor='white';
		item.style.color='black';
		IAssistant.hideLator(getTable(item).id);
	}

	/**
	 * 特定の項目が選択されたときに呼び出し，
	 * 選択された項目の値を inputId で指定された入力項目の値としてセットする。
	 */
	IAssistant.selected = function(item, input) {
		item = IWidget.getHtmlElement(item);
		input = IWidget.getHtmlElement(input);

		if(input.value!=undefined) {
			input.value = getSelectedValue();
		}
		else if(input.innerHTML!=undefined) {
			input.innerHTML = getSelectedValue();
		}

		getTable(item).style.display='none';

		function getSelectedValue() {
			if(item.tagName=="TR") {
				return item.cells[0].innerHTML
			}
			else {
				return item.innerHTML;
			}
		}
	}

	function showSelector(selector, input) {
		IWidget.hideInputHelper("IAssistant", selector);
		IWidget.setInputHelperPosition(input, selector);
		selector.style.display='block';
		input.iassistSelector = selector;
	}

	function send(requestId, input, func) {
		var request = new CongaRequest();
		request.async = true;
		request.send(requestId, getName(input) + "=" + getValue(input), func);
	}

	function getValue(input) {
		if(input.value!=undefined) {
			return input.value;
		}
		else if(input.innerHTML!=undefined) {
			return input.innerHTML;
		}
		else {
			return null;
		}
	}

	function getName(input) {
		if(input.name!=undefined) {
			return input.name;
		}
		else if(input.id!=undefined) {
			return input.id;
		}
		else {
			return null;
		}
	}

	function getTable(item) {
		var obj = item;
		while(obj) {
			if(obj.tagName=="TABLE") {
				break;
			}
			obj = obj.parentNode;
		}
		return obj;
	}

}
new IAssistant();


/**
 * IBorder を作成する。
 * @param container IBorder のコンテナ。
 * @param type 線タイプ。"v" は縦線。"h" は横線。
 * @param length 線の長さ。
 * @param width 線の幅。[デフォルト=2]
 */
function IBorder(container, type, length, width) {

	//// 変数定義
	var handler = this;
	var htmlElement = document.createElement("div");


	//// プロパティ定義

	/**
	 * HTML実体要素オブジェクト。
	 */
	this.htmlElement = htmlElement;


	//// public メソッド定義

	/**
	 * 表示位置を設定する。
	 * @param left 横位置。
	 * @param top 縦位置。
	 */
	this.setPosition = function(left, top) {
		htmlElement.style.left = left;
		htmlElement.style.top = top;
		return handler;
	}

	/**
	 * 線の長さを設定する。
	 * @param len 線の長さ。
	 */
	this.setLength = function(len) {
		if(type.charAt(0)=="v" || type.charAt(0)=="V" ) {
			htmlElement.style.height = len;
		}
		else {
			htmlElement.style.width = len;
		}
		return handler;
	}

	/**
	 * 背景色を設定する。
	 * @param color 背景色。
	 */
	this.setColor = function(color) {
		htmlElement.style.backgroundColor = color;
		return handler;
	}


	//// イベント定義

	/**
	 * このオブジェクトがドラッグ操作で移動されたときに呼び出される。
	 * @param left オブジェクト横位置。
	 * @param top オブジェクト縦位置。
	 * @param event マウスイベント。
	 */
	this.onmove = null;


	//// ローカルメソッド定義

	/**
	 * IBorder を初期化する。
	 * @param container IBorder のコンテナ。
	 * @param type 線タイプ。"v" は縦線。"h" は横線。
	 * @param length 線の長さ。
	 * @param width 線の幅。[デフォルト=2]
	 */
	function init(container, type, length, width) {
		if(!width) {
			width = 2;
		}

		IWidget.initHtmlElement(handler, htmlElement, true);

		htmlElement.handler = handler;
		htmlElement.style.backgroundColor = "";
		htmlElement.style.position = "absolute";
		htmlElement.style.fontSize = "1pt";
		htmlElement.style.lineHeight= "100%";

		container.style.position = "relative";
		container.appendChild(htmlElement);

		if(type.charAt(0)=="v" || type.charAt(0)=="V" ) {
			htmlElement.type = "x";
			htmlElement.style.width = width;
			htmlElement.style.height = length;
			htmlElement.style.cursor = "e-resize";
		}
		else {
			htmlElement.type = "y";
			htmlElement.style.width = length;
			htmlElement.style.height = width;
			htmlElement.style.cursor = "n-resize";
		}

		// for debug
		//htmlElement.style.backgroundColor = "red";
		//htmlElement.style.filter = "alpha(opacity=50)";
	}

	//// インスタンスの初期化
	init(container, type, length, width);
}


/**
 * ICalendar カレンダー。
 */
function ICalendar() {
	//// 変数定義
	var week;
	var content;
	var window;
	var firstDay;

	/**
	 * 本日として表示する日。
	 */
	var today;

	/**
	 * 書式。
	 */
	var format;

	/**
	 * 設定対象項目。
	 */
	var target;

	var initialized = false;


	//// public メソッド定義

	ICalendar.show = function(event, item) {
		show(event, item);
	}

	/**
	 * カレンダを表示する。
	 */
	function show(event, item) {

		event = IWidget.editEvent(event);

		if(!item) {
			item = event.target;
		}

		item = IWidget.getHtmlElement(item);

		initVariables();

		ICalendar.__update(0);

		IWidget.hideInputHelper("ICalendar", item);

		window.pack();
		window.show(event);

		function initVariables() {
			if(!initialized) {
				//// 変数定義
				week = new Array(
						IResource.get("ICalendar.sun", "Sun"),
						IResource.get("ICalendar.mon", "Mon"),
						IResource.get("ICalendar.tue", "Tue"),
						IResource.get("ICalendar.wed", "Wed"),
						IResource.get("ICalendar.thu", "Thu"),
						IResource.get("ICalendar.fri", "Fri"),
						IResource.get("ICalendar.sat", "Sat")
					);

				content = createContent();
				window = createWindow(content);
				format = IResource.get("ICalendar.format", "y/m/d");
				initialized = true;
			}

			firstDay = new Date();
			today = firstDay;
			target = item;
		}
	}

	/**
	 * カレンダを非表示にする。
	 */
	ICalendar.hide = function() {
		if(window) {
			window.hide();
		}
	}


	/**
	 * カレンダ表示内容を更新する。
	 */
	ICalendar.__update = function(ma) {
		nowdate  = firstDay.getDate();
		nowmonth = firstDay.getMonth();
		nowyear  = firstDay.getYear();

		// 月の変更
		if(nowmonth==11 && ma > 0) {
			nowmonth = ma - 1;
			nowyear++;
		}
		else if(nowmonth==0 && ma < 0) {
			nowmonth = ma + 12;
			nowyear--;
		}
		else {
			nowmonth += ma;
		}

		if(nowyear<1900) {
			nowyear=1900+nowyear;
		}

		// ヘッダ部の設定
		var tables = content.getElementsByTagName("table");
		var headerRow = tables.item(0).rows[0];
		headerRow.cells[0].onclick = function(){ICalendar.__update(-1);};
		headerRow.cells[2].onclick = function(){ICalendar.__update(1);};

		var m2 = (nowmonth + 1);
		headerRow.cells[1].innerHTML = nowyear+' . '+ ((m2<10)?('&nbsp;'+m2):m2);

		// ボディ部の設定
		firstDay = new Date(nowyear,nowmonth,1);
		var sunday = new Date(firstDay - ( firstDay.getDay() * 1000*60*60*24 ));
		var yearMonth = nowyear*100+nowmonth;

		var body = tables.item(1);

		for(var i=0; i<6; i++) {
			var row = body.rows[i + 1];

			for(var j=0; j<7; j++) {
				var cell = row.cells[j];

				var nextday = new Date(sunday.getTime() + (j * 1000*60*60*24));
				var tmpdate = nextday.getDate();
				var tmpmonth= nextday.getMonth();
				var tmpyear = nextday.getYear();

				if(tmpyear < 1900) {
					tmpyear=1900 + tmpyear;
				}

				cell.date = tmpdate;
				cell.month = tmpmonth;
				cell.year = tmpyear;
				cell.onclick = function(){ICalendar.__setDate(this);};
				cell.innerHTML = tmpdate;
				cell.title = "";

				var tmpYearMonth = tmpyear * 100 + tmpmonth;
				if(tmpYearMonth != yearMonth) {
					cell.className = "disabled";
				}
				else {
					if(tmpdate == today.getDate() && 
								tmpmonth == today.getMonth() && 
								nextday.getYear() == today.getYear()) {
						cell.className = "today";
					}
					else {
						var holiday = IResource.get("ICalendar." + createText(tmpyear, tmpmonth, tmpdate, "ymd"), "_");
						if(holiday!="_") {
							cell.className = "holiday";
							cell.title = holiday;
						}
						else {
							if(j==0) {
								cell.className = "sun";
							}
							else if(j==6) {
								cell.className = "sat";
							}
							else {
								cell.className = "oth";
							}
						}
					}
				}
			}

			sunday = nextday;
			sunday = sunday.getTime() + (1000*60*60*24);
			sunday = new Date(sunday);
		}

		// フッタ部の設定
		tables.item(2).rows[0].cells[0].onclick = function(){window.hide()};
	}

	/**
	 * 対象項目に日付をセットする。
	 */
	ICalendar.__setDate = function(cell) {
		var str = createText(cell.year, cell.month, cell.date, format);

		if(target.value!=undefined) {
			target.value = str;
		}
		else if(target.innerHTML!=undefined) {
			target.innerHTML = str;
		}
		window.hide();
	}

	function createText(tmpyear, tmpmonth, tmpdate, fmt) {
		m2 = tmpmonth+1;

		tmpmonth = ((m2<10)?'0'+m2:''+m2);
		tmpdate = ((tmpdate<10)?'0'+tmpdate:''+tmpdate);

		if(!fmt) {
			return tmpyear + "" + tmpmonth + "" + tmpdate;
		}
		else {
			var arr = new Array();
			var index = 0;
			for(var i=0; i<fmt.length; i++) {
				var c = fmt.charAt(i);
				if(c=='y' && tmpyear) {
					arr[index++] = tmpyear;
					tmpyear = null;
				}
				else if(c=='m' && tmpmonth) {
					arr[index++] = tmpmonth;
					tmpmonth = null;
				}
				else if(c=='d' && tmpdate) {
					arr[index++] = tmpdate;
					tmpdate = null;
				}
				else {
					arr[index++] = c;
				}
			}
			return arr.join("");
		}
	}


	function createContent() {
		var arr = new Array();
		var index = 0;
		arr[index++] =
			'<table style="margin-top:4px;" border="0" cellspacing="0" cellpadding="3" align="center">' +
			'  <tr align="center">' +
			'    <th class="button">&nbsp;&lt;</th>' +
			'    <th align="center"></th>' +
			'    <th class="button">&gt;&nbsp;</th>' +
			'  </tr>' +
			'</table>' +
			' <table align="center" border="0" cellspacing="1" cellpadding="1" class="calendar" width="96%"' +
			' style="margin:4px;border:1px solid black;background:#aaaaaa;">' +
			'  <tr>' +
			'    <th class="sun">' + week[0] + '</th>' +
			'    <th class="oth">' + week[1] + '</th>' +
			'    <th class="oth">' + week[2] + '</th>' +
			'    <th class="oth">' + week[3] + '</th>' +
			'    <th class="oth">' + week[4] + '</th>' +
			'    <th class="oth">' + week[5] + '</th>' +
			'    <th class="sat">' + week[6] + '</th>' +
			'  </tr>';

		for(var i = 0; i < 6; i++) {
			arr[index++] =
				' <tr>' +
				'   <td></td>' +
				'   <td></td>' +
				'   <td></td>' +
				'   <td></td>' +
				'   <td></td>' +
				'   <td></td>' +
				'   <td></td>' +
				' </tr>';
		}
		arr[index++] = '</table>';
		arr[index++] = '<table style="margin-top:4px;margin-bottom:4px;" border ="0" align="center">'+
						'<tr><td class="button">' + 
							IResource.get("close", "Close") + 
						'</td></tr>'+
						'</table>';

		var content = document.createElement("div");
		content.style.textAlign = "center";
		content.style.width = "220px";
		content.style.height = "210px";
		content.innerHTML = arr.join("");
		return content;
	}

	function createWindow(content) {
		var win = new IWindow("Calendar", content, true);
		win.setEnableMinimizeButton(false);
		win.onclosing = function(){win.hide(); return false};
		return win;
	}

}
new ICalendar();
/**
 * IContainer を作成する。
 * @param htmlElement コンテナに格納する HTML 要素。
 */
function IContainer(htmlElement) {

	//// 変数定義
	var handler = this;

	//// プロパティ定義

	/**
	 * HTML実体要素オブジェクト。
	 */
	this.htmlElement = htmlElement;

	//// public メソッド定義

	/**
	 * 幅設定を行なう。
	 * @param width 幅。
	 */
	this.setWidth = function(width) {
		htmlElement.onresize = null;
		IWidget.setWidth(htmlElement, width);
		return handler;
	}

	/**
	 * 高さ設定を行なう。
	 * @param height 高さ。
	 */
	this.setHeight = function(height) {
		htmlElement.onresize = null;
		IWidget.setHeight(htmlElement, height);
		return handler;
	}

	/**
	 * 幅と高さを設定する。
	 * @param width 幅。
	 * @param height 高さ。
	 */
	this.setSize = function(width, height) {
		handler.setWidth(width);
		handler.setHeight(height);
		return handler;
	}

	this.pack = function() {
		handler.setClientWidth(htmlElement.offsetWidth - IWidget.getMarginWidth(htmlElement));
		handler.setClientHeight(htmlElement.offsetHeight - IWidget.getMarginHeight(htmlElement));
	}

	/**
	 * コンテナ内コンポーネントの幅を設定する。
	 * @param width 幅。
	 * @return 指定された幅が設定できた場合は true。できなかった場合は false。
	 */
	this.setClientWidth = function(width) {
		if(htmlElement.childNodes.length == 1 && 
				htmlElement.firstChild.nodeType==1) {
			return IWidget.setWidth(htmlElement.firstChild, width);
		}
		else {
			return true;
		}
	}

	/**
	 * コンテナ内コンポーネントの高さを設定する。
	 * @param htmlElement 設定するHTML要素。
	 * @param height 高さ。
	 * @return 指定された高さが設定できた場合は true。できなかった場合は false。
	 */
	this.setClientHeight = function(height) {
		if(htmlElement.childNodes.length == 1 && 
				htmlElement.firstChild.nodeType==1) {
			return IWidget.setHeight(htmlElement.firstChild, height);
		}
		else {
			return true;
		}
	}


	//// ローカルメソッド定義

	/**
	 * IContainer を作成する。
	 * @param htmlElement コンテナに格納する HTML 要素。
	 */
	function init() {
		IWidget.initHtmlElement(handler, htmlElement);
		htmlElement.style.overflow = "auto";
		htmlElement.style.width = "100%";
		htmlElement.style.height = "100%";
		htmlElement.onresize = function(event) {
			try{handler.pack();}
			catch(e){}
		}
		handler.pack();
	}


	//// インスタンスの初期化
	init();
}


/**
 * IDebug。
 */
function IDebug() {

	//// public メソッド定義

	/**
	 * デバッグ出力を行なう。
	 * @param obj 出力内容。
	 */
	IDebug.print = function(obj) {
		var dbg = document.getElementById("_iwidgets_dbg");
		if(!dbg) {
			var debugWindow = new IWindow("debug");
			debugWindow.setContent(
				"<div id='_iwidgets_dbg'></div>"+
				"<input type='button' value='clear' onclick='IDebug.clear()'/>"
			);
			debugWindow.setBounds(600,20,200,200);
			debugWindow.onclosing = function(){return false;};
			dbg = document.getElementById("_iwidgets_dbg");
			dbg.window = debugWindow;
			dbg.innerHTML = obj;
		}
		else {
			if(dbg.window.isClosed()) {
				return;
			}
			else if(!dbg.window.isVisible()) {
				dbg.window.show();
			}
			dbg.innerHTML = dbg.innerHTML + "<br>" + obj;
			dbg.parentNode.scrollTop = dbg.parentNode.scrollHeight;
		}
	}

	/**
	 * デバッグ出力のクリアを行なう。
	 */
	IDebug.clear = function () {
		var dbg = document.getElementById("_iwidgets_dbg");
		if(dbg) {
			dbg.innerHTML="";
		}
	}
	
	/**
	 * オブジェクトをデバッグ出力する
	 */
	IDebug.printObject = function(obj) {
		IDebug.print(IDebug.object2String(obj));
	}

	/**
	 * オブジェクトを文字列表現に変換する。
	 */
	IDebug.object2String = function(obj) {
	    var val, output = "";
	    if (obj) {    
	        output += "{";
	        for (var i in obj) {
	            val = obj[i];
	            switch (typeof val) {
	                case ("object"):
	                    if (val[0]) {
	                        output += i + ":" + array2String(val) + ",";
	                    } else {
	                        output += i + ":" + object2String(val) + ",";
	                    }
	                    break;
	                case ("string"):
	                    output += i + ":'" + escape(val) + "',";
	                    break;
	                default:
	                    output += i + ":" + val + ",";
	            }
	        }
	        output = output.substring(0, output.length-1) + "}";
	    }
	    return output;
	}

	/**
	 * 配列を文字列表現に変換する。
	 */
	IDebug.array2String = function(array) {
	    var output = "";
	    if (array) {
	        output += "[";
	        for (var i in array) {
	            val = array[i];
	            switch (typeof val) {
	                case ("object"):
	                    if (val[0]) {
	                        output += array2String(val) + ",";
	                    } else {
	                        output += object2String(val) + ",";
	                    }
	                    break;
	                case ("string"):
	                    output += "'" + escape(val) + "',";
	                    break;
	                default:
	                    output += val + ",";
	            }
	        }
	        output = output.substring(0, output.length-1) + "]";
	    }
	    return output;
	}


	/**
	 * 文字列表現をオブジェクトに変換する。
	 */
	IDebug.string2Object = function(string) {
	    eval("var result = " + string);
	    return result;
	}

	/**
	 * 文字列表現の配列を配列オブジェクトに変換する。
	 */
	IDebug.string2Array = function(string) {
	    eval("var result = " + string);
	    return result;
	}
}
new IDebug();


/**
 * IDiv を作成する。
 * @param width 幅。[デフォルト=0]
 * @param height 高さ。[デフォルト=0]
 * @param container IDiv のコンテナ。[デフォルト=document.body]
 */
function IDiv(width, height, container) {

	//// 変数定義
	var handler = this;
	var htmlElement = document.createElement("div");


	//// プロパティ定義

	/**
	 * HTML実体要素オブジェクト。
	 */
	this.htmlElement = htmlElement;

	//// public メソッド定義

	/**
	 * 表示位置を設定する。
	 * @param left 横位置。
	 * @param top 縦位置。
	 */
	this.setPosition = function(left, top) {
		htmlElement.style.left = left;
		htmlElement.style.top = top;
		return handler;
	};

	/**
	 * サイズを設定する。
	 * @param width 幅。
	 * @param height 高さ。
	 */
	this.setSize = function(width, height) {
		htmlElement.style.width = width;
		htmlElement.style.height = height;
		return handler;
	}

	/**
	 * 表示位置とサイズをまとめて設定する。
	 * @param width 幅。
	 * @param height 高さ。
	 */
	this.setBounds = function(left, top, width, height) {
		htmlElement.handler.setPosition(left, top);
		htmlElement.handler.setSize(width, height);
		return handler;
	}

	/**
	 * 背景色を設定する。
	 */
	this.setColor = function(color) {
		htmlElement.style.backgroundColor = color;
		return handler;
	}

	/**
	 * 内容設定を行なう。
	 */
	this.setContent = function(obj) {
		IWidget.setContent(htmlElement, obj);
		return handler;
	}
	
	/**
	 * 幅を取得する。
	 * return width 幅。
	 */
	this.getWidth = function() {
		return htmlElement.style.width;
	}

	/**
	 * 高さを取得する。
	 * return height 高さ。
	 */
	this.getHeight = function() {
		return htmlElement.style.height;
	}

	//// イベント定義

	/**
	 * このオブジェクトがドラッグ操作で移動されたときに呼び出される。
	 * @param left オブジェクト横位置。
	 * @param top オブジェクト縦位置。
	 * @param event マウスイベント。
	 */
	this.onmove = null;

	/**
	 * このオブジェクトがダブルクリックされたときに呼び出される。
	 * @param event ダブルクリックイベント。
	 */
	this.ondblclick = null;



	//// ローカルメソッド定義

	/**
	 * IDiv を初期化する。
	 * @param width 幅。[デフォルト=0]
	 * @param height 高さ。[デフォルト=0]
	 * @param container IDiv のコンテナ。[デフォルト=document.body]
	 */
	function init(width, height, container) {
		if(!width) {
			width = 0;
		}

		if(!height) {
			height = 0;
		}

		if(!container) {
			container = document.body;
		}

		IWidget.initHtmlElement(handler, htmlElement, true);

		htmlElement.style.position = "absolute";
		htmlElement.style.fontSize = "1pt";
		if(width>0) {
			htmlElement.style.width = width;
		}
		if(height>0) {
			htmlElement.style.height = height;
		}

		container.style.position = "relative";
		container.appendChild(htmlElement);

		htmlElement.ondblclick = function(event) {
			event = IWidget.editEvent(event);
			var obj = event.target;
			while(obj) {
				if(obj.handler && obj.handler.ondblclick) {
					obj.handler.ondblclick(event);
					event.resultValue = false;
					event.cancelBubble = true;
					return false;
				}
				obj = obj.parentNode;
			}
		}
	}

	//// インスタンスの初期化
	init(width, height, container);
}


/**
 * IHelpMessage。
 */
function IHelpMessage() {

	var messageWindow;

	var initialized = false;

	var hideLator = null;


	//// public メソッド定義

	/**
	 * ヘルプメッセージの表示を行なう。
	 * @param target ヘルプを表示するオブジェクト。
	 * @param message 表示するメッセージ。
	 */
	IHelpMessage.show = function(target, message) {
		initVariables();

		if(hideLator) {
			clearTimeout(hideLator);
			hideLator = null;
		}

		target = IWidget.getHtmlElement(target);

		messageWindow.setContent(message);
		var htmlElement = messageWindow.htmlElement;
		htmlElement.className = "helpmessage";
		IWidget.setInputHelperPosition(target, htmlElement, 4);
		messageWindow.htmlElement.style.display = "block";


		hideLator = setTimeout(
			function() {
				if(hideLator) {
					hideLator = null;
					IHelpMessage.hide();
				}
			},
			6000
		);

		function initVariables() {
			if(!initialized) {
				messageWindow = new IDiv();
				var htmlElement = messageWindow.htmlElement;
				htmlElement.style.fontSize = "";
				htmlElement.className = "helpmessage";
				initialized = true;
			}
		}
	}

	/**
	 * ヘルプメッセージの消去を行なう。
	 */
	IHelpMessage.hide = function() {
		if(messageWindow) {
			messageWindow.htmlElement.style.display = "none";
		}
	}

}
new IHelpMessage();


/**
 * IMenu を作成する。
 */
function IMenu() {

	//// 変数定義
	var handler = this;
	var htmlElement = document.createElement("div");
	var container = document.body;
	var menuItem = new Array();

	//// プロパティ定義

	/**
	 * HTML実体要素オブジェクト。
	 */
	this.htmlElement = htmlElement;

	//// public メソッド定義

	/**
	 * メニューを表示する。
	 * @param event メニュー表示の起因となったイベント。
	 */
	this.show = function(event) {
		event = IWidget.editEvent(event);
		event = IWidget.getWindowPosition(event, htmlElement);
		htmlElement.style.zIndex = IWidget.getMaxZIndex();
		htmlElement.style.left = event.posX;
		htmlElement.style.top = event.posY;
		htmlElement.style.visibility = "visible";
		htmlElement.style.filter = "alpha(opacity=100)";
		return handler;
	};

	/**
	 * メニューを非表示にする。
	 */
	this.hide = function() {
		IWidget.hideSlowly(htmlElement);
		return handler;
	}


	/**
	 * メニューへ項目追加を行なう。
	 * @param text 項目名称。
	 * @param func メニュークリック時に実行するファンクション。
	 * @param accessKey アクセスキー。
	 */
	this.addItem = function(text, func, accessKey) {
		var input = document.createElement("input");
		input.type = "button";
		input.value = IResource.get(text);
		if(accessKey) {
			input.accessKey = accessKey;
		}
		input.onclick = function() {
			htmlElement.style.visibility = "hidden";
			func();
			return false;
		}

		input.onmouseover = function(event) {
			event = IWidget.editEvent(event);
			event.target.originalBackgroundColor = event.target.style.backgroundColor;
			event.target.style.backgroundColor = "#000066";
			event.target.style.color = "white";
		}

		input.onmouseout = function(event) {
			event = IWidget.editEvent(event);
			event.target.style.backgroundColor = event.target.originalBackgroundColor;
			event.target.style.color = "black";
		}


		input.style.border = "none";
		input.style.textAlign = "left";
		input.style.paddingLeft = "14px";
		htmlElement.appendChild(input);
		htmlElement.appendChild(document.createElement("br"));

		var maxWidth = 0;
		for(var i=0; i < htmlElement.childNodes.length; i++) {
			var child = htmlElement.childNodes.item(i);
			maxWidth = Math.max(maxWidth, child.offsetWidth);
		}

		for(var i=0; i < htmlElement.childNodes.length; i++) {
			var child = htmlElement.childNodes.item(i);
			if(child.nodeName=="INPUT") {
				child.style.width = maxWidth;
			}
		}

		menuItem[text] = input;
		return handler;
	}

	/**
	 * メニュー項目の選択可不可を設定する。
	 * @param text メニュー項目名。
	 * @param enabled 選択可にするときは true。不可にするときは false。
	 */
	this.setEnabled = function(text, enabled) {
		menuItem[text].disabled = !enabled;
		return handler;
	}

	/**
	 * メニューを表示するオブジェクトを指定する。
	 * @param obj メニューを表示するオブジェクト。
	 */
	this.addTarget = function(obj) {
		obj.oncontextmenu = function(event) {
			event = IWidget.editEvent(event);
			handler.show(event);
			event.returnValue = false;
			event.cancelBubble = !event.returnValue;
			return event.returnValue;
		}
		return handler;
	}

	/**
	 * IMenu を初期化する。
	 */
	function init() {
		if(!IMenu.prototype.blockedDefaultMenu) {
			document.oncontextmenu = function(event) {
				return false;
			}

			IMenu.prototype.blockedDefaultMenu = true;
		}

		IWidget.initHtmlElement(handler, htmlElement, false);

		htmlElement.style.border = "outset 2px";
		htmlElement.className = "imenu";
		htmlElement.style.position = "absolute";

		container.appendChild(htmlElement);

		htmlElement.style.visibility = "hidden";
		htmlElement.onmouseout = function(event) {
			handler.mouseover = false;
			setTimeout(
				function(){
					if(!handler.mouseover){
						handler.hide();
					}
				}
			, 200);
		}

		htmlElement.onmouseover = function(event) {
			handler.mouseover = true;
			htmlElement.style.visibility = "visible";
		}
	}


	//// インスタンスの初期化
	init();
}
/**
 * IResource。
 */
function IResource() {

	//// public メソッド定義

	/**
	 * リソース登録されている値を取得する。
	 * @param key 登録キー。
	 * @param defaultValue 登録されていない場合にデフォルト表示する値。
	 *                    [デフォルト=登録キー]
	 * @return リソース登録されている値。
	 */
	IResource.get = getText;
	IResource.getText = getText;

	/**
	 * メッセージを取得する。
	 * @param key メッセージ登録キー。
	 * @return リソース登録されているメッセージ。
	 */
	IResource.message = getMessage;
	IResource.getMessage = getMessage;

	/**
	 * リソース登録を行なう。
	 * @param key 登録キー。
	 * @param value 登録する値。
	 */
	IResource.add = function(key, value, lang) {
		if(!IResource.resource) {
			IResource.resource = new Array();
		}
		if(!lang) {
			lang = "__"; // デフォルト
		}

		var resource = IResource.resource[lang];
		if(!resource) {
			resource = new Array();
			IResource.resource[lang] = resource;
		}
		resource[key] = value;
	}


	//// ローカルメソッド定義

	/**
	 * リソース登録されている値を取得する。
	 * @param key 登録キー。
	 * @param defaultValue 登録されていない場合にデフォルト表示する値。
	 *                    [デフォルト=登録キー]
	 * @return リソース登録されている値。
	 */
	function getText(key, defaultValue) {
		var lang;
		if(navigator.language) {
			lang = navigator.language;
		}
		else {
			lang = navigator.userLanguage;
		}

		var resource = IResource.resource[lang];
		if(!resource) {
			if(lang.length > 2) {
				resource  = IResource.resource[lang.substring(0, 2)];
			}
			if(!resource) {
				resource = IResource.resource["__"]; // デフォルト
			}
		}

		if(resource) {
			if(!defaultValue) {
				defaultValue = key;
			}

			var value = resource[key];
			if(!value) {
				if(lang.length > 2) {
					value = IResource.resource[lang.substring(0, 2)][key];
				}
				if(!value) {
					value = IResource.resource["__"][key]
				}
			}
			return (value)? value : defaultValue;
		}
		else {
			return key;
		}
	}


	/**
	 * メッセージを取得する。
	 * @param key メッセージ登録キー。
	 * @return リソース登録されているメッセージ。
	 */
	function getMessage(key) {
		var text = getText(key);
		if(arguments.length > 1) {
			for(var i=1; i<arguments.length; i++) {
				text = text.replace("%"+i, arguments[i]);
			}
		}
		return text;
	}
}
new IResource();


/**
 * ISplitPane を作成する。
 * @param htmlElement オブジェクト。
 * @param orientation 分割方向。"v" は縦分割。"h" は横分割。
 */
function ISplitPane(htmlElement, orientation) {

	//// 継承 : IContainer
	IContainer.call(this, htmlElement);
	this.constructor = ISplitPane;

	//// ローカル変数定義

	var handler = this;

	if(!ISplitPane.prototype.BORDER_WIDTH) {
		ISplitPane.prototype.BORDER_WIDTH = 10;
		ISplitPane.prototype.VISIBLE_BORDER_WIDTH = 2;
	}
	var bw = ISplitPane.prototype.BORDER_WIDTH;
	var vbw = ISplitPane.prototype.VISIBLE_BORDER_WIDTH;
	var hbw = bw/2;
	var divider;
	var a = null;
	var b = null;
	var vertical = orientation.charAt(0)=="v" || orientation.charAt(0)=="V" ;


	//// public メソッド定義


	/**
	 * コンテナ内コンポーネントの幅を設定する。
	 * @param width 幅。
	 * @return 指定された幅が設定できた場合は true。できなかった場合は false。
	 */
	this.setClientWidth = function(width) {
		if(vertical) {
			if(width <= a.offsetWidth) {
				return false;
			}
			if(!IWidget.setWidth(b, width - a.offsetWidth - vbw)) {
				return false;
			}

			b.style.left = a.offsetWidth;
			divider.setPosition(a.offsetWidth-hbw, 0);
		}
		else {
			if(!IWidget.setWidth(a, width-1)) {
				return false;
			}
			if(!IWidget.setWidth(b, width-1)) {
				return false;
			}
			divider.setLength(width);
		}
		return true;
	}

	/**
	 * コンテナ内コンポーネントの高さを設定する。
	 * @param htmlElement 設定するHTML要素。
	 * @param height 高さ。
	 * @return 指定された高さが設定できた場合は true。できなかった場合は false。
	 */
	this.setClientHeight = function(height) {
		if(!vertical) {
			if(height <= a.offsetHeight) {
				return false;
			}
			if(!IWidget.setHeight(b, height - a.offsetHeight - vbw)) {
				return false;
			}

			b.style.top = a.offsetHeight;
			divider.setPosition(0, a.offsetHeight-hbw);
		}
		else {
			if(!IWidget.setHeight(a, height-1)) {
				return false;
			}
			if(!IWidget.setHeight(b, height-1)) {
				return false;
			}
			divider.setLength(height);
		}
		return true;
	}

	/**
	 * ディバイダの位置を設定する。
	 * @param location ディバイダの位置。
	 */
	this.setDividerLocation = function(location) {
		if(vertical) {
			divider.onmove(location, 0);
			divider.setPosition(location, 0);
		}
		else {
			divider.onmove(0, location);
			divider.setPosition(0, location);
		}
	}

	/**
	 * 子要素が更新されたときに呼び出される。
	 * @param elem 子要素。
	 */
	this.updateChild = function(elem) {
		if(elem.id==a.id) {
			a = elem;
		}
		else if(elem.id==b.id) {
			b = elem;
		}
		initChildren();
	}


	/**
	 * ISplitPane を初期化する。
	 */
	function init() {
		htmlElement.handler = handler;
		htmlElement.style.position = "relative";
		htmlElement.style.padding = "0px";
		htmlElement.style.margin = "0px";

		var children = htmlElement.childNodes;
		for(var i=0; i<children.length; i++) {
			var child = children.item(i);
			if(child.tagName!="DIV") {
				continue;
			}
			child.style.position = "absolute";

			if(a==null) {
				a = child;
			}
			else if(b==null) {
				b = child;
				break;
			}
		}
		initChildren();
	}

	function initChildren() {
		if(vertical) {
			b.style.left = a.offsetWidth;
			b.style.borderLeft = "ridge 2px gray";
			divider = new IBorder(htmlElement, orientation, "100%", bw);
			divider.setPosition(a.offsetWidth-hbw, 0);
		}
		else {
			b.style.top = a.offsetHeight;
			b.style.borderTop = "ridge 2px gray";
			divider = new IBorder(htmlElement, orientation, "100%", bw);
			divider.setPosition(0, a.offsetHeight-hbw);
		}

		divider.onmove = function(left, top) {
			if(vertical) {
				var aWidth = a.offsetWidth;
				var bWidth = b.offsetWidth;

				if(left <= bw || left >= htmlElement.offsetWidth-bw) {
					divider.setPosition(aWidth-hbw, 0);
					return;
				}

				var diff = left - aWidth + hbw;
				if(!IWidget.setWidth(a, aWidth + diff)) {
					divider.setPosition(aWidth-hbw, 0);
					return;
				}
				if(!IWidget.setWidth(b, bWidth - diff)) {
					a.style.width = aWidth;
					divider.setPosition(aWidth-hbw, 0);
					return;
				}

				b.style.left = aWidth + diff;
			}
			else {
				var aHeight = a.offsetHeight;
				var bHeight = b.offsetHeight;

				if(top <= bw || top >= htmlElement.offsetHeight-bw) {
					divider.setPosition(0, aHeight-hbw);
					return;
				}

				var diff = top - aHeight + hbw;
				if(!IWidget.setHeight(a, aHeight + diff)) {
					divider.setPosition(0, aHeight-hbw);
					return;
				}
				if(!IWidget.setHeight(b, bHeight - diff)) {
					a.style.height = aHeight;
					divider.setPosition(0, aHeight-hbw);
					return;
				}
				b.style.top = aHeight + diff;
			}
		}
	}

	//// インスタンスの初期化
	init();
}


/**
 * ITextInput。
 */
function ITextInput() {

	//// public メソッド定義

	/**
	 * 指定された入力項目の値が英数字として妥当かどうかチェックする。
	 * @param event イベント。
	 * @param obj チェック対象入力項目。
	 * @return 英数字のみの場合は true。それ以外の文字が含まれる場合は false。
	 */
	ITextInput.chkAlnum = function(event, obj) {
		var str = obj.value;

		if (str.length == 0) {
			return true;
		}

		flg = false;
		for(i=0; i<str.length; i++) {
			c = str.charCodeAt(i);
			if((c >=0x0021 && c <= 0x007e)) {
				flg = true;
			}
			else {
				flg = false;
				break;
			}
		}

		if(flg==false) {
			showAlert(event, IResource.message("ITextInput.input_alnum"), obj);
		}

		return flg;
	}

	/**
	 * 指定された入力項目の値が数値として妥当かどうかをチェックする。
	 * @param event イベント。
	 * @param obj チェック対象入力項目。
	 * @param ilen 整数部桁数。
	 * @param flen 小数部桁数。
	 * @param positiveOnly マイナス値をエラーとする場合は true を設定。
	 * @return 数値のみの場合は true。それ以外の文字が含まれる場合は false。
	 */
	ITextInput.chkNum = function(event, obj, ilen, flen, positiveOnly) {
		var str = obj.value;

		if(str.length == 0) {
			return true;
		}

		/* カンマははじく*/
		nonCommaValue = str;
		while(nonCommaValue.indexOf(",") != -1) {
			nonCommaValue = nonCommaValue.replace(",","");
		}
		str = nonCommaValue;

		if(isNaN(str)) {
			showAlert(event, IResource.message("ITextInput.input_num"), obj);
			return false;
		}
		else {
			if(str.indexOf("x") > -1 || str.indexOf("X") > -1 || 
					str.indexOf("e") > -1 || str.indexOf("E") > -1) {
				showAlert(event, IResource.message("ITextInput.input_num"), obj);
				return false;
			}
		}

		var pindex = str.indexOf(".");
		var mindex = str.indexOf("-");

		/* 整数指定がある場合は，ピリオドをチェックする */
		if((flen==0) && (pindex > -1)) {
			showAlert(event, IResource.message("ITextInput.input_int"), obj);
			return false;
		}

		/* 正数指定がある場合は，マイナス値をチェックする */
		if(positiveOnly && (mindex > -1)) {
			showAlert(event, IResource.message("ITextInput.input_pos"), obj);
			return false;
		}

		/* マイナス記号の位置をチェックする */
		if(mindex>0) {
			showAlert(event, IResource.message("ITextInput.input_num"), obj);
			return false;
		}

		/* 整数部桁数をチェックする */
		var len = str.length;
		if(pindex>-1) {
			len = pindex;
		}
		if(mindex>-1) {
			len--;
		}
		if(len>ilen) {
			showAlert(event, IResource.message("ITextInput.input_int_with", ilen), obj);
			return false;
		}

		/* 小数部桁数をチェックする */
		if(pindex>-1) {
			len = str.length - (pindex+1);
			if(len>flen) {
				showAlert(event, IResource.message("ITextInput.input_dec_with", flen), obj);
				return false;
			}
		}

		/* カンマをはじいた値を設定する */
		obj.value = str;
		return true;
	}

	/*
	 * 整数項目用インプット値チェック。
	 * 押されたキーが数字または '-' である場合のみ入力を許します。
	 * @param event イベント。
	 */
	ITextInput.chkInt = function(event) {
		event = IWidget.editEvent(event);
		if(event.keyCode>47 && event.keyCode<58) {
			event.returnValue=true;
		}
		else if(event.keyCode==45 && event.srcElement.value.indexOf("-",0)==-1) {
			event.returnValue=true;
		}
		else {
			event.returnValue=false;
		}
		return event.returnValue;
	}

	/*
	 * 小数項目用インプット値チェック。
	 * 押されたキーが数字または '-' であるか，'.' である場合のみ入力を許します。
	 * @param event イベント。
	 */
	ITextInput.chkDec = function(event) {
		event = IWidget.editEvent(event);
		if(chkInt(event)) {
			return true;
		}
		else if(event.keyCode==46 && event.srcElement.value.indexOf(".",0)==-1) {
			event.returnValue=true;
		}
		else {
			event.returnValue=false;
		}
		return event.returnValue;
	}

	/**
	 * カンマ削除。
	 * 指定された obj 項目の中からカンマ文字を削除します。
	 * @param obj カンマ文字削除対象項目。
	 */
	ITextInput.delComma = function(obj) {
		try {
			obj.value = obj.value.split(",").join("");
			obj.select();
		}
		catch(e) {
		}
	}

	/**
	 * エンターキーでリクエスト実行を行なう。
	 */
	ITextInput.requestOnEnter = function(event, requestId) {
		event = IWidget.editEvent(event);
		if(event.keyCode == 13) {
			new CongaRequest().send(requestId);
		}
	}

	//// ローカルメソッド定義

	/*
	 * アラート表示。
	 * アラートを表示し，obj 項目にカーソルを移動します。
	 * @param errorMsg エラーメッセージ。
	 * @param obj カーソルを移動する項目。
	 */
	function showAlert(event, errorMsg, obj) {
		event = IWidget.editEvent(event);
		alert(errorMsg);
		event.returnValue = false;
		obj.value = "";
		obj.focus();
	}

}
new ITextInput();


/**
 * ITree を作成する。
 * @param htmlElement 
 */
function ITree(htmlElement) {

	//// ローカル変数定義

	var handler = this;


	//// プロパティ定義

	/**
	 * HTML実体要素オブジェクト。
	 */
	this.htmlElement = htmlElement;

	/**
	 * 選択ノードを格納するための hidden タグ。
	 */
	this.selectedNodeHolder = null;

	/**
	 * ノード選択時（非オープン/非クローズ時）に呼び出されるリクエスト。
	 */
	this.requestOnselected = null;

	/**
	 * ノードオープン時に呼び出されるリクエスト。
	 */
	this.requestOnopening = null;

	/**
	 * ノードクローズ時に呼び出されるリクエスト。
	 */
	this.requestOnclosing = null;

	/**
	 * 子ノード読み込み時（子ノードの存在しないノードがオープンされたとき）
	 * に呼び出されるリクエスト。
	 */
	this.requestOnloading = null;


	//// public メソッド定義

	/**
	 * 指定されたノードに子ノードを追加する。
	 * @param parentNode 追加先ノード。
	 * @param childNode 追加する子ノード。
	 */
	this.addChildNode = function(parentNode, childNode) {
		var children = parentNode.childNodes;
		for(var i=children.length-1; i >= 0; i--) {
			var child = children.item(i);
			if(child.className=="children") {
				child.appendChild(childNode);
				break;
			}
		}
	}

	/**
	 * 指定されたノードを更新する。
	 * @param currentNode 更新するノード。
	 * @param newNode 更新内容。HTML文字列でも指定可。
	 */
	this.updateNode = function(currentNode, newNode) {
		var parent = currentNode.parentNode;
		var children = parent.childNodes;
		var index = -1;
		for(var i=0; i<children.length; i++) {
			if(children.item(i)==currentNode) {
				index = i;
				break;
			}
		}
		if(typeof(newNode)=="string") {
			currentNode.outerHTML = newNode;
		}
		else {
			currentNode.outerHTML = newNode.outerHTML;
		}
		if(index>-1) {
			var children = parent.childNodes.item(index).childNodes;
			for(var i=0; i<children.length; i++) {
				if(children.item(i).tagName=="INPUT") {
					children.item(i).focus();
					children.item(i).select();
				}
			}
		}
	}

	/**
	 * ノードの input 部分をクリックしたときに呼び出すための
	 * イベントハンドラ。
	 */
	ITree.input_onclick = function(event) {
		event = IWidget.editEvent(event);
		var input = event.target;
		selectNode(input.parentNode, true, false);
	}

	/**
	 * ノードの input 部分でキー入力したときに呼び出すための
	 * イベントハンドラ。
	 */
	ITree.input_onkeydown = function(event) {
		event = IWidget.editEvent(event);
		var input = event.target;

		if(event.keyCode==9) { // TAB
			selectNode(input.parentNode, false, false);
		}
		else if(event.keyCode==40) { // down
			event.keyCode = 9;
		}
		else if(event.keyCode==38) { // up
			input = prevFocus(input);
			if(input) {
				selectNode(input.parentNode, false, false);
			}
		}
		else if(event.keyCode==37) { // left
			selectNode(input.parentNode, false, true);
		}
		else if(event.keyCode==39) { // right
			selectNode(input.parentNode, true, false);
		}
	}

	/**
	 * ノードのアイコン画像部分をクリックしたときに呼び出すための
	 * イベントハンドラ。
	 */
	ITree.icon_onclick = function (event) {
		event = IWidget.editEvent(event);
		var node = event.target;
		if(node.tagName!="SPAN") {
			node = node.parentNode;
		}
		var input = getInput(node);
		if(input) {
			input.focus();
			selectNode(node.parentNode, true, false);
		}
	}

	/**
	 * ノードの＋－画像をクリックしたときに呼び出すための
	 * イベントハンドラ。
	 */
	ITree.opener_onmousedown = function(event) {
		event = IWidget.editEvent(event);
		var node = event.target;
		if(node.tagName!="SPAN") {
			node = node.parentNode;
		}
		var input = getInput(node);
		if(input) {
			input.focus();
			selectNode(node.parentNode, true, true);
		}
	}



	//// イベント定義

	/**
	 * ノード選択時（非オープン/非クローズ時）に呼び出されるイベントハンドラ。
	 */
	this.onselected = null;

	/**
	 * ノードオープン時に呼び出されるイベントハンドラ。
	 */
	this.onopening = null;

	/**
	 * ノードクローズ時に呼び出されるイベントハンドラ。
	 */
	this.onclosing = null;

	/**
	 * 子ノード読み込み時（子ノードの存在しないノードがオープンされたとき）
	 * に呼び出されるリクエスト。
	 */
	this.onloading = null;



	//// ローカルメソッド定義

	/**
	 * ITree の初期化を行なう。
	 */
	function init() {
		if(htmlElement) {
			IWidget.initHtmlElement(handler, htmlElement);
			htmlElement.style.overflow = "auto";
			handler.selectedNodeHolder = IWidget.getValueHolder(htmlElement, "tree");
		}
	}

	/**
	 * 指定されたタグ内にある画像の表示非表示を切り替える。
	 */
	function changeIconImage(node) {
		var children = node.getElementsByTagName("img");
		if(children.length < 2) {
			return;
		}
		for(var i=0; i < children.length; i++) {
			var child = children.item(i);
			if(child.style.display=="none") {
				child.style.display = "inline";
			}
			else {
				child.style.display = "none";
			}
		}
	}

	/**
	 * 指定されたタグ内にある画像の表示非表示を切り替える。
	 */
	function changeIcons(node) {
		var children = node.childNodes;
		for(var i=0; i < children.length; i++) {
			var child = children.item(i);
			if(child.className=="opener" || child.className=="icon") {
				changeIconImage(child);
			}
		}
	}

	/**
	 * 指定されたノードの子ノードを開く。
	 * @param node 対象となるノード。
	 * @param child 子ノード。
	 */
	function openNode(node, child) {
		child.style.display = "block";
		changeIcons(node);
	}

	/**
	 * 指定されたノードの子ノードを閉じる。
	 * @param node 対象となるノード。
	 * @param child 子ノード。
	 */
	function closeNode(node, child) {
		child.style.display = "none";
		changeIcons(node);
	}


	/**
	 * 指定された HTML 要素オブジェクトからハンドラの検索を行なう。
	 */
	function findHandler(obj) {
		while(obj) {
			if(obj.handler) {
				return obj.handler;
			}
			obj = obj.parentNode;
		}
	}

	/**
	 * selectedNodeHandler.value に selectedNode.id を追加し，
	 * eventHandler で指定されたファンクション呼び出し，もしくは，
	 * request で指定されたリクエスト呼び出しを実行する。
	 */
	function execute(handler, selectedNode, eventHandler, request) {
		var id = selectedNode.id;
		handler.selectedNodeHolder.value = handler.selectedNodeHolder.value +","+ id;
		var name = handler.selectedNodeHolder.name;
		var value = handler.selectedNodeHolder.value;

		if(eventHandler) {
			eventHandler(selectedNode);
		}
		if(request) {
			if(new CongaRequest().send(request, name + "=" + value)) {
				// 通信が正常終了した場合のみ，クリアする。
				handler.selectedNodeHolder.value = "";
			}
		}
	}


	/**
	 * 指定されたノードを選択状態にする。
	 * @param node 選択状態にするノード。
	 * @param open ノードのオープンが可能かどうか。
	 * @param close ノードのクローズが可能かどうか。
	 */
	function selectNode(node, open, close) {
		var child = node.getElementsByTagName("div").item(0);

		var handler = findHandler(node);
		var eventHandler = null;
		var request = null;

		if(child) {
			if(child.style.display == "block") {
				if(close) {
					closeNode(node, child);
					eventHandler = handler.onclosing;
					request = handler.requestOnclosing;
				}
			}
			else {
				if(open) {
					openNode(node, child);
					if(!hasChild(child)) {
						eventHandler = handler.onloading;
						request = handler.requestOnloading;
					}
					if(eventHandler==null) {
						eventHandler = handler.onopening;
					}
					if(request==null) {
						request = handler.requestOnopening;
					}
				}
			}
		}

		if(!eventHandler) {
			eventHandler = handler.onselected;
		}

		if(!request) {
			request = handler.requestOnselected;
		}

		execute(handler, node, eventHandler, request);
	}

	/**
	 * 指定されたノードに子ノードが含まれるかどうかを調べる。
	 */
	function hasChild(node) {
		return node.firstChild && node.firstChild.tagName=="DIV";
	}

	/**
	 * 指定されたノードの中から input タグオブジェクトを取得する。
	 */
	function getInput(node) {
		node = node.nextSibling;
		while(node) {
			if(node.tagName=="INPUT") {
				return node;
			}
			node = node.nextSibling;
		}
		return null;
	}

	/**
	 * より上のセルにフォーカスを当てる。
	 */
	function prevFocus(input) {
		if(!input) {
			return null;
		}
		var originalInput = input;
		var node = input.parentNode;

		var foundOriginal = false;
		var prevNode = node.previousSibling;
		if(prevNode) {
			foundOriginal = true;
			node = prevNode;
		}
		else {
			node = node.parentNode; // children
			if(node && node.parentNode) {
				node = node.parentNode;
			}
			else {
				node = null;
			}
		}

		if(!node) {
			return;
		}

		// input を探す。
		if(node.getElementsByTagName) {
			var children = node.getElementsByTagName("input");
			for(var i=children.length-1; i >=0 ; i--) {
				var input = children.item(i);
				if(foundOriginal) {
					try {
						input.focus();
						return input;
					}
					catch(e) {
					}
				}
				else {
					if(input==originalInput) {
						foundOriginal = true;
					}
				}
			}
		}

		return null;
	}

	//// インスタンスの初期化
	init();
}
new ITree();


/**
 * IWidget。
 */
function IWidget() {

	//// public メソッド定義

	/**
	 * イベント発生元オブジェクトおよびイベント発生元位置をプロパティ設定した
	 * イベントオブジェクトを取得する。
	 * このメソッドを使用して取得したイベントオブジェクトは以下の状態になる。
	 *   ・イベント発生元オブジェクトが target プロパティとして設定されている。
	 *   ・イベント発生元位置が posX, posY プロパティとして設定されている。
	 * @param event 編集元イベント。
	 * @return 編集されたイベント。
	 */
	IWidget.editEvent = function(event) {
		if(!event) {
			event = window.event;
		}

		if(!event.target) {
			event.target = event.srcElement;
		}

		if(event.pageX) {
			event.posX = event.pageX;
			event.posY = event.pageY;
		}
		else {
			event.posX = event.clientX;
			event.posY = event.clientY;
		}

		return event;
	}


	/**
	 * 指定されたコンテナの内容として指定されたオブジェクトを設定する。
	 * @param container 設定先。
	 * @param obj 設定するオブジェクト。
	 *  HTML 文字列，HTML要素オブジェクト，
	 *  htmlElement プロパティを持つオブジェクトのいずれかを指定可能。
	 */
	IWidget.setContent = function(container, obj) {
		if(typeof(obj)=="string") {
			container.innerHTML = obj;
		}
		else if(obj.htmlElement) {
			container.appendChild(obj.htmlElement);
		}
		else {
			container.appendChild(obj);
		}
	}

	/**
	 * 指定されたHTML要素の幅設定を行なう。
	 * @param htmlElement 設定するHTML要素。
	 * @param width 幅。
	 * @return 指定された幅が設定できた場合は true。できなかった場合は false。
	 */
	IWidget.setWidth = function(htmlElement, width) {
		width = parseInt(width);
		if(width < 0) {
			return false;
		}

		if(width > htmlElement.offsetWidth) {
			htmlElement.style.width = width;
			if(htmlElement.handler && htmlElement.handler.setClientWidth) {
//				htmlElement.handler.setClientWidth(width - IWidget.getMarginWidth(htmlElement));
				htmlElement.handler.setClientWidth(width);
			}
		}
		else {
			if(htmlElement.handler && htmlElement.handler.setClientWidth) {
				if(!htmlElement.handler.setClientWidth(width - IWidget.getMarginWidth(htmlElement))) {
					if(htmlElement.style.overflow!="auto" && 
							htmlElement.style.overflow!="scroll"){
						return false;
					}
				}
			}
			var oWidth = htmlElement.offsetWidth;

			htmlElement.style.width = width;
			if(htmlElement.offsetWidth!=width) {
				htmlElement.style.width = oWidth;
				return false;
			}
		}
		return true;
	}

	/**
	 * 指定された HTML 要素の 
	 * borderLeftWidth + borderRightWidth + paddingLeft + paddingRight
	 * を返す。
	 */
	IWidget.getMarginWidth = function(htmlElement) {
		return getInt(htmlElement.style.borderLeftWidth) + 
				getInt(htmlElement.style.borderRightWidth) +
				getInt(htmlElement.style.paddingLeft) +
				getInt(htmlElement.style.paddingRight);
	}


	/**
	 * 指定されたHTML要素の高さ設定を行なう。
	 * @param htmlElement 設定するHTML要素。
	 * @param height 高さ。
	 * @return 指定された高さが設定できた場合は true。できなかった場合は false。
	 */
	IWidget.setHeight = function(htmlElement, height) {
		height = parseInt(height);
		if(height < 0) {
			return false;
		}

		if(height > htmlElement.offsetHeight) {
			htmlElement.style.height = height;
			if(htmlElement.handler && htmlElement.handler.setClientHeight) {
//				htmlElement.handler.setClientHeight(height - IWidget.getMarginHeight(htmlElement));
				htmlElement.handler.setClientHeight(height);
			}
		}
		else {
			if(htmlElement.handler && htmlElement.handler.setClientHeight) {
				if(!htmlElement.handler.setClientHeight(height - IWidget.getMarginHeight(htmlElement))) {
					if(htmlElement.style.overflow!="auto"&& 
							htmlElement.style.overflow!="scroll") {
						return false;
					}
				}
			}
			var oHeight = htmlElement.offsetHeight;
			htmlElement.style.height = height;
			if(htmlElement.offsetHeight!=height) {
				htmlElement.style.height = oHeight;
				return false;
			}
		}
		return true;
	}

	/**
	 * 指定された HTML 要素の 
	 * borderTopWidth + borderBottomWidth + paddingTop + paddingBottom
	 * を返す。
	 */
	IWidget.getMarginHeight = function(htmlElement) {
		return getInt(htmlElement.style.borderTopWidth) + 
				getInt(htmlElement.style.borderBottomWidth) +
				getInt(htmlElement.style.paddingTop) +
				getInt(htmlElement.style.paddingBottom);
	}

	function getInt(value) {
		var v = parseInt(value);
		if(isNaN(v)) {
			return 0;
		}
		else {
			return v;
		}
	}

	/**
	 * htmlElement で指定されたウインドウやコンテキストメニューを
	 * マウスポインタ付近に表示できるようにするために，
	 * 指定されたイベントの posX, posY を設定する。
	 * @param event 設定するイベント。
	 * @param htmlElement HTML要素。
	 * @return posX, posY が設定されたイベント。
	 */
	IWidget.getWindowPosition = function(event, htmlElement) {
		var screenWidth;
		var screenHeight;
		if(document.body.clientWidth) {
			screenWidth = document.body.clientWidth;
			screenHeight = document.body.clientHeight;
		}
		else {
			screenWidth = window.innerWidth;
			screenHeight = window.innerHeight;
		}

		if(screenWidth - event.posX < htmlElement.offsetWidth){
			event.posX = document.body.scrollLeft + event.posX - htmlElement.offsetWidth;
		} else {
			event.posX = document.body.scrollLeft + event.posX;
		}
		if(screenHeight - event.posY < htmlElement.offsetHeight){
			event.posY = document.body.scrollTop + event.posY - htmlElement.offsetHeight;
		} else {
			event.posY = document.body.scrollTop + event.posY;
		}

		return event;
	}

	/**
	 * ハンドラオブジェクトをHTML要素オブジェクトの handler プロパティとして設定する。
	 * @param handler ハンドラオブジェクト。
	 * @param htmlElement HTML要素オブジェクト。
	 * @param movable 移動可能オブジェクトとするかどうか。[デフォルト=false]
	 */
	IWidget.initHtmlElement = function(handler, htmlElement, movable) {
		htmlElement.handler = handler;
		if(movable) {
			htmlElement.movable = true;
		}
		else {
			htmlElement.movable = false;
		}
	}

	/**
	 * 最も大きい zindex を返す。
	 * @return 最も大きい zindex を返す。
	 */
	IWidget.getMaxZIndex = function() {
		if(IWindow.prototype.maxZIndex) {
			IWindow.prototype.maxZIndex = parseInt(IWindow.prototype.maxZIndex) + 1;
		}
		else {
			IWindow.prototype.maxZIndex = 1;
		}
		return IWindow.prototype.maxZIndex;
	}

	/**
	 * 指定されたオブジェクトをゆっくりと非表示にする。
	 * @param htmlElement 非表示にするオブジェクト。
	 * @param func 非表示になったときに呼び出される関数（引数はなし）。
	 */
	IWidget.hideSlowly = function(htmlElement, func) {
		htmlElement.style.filter = "alpha(opacity=40)"
		setTimeout(
			function(){
				htmlElement.style.filter = "alpha(opacity=20)"
				setTimeout(function() {
					htmlElement.style.visibility = "hidden";
					if(func) {
						func();
					}
				},80);
			}
		, 80);
	}

	/**
	 * 値の格納用オブジェクトを取得する。
	 * input type="hidden" タグを htmlElement の中から検索し，
	 * 見つかった場合は，それを返し，見つからなかった場合は，
	 * 新規で作成して返す。
	 * @param htmlElement HTML要素オブジェクト。
	 * @param name input タグを新規作成する場合につける名前
	 */
	IWidget.getValueHolder = function(htmlElement, name) {
		var input = findHidden(htmlElement, name);

		// 見つからなかったら作成する。
		if(input==null) {
			input = document.createElement("input");
			input.type = "hidden";
			input.name = name;
		}

		input.value = "";
		return input;
	}

	/**
	 * 画面カバーを作成する。
	 */
	IWidget.createDisplayCover = function(parent, opac) {
		var cover = document.createElement("div");
		cover.style.backgroundColor = "white";
		if(opac) {
			cover.style.filter = "alpha(opacity=50)";
		}
		cover.style.display = "block";
		cover.style.position = "absolute";
		cover.style.left = 0;
		cover.style.top = 0;
		if(!parent) {
			parent = document.body;
		}
		parent.appendChild(cover);
		cover.style.width = document.body.clientWidth;
		cover.style.height = document.body.clientHeight;
		cover.style.cursor = "wait";
		return cover;
	}

	/**
	 * オブジェクト取得を行なう。
	 */
	IWidget.getHtmlElement = function(obj) {
		if(!obj) {
			return null;
		}

		if(typeof(obj)=="string") {
			return document.getElementById(obj);
		}
		else {
			return obj;
		}
	}

	/**
	 * 入力ヘルパを消去する。
	 */
	IWidget.hideInputHelper = function(exception, target) {
		if(exception!="IHelpMessage") {
			IHelpMessage.hide();
		}
		if(exception!="IAssistant") {
			IAssistant.hide(target);
		}
		if(exception!="ICalendar") {
			ICalendar.hide();
		}
	}

	/**
	 * 入力ヘルパの位置設定を行なう。
	 */
	IWidget.setInputHelperPosition = function(target, htmlElement, marginLeft, marginTop) {
		if(marginLeft==undefined) {
			marginLeft = 0;
		}
		if(marginTop==undefined) {
			marginTop = 0;
		}

		var left = target.offsetLeft + marginLeft;
		var top = target.offsetTop + target.offsetHeight + marginTop;
		var obj = target.offsetParent;


		while(obj!=null) {
			left += obj.offsetLeft;
			top += obj.offsetTop;
			obj = obj.offsetParent;
		}
		htmlElement.style.left = left;
		htmlElement.style.top = top;
	}

	//// ローカルメソッド定義

	/**
	 * IWidget を初期化する。
	 */
	function init() {

		document.onmousedown = function (event) {
			event = IWidget.editEvent(event);
			this.movable = getMovableObject(event.target);
			if (!this.movable) {
				return true;
			}

			this.movable.startX = event.posX - this.movable.handler.htmlElement.offsetLeft;
			this.movable.startY = event.posY - this.movable.handler.htmlElement.offsetTop;

			document.onmousemove = movable_onmousemove;
			document.onmouseup = movable_onmouseup;
			event.cancelBubble = true;
			return false;
		}

		function movable_onmouseup(event) {
			hideCover();

			document.onmousemove = null;
			document.onmouseup = null;
			this.movable = null;
			return true;
		}

		function movable_onmousemove(event) {
			if (!this.movable || !this.movable.handler || 
					!this.movable.handler.htmlElement || !this.movable.startX) {
				hideCover();
				this.movable = null;
				return true;
			}

			event = IWidget.editEvent(event);

			if(!event.which && event.button!=1) {
				hideCover();
				this.movable = null;
				return true;
			}

			initCover();

			if(!this.movable.type || this.movable.type=="x") {
				this.movable.handler.htmlElement.style.left = 
					event.posX - this.movable.startX;
			}
			if(!this.movable.type || this.movable.type=="y") {
				this.movable.handler.htmlElement.style.top = 
					event.posY - this.movable.startY;
			}

			if(this.movable.handler.onmove) {
				this.movable.handler.onmove(
					parseInt(this.movable.handler.htmlElement.style.left), 
					parseInt(this.movable.handler.htmlElement.style.top), 
				event);
			}

			event.cancelBubble = true;
			return false;
		}

		function initCover() {
			if(!document.cover) {
				var cover = document.createElement("div");
				//cover.style.backgroundColor = "transparent";
				cover.style.backgroundColor = "yellow";
				cover.style.filter = "alpha(opacity=0)"
				cover.style.display = "none";
				cover.style.position = "absolute";
				cover.style.left = 0;
				cover.style.top = 0;
				if(document.body.clientWidth) {
					cover.style.width = document.body.clientWidth;
					cover.style.height = document.body.clientHeight;
				}
				else {
					cover.style.width = window.innerWidth;
					cover.style.height = window.innerHeight;
				}
				document.body.appendChild(cover);
				document.cover = cover;
			}
			document.cover.style.zIndex = 999999999999;
			document.cover.style.display = "block";
		}

		function hideCover() {
			if(document.cover) {
				document.cover.style.display = "none";
			}
		}


		function getMovableObject(obj) {
			while(obj) {
				if(obj.handler && obj.handler.htmlElement && obj.movable) {
					return obj;
				}
				obj = obj.parentNode;
			}
			return null;
		}
	}

	/**
	 * input type="hidden" タグを htmlElement の中から検索する。
	 * @param htmlElement HTML要素オブジェクト。
	 * @return input オブジェクト。
	 */
	function findHidden(htmlElement, name) {
		var children = htmlElement.childNodes;
		for(var i=0; i < children.length; i++) {
			var child = children.item(i);
			if(child.tagName=="INPUT" || child.type=="hidden") {
				return child;
			}
		}
		return null;
	}

	//// インスタンスの初期化
	init();
}
new IWidget();


/**
 * 内部ウインドウを作成する。
 * @param titleText ウインドウタイトル文字列。
 * @param content ウインドウ内に表示する内容。
 * @param hidden 初期表示の際にウインドウを隠すかどうか。(未指定の場合は false)
 * @param parent ウインドウの親要素。
 * @param modal モーダルウインドウとして使用するかどうか。
 */
function IWindow(titleText, content, hidden, parent, modal) {
	if(!document.body) {
		return;
	}

	//// 変数定義
	var handler = this;
	var modalCover = createModalCover(parent, modal);
	var win = createWindow(parent); // ウインドウ
	var titleBar = createTitleBar(win);// タイトルバー
	var icon = createTitleBarElement(titleBar, "icon"); // タイトルバーアイコン
	var title = createTitleBarElement(titleBar, "title"); // タイトル文字列
	var minimizeButton = createTitleBarElement(titleBar, "minimize-button"); // 最小化ボタン
	var closeButton = createTitleBarElement(titleBar, "close-button"); // Close ボタン
	var contentContainer = createContentContainer(win); // 内容表示領域

	var bottomBorder = createBorder("h", // 下枠
		function(left, top) {
			var h = top - parseInt(win.style.top);
			handler.setBounds(win.style.left, win.style.top, win.style.width, h);
		}
	);

	var rightBorder = createBorder("v", // 右枠
		function(left, top) {
			var w = left - parseInt(win.style.left);
			handler.setBounds(win.style.left, win.style.top, w, win.style.height);
		}
	);

	var topBorder = createBorder("h", // 上枠
		function(left, top) {
			var h = parseInt(win.style.height) - (top - parseInt(win.style.top));
			handler.setBounds(win.style.left, top, win.style.width, h);
		}
	);

	var leftBorder = createBorder("v", // 左枠
		function(left, top) {
			var w = parseInt(win.style.width) - (left - parseInt(win.style.left));
			handler.setBounds(left, win.style.top, w, win.style.height);
		}
	);

	var rightBottom = createCorner("nw-resize", // 右下角
		function(left, top) {
			var h = top - parseInt(win.style.top);
			var w = left - parseInt(win.style.left);
			handler.setBounds(parseInt(win.style.left), parseInt(win.style.top), w, h);
		}
	);

	var leftBottom = createCorner("sw-resize", // 左下角
		 function(left, top) {
			var h = top - parseInt(win.style.top);
			var w = parseInt(win.style.width) - (left - parseInt(win.style.left));
			handler.setBounds(left, parseInt(win.style.top), w, h);
		}
	);

	var rightTop = createCorner("ne-resize", // 右上角
		function(left, top) {
			var h = parseInt(win.style.height) - (top - parseInt(win.style.top));
			var w = left - parseInt(win.style.left);
			handler.setBounds(parseInt(win.style.left), top, w, h);
		}
	);

	var leftTop = createCorner("nw-resize", // 左上角
		function(left, top) {
			var h = parseInt(win.style.height) - (top - parseInt(win.style.top));
			var w = parseInt(win.style.width) - (left - parseInt(win.style.left));
			handler.setBounds(left, top, w, h);
		}
	);

	var cover = createCover(win);
	this._cover = cover;
	this._modalCover = modalCover;

	//// プロパティ定義

	/**
	 * HTML実体要素オブジェクト。
	 */
	this.htmlElement = win;

	//// プロパティ定義

	/**
	 * ウインドウ表示内容を設定するコンテナオブジェクト。
	 */
	this.contentContainer = contentContainer;

	/**
	 * リサイズ可能かどうか。
	 */
	this.resizable = true;
	
	/**
	 * 最小化表示中かどうか。
	 * ウィンドウが表示中で最小化表示されているときのみ true が格納される。
	 */
	this.minimize = false;



	//// public メソッド定義

	/**
	 * ウインドウの表示を行なう。
	 */
	this.show = function(event) {
		if(event) {
			event = IWidget.editEvent(event);
			event = IWidget.getWindowPosition(event, win);
			handler.setPosition(event.posX, event.posY);
		}
		win.style.visibility = "visible";
		if(handler.resizable) {
			setVisibleBorders(win.style.visibility);
		}
		initzIndex();
		win.style.filter = "alpha(opacity=100)"
		this.minimize = false;
		return handler;
	}

	/**
	 * ウインドウを非表示にする。
	 */
	this.hide = function() {
		this.minimize = true;
		IWidget.hideSlowly(win,
			function() {
				setVisibleBorders(win.style.visibility);
				if(handler.onhide) {
					handler.onhide();
				}
			}
		);
		return handler;
	}
	
	/**
	 * ウィンドウが表示中かどうか取得する。
	 * @return true 表示中, false 非表示中
	 */
	this.isVisible = function() {
		return (win.style.visibility == "visible");
	}

	/**
	 * ウインドウの表示状態を変更する。
	 * @param visible true の場合は表示，false の場合は非表示。
	 * 
	 */
	this.setVisible = function(visible) {
		if(visible) {
			handler.show();
		}
		else {
			win.style.visibility = "hidden";
			setVisibleBorders(win.style.visibility);
		}
	}

	/**
	 * このウインドウを最前面に表示する。
	 */
	this.toFront = initzIndex;

	/**
	 * タイトルバーを非表示にする。
	 */
	this.hideTitleBar = function() {
		titleBar.style.display = "none";
		handler.pack();
		return handler;
	}

	/**
	 * 表示位置を設定する。
	 * @param left 横位置。
	 * @param top 縦位置。
	 */
	this.setPosition = function(left, top) {
		setPosition0(win, left, top);
		initBorders();
		return handler;
	}

	/**
	 * サイズを設定する。
	 * @param width 幅。
	 * @param height 高さ。
	 */
	this.setSize = function(width, height) {
		width = parseInt(width);
		height = parseInt(height);
		if(height <= getTitleBarHeight()) {
			initBorders();
			return handler;
		}
		if(width <= 40) {
			initBorders();
			return handler;
		}

		win.style.width = width;
		win.style.height = height;
		IWidget.setWidth(contentContainer, width - 4);
		IWidget.setHeight(contentContainer, height - getTitleBarHeight() - 2);

		initBorders();
		return handler;
	}


	/**
	 * 表示位置とサイズをまとめて設定する。
	 * @param left 横位置。
	 * @param top 縦位置。
	 * @param width 幅。
	 * @param height 高さ。
	 */
	this.setBounds = function(left, top, width, height) {
		setPosition0(win, left, top);
		handler.setSize(width, height);
		return handler;
	}

	/**
	 * 表示内容のサイズに合わせて，ウインドウのサイズを調整する。
	 */
	this.pack = function() {
		var obj = contentContainer.firstChild;

		if(!obj) {
			return handler;
		}

		var objWidth = obj.offsetWidth;
		var objHeight = obj.offsetHeight;
		if(!objWidth) {
			objWidth = 100;
		}
		if(! objHeight) {
			objHeight = 100;
		}

		handler.setBounds(win.offsetLeft, win.offsetTop, 
			objWidth + 5, getTitleBarHeight() + objHeight + 2);
		return handler;
	}

	/**
	 * ウインドウのサイズを可変にするかどうかを設定する。
	 * @param resizable 可変にする場合は true。
	 */
	this.setResizable = function(resizable) {
		setVisibleBorders((resizable)?"visible":"hidden");
		handler.resizable = resizable;
		return handler;
	}
	
	/**
	 * ウインドウが最小化表示中かどうか。
	 * @return true 最小化, false 最小化以外
	 */
	this.isMinimize = function() {
		return this.minimize;
	}
	
	/**
	 * スクロールバー表示をするかしないかを設定する。
	 * @param scroll スクロールバー表示をする場合は true を指定する。
	 */
	this.setScroll = function(scroll) {
		if(scroll=="auto" || scroll=="scroll" || scroll=="hidden") {
			contentContainer.style.overflow = scroll;
		}
		else if(scroll=="true") {
			contentContainer.style.overflow = "scroll";
		}
		else if(scroll=="false") {
			contentContainer.style.overflow = "hidden";
		}
		else if(scroll) {
			contentContainer.style.overflow = "scroll";
		}
		else {
			contentContainer.style.overflow = "hidden";
		}
		return handler;
	}



	/**
	 * ウインドウのタイトルを設定する。
	 * @param text ウインドウのタイトル文字列。
	 */
	this.setTitle = function(text) {
		title.innerHTML = text;
		return handler;
	}

	/**
	 * ウインドウのタイトルを取得する。
	 * @return ウインドウのタイトル文字列。
	 */
	this.getTitle = function() {
		return title.innerHTML;
	}

	/**
	 * タイトルバーのアイコンを設定する。
	 * @param iconURL アイコン画像を示す URL。
	 */
	this.setIcon = function(iconURL) {
		icon.style.backgroundImage = iconURL;
		return handler;
	}

	/**
	 * タイトルバーのアイコンを取得するする。
	 * @return iconURL アイコン画像を示す URL。
	 */
	this.getIcon = function() {
		return icon.style.backgroundImage;
	}

	/**
	 * 表示内容設定を行なう。
	 * @param obj HTML文字列 / HTML 要素 / htmlElement プロパティを持つオブジェクト。
	 */
	this.setContent = function(obj) {
		IWidget.setContent(contentContainer, obj);
		handler.pack();
		return handler;
	}

	/**
	 * 指定された URL のドキュメントを表示内容として設定する。
	 * @param url 表示する URL。
	 */
	this.setContentByURL = function(url) {
		var iframe = document.createElement("iframe");
		iframe.src = url;
		handler.setContent(iframe);
		return handler;
	}

	/**
	 * ウインドウを画面中央に表示する。
	 */
	this.center = function() {
		var left = (document.body.clientWidth - win.offsetWidth) / 2;
		var top = (document.body.clientHeight - win.offsetHeight) / 2;
		handler.setPosition(left, top);
	}

	/**
	 * 最小化ボタンを利用可能とするかどうかを設定する。
	 * @param value 利用可能とする場合は true。
	 */
	this.setEnableMinimizeButton = function(value) {
		if(value) {
			minimizeButton.style.display = 'inline';
			minimizeButton.onlick = function() {handler.hide();};
		}
		else {
			minimizeButton.style.display = 'none';
		}
	}

	/**
	 * クローズボタンを利用可能とするかどうかを設定する。
	 * @param value 利用可能とする場合は true。
	 */
	this.setEnableCloseButton = function(value) {
		if(value) {
			closeButton.style.display = 'inline';
			closeButton.onlick = function() {handler.close();};
		}
		else {
			closeButton.style.display = 'none';
		}
	}

	/**
	 * ウインドウが非表示になったときに呼び出される。
	 */
	this.onhide = null;

	/**
	 * ウインドウがクローズされる直前に呼び出される。
	 * このイベントハンドラが false を返す場合は，クローズしない。
	 */
	this.onclosing = null;

	/**
	 * ウインドウがクローズされた直後に呼び出される。
	 */
	this.onclosed = null;



	/**
	 * 指定されたタイトルバー文字列のウインドウを取得する。
	 * 同じタイトルバー文字列のウインドウが複数ある場合はどちらかひとつを返す。
	 * @param title タイトルバー文字列。
	 * @return IWindow インスタンス。見つからなかった場合は null。
	 */
	IWindow.findByTitle = function(title) {
		var list = IWindow.prototype.winList;
		for(var i=0; i<list.length; i++) {
			if(title==list[i].getTitle()) {
				return list[i];
			}
		}
		return null;
	}

	/**
	 * ウインドウをクローズする。
	 */
	this.close = function() {
		this.minimize = false;
		if(handler.onclosing) {
			if(!handler.onclosing()) {
				return handler;
			}
		}

		IWidget.hideSlowly(win,
			function() {
				handler.onhide = null;
				handler.onclosing = null;
				handler._cover = null;
				remove(handler._modalCover);
				handler._modalCover = null;

				remove(handler.htmlElement);
				handler.htmlElement = null;

				remove(bottomBorder.htmlElement);
				remove(rightBorder.htmlElement);
				remove(topBorder.htmlElement);
				remove(leftBorder.htmlElement);
				remove(rightBottom.htmlElement);
				remove(leftBottom.htmlElement);
				remove(rightTop.htmlElement);
				remove(leftTop.htmlElement);

				handler.contentContainer = null;

				var list = IWindow.prototype.winList;
				for(var i=0; i<list.length; i++) {
					if(list[i]==handler) {
						list.splice(i, 1);
					}
				}
				if(handler.onclosed) {
					handler.onclosed();
					handler.onclosed = null;
				}
				handler = null;

				function remove(node) {
					if(node && node.handler) {
						if(node.handler.htmlElement) {
							node.handler.htmlElement = null;
						}
						node.handler = null;
					}
					if(node && node.parentNode) {
						node.parentNode.removeChild(node);
					}
				}
			}
		);
		return null;
	}

	/**
	 * ウインドウがクローズされているかどうかを調べる
	 */
	this.isClosed = function() {
		return handler==null;
	}


	//// ローカルメソッド定義

	/**
	 * IWindow の初期化を行なう。
	 */
	function init(titleText) {
		if(!titleText) {
			titleText = "";
		}
		title.innerHTML = titleText;


		if(!IWindow.prototype.winList) {
			IWindow.prototype.winList = new Array();
			IWindow.prototype.hideAll = function() {
				var list = IWindow.prototype.winList;
				for(var i=0; i<list.length; i++) {
					list[i].hide();
				}
			};
			IWindow.prototype.showAll = function() {
				var list = IWindow.prototype.winList;
				for(var i=0; i<list.length; i++) {
					list[i].show();
				}
			};
		}
		IWindow.prototype.winList.push(handler);

		IWidget.initHtmlElement(handler, win);

		titleBar.handler = handler;
		titleBar.movable = true;

		initBorderzIndex();
		minimizeButton.onclick = function() {handler.hide();};
		closeButton.onclick = function() {handler.close();};

		handler.onmove = function(left, top) {
			initBorders();
		}

		if(content) {
			if(typeof(content)=="string") {
				handler.setContent(content);
			}
			else if(content.htmlElement) {
				content = content.htmlElement;
			}

			var left = content.offsetLeft;
			var top = content.offsetTop;
			handler.setContent(content);
			handler.setPosition(left, top);
		}

		if(hidden) {
			handler.setVisible(false);
		}
		initzIndex();
	}


	/**
	 * ウインドウ用 HTML 要素を作成する。
	 * @param parent 親要素。
	 * @return ウインドウ用 HTML 要素。
	 */
	function createWindow(parent) {
		var win = document.createElement("div");
		win.className = "iwindow";
		win.style.position = "absolute";
		win.style.zIndex = IWidget.getMaxZIndex();
		win.onmousedown = function() {initzIndex()};
		if(!parent) {
			parent = document.body;
		}
		parent.appendChild(win);
		return win;
	}

	/**
	 * モーダルウインドウ用カバーを作成する。
	 */
	function createModalCover(parent, modal) {
		var cover = null;
		if(modal) {
			cover = IWidget.createDisplayCover(parent, true);
		}
		return cover;
	}

	/**
	 * ウインドウのタイトルバーを作成する。
	 * @param win ウインドウ用 HTML 要素。
	 * @return ウインドウのタイトルバー。
	 */
	function createTitleBar(win) {
		var titleBar = document.createElement("div");
		titleBar.className = "title-bar"
		win.appendChild(titleBar);
		return titleBar;
	}

	/**
	 * ウインドウタイトルバー内HTML 要素を作成する。
	 * @param titleBar タイトルバー。
	 * @param className クラス名。
	 * @return ウインドウタイトルバー内HTML要素。
	 */
	function createTitleBarElement(titleBar, className) {
		var elem = document.createElement("span");
		elem.display = "inline";
		elem.className = className;
		titleBar.appendChild(elem);
		return elem;
	}

	/**
	 * 内容表示領域を作成する。
	 * @param win ウインドウ用 HTML 要素。
	 * @return 内容表示領域。
	 */
	function createContentContainer(win) {
		var contentContainer = document.createElement("div");
		contentContainer.className = "content";
		contentContainer.style.overflow = "auto";
		win.appendChild(contentContainer);
		new IContainer(contentContainer);
		return contentContainer;
	}

	/**
	 * ウインドウ枠線を作成する。
	 * @param type 枠線タイプ。
	 * @param onmovefunc 枠線が移動されたときに呼び出される関数。
	 */
	function createBorder(type, onmovefunc) {
		var border = new IBorder(document.body, type, 100,10);
		border.onmove = onmovefunc;
		return border;
	}

	/**
	 * ウインドウの角を作成する。
	 * @param cursor カーソルタイプ。
	 * @param onmovefunc 角が移動されたときに呼び出される関数。
	 */
	function createCorner(cursor, onmovefunc) {
		var corner = new IDiv(10,10);
		corner.htmlElement.style.cursor = cursor;
		corner.htmlElement.style.backgroundColor = "";
		corner.onmove = onmovefunc;
		return corner;
	}

	/**
	 * カバー作成。
	 */
	function createCover(win) {
		var cover = document.createElement("div");
		cover.style.backgroundColor = "white";
		cover.style.filter = "alpha(opacity=0)";
		contentContainer.style.filter = "alpha(opacity=100)";
		cover.style.display = "none";
		cover.style.position = "absolute";
		cover.style.left = 0;
		cover.style.top = 0;
		win.appendChild(cover);
		return cover;
	}

	/**
	 * ウインドウの位置を設定する。
	 * @param win 設定するウインドウ。
	 * @param left 横位置。
	 * @param top 縦位置。
	 */
	function setPosition0(win, left, top) {
		left = parseInt(left);
		top = parseInt(top);
		if(!left || left < 1) {
			left = 1;
		}
		if(!top || top < 1) {
			top = 1;
		}

		win.style.left = left;
		win.style.top = top;
	}

	/**
	 * 枠線の表示非表示を設定する。
	 */
	function setVisibleBorders(visibility) {
		bottomBorder.htmlElement.style.visibility = visibility;
		rightBorder.htmlElement.style.visibility = visibility;
		topBorder.htmlElement.style.visibility = visibility;
		leftBorder.htmlElement.style.visibility = visibility;
		rightBottom.htmlElement.style.visibility = visibility;
		leftBottom.htmlElement.style.visibility = visibility;
		rightTop.htmlElement.style.visibility = visibility;
		leftTop.htmlElement.style.visibility = visibility;
	}

	/**
	 * タイトルバーの高さを取得する。
	 */
	function getTitleBarHeight() {
		if(titleBar.style.visibility == "hidden") {
			return 0;
		}
		else {
			return titleBar.offsetHeight;
		}
	}

	/**
	 * 枠線の表示状態を設定する。
	 */
	function initBorders() {
		var left = parseInt(win.style.left);
		var top = parseInt(win.style.top);
		var width = parseInt(win.style.width);
		var height = parseInt(win.style.height);

		bottomBorder.setPosition(left, top+height-5);
		bottomBorder.setLength(width);
		rightBorder.setPosition(left+width-5, top);
		rightBorder.setLength(height);

		topBorder.setPosition(left, top-5);
		topBorder.setLength(width);

		leftBorder.setPosition(left-5, top);
		leftBorder.setLength(height);

		rightBottom.setPosition(left+width-5, top+height-5);
		leftBottom.setPosition(left-5, top+height-5);

		rightTop.setPosition(left+width-5, top-5);
		leftTop.setPosition(left-5, top-5);

		cover.style.top = contentContainer.offsetTop;
		cover.style.left = contentContainer.offsetLeft;
		cover.style.width = contentContainer.style.width;
		cover.style.height = contentContainer.style.height;
	}

	/**
	 * ウインドウの zIndex を設定する。
	 */
	function initzIndex() {
		IWindow.prototype.maxZIndex = parseInt(IWindow.prototype.maxZIndex) + 10;
		win.style.zIndex = IWindow.prototype.maxZIndex;
		initBorderzIndex();
		cover.style.display = "none";
		var list = IWindow.prototype.winList;
		for(var i=0; i<list.length; i++) {
			if(list[i]!=handler) {
				list[i]._cover.style.display = "block";
			}
		}
	}

	/**
	 * border の zIndex を設定する。
	 */
	function initBorderzIndex() {
		bottomBorder.htmlElement.style.zIndex = win.style.zIndex + 1;
		rightBorder.htmlElement.style.zIndex = win.style.zIndex + 2;
		topBorder.htmlElement.style.zIndex = win.style.zIndex + 3;
		leftBorder.htmlElement.style.zIndex = win.style.zIndex + 4;
		rightBottom.htmlElement.style.zIndex = win.style.zIndex + 5;
		leftBottom.htmlElement.style.zIndex = win.style.zIndex + 6;
		rightTop.htmlElement.style.zIndex = win.style.zIndex + 7;
		leftTop.htmlElement.style.zIndex = win.style.zIndex + 8;
	}


	//// インスタンスの初期化
	init(titleText);
}
new IWindow();

/**
 * ICalendar 用リソース登録。
 */
with(IResource) {

	// 表示フォーマット
	add("ICalendar.format", "y-m-d");
	add("ICalendar.format", "y/m/d", "ja");

	// 曜日
	add("ICalendar.sun", "Sun");
	add("ICalendar.mon", "Mon");
	add("ICalendar.tue", "Tue");
	add("ICalendar.wed", "Wed");
	add("ICalendar.thu", "Thu");
	add("ICalendar.fri", "Fri");
	add("ICalendar.sat", "Sat");

	add("ICalendar.sun", "日", "ja");
	add("ICalendar.mon", "月", "ja");
	add("ICalendar.tue", "火", "ja");
	add("ICalendar.wed", "水", "ja");
	add("ICalendar.thu", "木", "ja");
	add("ICalendar.fri", "金", "ja");
	add("ICalendar.sat", "土", "ja");

	// 休日
	add("ICalendar.20060101", "元日", "ja");
	add("ICalendar.20060109", "成人の日", "ja");
	add("ICalendar.20060211", "建国記念日", "ja");
	add("ICalendar.20060321", "春分の日", "ja");
	add("ICalendar.20060429", "みどりの日", "ja");
	add("ICalendar.20060503", "憲法記念日", "ja");
	add("ICalendar.20060505", "子供の日", "ja");
	add("ICalendar.20060717", "海の日", "ja");
	add("ICalendar.20060918", "敬老の日", "ja");
	add("ICalendar.20060923", "秋分の日", "ja");
	add("ICalendar.20061009", "体育の日", "ja");
	add("ICalendar.20061103", "文化の日", "ja");
	add("ICalendar.20061123", "勤労感謝の日", "ja");
	add("ICalendar.20061223", "天皇誕生日", "ja");

}
/**
 * リソース登録。
 */
with(IResource) {

	// Context Menu
	add("copyAll", "Copy All");
	add("copyAll", "全体コピー", "ja");

	add("copy", "Copy");
	add("copy", "コピー", "ja");

	add("cut", "Cut");
	add("cut", "切り取り", "ja");

	add("paste", "Paste");
	add("paste", "貼り付け", "ja");

	add("clear", "Clear");
	add("clear", "クリア", "ja");

	add("close", "Close");
	add("close", "閉じる", "ja");

	// ITextInput 用 Message
	add("ITextInput.input_alnum", "半角英数字を入力してください。", "ja");

	add("ITextInput.input_num", "数値を入力してください。", "ja");

	add("ITextInput.input_int", "整数を入力してください。", "ja");

	add("ITextInput.input_pos", "正の数値を入力して下さい。", "ja");

	add("ITextInput.input_int_with", "整数部%1桁以内で入力してください。", "ja");

	add("ITextInput.input_dec_with", "小数部%1桁以内で入力してください。", "ja");

}
