/**
 * ITree を作成する。
 * @param htmlElement 
 */
function ITree(htmlElement) {

	//// ローカル変数定義

	var handler = this;


	//// プロパティ定義

	/**
	 * HTML実体要素オブジェクト。
	 */
	this.htmlElement = htmlElement;

	/**
	 * 選択ノードを格納するための hidden タグ。
	 */
	this.selectedNodeHolder = null;

	/**
	 * ノード選択時（非オープン/非クローズ時）に呼び出されるリクエスト。
	 */
	this.requestOnselected = null;

	/**
	 * ノードオープン時に呼び出されるリクエスト。
	 */
	this.requestOnopening = null;

	/**
	 * ノードクローズ時に呼び出されるリクエスト。
	 */
	this.requestOnclosing = null;

	/**
	 * 子ノード読み込み時（子ノードの存在しないノードがオープンされたとき）
	 * に呼び出されるリクエスト。
	 */
	this.requestOnloading = null;


	//// public メソッド定義

	/**
	 * 指定されたノードに子ノードを追加する。
	 * @param parentNode 追加先ノード。
	 * @param childNode 追加する子ノード。
	 */
	this.addChildNode = function(parentNode, childNode) {
		var children = parentNode.childNodes;
		for(var i=children.length-1; i >= 0; i--) {
			var child = children.item(i);
			if(child.className=="children") {
				child.appendChild(childNode);
				break;
			}
		}
	}

	/**
	 * 指定されたノードを更新する。
	 * @param currentNode 更新するノード。
	 * @param newNode 更新内容。HTML文字列でも指定可。
	 */
	this.updateNode = function(currentNode, newNode) {
		var parent = currentNode.parentNode;
		var children = parent.childNodes;
		var index = -1;
		for(var i=0; i<children.length; i++) {
			if(children.item(i)==currentNode) {
				index = i;
				break;
			}
		}
		if(typeof(newNode)=="string") {
			currentNode.outerHTML = newNode;
		}
		else {
			currentNode.outerHTML = newNode.outerHTML;
		}
		if(index>-1) {
			var children = parent.childNodes.item(index).childNodes;
			for(var i=0; i<children.length; i++) {
				if(children.item(i).tagName=="INPUT") {
					children.item(i).focus();
					children.item(i).select();
				}
			}
		}
	}

	/**
	 * ノードの input 部分をクリックしたときに呼び出すための
	 * イベントハンドラ。
	 */
	ITree.input_onclick = function(event) {
		event = IWidget.editEvent(event);
		var input = event.target;
		selectNode(input.parentNode, true, false);
	}

	/**
	 * ノードの input 部分でキー入力したときに呼び出すための
	 * イベントハンドラ。
	 */
	ITree.input_onkeydown = function(event) {
		event = IWidget.editEvent(event);
		var input = event.target;

		if(event.keyCode==9) { // TAB
			selectNode(input.parentNode, false, false);
		}
		else if(event.keyCode==40) { // down
			event.keyCode = 9;
		}
		else if(event.keyCode==38) { // up
			input = prevFocus(input);
			if(input) {
				selectNode(input.parentNode, false, false);
			}
		}
		else if(event.keyCode==37) { // left
			selectNode(input.parentNode, false, true);
		}
		else if(event.keyCode==39) { // right
			selectNode(input.parentNode, true, false);
		}
	}

	/**
	 * ノードのアイコン画像部分をクリックしたときに呼び出すための
	 * イベントハンドラ。
	 */
	ITree.icon_onclick = function (event) {
		event = IWidget.editEvent(event);
		var node = event.target;
		if(node.tagName!="SPAN") {
			node = node.parentNode;
		}
		var input = getInput(node);
		if(input) {
			input.focus();
			selectNode(node.parentNode, true, false);
		}
	}

	/**
	 * ノードの＋－画像をクリックしたときに呼び出すための
	 * イベントハンドラ。
	 */
	ITree.opener_onmousedown = function(event) {
		event = IWidget.editEvent(event);
		var node = event.target;
		if(node.tagName!="SPAN") {
			node = node.parentNode;
		}
		var input = getInput(node);
		if(input) {
			input.focus();
			selectNode(node.parentNode, true, true);
		}
	}



	//// イベント定義

	/**
	 * ノード選択時（非オープン/非クローズ時）に呼び出されるイベントハンドラ。
	 */
	this.onselected = null;

	/**
	 * ノードオープン時に呼び出されるイベントハンドラ。
	 */
	this.onopening = null;

	/**
	 * ノードクローズ時に呼び出されるイベントハンドラ。
	 */
	this.onclosing = null;

	/**
	 * 子ノード読み込み時（子ノードの存在しないノードがオープンされたとき）
	 * に呼び出されるリクエスト。
	 */
	this.onloading = null;



	//// ローカルメソッド定義

	/**
	 * ITree の初期化を行なう。
	 */
	function init() {
		if(htmlElement) {
			IWidget.initHtmlElement(handler, htmlElement);
			htmlElement.style.overflow = "auto";
			handler.selectedNodeHolder = IWidget.getValueHolder(htmlElement, "tree");
		}
	}

	/**
	 * 指定されたタグ内にある画像の表示非表示を切り替える。
	 */
	function changeIconImage(node) {
		var children = node.getElementsByTagName("img");
		if(children.length < 2) {
			return;
		}
		for(var i=0; i < children.length; i++) {
			var child = children.item(i);
			if(child.style.display=="none") {
				child.style.display = "inline";
			}
			else {
				child.style.display = "none";
			}
		}
	}

	/**
	 * 指定されたタグ内にある画像の表示非表示を切り替える。
	 */
	function changeIcons(node) {
		var children = node.childNodes;
		for(var i=0; i < children.length; i++) {
			var child = children.item(i);
			if(child.className=="opener" || child.className=="icon") {
				changeIconImage(child);
			}
		}
	}

	/**
	 * 指定されたノードの子ノードを開く。
	 * @param node 対象となるノード。
	 * @param child 子ノード。
	 */
	function openNode(node, child) {
		child.style.display = "block";
		changeIcons(node);
	}

	/**
	 * 指定されたノードの子ノードを閉じる。
	 * @param node 対象となるノード。
	 * @param child 子ノード。
	 */
	function closeNode(node, child) {
		child.style.display = "none";
		changeIcons(node);
	}


	/**
	 * 指定された HTML 要素オブジェクトからハンドラの検索を行なう。
	 */
	function findHandler(obj) {
		while(obj) {
			if(obj.handler) {
				return obj.handler;
			}
			obj = obj.parentNode;
		}
	}

	/**
	 * selectedNodeHandler.value に selectedNode.id を追加し，
	 * eventHandler で指定されたファンクション呼び出し，もしくは，
	 * request で指定されたリクエスト呼び出しを実行する。
	 */
	function execute(handler, selectedNode, eventHandler, request) {
		var id = selectedNode.id;
		handler.selectedNodeHolder.value = handler.selectedNodeHolder.value +","+ id;
		var name = handler.selectedNodeHolder.name;
		var value = handler.selectedNodeHolder.value;

		if(eventHandler) {
			eventHandler(selectedNode);
		}
		if(request) {
			if(new CongaRequest().send(request, name + "=" + value)) {
				// 通信が正常終了した場合のみ，クリアする。
				handler.selectedNodeHolder.value = "";
			}
		}
	}


	/**
	 * 指定されたノードを選択状態にする。
	 * @param node 選択状態にするノード。
	 * @param open ノードのオープンが可能かどうか。
	 * @param close ノードのクローズが可能かどうか。
	 */
	function selectNode(node, open, close) {
		var child = node.getElementsByTagName("div").item(0);

		var handler = findHandler(node);
		var eventHandler = null;
		var request = null;

		if(child) {
			if(child.style.display == "block") {
				if(close) {
					closeNode(node, child);
					eventHandler = handler.onclosing;
					request = handler.requestOnclosing;
				}
			}
			else {
				if(open) {
					openNode(node, child);
					if(!hasChild(child)) {
						eventHandler = handler.onloading;
						request = handler.requestOnloading;
					}
					if(eventHandler==null) {
						eventHandler = handler.onopening;
					}
					if(request==null) {
						request = handler.requestOnopening;
					}
				}
			}
		}

		if(!eventHandler) {
			eventHandler = handler.onselected;
		}

		if(!request) {
			request = handler.requestOnselected;
		}

		execute(handler, node, eventHandler, request);
	}

	/**
	 * 指定されたノードに子ノードが含まれるかどうかを調べる。
	 */
	function hasChild(node) {
		return node.firstChild && node.firstChild.tagName=="DIV";
	}

	/**
	 * 指定されたノードの中から input タグオブジェクトを取得する。
	 */
	function getInput(node) {
		node = node.nextSibling;
		while(node) {
			if(node.tagName=="INPUT") {
				return node;
			}
			node = node.nextSibling;
		}
		return null;
	}

	/**
	 * より上のセルにフォーカスを当てる。
	 */
	function prevFocus(input) {
		if(!input) {
			return null;
		}
		var originalInput = input;
		var node = input.parentNode;

		var foundOriginal = false;
		var prevNode = node.previousSibling;
		if(prevNode) {
			foundOriginal = true;
			node = prevNode;
		}
		else {
			node = node.parentNode; // children
			if(node && node.parentNode) {
				node = node.parentNode;
			}
			else {
				node = null;
			}
		}

		if(!node) {
			return;
		}

		// input を探す。
		if(node.getElementsByTagName) {
			var children = node.getElementsByTagName("input");
			for(var i=children.length-1; i >=0 ; i--) {
				var input = children.item(i);
				if(foundOriginal) {
					try {
						input.focus();
						return input;
					}
					catch(e) {
					}
				}
				else {
					if(input==originalInput) {
						foundOriginal = true;
					}
				}
			}
		}

		return null;
	}

	//// インスタンスの初期化
	init();
}
new ITree();


