/**
 * IWidget。
 */
function IWidget() {

	//// public メソッド定義

	/**
	 * イベント発生元オブジェクトおよびイベント発生元位置をプロパティ設定した
	 * イベントオブジェクトを取得する。
	 * このメソッドを使用して取得したイベントオブジェクトは以下の状態になる。
	 *   ・イベント発生元オブジェクトが target プロパティとして設定されている。
	 *   ・イベント発生元位置が posX, posY プロパティとして設定されている。
	 * @param event 編集元イベント。
	 * @return 編集されたイベント。
	 */
	IWidget.editEvent = function(event) {
		if(!event) {
			event = window.event;
		}

		if(!event.target) {
			event.target = event.srcElement;
		}

		if(event.pageX) {
			event.posX = event.pageX;
			event.posY = event.pageY;
		}
		else {
			event.posX = event.clientX;
			event.posY = event.clientY;
		}

		return event;
	}


	/**
	 * 指定されたコンテナの内容として指定されたオブジェクトを設定する。
	 * @param container 設定先。
	 * @param obj 設定するオブジェクト。
	 *  HTML 文字列，HTML要素オブジェクト，
	 *  htmlElement プロパティを持つオブジェクトのいずれかを指定可能。
	 */
	IWidget.setContent = function(container, obj) {
		if(typeof(obj)=="string") {
			container.innerHTML = obj;
		}
		else if(obj.htmlElement) {
			container.appendChild(obj.htmlElement);
		}
		else {
			container.appendChild(obj);
		}
	}

	/**
	 * 指定されたHTML要素の幅設定を行なう。
	 * @param htmlElement 設定するHTML要素。
	 * @param width 幅。
	 * @return 指定された幅が設定できた場合は true。できなかった場合は false。
	 */
	IWidget.setWidth = function(htmlElement, width) {
		width = parseInt(width);
		if(width < 0) {
			return false;
		}

		if(width > htmlElement.offsetWidth) {
			htmlElement.style.width = width;
			if(htmlElement.handler && htmlElement.handler.setClientWidth) {
//				htmlElement.handler.setClientWidth(width - IWidget.getMarginWidth(htmlElement));
				htmlElement.handler.setClientWidth(width);
			}
		}
		else {
			if(htmlElement.handler && htmlElement.handler.setClientWidth) {
				if(!htmlElement.handler.setClientWidth(width - IWidget.getMarginWidth(htmlElement))) {
					if(htmlElement.style.overflow!="auto" && 
							htmlElement.style.overflow!="scroll"){
						return false;
					}
				}
			}
			var oWidth = htmlElement.offsetWidth;

			htmlElement.style.width = width;
			if(htmlElement.offsetWidth!=width) {
				htmlElement.style.width = oWidth;
				return false;
			}
		}
		return true;
	}

	/**
	 * 指定された HTML 要素の 
	 * borderLeftWidth + borderRightWidth + paddingLeft + paddingRight
	 * を返す。
	 */
	IWidget.getMarginWidth = function(htmlElement) {
		return getInt(htmlElement.style.borderLeftWidth) + 
				getInt(htmlElement.style.borderRightWidth) +
				getInt(htmlElement.style.paddingLeft) +
				getInt(htmlElement.style.paddingRight);
	}


	/**
	 * 指定されたHTML要素の高さ設定を行なう。
	 * @param htmlElement 設定するHTML要素。
	 * @param height 高さ。
	 * @return 指定された高さが設定できた場合は true。できなかった場合は false。
	 */
	IWidget.setHeight = function(htmlElement, height) {
		height = parseInt(height);
		if(height < 0) {
			return false;
		}

		if(height > htmlElement.offsetHeight) {
			htmlElement.style.height = height;
			if(htmlElement.handler && htmlElement.handler.setClientHeight) {
//				htmlElement.handler.setClientHeight(height - IWidget.getMarginHeight(htmlElement));
				htmlElement.handler.setClientHeight(height);
			}
		}
		else {
			if(htmlElement.handler && htmlElement.handler.setClientHeight) {
				if(!htmlElement.handler.setClientHeight(height - IWidget.getMarginHeight(htmlElement))) {
					if(htmlElement.style.overflow!="auto"&& 
							htmlElement.style.overflow!="scroll") {
						return false;
					}
				}
			}
			var oHeight = htmlElement.offsetHeight;
			htmlElement.style.height = height;
			if(htmlElement.offsetHeight!=height) {
				htmlElement.style.height = oHeight;
				return false;
			}
		}
		return true;
	}

	/**
	 * 指定された HTML 要素の 
	 * borderTopWidth + borderBottomWidth + paddingTop + paddingBottom
	 * を返す。
	 */
	IWidget.getMarginHeight = function(htmlElement) {
		return getInt(htmlElement.style.borderTopWidth) + 
				getInt(htmlElement.style.borderBottomWidth) +
				getInt(htmlElement.style.paddingTop) +
				getInt(htmlElement.style.paddingBottom);
	}

	function getInt(value) {
		var v = parseInt(value);
		if(isNaN(v)) {
			return 0;
		}
		else {
			return v;
		}
	}

	/**
	 * htmlElement で指定されたウインドウやコンテキストメニューを
	 * マウスポインタ付近に表示できるようにするために，
	 * 指定されたイベントの posX, posY を設定する。
	 * @param event 設定するイベント。
	 * @param htmlElement HTML要素。
	 * @return posX, posY が設定されたイベント。
	 */
	IWidget.getWindowPosition = function(event, htmlElement) {
		var screenWidth;
		var screenHeight;
		if(document.body.clientWidth) {
			screenWidth = document.body.clientWidth;
			screenHeight = document.body.clientHeight;
		}
		else {
			screenWidth = window.innerWidth;
			screenHeight = window.innerHeight;
		}

		if(screenWidth - event.posX < htmlElement.offsetWidth){
			event.posX = document.body.scrollLeft + event.posX - htmlElement.offsetWidth;
		} else {
			event.posX = document.body.scrollLeft + event.posX;
		}
		if(screenHeight - event.posY < htmlElement.offsetHeight){
			event.posY = document.body.scrollTop + event.posY - htmlElement.offsetHeight;
		} else {
			event.posY = document.body.scrollTop + event.posY;
		}

		return event;
	}

	/**
	 * ハンドラオブジェクトをHTML要素オブジェクトの handler プロパティとして設定する。
	 * @param handler ハンドラオブジェクト。
	 * @param htmlElement HTML要素オブジェクト。
	 * @param movable 移動可能オブジェクトとするかどうか。[デフォルト=false]
	 */
	IWidget.initHtmlElement = function(handler, htmlElement, movable) {
		htmlElement.handler = handler;
		if(movable) {
			htmlElement.movable = true;
		}
		else {
			htmlElement.movable = false;
		}
	}

	/**
	 * 最も大きい zindex を返す。
	 * @return 最も大きい zindex を返す。
	 */
	IWidget.getMaxZIndex = function() {
		if(IWindow.prototype.maxZIndex) {
			IWindow.prototype.maxZIndex = parseInt(IWindow.prototype.maxZIndex) + 1;
		}
		else {
			IWindow.prototype.maxZIndex = 1;
		}
		return IWindow.prototype.maxZIndex;
	}

	/**
	 * 指定されたオブジェクトをゆっくりと非表示にする。
	 * @param htmlElement 非表示にするオブジェクト。
	 * @param func 非表示になったときに呼び出される関数（引数はなし）。
	 */
	IWidget.hideSlowly = function(htmlElement, func) {
		htmlElement.style.filter = "alpha(opacity=40)"
		setTimeout(
			function(){
				htmlElement.style.filter = "alpha(opacity=20)"
				setTimeout(function() {
					htmlElement.style.visibility = "hidden";
					if(func) {
						func();
					}
				},80);
			}
		, 80);
	}

	/**
	 * 値の格納用オブジェクトを取得する。
	 * input type="hidden" タグを htmlElement の中から検索し，
	 * 見つかった場合は，それを返し，見つからなかった場合は，
	 * 新規で作成して返す。
	 * @param htmlElement HTML要素オブジェクト。
	 * @param name input タグを新規作成する場合につける名前
	 */
	IWidget.getValueHolder = function(htmlElement, name) {
		var input = findHidden(htmlElement, name);

		// 見つからなかったら作成する。
		if(input==null) {
			input = document.createElement("input");
			input.type = "hidden";
			input.name = name;
		}

		input.value = "";
		return input;
	}

	/**
	 * 画面カバーを作成する。
	 */
	IWidget.createDisplayCover = function(parent, opac) {
		var cover = document.createElement("div");
		cover.style.backgroundColor = "white";
		if(opac) {
			cover.style.filter = "alpha(opacity=50)";
		}
		cover.style.display = "block";
		cover.style.position = "absolute";
		cover.style.left = 0;
		cover.style.top = 0;
		if(!parent) {
			parent = document.body;
		}
		parent.appendChild(cover);
		cover.style.width = document.body.clientWidth;
		cover.style.height = document.body.clientHeight;
		cover.style.cursor = "wait";
		return cover;
	}

	/**
	 * オブジェクト取得を行なう。
	 */
	IWidget.getHtmlElement = function(obj) {
		if(!obj) {
			return null;
		}

		if(typeof(obj)=="string") {
			return document.getElementById(obj);
		}
		else {
			return obj;
		}
	}

	/**
	 * 入力ヘルパを消去する。
	 */
	IWidget.hideInputHelper = function(exception, target) {
		if(exception!="IHelpMessage") {
			IHelpMessage.hide();
		}
		if(exception!="IAssistant") {
			IAssistant.hide(target);
		}
		if(exception!="ICalendar") {
			ICalendar.hide();
		}
	}

	/**
	 * 入力ヘルパの位置設定を行なう。
	 */
	IWidget.setInputHelperPosition = function(target, htmlElement, marginLeft, marginTop) {
		if(marginLeft==undefined) {
			marginLeft = 0;
		}
		if(marginTop==undefined) {
			marginTop = 0;
		}

		var left = target.offsetLeft + marginLeft;
		var top = target.offsetTop + target.offsetHeight + marginTop;
		var obj = target.offsetParent;


		while(obj!=null) {
			left += obj.offsetLeft;
			top += obj.offsetTop;
			obj = obj.offsetParent;
		}
		htmlElement.style.left = left;
		htmlElement.style.top = top;
	}

	//// ローカルメソッド定義

	/**
	 * IWidget を初期化する。
	 */
	function init() {

		document.onmousedown = function (event) {
			event = IWidget.editEvent(event);
			this.movable = getMovableObject(event.target);
			if (!this.movable) {
				return true;
			}

			this.movable.startX = event.posX - this.movable.handler.htmlElement.offsetLeft;
			this.movable.startY = event.posY - this.movable.handler.htmlElement.offsetTop;

			document.onmousemove = movable_onmousemove;
			document.onmouseup = movable_onmouseup;
			event.cancelBubble = true;
			return false;
		}

		function movable_onmouseup(event) {
			hideCover();

			document.onmousemove = null;
			document.onmouseup = null;
			this.movable = null;
			return true;
		}

		function movable_onmousemove(event) {
			if (!this.movable || !this.movable.handler || 
					!this.movable.handler.htmlElement || !this.movable.startX) {
				hideCover();
				this.movable = null;
				return true;
			}

			event = IWidget.editEvent(event);

			if(!event.which && event.button!=1) {
				hideCover();
				this.movable = null;
				return true;
			}

			initCover();

			if(!this.movable.type || this.movable.type=="x") {
				this.movable.handler.htmlElement.style.left = 
					event.posX - this.movable.startX;
			}
			if(!this.movable.type || this.movable.type=="y") {
				this.movable.handler.htmlElement.style.top = 
					event.posY - this.movable.startY;
			}

			if(this.movable.handler.onmove) {
				this.movable.handler.onmove(
					parseInt(this.movable.handler.htmlElement.style.left), 
					parseInt(this.movable.handler.htmlElement.style.top), 
				event);
			}

			event.cancelBubble = true;
			return false;
		}

		function initCover() {
			if(!document.cover) {
				var cover = document.createElement("div");
				//cover.style.backgroundColor = "transparent";
				cover.style.backgroundColor = "yellow";
				cover.style.filter = "alpha(opacity=0)"
				cover.style.display = "none";
				cover.style.position = "absolute";
				cover.style.left = 0;
				cover.style.top = 0;
				if(document.body.clientWidth) {
					cover.style.width = document.body.clientWidth;
					cover.style.height = document.body.clientHeight;
				}
				else {
					cover.style.width = window.innerWidth;
					cover.style.height = window.innerHeight;
				}
				document.body.appendChild(cover);
				document.cover = cover;
			}
			document.cover.style.zIndex = 999999999999;
			document.cover.style.display = "block";
		}

		function hideCover() {
			if(document.cover) {
				document.cover.style.display = "none";
			}
		}


		function getMovableObject(obj) {
			while(obj) {
				if(obj.handler && obj.handler.htmlElement && obj.movable) {
					return obj;
				}
				obj = obj.parentNode;
			}
			return null;
		}
	}

	/**
	 * input type="hidden" タグを htmlElement の中から検索する。
	 * @param htmlElement HTML要素オブジェクト。
	 * @return input オブジェクト。
	 */
	function findHidden(htmlElement, name) {
		var children = htmlElement.childNodes;
		for(var i=0; i < children.length; i++) {
			var child = children.item(i);
			if(child.tagName=="INPUT" || child.type=="hidden") {
				return child;
			}
		}
		return null;
	}

	//// インスタンスの初期化
	init();
}
new IWidget();


