/**
 * 内部ウインドウを作成する。
 * @param titleText ウインドウタイトル文字列。
 * @param content ウインドウ内に表示する内容。
 * @param hidden 初期表示の際にウインドウを隠すかどうか。(未指定の場合は false)
 * @param parent ウインドウの親要素。
 * @param modal モーダルウインドウとして使用するかどうか。
 */
function IWindow(titleText, content, hidden, parent, modal) {
	if(!document.body) {
		return;
	}

	//// 変数定義
	var handler = this;
	var modalCover = createModalCover(parent, modal);
	var win = createWindow(parent); // ウインドウ
	var titleBar = createTitleBar(win);// タイトルバー
	var icon = createTitleBarElement(titleBar, "icon"); // タイトルバーアイコン
	var title = createTitleBarElement(titleBar, "title"); // タイトル文字列
	var minimizeButton = createTitleBarElement(titleBar, "minimize-button"); // 最小化ボタン
	var closeButton = createTitleBarElement(titleBar, "close-button"); // Close ボタン
	var contentContainer = createContentContainer(win); // 内容表示領域

	var bottomBorder = createBorder("h", // 下枠
		function(left, top) {
			var h = top - parseInt(win.style.top);
			handler.setBounds(win.style.left, win.style.top, win.style.width, h);
		}
	);

	var rightBorder = createBorder("v", // 右枠
		function(left, top) {
			var w = left - parseInt(win.style.left);
			handler.setBounds(win.style.left, win.style.top, w, win.style.height);
		}
	);

	var topBorder = createBorder("h", // 上枠
		function(left, top) {
			var h = parseInt(win.style.height) - (top - parseInt(win.style.top));
			handler.setBounds(win.style.left, top, win.style.width, h);
		}
	);

	var leftBorder = createBorder("v", // 左枠
		function(left, top) {
			var w = parseInt(win.style.width) - (left - parseInt(win.style.left));
			handler.setBounds(left, win.style.top, w, win.style.height);
		}
	);

	var rightBottom = createCorner("nw-resize", // 右下角
		function(left, top) {
			var h = top - parseInt(win.style.top);
			var w = left - parseInt(win.style.left);
			handler.setBounds(parseInt(win.style.left), parseInt(win.style.top), w, h);
		}
	);

	var leftBottom = createCorner("sw-resize", // 左下角
		 function(left, top) {
			var h = top - parseInt(win.style.top);
			var w = parseInt(win.style.width) - (left - parseInt(win.style.left));
			handler.setBounds(left, parseInt(win.style.top), w, h);
		}
	);

	var rightTop = createCorner("ne-resize", // 右上角
		function(left, top) {
			var h = parseInt(win.style.height) - (top - parseInt(win.style.top));
			var w = left - parseInt(win.style.left);
			handler.setBounds(parseInt(win.style.left), top, w, h);
		}
	);

	var leftTop = createCorner("nw-resize", // 左上角
		function(left, top) {
			var h = parseInt(win.style.height) - (top - parseInt(win.style.top));
			var w = parseInt(win.style.width) - (left - parseInt(win.style.left));
			handler.setBounds(left, top, w, h);
		}
	);

	var cover = createCover(win);
	this._cover = cover;
	this._modalCover = modalCover;

	//// プロパティ定義

	/**
	 * HTML実体要素オブジェクト。
	 */
	this.htmlElement = win;

	//// プロパティ定義

	/**
	 * ウインドウ表示内容を設定するコンテナオブジェクト。
	 */
	this.contentContainer = contentContainer;

	/**
	 * リサイズ可能かどうか。
	 */
	this.resizable = true;
	
	/**
	 * 最小化表示中かどうか。
	 * ウィンドウが表示中で最小化表示されているときのみ true が格納される。
	 */
	this.minimize = false;



	//// public メソッド定義

	/**
	 * ウインドウの表示を行なう。
	 */
	this.show = function(event) {
		if(event) {
			event = IWidget.editEvent(event);
			event = IWidget.getWindowPosition(event, win);
			handler.setPosition(event.posX, event.posY);
		}
		win.style.visibility = "visible";
		if(handler.resizable) {
			setVisibleBorders(win.style.visibility);
		}
		initzIndex();
		win.style.filter = "alpha(opacity=100)"
		this.minimize = false;
		return handler;
	}

	/**
	 * ウインドウを非表示にする。
	 */
	this.hide = function() {
		this.minimize = true;
		IWidget.hideSlowly(win,
			function() {
				setVisibleBorders(win.style.visibility);
				if(handler.onhide) {
					handler.onhide();
				}
			}
		);
		return handler;
	}
	
	/**
	 * ウィンドウが表示中かどうか取得する。
	 * @return true 表示中, false 非表示中
	 */
	this.isVisible = function() {
		return (win.style.visibility == "visible");
	}

	/**
	 * ウインドウの表示状態を変更する。
	 * @param visible true の場合は表示，false の場合は非表示。
	 * 
	 */
	this.setVisible = function(visible) {
		if(visible) {
			handler.show();
		}
		else {
			win.style.visibility = "hidden";
			setVisibleBorders(win.style.visibility);
		}
	}

	/**
	 * このウインドウを最前面に表示する。
	 */
	this.toFront = initzIndex;

	/**
	 * タイトルバーを非表示にする。
	 */
	this.hideTitleBar = function() {
		titleBar.style.display = "none";
		handler.pack();
		return handler;
	}

	/**
	 * 表示位置を設定する。
	 * @param left 横位置。
	 * @param top 縦位置。
	 */
	this.setPosition = function(left, top) {
		setPosition0(win, left, top);
		initBorders();
		return handler;
	}

	/**
	 * サイズを設定する。
	 * @param width 幅。
	 * @param height 高さ。
	 */
	this.setSize = function(width, height) {
		width = parseInt(width);
		height = parseInt(height);
		if(height <= getTitleBarHeight()) {
			initBorders();
			return handler;
		}
		if(width <= 40) {
			initBorders();
			return handler;
		}

		win.style.width = width;
		win.style.height = height;
		IWidget.setWidth(contentContainer, width - 4);
		IWidget.setHeight(contentContainer, height - getTitleBarHeight() - 2);

		initBorders();
		return handler;
	}


	/**
	 * 表示位置とサイズをまとめて設定する。
	 * @param left 横位置。
	 * @param top 縦位置。
	 * @param width 幅。
	 * @param height 高さ。
	 */
	this.setBounds = function(left, top, width, height) {
		setPosition0(win, left, top);
		handler.setSize(width, height);
		return handler;
	}

	/**
	 * 表示内容のサイズに合わせて，ウインドウのサイズを調整する。
	 */
	this.pack = function() {
		var obj = contentContainer.firstChild;

		if(!obj) {
			return handler;
		}

		var objWidth = obj.offsetWidth;
		var objHeight = obj.offsetHeight;
		if(!objWidth) {
			objWidth = 100;
		}
		if(! objHeight) {
			objHeight = 100;
		}

		handler.setBounds(win.offsetLeft, win.offsetTop, 
			objWidth + 5, getTitleBarHeight() + objHeight + 2);
		return handler;
	}

	/**
	 * ウインドウのサイズを可変にするかどうかを設定する。
	 * @param resizable 可変にする場合は true。
	 */
	this.setResizable = function(resizable) {
		setVisibleBorders((resizable)?"visible":"hidden");
		handler.resizable = resizable;
		return handler;
	}
	
	/**
	 * ウインドウが最小化表示中かどうか。
	 * @return true 最小化, false 最小化以外
	 */
	this.isMinimize = function() {
		return this.minimize;
	}
	
	/**
	 * スクロールバー表示をするかしないかを設定する。
	 * @param scroll スクロールバー表示をする場合は true を指定する。
	 */
	this.setScroll = function(scroll) {
		if(scroll=="auto" || scroll=="scroll" || scroll=="hidden") {
			contentContainer.style.overflow = scroll;
		}
		else if(scroll=="true") {
			contentContainer.style.overflow = "scroll";
		}
		else if(scroll=="false") {
			contentContainer.style.overflow = "hidden";
		}
		else if(scroll) {
			contentContainer.style.overflow = "scroll";
		}
		else {
			contentContainer.style.overflow = "hidden";
		}
		return handler;
	}



	/**
	 * ウインドウのタイトルを設定する。
	 * @param text ウインドウのタイトル文字列。
	 */
	this.setTitle = function(text) {
		title.innerHTML = text;
		return handler;
	}

	/**
	 * ウインドウのタイトルを取得する。
	 * @return ウインドウのタイトル文字列。
	 */
	this.getTitle = function() {
		return title.innerHTML;
	}

	/**
	 * タイトルバーのアイコンを設定する。
	 * @param iconURL アイコン画像を示す URL。
	 */
	this.setIcon = function(iconURL) {
		icon.style.backgroundImage = iconURL;
		return handler;
	}

	/**
	 * タイトルバーのアイコンを取得するする。
	 * @return iconURL アイコン画像を示す URL。
	 */
	this.getIcon = function() {
		return icon.style.backgroundImage;
	}

	/**
	 * 表示内容設定を行なう。
	 * @param obj HTML文字列 / HTML 要素 / htmlElement プロパティを持つオブジェクト。
	 */
	this.setContent = function(obj) {
		IWidget.setContent(contentContainer, obj);
		handler.pack();
		return handler;
	}

	/**
	 * 指定された URL のドキュメントを表示内容として設定する。
	 * @param url 表示する URL。
	 */
	this.setContentByURL = function(url) {
		var iframe = document.createElement("iframe");
		iframe.src = url;
		handler.setContent(iframe);
		return handler;
	}

	/**
	 * ウインドウを画面中央に表示する。
	 */
	this.center = function() {
		var left = (document.body.clientWidth - win.offsetWidth) / 2;
		var top = (document.body.clientHeight - win.offsetHeight) / 2;
		handler.setPosition(left, top);
	}

	/**
	 * 最小化ボタンを利用可能とするかどうかを設定する。
	 * @param value 利用可能とする場合は true。
	 */
	this.setEnableMinimizeButton = function(value) {
		if(value) {
			minimizeButton.style.display = 'inline';
			minimizeButton.onlick = function() {handler.hide();};
		}
		else {
			minimizeButton.style.display = 'none';
		}
	}

	/**
	 * クローズボタンを利用可能とするかどうかを設定する。
	 * @param value 利用可能とする場合は true。
	 */
	this.setEnableCloseButton = function(value) {
		if(value) {
			closeButton.style.display = 'inline';
			closeButton.onlick = function() {handler.close();};
		}
		else {
			closeButton.style.display = 'none';
		}
	}

	/**
	 * ウインドウが非表示になったときに呼び出される。
	 */
	this.onhide = null;

	/**
	 * ウインドウがクローズされる直前に呼び出される。
	 * このイベントハンドラが false を返す場合は，クローズしない。
	 */
	this.onclosing = null;

	/**
	 * ウインドウがクローズされた直後に呼び出される。
	 */
	this.onclosed = null;



	/**
	 * 指定されたタイトルバー文字列のウインドウを取得する。
	 * 同じタイトルバー文字列のウインドウが複数ある場合はどちらかひとつを返す。
	 * @param title タイトルバー文字列。
	 * @return IWindow インスタンス。見つからなかった場合は null。
	 */
	IWindow.findByTitle = function(title) {
		var list = IWindow.prototype.winList;
		for(var i=0; i<list.length; i++) {
			if(title==list[i].getTitle()) {
				return list[i];
			}
		}
		return null;
	}

	/**
	 * ウインドウをクローズする。
	 */
	this.close = function() {
		this.minimize = false;
		if(handler.onclosing) {
			if(!handler.onclosing()) {
				return handler;
			}
		}

		IWidget.hideSlowly(win,
			function() {
				handler.onhide = null;
				handler.onclosing = null;
				handler._cover = null;
				remove(handler._modalCover);
				handler._modalCover = null;

				remove(handler.htmlElement);
				handler.htmlElement = null;

				remove(bottomBorder.htmlElement);
				remove(rightBorder.htmlElement);
				remove(topBorder.htmlElement);
				remove(leftBorder.htmlElement);
				remove(rightBottom.htmlElement);
				remove(leftBottom.htmlElement);
				remove(rightTop.htmlElement);
				remove(leftTop.htmlElement);

				handler.contentContainer = null;

				var list = IWindow.prototype.winList;
				for(var i=0; i<list.length; i++) {
					if(list[i]==handler) {
						list.splice(i, 1);
					}
				}
				if(handler.onclosed) {
					handler.onclosed();
					handler.onclosed = null;
				}
				handler = null;

				function remove(node) {
					if(node && node.handler) {
						if(node.handler.htmlElement) {
							node.handler.htmlElement = null;
						}
						node.handler = null;
					}
					if(node && node.parentNode) {
						node.parentNode.removeChild(node);
					}
				}
			}
		);
		return null;
	}

	/**
	 * ウインドウがクローズされているかどうかを調べる
	 */
	this.isClosed = function() {
		return handler==null;
	}


	//// ローカルメソッド定義

	/**
	 * IWindow の初期化を行なう。
	 */
	function init(titleText) {
		if(!titleText) {
			titleText = "";
		}
		title.innerHTML = titleText;


		if(!IWindow.prototype.winList) {
			IWindow.prototype.winList = new Array();
			IWindow.prototype.hideAll = function() {
				var list = IWindow.prototype.winList;
				for(var i=0; i<list.length; i++) {
					list[i].hide();
				}
			};
			IWindow.prototype.showAll = function() {
				var list = IWindow.prototype.winList;
				for(var i=0; i<list.length; i++) {
					list[i].show();
				}
			};
		}
		IWindow.prototype.winList.push(handler);

		IWidget.initHtmlElement(handler, win);

		titleBar.handler = handler;
		titleBar.movable = true;

		initBorderzIndex();
		minimizeButton.onclick = function() {handler.hide();};
		closeButton.onclick = function() {handler.close();};

		handler.onmove = function(left, top) {
			initBorders();
		}

		if(content) {
			if(typeof(content)=="string") {
				handler.setContent(content);
			}
			else if(content.htmlElement) {
				content = content.htmlElement;
			}

			var left = content.offsetLeft;
			var top = content.offsetTop;
			handler.setContent(content);
			handler.setPosition(left, top);
		}

		if(hidden) {
			handler.setVisible(false);
		}
		initzIndex();
	}


	/**
	 * ウインドウ用 HTML 要素を作成する。
	 * @param parent 親要素。
	 * @return ウインドウ用 HTML 要素。
	 */
	function createWindow(parent) {
		var win = document.createElement("div");
		win.className = "iwindow";
		win.style.position = "absolute";
		win.style.zIndex = IWidget.getMaxZIndex();
		win.onmousedown = function() {initzIndex()};
		if(!parent) {
			parent = document.body;
		}
		parent.appendChild(win);
		return win;
	}

	/**
	 * モーダルウインドウ用カバーを作成する。
	 */
	function createModalCover(parent, modal) {
		var cover = null;
		if(modal) {
			cover = IWidget.createDisplayCover(parent, true);
		}
		return cover;
	}

	/**
	 * ウインドウのタイトルバーを作成する。
	 * @param win ウインドウ用 HTML 要素。
	 * @return ウインドウのタイトルバー。
	 */
	function createTitleBar(win) {
		var titleBar = document.createElement("div");
		titleBar.className = "title-bar"
		win.appendChild(titleBar);
		return titleBar;
	}

	/**
	 * ウインドウタイトルバー内HTML 要素を作成する。
	 * @param titleBar タイトルバー。
	 * @param className クラス名。
	 * @return ウインドウタイトルバー内HTML要素。
	 */
	function createTitleBarElement(titleBar, className) {
		var elem = document.createElement("span");
		elem.display = "inline";
		elem.className = className;
		titleBar.appendChild(elem);
		return elem;
	}

	/**
	 * 内容表示領域を作成する。
	 * @param win ウインドウ用 HTML 要素。
	 * @return 内容表示領域。
	 */
	function createContentContainer(win) {
		var contentContainer = document.createElement("div");
		contentContainer.className = "content";
		contentContainer.style.overflow = "auto";
		win.appendChild(contentContainer);
		new IContainer(contentContainer);
		return contentContainer;
	}

	/**
	 * ウインドウ枠線を作成する。
	 * @param type 枠線タイプ。
	 * @param onmovefunc 枠線が移動されたときに呼び出される関数。
	 */
	function createBorder(type, onmovefunc) {
		var border = new IBorder(document.body, type, 100,10);
		border.onmove = onmovefunc;
		return border;
	}

	/**
	 * ウインドウの角を作成する。
	 * @param cursor カーソルタイプ。
	 * @param onmovefunc 角が移動されたときに呼び出される関数。
	 */
	function createCorner(cursor, onmovefunc) {
		var corner = new IDiv(10,10);
		corner.htmlElement.style.cursor = cursor;
		corner.htmlElement.style.backgroundColor = "";
		corner.onmove = onmovefunc;
		return corner;
	}

	/**
	 * カバー作成。
	 */
	function createCover(win) {
		var cover = document.createElement("div");
		cover.style.backgroundColor = "white";
		cover.style.filter = "alpha(opacity=0)";
		contentContainer.style.filter = "alpha(opacity=100)";
		cover.style.display = "none";
		cover.style.position = "absolute";
		cover.style.left = 0;
		cover.style.top = 0;
		win.appendChild(cover);
		return cover;
	}

	/**
	 * ウインドウの位置を設定する。
	 * @param win 設定するウインドウ。
	 * @param left 横位置。
	 * @param top 縦位置。
	 */
	function setPosition0(win, left, top) {
		left = parseInt(left);
		top = parseInt(top);
		if(!left || left < 1) {
			left = 1;
		}
		if(!top || top < 1) {
			top = 1;
		}

		win.style.left = left;
		win.style.top = top;
	}

	/**
	 * 枠線の表示非表示を設定する。
	 */
	function setVisibleBorders(visibility) {
		bottomBorder.htmlElement.style.visibility = visibility;
		rightBorder.htmlElement.style.visibility = visibility;
		topBorder.htmlElement.style.visibility = visibility;
		leftBorder.htmlElement.style.visibility = visibility;
		rightBottom.htmlElement.style.visibility = visibility;
		leftBottom.htmlElement.style.visibility = visibility;
		rightTop.htmlElement.style.visibility = visibility;
		leftTop.htmlElement.style.visibility = visibility;
	}

	/**
	 * タイトルバーの高さを取得する。
	 */
	function getTitleBarHeight() {
		if(titleBar.style.visibility == "hidden") {
			return 0;
		}
		else {
			return titleBar.offsetHeight;
		}
	}

	/**
	 * 枠線の表示状態を設定する。
	 */
	function initBorders() {
		var left = parseInt(win.style.left);
		var top = parseInt(win.style.top);
		var width = parseInt(win.style.width);
		var height = parseInt(win.style.height);

		bottomBorder.setPosition(left, top+height-5);
		bottomBorder.setLength(width);
		rightBorder.setPosition(left+width-5, top);
		rightBorder.setLength(height);

		topBorder.setPosition(left, top-5);
		topBorder.setLength(width);

		leftBorder.setPosition(left-5, top);
		leftBorder.setLength(height);

		rightBottom.setPosition(left+width-5, top+height-5);
		leftBottom.setPosition(left-5, top+height-5);

		rightTop.setPosition(left+width-5, top-5);
		leftTop.setPosition(left-5, top-5);

		cover.style.top = contentContainer.offsetTop;
		cover.style.left = contentContainer.offsetLeft;
		cover.style.width = contentContainer.style.width;
		cover.style.height = contentContainer.style.height;
	}

	/**
	 * ウインドウの zIndex を設定する。
	 */
	function initzIndex() {
		IWindow.prototype.maxZIndex = parseInt(IWindow.prototype.maxZIndex) + 10;
		win.style.zIndex = IWindow.prototype.maxZIndex;
		initBorderzIndex();
		cover.style.display = "none";
		var list = IWindow.prototype.winList;
		for(var i=0; i<list.length; i++) {
			if(list[i]!=handler) {
				list[i]._cover.style.display = "block";
			}
		}
	}

	/**
	 * border の zIndex を設定する。
	 */
	function initBorderzIndex() {
		bottomBorder.htmlElement.style.zIndex = win.style.zIndex + 1;
		rightBorder.htmlElement.style.zIndex = win.style.zIndex + 2;
		topBorder.htmlElement.style.zIndex = win.style.zIndex + 3;
		leftBorder.htmlElement.style.zIndex = win.style.zIndex + 4;
		rightBottom.htmlElement.style.zIndex = win.style.zIndex + 5;
		leftBottom.htmlElement.style.zIndex = win.style.zIndex + 6;
		rightTop.htmlElement.style.zIndex = win.style.zIndex + 7;
		leftTop.htmlElement.style.zIndex = win.style.zIndex + 8;
	}


	//// インスタンスの初期化
	init(titleText);
}
new IWindow();

