

--
-- Copyright (C) 2022  <fastrgv@gmail.com>
--
-- This program is free software: you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation, either version 3 of the License, or
-- (at your option) any later version.
--
-- This program is distributed in the hope that it will be useful,
-- but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
-- GNU General Public License for more details.
--
-- You may read the full text of the GNU General Public License
-- at <http://www.gnu.org/licenses/>.
--



--
--*****************************************************************
--
-- splaypq Package description:
-- Priority Queue specialized for hbox4 sokoban solver.
--
-- New splaytree with 4 added priority fields that allow:
-- a) splaytree random access by unique key value, just as before;
-- b) 4 sequential access queues, each with a
--    priority field that allows duplicates, with the
--    typical properties of a Priority Queue.
--    Equal priorities are inserted AFTER others.
--
-- This package implements an extremely efficient self-adjusting
-- binary search tree called a splay tree with very little overhead
-- to maintain the balance.  The ordering by IdType is maintained
-- to permit fast access by Id and fast checks for duplicates.
-- Linear access and traversal of the tree elements according to
-- priority order is also supported.
--
-- Reference:
-- See the Journal of the A.C.M., July 1985,
-- Vol. 32, No. 3, pg. 652, library call # QA 76 A77
--
--*************************************************************************
with text_io; --debug only
with Unchecked_Deallocation;

package body splaypq is


  procedure dispose is new unchecked_deallocation(splayrec,splayptr);
  procedure dispose is new unchecked_deallocation(hashnode,hashptr);



	procedure myassert( 
		condition : boolean;  
		flag: integer:=0;
		msg: string := ""
		) is
		use text_io;
	begin
	  if condition=false then
			put("ASSERTION Failed!  ");
			if flag /= 0 then
				put( "@ " & integer'image(flag) &" : " );
			end if;
			put_line(msg);
			new_line;
			raise program_error;
	  end if;
	end myassert;



  -----------------------------------------------------
  --without changing tree structure this searches for
  --the node pointer for Id;  returns null if not found
  -----------------------------------------------------
  function getptr( Id       : in IdType;
                   List     : in ListType ) return splayptr is
    p: splayptr;
  begin
   if       list.header = null
   or else  list.header.size=0  then
     return null;
   else
    p:=list.header.root;
    while( p /= null ) and then ( p.Id /= Id ) loop
      if  Id < p.Id  then
        p:=p.left_child;
      else
        p:=p.right_child;
      end if;
    end loop;
    return p;
   end if;
  end getptr;


   -- temporary routine for debugging purposes only;
   -- allows users to deduce the tree's structure
  procedure GetParentKey( k: IdType;
                          list: in out listtype;
                          kp: out IdType ) is
    p: splayptr;
  begin
    p := getptr(k,list);
    if( p /= null ) and then ( p.Id = k )  then
      if  p.parent /= null  then
        kp := p.parent.Id;
      else
        kp := k;
      end if;
    end if;
  end getparentkey;



  --------------------------------------
  -- the main balancing mechanism...
  -- see "splay trees" in the literature.
  -- Careful, only experts should attempt
  -- modifying this routine.
  ----------------------------------------------
  procedure splay( p: in out splayptr; list: in listtype ) is
    q,r: splayptr;

    procedure rotate( thisnode: in out splayptr ) is
      dad,son,grandad:splayptr;
    begin
      dad := thisnode.parent;
      if dad = null then
        raise constraint_error;
      end if;
      grandad:=dad.parent;

      if  thisnode = dad.left_child  then
        son:=thisnode.right_child;
        thisnode.right_child := dad;
        thisnode.parent := grandad;
        dad.parent:=thisnode;
        dad.left_child := son;
        if  son /= null  then
          son.parent := dad;
        end if;
        if  grandad=null  then  --dad was old root
          list.header.root:=thisnode;
        else -- grandad /= null
          if  grandad.right_child=dad  then
             grandad.right_child:=thisnode;
          elsif  grandad.left_child=dad  then
             grandad.left_child:=thisnode;
          else
            raise constraint_error;
          end if;
        end if;
      elsif  thisnode = dad.right_child  then
        son:=thisnode.left_child;
        thisnode.left_child := dad;
        thisnode.parent := grandad;
        dad.parent := thisnode;
        dad.right_child := son;
        if  son /= null  then
          son.parent := dad;
        end if;
        if  grandad=null  then  --dad was old root
          list.header.root:=thisnode;
        else
          if  grandad.right_child=dad  then
             grandad.right_child:=thisnode;
          elsif  grandad.left_child=dad  then
             grandad.left_child:=thisnode;
          else
             raise constraint_error;
          end if;
        end if;
      else
        raise constraint_error;
      end if;
    end rotate;

  begin  -- splay
   if  ( p /= null  and  list.header /= null )
   and then  list.header.size>0
   then
    while  p.parent /= null  loop
      q:=p.parent;
      if  q.parent = null  then -- q is root
        rotate(p);
      else -- q is not root
        r:=q.parent;
        if ( ( q=r.left_child  ) and ( p=q.left_child  ) )
        or ( ( q=r.right_child ) and ( p=q.right_child ) )
        then -- ZIG-ZIG
          rotate(q);
          rotate(p);
        elsif ( ( q=r.left_child  ) and ( p=q.right_child ) )
        or    ( ( q=r.right_child ) and ( p=q.left_child  ) )
        then  -- ZIG-ZAG
          rotate(p);
          rotate(p);
        else
          raise constraint_error; --logically impossible
        end if;
      end if;
    end loop;
   end if;
  end splay;


--------- end of utility routines ------------------------

----------- begin main body ------------------------------

  ------------------------------------------
  -- returns the number of nodes in the list
  ------------------------------------------
  function length( List: in ListType ) return integer is
  begin
    if       list.header = null
    or else  list.header.size=0  then
      return 0;
    else
      return list.header.size;
    end if;
  end length;




  ------------------------------------------------
  -- gets the nodal data belonging to specified Id;
  -- in the process, we splay, bringing Id to root.
  ------------------------------------------------
  procedure search( Id       : in IdType;
                     List     : in     ListType;
                     Data     : out DataType;
                     Status :    out StatusType) is
    p: splayptr;
  begin -- search
    p := getptr(Id,List);
    if  p=null  then
      Status := NotFound;
    elsif ( p.Id = Id )  then
      Status := Found;
      Data := p.Data;
      splay(p,list);
    else
      Status := NotFound;  --impossible branch!
    end if;
    -- pure implementations always splay at the
    -- last non-null node visited, even on failure !

  end search;




 





  ---------------------------------------
  -- deletes the node with specified Id
  -- Any action to other list links must
  -- be accomplished BEFORE calling this.
  -- This proc avoids changing curr ptr.
  ---------------------------------------
  procedure DelNodeAtPtr( q     : in splayptr;
                     List   : in out ListType;
                     Status :    out StatusType) is
	use text_io;
    idnext : IdType;
    d: DataType;
    p: splayptr;
	 r: splayptr:=q;
    localstatus: statustype;
  begin


  	splay(r,list); --moves r to root

	status:=Ok;
   list.header.size := list.header.size - 1;


    if  list.header.root.right_child=null  then
      -- tree deletion is easy
      list.header.root := list.header.root.left_child;
      if  list.header.root /= null  then
        list.header.root.parent := null;
      end if;

      --list delete and dispose
		if r/=null then
			dispose(r);
		end if;



    else
      p := list.header.root.right_child;

      while ( p.left_child /= null )  loop
         p:=p.left_child;
      end loop;
      idnext := p.Id; --immediate successor to Id
      search(idnext,list,d,localstatus);
      if  localstatus /= found  then
        raise program_error; --should never happen!
      end if;
      -- at this point r is the leftson of its successor (which is
      -- now at the root) and, therefore, has no rightson itself
      list.header.root.left_child := r.left_child;
      if  list.header.root.left_child /= null  then
        list.header.root.left_child.parent := list.header.root;
      end if;

      --list delete and dispose
		if r/=null then
      	dispose(r);
		end if;

    end if;

	if list.header.size<1 then
		list.header:=null;
		status:=empty;
	end if;

  end delnodeatptr;





--detach "thishash" from the head of the hash(pri) list
procedure unhook(
	list: in out listtype; 
	thishash: hashptr;
	ip: p4range; --which PQ {1..4}
	opri: p1range;
	opri0: p2range
	) is
	prevh,nexth: hashptr;
begin
		nexth:=thishash.hnext(ip); --possibly null
		prevh:=thishash.hprev(ip); --possibly null
		if prevh = null then --flag for head of hash list
			myassert(list.header.hash(ip)(opri,opri0)=thishash,8008,"spq.unhook");
			list.header.hash(ip)(opri,opri0):=nexth;
		else
			prevh.hnext(ip) := nexth;
		end if;
		if nexth /= null then
			nexth.hprev(ip) := prevh;
		end if;
end unhook;




--Push "thishash" onto the head of the hash(pri) list
procedure push(
	list:in out listtype; 
	thishash: hashptr;
	ip: p4range; --which PQ {1..4}
	opri: p1range;
	nupri0: p2range
	) is
	r: hashptr;
begin

-- insert thishash into different doubly linked list @head:
	r:=list.header.hash(ip)(opri,nupri0); --may be null
	thishash.hnext(ip):=r;
	thishash.hprev(ip):=null; --head flag
	if r/=null then
		r.hprev(ip) := thishash; -- r becomes 2nd item
	end if;
	list.header.hash(ip)(opri,nupri0):=thishash; --new node becomes 1st

end push;




--with this algorithm, the primary priority does NOT change;
--but the secondary [totpulz+hunEst] sometimes gets smaller.
--This proc handles all 4 PQs.
procedure bumpKey(
	List   : in out ListType;
	Id   : IdType;
	data : datatype;
	nupri0 : p2range; -- 2ndary pri
	Status :    out StatusType) is

	thishash: hashptr;
	thissplay: splayptr;
	opri0: p2range;
	opri1,opri2,opri3,opri4: p1range;

begin

	thissplay := getptr(Id,List);
	if thissplay=null then
		status:=notfound;
	else
		status:=Ok;

		thishash:=thissplay.hptr;

		opri1:=thissplay.priority1; -- will be unchanged
		opri2:=thissplay.priority2; -- will be unchanged
		opri3:=thissplay.priority3; -- will be unchanged
		opri4:=thissplay.priority4; -- will be unchanged
		opri0:=thissplay.priority0; -- should be bigger than nupri0

		unhook(list,thishash, 1,opri1,opri0);
		push(list,thishash,1,opri1,nupri0);

		unhook(list,thishash, 2,opri2,opri0);
		push(list,thishash,2,opri2,nupri0);

		unhook(list,thishash, 3,opri3,opri0);
		push(list,thishash,3,opri3,nupri0);

		unhook(list,thishash, 4,opri4,opri0);
		push(list,thishash,4,opri4,nupri0);


		thissplay.data := data; --updated totpulz, etc
		thissplay.priority0 := nupri0;

		-- the 2 pointers between splaynode & hashnode 
		-- were unchanged but should still correspond:
		myassert(thissplay.hptr=thishash, 808, "427:bumpKey");
		myassert(thishash.down=thissplay, 809, "428:bumpKey");

	end if;


end bumpKey;








procedure popNode(
	List   : in out ListType;
	Id   : out IdType;
	Data : out DataType;
	ip : in p4range; -- which PQ {1,2,3,4} is to be popped
	pri1,pri2,pri3,pri4, pri0  : out integer;
	Status :    out StatusType) is
	p: hashptr;
	q: splayptr;
	use text_io;
begin

	if list.header/= null
	then
		status:=Ok;

		outer:
		for i in p1range loop
			for j in p2range loop
				p := list.header.hash(ip)(i,j);
				if p /= null then
					q    := p.down; --splayptr node to pop
					exit outer;
				end if;
			end loop; --for j
		end loop outer; -- for i

myassert(q/=null, 59595, "splaypq.pop::484");

--------------------------------------------------------------

		id   := q.id;
		pri1 := integer(q.priority1);
		pri2 := integer(q.priority2);
		pri3 := integer(q.priority3);
		pri4 := integer(q.priority4);
		pri0 := integer(q.priority0);
		Data := q.Data;

--------------------------------------------------------------

		--first, disconnect q from all 4 PQs
		p:=q.hptr;
		unhook(list,p, 1,pri1,pri0);
		unhook(list,p, 2,pri2,pri0);
		unhook(list,p, 3,pri3,pri0);
		unhook(list,p, 4,pri4,pri0);

--------------------------------------------------------------

		dispose(p); -- delete hashnode with 4PQ pointers

		delNodeAtPtr(q,list,status); -- delete splaytree node

	else
		status:=NilPtr;
	end if;

end popnode;














----------------------------------------------------
-- inserts into random access splaytree by unique Id
-- AND into 4 queues by integer priority
---------------------------------------
procedure AddNode( Id     : in IdType;
                   Data   : in DataType;
						 pri1   : in p1range;
						 pri2   : in p1range;
						 pri3   : in p1range;
						 pri4   : in p1range;
						 pri0   : in p2range;
                   List   : in out ListType;
                   Status :    out StatusType) is

  p,q:splayptr;  bok: boolean := true;
  use text_io;
  found: boolean := false;
  r, nuhptr: hashptr;
begin


  Status := Ok;

  if  list.header=null
  or else  list.header.size=0  then  --empty tree (list)

    if  list.header=null  then
      list.header := new listheader;
    end if;

    list.header.size := 1;
    list.header.root := new splayrec'(Id,data,
	 	pri1,pri2,pri3,pri4,pri0, null,null,null, null );

--======================================================================
-- now we deal with list structure...insert node in priority order
--======================================================================

		--load 1st elt into hash array
		nuhptr := new hashnode;
		nuhptr.down:= list.header.root;

		for j in p4range loop --list-heads=>these ptrs are null
			nuhptr.hnext(j):=null;
			nuhptr.hprev(j):=null;
		end loop;

		list.header.hash(1)(pri1,pri0):=nuhptr;
		list.header.hash(2)(pri2,pri0):=nuhptr;
		list.header.hash(3)(pri3,pri0):=nuhptr;
		list.header.hash(4)(pri4,pri0):=nuhptr;

		list.header.root.hptr := nuhptr;



  else --already 1 or more nodes in splaytree

    p:=list.header.root;
    search_loop:
    loop --search by Id
      exit search_loop when p=null;
      q:=p;
      if( Id < p.Id ) then
        p:=p.left_child;
      elsif  Id > p.Id  then
        p:=p.right_child;
      else  -- Id=p.Id...duplicate Id!
        status := dupid;
        bok := false;
        exit search_loop;
      end if;
    end loop search_loop;
    -- q is parent-to-be

    if  bok  then -- this is a new unique Id [key]...insert it
      list.header.size := list.header.size + 1;
      p := new splayrec'(Id,data,
			pri1,pri2,pri3,pri4,pri0, q, null,null, null); --q:=parent
      if  Id < q.Id  then
        q.left_child  := p;
      else
        q.right_child := p;
      end if;

      splay(p,list);  --26 jul 94 (expedites subsequent calls to addnode)
      -- tree operations complete


--======================================================================
-- now we deal with 4 list structures...insert node in priority order
--======================================================================

		nuhptr := new hashnode;
		nuhptr.down:=p;
		p.hptr := nuhptr; --ptr from new splayrec back to hashnode
------------------------------------------------------------------
--pri1
		r:=list.header.hash(1)(pri1,pri0);
		nuhptr.hnext(1) := r;
		if r/=null then
			r.hprev(1) := nuhptr;
		end if;
		list.header.hash(1)(pri1,pri0):=nuhptr;
------------------------------------------------------------------
--pri2
		r:=list.header.hash(2)(pri2,pri0);
		nuhptr.hnext(2) := r;
		if r/=null then
			r.hprev(2) := nuhptr;
		end if;
		list.header.hash(2)(pri2,pri0):=nuhptr;
------------------------------------------------------------------
--pri3
		r:=list.header.hash(3)(pri3,pri0);
		nuhptr.hnext(3) := r;
		if r/=null then
			r.hprev(3) := nuhptr;
		end if;
		list.header.hash(3)(pri3,pri0):=nuhptr;
------------------------------------------------------------------
--pri4
		r:=list.header.hash(4)(pri4,pri0);
		nuhptr.hnext(4) := r;
		if r/=null then
			r.hprev(4) := nuhptr;
		end if;
		list.header.hash(4)(pri4,pri0):=nuhptr;
------------------------------------------------------------------


    end if; -- bok

  end if; -- one or more nodes already in splaytree


exception
	when storage_error =>
		text_io.put_line("AddNode memory maximum reached");
		raise;


end AddNode;

















-- prepare for reuse:
procedure make_empty(list: in out listtype; status: out statustype) is
	lstat: statustype;
	id: idtype;
	data: datatype;
	pri1,pri2,pri3,pri4,pri0: integer;
begin
	status:=ok;
	while length(list)>0 loop
		popnode(list,id,data,1,pri1,pri2,pri3,pri4,pri0,lstat);
	end loop;
	if lstat=NilPtr and length(list)>0 then
		raise program_error;
	end if;
	status:=empty;
end make_empty;



function getsize return integer is
begin
	return splayrec'size/8;
end getsize;



end splaypq;

