// ELROS本体：C++(Windows)からの移植
// クラス名などの命名規則が滅茶苦茶ですので参考にしないで下さい

class TRobotConfig {
  int    LinkNumber;             // リンク数 (1,2,3)
  float  L1_Length, L2_Length, L3_Length;    // [m]
  float  L1_Width,  L2_Width,  L3_Width;     // [m]
  float  J1_Radius, J2_Radius, J3_Radius;    // [m]
  TRobotConfig() {
    LinkNumber = 3;
    L1_Length = L2_Length = L3_Length = 1.0;
    J1_Radius = J2_Radius = J3_Radius = 0.1;
    L1_Width = J1_Radius;
    L2_Width = J2_Radius;
    L3_Width = J3_Radius;
  }
}

class TWorld {
  float  PixelRatio;         // [Pixel/m]
  float  OrgX, OrgY;         // [%], 0[%]～100[%]
  int    WindowX, WindowY;   // WindowSize [Pixel]
  int    OriginX, OriginY;   // Origin [Pixel]
  int    Layer;              // 描画するレイヤー（0, 1, 2,...)
  protected int _Depth = 1;  // レイヤーの深さ

  TWorld() {
    PixelRatio  = 200.0;
    OrgX        = 50.0;
    OrgY        = 90.0;
    Layer       = 0;
    WindowX = width;
    WindowY = height;
  }
  void ReCalc() {
    OriginX = (int)((OrgX / 100) * WindowX);
    OriginY = (int)((OrgY / 100) * WindowY);
  }
  int RealToWinX(float x0) {
    float x;
    x = x0 * PixelRatio + OriginX;
    return (int)x;
  }
  int RealToWinY(float y0) {
    float y;
    y = -y0 * PixelRatio + OriginY;
    return (int)y;
  }
  float WinToRealX(int x0) {
    float x;
    x = (x0 - OriginX) / PixelRatio;
    return x;
  }
  float WinToRealY(int y0) {
    float y;
    y = -(y0 - OriginY) / PixelRatio;
    return y;
  }
}

class TRobotController {
  D2T   Ptarget = new D2T();         // 目標のロボットの手先座標 [m]
  float theta11, theta21, theta31;   // 目標のロボットの相対関節角度 [rad]
  D2T   TipVt = new D2T();           // 手先の目標速度ベクトル [m/s]
  // 位置制御用パラメータ
  float	omega1,  omega2,  omega3;    // 現在の各関節の角速度 [rad/s]
  float	omegad1, omegad2, omegad3;   // 現在の各関節の角加速度 [rad/s^2]
  D2T   TipV = new D2T();            // 現在の手先の速度と加速度 [m/s], [m/s^2]
  D2T   TipA = new D2T();
  float Vmax, Amax;                  // 手先位置の速度制限と加速度制限 [m/s], [m/s^2]
  float  allowableError;             // [m]

  // コンストラクタ
  TRobotController() {
    Vmax = 1.0;  // 1[m/s]
    Amax = 5.0;  // 5[m/s^2]
    allowableError = 0.001;  // 1[mm]
    ResetRC();
  }
  // ロボットの制御状態をリセット
  void ResetRC() {
    omega1  = omega2  = omega3  = 0.0;  // 現在の角速度
    omegad1 = omegad2 = omegad3 = 0.0;  // 現在の角加速度
    TipV.x  = TipV.y  = 0.0;            // 現在の手先の速度
    TipA.x  = TipA.y  = 0.0;            // 現在の手先の加速度
    TipVt.x = TipVt.y = 0.0;            // 手先の目標速度ベクトル
  }
}

// ロボット全体のクラス
class TRobot {
  TRobotController	RC;
  TRobotConfig    	RobotConfig;
  D2T 	Pj1, Pj2, Pj3, Pj4;         	// 現在のロボットの関節位置 [m]
  float theta1, theta2, theta3;         // 現在のロボットの相対関節角度 [rad]
  D2T 	Pj10, Pj20, Pj30, Pj40;     	// 軌道表示開始時のロボットの関節位置 [m]
  float theta10, theta20, theta30;      // 軌道表示開始時のロボットの相対関節角度 [rad]
  float theta1r, theta2r, theta3r;      // 記憶したロボットの相対関節角度 [rad]
  // ヤコビ行列
  Tm22  J22;
  Tm33  J23;

  // コンストラクタ
  TRobot() {
    RC           = new TRobotController();
    RobotConfig  = new TRobotConfig();
    Pj1  = new D2T();
    Pj2  = new D2T();
    Pj3  = new D2T();
    Pj4  = new D2T();
    Pj10 = new D2T();
    Pj20 = new D2T();
    Pj30 = new D2T();
    Pj40 = new D2T();
    J22  = new Tm22();
    J23  = new Tm33();
    theta1  = radians(60.0);
    theta2  = radians(30.0);
    theta3  = radians(30.0);
    theta10 = theta1;
    theta20 = theta2;
    theta30 = theta3;
    ReCalc();
  }
  void CalcJ22() {
    J22.m[0][0] = - RobotConfig.L1_Length * sin(theta1) - RobotConfig.L2_Length * sin(theta1 + theta2);
    J22.m[0][1] = - RobotConfig.L2_Length * sin(theta1 + theta2);
    J22.m[1][0] =   RobotConfig.L1_Length * cos(theta1) + RobotConfig.L2_Length * cos(theta1 + theta2);
    J22.m[1][1] =   RobotConfig.L2_Length * cos(theta1 + theta2);
  }
  void CalcJ23() {
    float  s1, s12, s123;
    float  c1, c12, c123;
    float  l1, l2, l3;
    s1 = sin(theta1);	s12 = sin(theta1 + theta2);	s123 = sin(theta1 + theta2 + theta3);
    c1 = cos(theta1);	c12 = cos(theta1 + theta2);	c123 = cos(theta1 + theta2 + theta3);
    l1 = RobotConfig.L1_Length;	l2 = RobotConfig.L2_Length;	l3 = RobotConfig.L3_Length;
    J23.m23.m[0][0] = -l1 * s1 - l2 * s12 - l3 * s123;
    J23.m23.m[0][1] = -l2 * s12 -l3 * s123;
    J23.m23.m[0][2] = -l3 * s123;
    J23.m23.m[1][0] =  l1 * c1 + l2 * c12 + l3 * c123;
    J23.m23.m[1][1] =  l2 * c12 + l3 * c123;
    J23.m23.m[1][2] =  l3 * c123;
  }
  // 順運動学の再計算
  void ReCalc() {
    Pj1.x = Pj1.y = 0.0;
    Pj2.x = Pj1.x + RobotConfig.L1_Length * cos(theta1);
    Pj2.y = Pj1.y + RobotConfig.L1_Length * sin(theta1);
    Pj3.x = Pj2.x + RobotConfig.L2_Length * cos(theta1 + theta2);
    Pj3.y = Pj2.y + RobotConfig.L2_Length * sin(theta1 + theta2);
    Pj4.x = Pj3.x + RobotConfig.L3_Length * cos(theta1 + theta2 + theta3);
    Pj4.y = Pj3.y + RobotConfig.L3_Length * sin(theta1 + theta2 + theta3);
    Pj10.x = Pj10.y = 0.0;
    Pj20.x = Pj10.x + RobotConfig.L1_Length * cos(theta10);
    Pj20.y = Pj10.y + RobotConfig.L1_Length * sin(theta10);
    Pj30.x = Pj20.x + RobotConfig.L2_Length * cos(theta10 + theta20);
    Pj30.y = Pj20.y + RobotConfig.L2_Length * sin(theta10 + theta20);
    Pj40.x = Pj30.x + RobotConfig.L3_Length * cos(theta10 + theta20 + theta30);
    Pj40.y = Pj30.y + RobotConfig.L3_Length * sin(theta10 + theta20 + theta30);
    CalcJ22();
    CalcJ23();
  }
  // 現在位置の記憶
  void SetOriginalPosture() {
    theta10 = theta1;
    theta20 = theta2;
    theta30 = theta3; 
  }
  void RestoreOriginalPosture() {
    theta1 = theta10;
    theta2 = theta20;
    theta3 = theta30;
    ReCalc();
  }
  // 現在のリンク先端座標
  D2T TipPosition() {
    switch (RobotConfig.LinkNumber) {
      case 1 : return Pj2;
      case 2 : return Pj3;
    }
    return (Pj4);
  }
  // 初期のリンク先端座標
  D2T TipPosition0() {
    switch (RobotConfig.LinkNumber) {
      case 1 : return Pj20;
      case 2 : return Pj30;
    }
    return (Pj40);
  }
  // 位置制御関連
  // 位置誤差
  float PositionError() {
    float dx, dy;
    D2T   Ptip = new D2T();

    Ptip = TipPosition();
    dx = Ptip.x - RC.Ptarget.x;
    dy = Ptip.y - RC.Ptarget.y;
    return (sqrt(dx*dx + dy*dy));
  }
}

class TDisplay {
  boolean Trajectory;
  boolean Strobo;
  D2T     LP;      // 最後に線を引いた座標（終点），あるいはMovetoした点
  D2T     LPo;     // オーバーレイレイヤー用
  D2T     Trj;     // 軌跡の終点
  TDisplay() {
    Trajectory  = false;
    Strobo      = false;
    LP          = new D2T();
    LPo         = new D2T();
    Trj         = new D2T();
  }
  void setTrj(D2T Tip) { Trj.x = Tip.x; Trj.y = Tip.y; }
  void setLP(float x, float y)  { LP.x  = x; LP.y  = y; }
  void setLPo(float x, float y) { LPo.x = x; LPo.y = y; }
}

class textInfoClass {
  String J1_Label,  J2_Label,  J3_Label;   // 現在の位置（関節角度）
  String J10_Label, J20_Label, J30_Label;  // 初期位置（関節角度）
  String J11_Label, J21_Label, J31_Label;  // 目標位置（関節角度）
  String Tx_Label,  Ty_Label;              // 現在位置（手先）
  String Tx0_Label, Ty0_Label;             // 初期位置（手先）
  String Tx1_Label, Ty1_Label;             // 目標位置（手先）
  String PositionError_Label;              // 目標位置と現在位置の差
  String TV_Label;                         // 手先座標の速度
  String SimulationTime_Label;             // シミュレーション経過時間
  String ErrorMessage;                     // 特異姿勢発生，など
  // コンストラクタ
  textInfoClass() {
    J1_Label  = J2_Label  = J3_Label  = "";
    J10_Label = J20_Label = J30_Label = "none";
    J11_Label = J21_Label = J31_Label = "none";
    Tx_Label  = Ty_Label  = "";
    Tx0_Label = Ty0_Label = "none";
    Tx1_Label = Ty1_Label = "none";
    PositionError_Label   = "none";
    TV_Label              = "none";
    SimulationTime_Label  = "0.0 [s]";
    ErrorMessage = "";
  }
  void setErrorMessage(String str) { ErrorMessage = str; }
  void resetErrorMessage()         { ErrorMessage = ""; }
  // 常に表示更新
  void displayTextInfo() {
    int  x = 45, y = 0;
    // 現在値
    crowbar._write(x, y++, "【現在値】");
    crowbar._write(x, y++, "第一関節：" + J1_Label);
    crowbar._write(x, y++, "第二関節：" + J2_Label);
    if (ELROS.LinkNumber() == 3) {
      crowbar._write(x, y++, "第三関節：" + J3_Label);
    }
    crowbar._write(x, y++, "手先位置：(" + Tx_Label + "," + Ty_Label + ")");
    y++;
    // 初期位置
    crowbar._write(x, y++, "【初期位置】");
    crowbar._write(x, y++, "第一関節：" + J10_Label);
    crowbar._write(x, y++, "第二関節：" + J20_Label);
    if (ELROS.LinkNumber() == 3) {
      crowbar._write(x, y++, "第三関節：" + J30_Label);
    }
    crowbar._write(x, y++, "手先位置：(" + Tx0_Label + "," + Ty0_Label + ")");
    y++;
    // 目標位置
    crowbar._write(x, y++, "【目標位置】");
    crowbar._write(x, y++, "第一関節：" + J11_Label);
    crowbar._write(x, y++, "第二関節：" + J21_Label);
    if (ELROS.LinkNumber() == 3) {
      crowbar._write(x, y++, "第三関節：" + J31_Label);
    }
    crowbar._write(x, y++, "手先位置：(" + Tx1_Label + "," + Ty1_Label + ")");
//  if (ELROS.Sim.SimulationRunning) {
      y++;
      crowbar._write(x, y++, "（制御情報）");
      crowbar._write(x, y++, "手先速度：" + TV_Label);
      crowbar._write(x, y++, "位置誤差：" + PositionError_Label);
      crowbar._write(x, y++, "経過時間：" + SimulationTime_Label);
//  }
    if (ErrorMessage.length() > 0) crowbar._write(x, y++, ErrorMessage);
    
    // フラグ表示
    x = 0; y = 0;
    crowbar._write(x, y++, "【画面表示設定】");
    crowbar._write(x, y++, "[T] 軌跡表示　　：" + (ELROS.Display.Trajectory ? "する" : "しない"));
    crowbar._write(x, y++, "[S] ストロボ表示：" + (ELROS.Display.Strobo     ? "する" : "しない"));
    crowbar._write(x, y++, "[H] 中断");
    crowbar._write(x, y++, "[ESC] 終了");

    // 操作説明
    x = 0; y = 38;
    crowbar._write(x, y++, "（手動操作）：関節１ [Z][X]，関節２ [C][V]，関節３ [B][N]");
    crowbar._write(x, y++, "（自動動作）：マウス左クリック 目標値設定，[SPACE] 位置制御");

   // コピーライト
   x = 43; y = 38;
   crowbar._write(x, y++, "ELROS(Easy Link RObot Simulator)");
   crowbar._write(x, y++, "by T.Shirai(@tatsuva) 2012/04/25");
  }
}

class TSim {
  boolean  SimulationRunning;
  boolean  RedrawTiming;
  float    RunningTime;			// シミュレーション開始後の積算時間
  float    RunningCycleTime;		// 画面再描画までに行う時間（この積算時間がRedrawIntervalを超えたら一時停止)
  float    KeyStep;        		// キー入力時の関節角度変化の刻み [deg]
  char     J1P, J1M, J2P, J2M, J3P, J3M;      // キー割当
  boolean  TargetTeached;       // 目標位置設定済み（手動操作でリセット）
  float    SamplingTime;	// [s]
  float    RedrawInterval;	// [s]

  int      status = 0;          // 0: Initialize, 1: Running 2: Finishing
  // コンストラクタ
  TSim() {
    KeyStep = 1.0;          // [deg]
    J1P = 'Z';  J1M = 'X';
    J2P = 'C';  J2M = 'V';
    J3P = 'B';  J3M = 'N';
    TargetTeached = false;
    SamplingTime = 0.001;	// 1[ms]
    // 画面再描画間隔の設定
    RedrawInterval = 1.0 / frameRate;
    SimulationRunning = RedrawTiming = false;
  }
  // シミュレーション開始
  void SimulationStart() {
    // シミュレーション経過時間のクリア
    RunningTime = RunningCycleTime = 0.0;
    // シミュレーション中フラグのセット
    SimulationRunning = true;
    // 状態管理のリセット
    status = 0;
  }
  // シミュレーション終了（および強制終了）
  void SimulationStop() {
    // シミュレーション中フラグのリセット
    SimulationRunning = false;
    // 状態管理
    status = 0;
  }
  void ResetRedrawTiming() {
    RedrawTiming = false;
    RunningCycleTime = 0.0;
  }
  void SetRedrawTiming() {
    RedrawTiming = true;
  }
}

// システム統括
class TELROS {
  TWorld          World;
  TRobot          Robot;
  TDisplay        Display;
  TSim            Sim;
  textInfoClass   TextInfo;
  PGraphics       targetLayer;
  PGraphics       PGover;
  // コンストラクタ
  TELROS() {
    World    = new TWorld();
    Robot    = new TRobot();
    Display  = new TDisplay();
    Sim      = new TSim();
    TextInfo = new textInfoClass();
    targetLayer = null;
    PGover   = createGraphics(World.WindowX, World.WindowY, P2D);  // Overlay
  }
  int LinkNumber() { return (Robot.RobotConfig.LinkNumber); }
  boolean IsOneLink() {
    if (Robot.RobotConfig.LinkNumber == 1) return true;
    return false;
  }
  boolean IsTwoLink() {
    if (Robot.RobotConfig.LinkNumber == 2) return true;
    return false;
  }
  boolean IsThreeLink() {
    if (Robot.RobotConfig.LinkNumber == 3) return true;
    return false;
  }
  // レイヤー
  void overlayLayer(boolean layer) {
    if (layer) targetLayer = PGover;
      else     targetLayer = null;
  }
  void BeginDraw(boolean layer) {
    overlayLayer(layer);
    if (targetLayer != null) targetLayer.beginDraw();
  }
  void EndDraw() {
    if (targetLayer != null) targetLayer.endDraw();
  }

  // Canvasへの描画命令
  void Line(float x1, float y1, float x2, float y2) {
    _line(World.RealToWinX(x1), World.RealToWinY(y1), World.RealToWinX(x2), World.RealToWinY(y2));
  }
  void LineTo(float x1, float y1) {
    _lineTo(World.RealToWinX(x1), World.RealToWinY(y1));
  }
  void MoveTo(float x1, float y1) {
    _moveTo(World.RealToWinX(x1), World.RealToWinY(y1));
  }
  void Circle(float x, float y, float r) {
    int xc, yc, rc;
    xc = World.RealToWinX(x);
    yc = World.RealToWinY(y);
    rc = int(r * World.PixelRatio);
    _circle(xc, yc, rc);
  }
  void PenColor(int c) {
    if (targetLayer != null) targetLayer.stroke(c);
      else                   stroke(c);
  }
  void PenWidth(int w) {
    if (targetLayer != null) targetLayer.strokeWeight(w);
      else                   strokeWeight(w);
  }
  void _noFill() {
    if (targetLayer != null) targetLayer.noFill();
      else                   noFill();
  }
  void clrscr() {
    if (targetLayer != null) {
      targetLayer.background(#000000, 0);     // 背景色
      targetLayer.fill(crowbar.textColor);    // 塗りつぶし色
    } else {
      background(crowbar.backgroundColor);    // 背景色
      fill(crowbar.textColor);                // 塗りつぶし色
    }
  }
  void _line(float x1, float y1, float x2, float y2) {
    if (targetLayer != null) {
      targetLayer.line(x1, y1, x2, y2);
      Display.setLPo(x2, y2);
    } else {
      line(x1, y1, x2, y2);
      Display.setLP(x2, y2);
    }
  }
  void _lineTo(float x1, float y1) {
    if (targetLayer != null) {
      targetLayer.line(Display.LPo.x, Display.LPo.y, x1, y1);
      Display.setLPo(x1, y1);
    } else {
      line(Display.LP.x, Display.LP.y, x1, y1);
      Display.setLP(x1, y1);
    }
  }
  void _moveTo(float x1, float y1) {
    if (targetLayer != null) Display.setLPo(x1, y1);
      else                   Display.setLP(x1, y1);
  }
  void _circle(float x, float y, float r) {
    if (targetLayer != null) {
      targetLayer.ellipse(x, y, r * 2, r * 2);
      Display.setLPo(x, y);
    } else {
      ellipse(x, y, r * 2, r * 2);
      Display.setLP(x, y);
    }
  }
  // リンクの四角を描く
  void LinkBox(float x1, float y1, float x2, float y2, float LW) {
    D2T  P1, P2, P3, P4;
    float angle;   // [rad]

    P1 = new D2T();
    P2 = new D2T();
    P3 = new D2T();
    P4 = new D2T();
    angle = atan2(y2 - y1, x2 - x1) + (HALF_PI);
    P1.x = P3.x = LW * cos(angle);
    P2.x = P4.x = - P1.x;
    P1.y = P3.y = LW * sin(angle);
    P2.y = P4.y =- P1.y;
    P1.x += x1; P1.y += y1;
    P2.x += x1; P2.y += y1;
    P3.x += x2; P3.y += y2;
    P4.x += x2; P4.y += y2;
    Line(P1.x, P1.y, P2.x, P2.y);
    LineTo(P4.x, P4.y);
    LineTo(P3.x, P3.y);
    LineTo(P1.x, P1.y);
  }
  // ロボットの描画
  void DrawRobotStrobo() {
    DrawRobot(true);
  }
  // 背景画面： false, オーバーレイ画面（ストロボ）： true
  void DrawRobot(boolean layer) {
    overlayLayer(false);
    Robot.ReCalc();
    DrawAxis();
    BeginDraw(layer);
    // 関節
    PenColor(#000000);
    _noFill();
    Circle(Robot.Pj1.x, Robot.Pj1.y, Robot.RobotConfig.J1_Radius);
    if (LinkNumber() >= 2) {
      Circle(Robot.Pj2.x, Robot.Pj2.y, Robot.RobotConfig.J2_Radius);
      if (LinkNumber() >= 3) {
        Circle(Robot.Pj3.x, Robot.Pj3.y, Robot.RobotConfig.J3_Radius);
      }
    }
    // リンク
    PenColor(#ff0000);
    Line(Robot.Pj1.x, Robot.Pj1.y, Robot.Pj2.x, Robot.Pj2.y);
    LinkBox(Robot.Pj1.x, Robot.Pj1.y, Robot.Pj2.x, Robot.Pj2.y, Robot.RobotConfig.L1_Width);
    if (LinkNumber() >= 2) {
      Line(Robot.Pj2.x, Robot.Pj2.y, Robot.Pj3.x, Robot.Pj3.y);
      LinkBox(Robot.Pj2.x, Robot.Pj2.y, Robot.Pj3.x, Robot.Pj3.y, Robot.RobotConfig.L2_Width);
      if (LinkNumber() >= 3) {
        Line(Robot.Pj3.x, Robot.Pj3.y, Robot.Pj4.x, Robot.Pj4.y);
        LinkBox(Robot.Pj3.x, Robot.Pj3.y, Robot.Pj4.x, Robot.Pj4.y, Robot.RobotConfig.L3_Width);
      }
    }
    EndDraw();
  }
  // xy軸を描画
  void DrawAxis() {
    BeginDraw(false);
    PenColor(#000000);
    line(0, World.OriginY, World.WindowX, World.OriginY);
    line(World.OriginX, 0, World.OriginX, World.WindowY);
    EndDraw();
  }
  void DrawTrajectory() {
    D2T	Ptip;

    Ptip = Robot.TipPosition();
    if (Ptip.Sub(ELROS.Display.Trj).Norm() > 0.0) {
      BeginDraw(true);
      PenColor(#0000ff);
      Line(ELROS.Display.Trj.x, ELROS.Display.Trj.y, Ptip.x, Ptip.y);
      EndDraw();
      ELROS.Display.setTrj(Ptip);
    }
  }
  // 目標位置の十文字描画＋直線軌道描画
  void TargetBullet() {
    int X, Y;
    D2T	Ptip;

    BeginDraw(true);
    // 十文字描画
    X = World.RealToWinX(Robot.RC.Ptarget.x);
    Y = World.RealToWinY(Robot.RC.Ptarget.y);
    PenColor(#00ff00);
    _line(X - 20, Y, X + 20, Y);
    _line(X, Y - 20, X, Y + 20);
    _noFill();
    _circle(X, Y, 10);
    // 直線軌道描画
    Ptip = Robot.TipPosition();
    Line(Robot.RC.Ptarget.x, Robot.RC.Ptarget.y, Ptip.x, Ptip.y);
    EndDraw();
  }

  // 位置制御関連
  // 位置制御本体(軌道制御なし, Vmaxで移動)
  void PositionControl_NonCP(boolean JointConstraint) {
    Tm33    InvJ32 = new Tm33();
    Tm22    InvJ22 = new Tm22();
    String  str;

    if (LinkNumber() == 1) return;
    if (Sim.status == 0) {
      // ロボットコントローラ（現在の状態）のリセット
      Robot.RC.ResetRC();
      Sim.SimulationStart();
      Sim.status++;
      return;
    } else if (Sim.status == 1) {
//    println(Robot.PositionError());
      while (Sim.RunningCycleTime <= Sim.RedrawInterval) {
        if (Sim.SimulationRunning == false) return;
        if (Robot.PositionError() < Robot.RC.allowableError) {
          Sim.status++;
          break;
        }
        // 目標速度ベクトルのセット
        Robot.RC.TipVt = Robot.RC.Ptarget.Sub(Robot.TipPosition()).Unit().Scale(Robot.RC.Vmax);
        if (LinkNumber() == 2) {
          return;
        } else if (LinkNumber() == 3) {
          Robot.ReCalc();
          Robot.CalcJ23();
          InvJ32.m32 = Robot.J23.Inv();
          if (InvJ32.m32.InvError == true) {
            // 特異姿勢発生
            ELROS.TextInfo.setErrorMessage("特異姿勢が発生しました．");
            Robot.theta2 += 0.01;
            Robot.ReCalc();
            Sim.status++;
            break;
          }
          Robot.theta1 += (InvJ32.m32.m[0][0] * Robot.RC.TipVt.x + InvJ32.m32.m[0][1] * Robot.RC.TipVt.y) * Sim.SamplingTime;
          Robot.theta2 += (InvJ32.m32.m[1][0] * Robot.RC.TipVt.x + InvJ32.m32.m[1][1] * Robot.RC.TipVt.y) * Sim.SamplingTime;
          Robot.theta3 += (InvJ32.m32.m[2][0] * Robot.RC.TipVt.x + InvJ32.m32.m[2][1] * Robot.RC.TipVt.y) * Sim.SamplingTime;
        } else {
          return;
        }
        Sim.RunningTime      += Sim.SamplingTime;
        Sim.RunningCycleTime += Sim.SamplingTime;
      }
      Sim.ResetRedrawTiming();
      // シミュレーション中の情報更新
      refreshSimulationInfo();
    } else if (Sim.status == 2) {
//    DrawRobot();
      Sim.SimulationStop();
    }
  }
  // 目標姿勢での関節角度を事前に調べる
  void CalculateTargetPosture() {
    Tm33    InvJ32 = new Tm33();
//    Tm22    InvJ22 = new Tm22();
    D2T	    TipVt;
    String  str;
    boolean InvError = false;

    // 現在の姿勢のバックアップ
    Robot.SetOriginalPosture();

    if (LinkNumber() == 1) return;
    while (Robot.PositionError() > Robot.RC.allowableError) {
      // 目標速度ベクトルのセット
      TipVt = Robot.RC.Ptarget.Sub(Robot.TipPosition()).Unit().Scale(Robot.RC.Vmax);
      if (LinkNumber() == 2) {
/*
        Robot.ReCalc();
        Robot.CalcJ22();
        InvJ22.m22 = Robot.J22.Inv();
        if (InvJ22.m22.InvError == true) {
          TextInfo.setErrorMessage("警告：特異姿勢が発生します．");
          break;
        } else {
          Robot.theta1 += (InvJ22.m22.m[0][0] * TipVt.x + InvJ22.m22.m[0][1] * TipVt.y) * Sim.SamplingTime;
          Robot.theta2 += (InvJ22.m22.m[1][0] * TipVt.x + InvJ22.m22.m[1][1] * TipVt.y) * Sim.SamplingTime;
        }
        break;
*/
      } else if (LinkNumber() == 3) {
        Robot.ReCalc();
        Robot.CalcJ23();
        InvJ32.m32 = Robot.J23.Inv();
        if (InvJ32.m32.InvError == true) {
          TextInfo.setErrorMessage("警告：特異姿勢が発生します．");
          break;
        } else {
          Robot.theta1 += (InvJ32.m32.m[0][0] * TipVt.x + InvJ32.m32.m[0][1] * TipVt.y) * Sim.SamplingTime;
          Robot.theta2 += (InvJ32.m32.m[1][0] * TipVt.x + InvJ32.m32.m[1][1] * TipVt.y) * Sim.SamplingTime;
          Robot.theta3 += (InvJ32.m32.m[2][0] * TipVt.x + InvJ32.m32.m[2][1] * TipVt.y) * Sim.SamplingTime;
        }
      } else {
        break;
      }
    }
    // 目標角度の保存
    Robot.RC.theta11 = Robot.theta1;
    Robot.RC.theta21 = Robot.theta2;
    Robot.RC.theta31 = Robot.theta3;
    if (!InvError) {
      // テキスト情報の更新
      setTargetInfo();
    } else {
      TextInfo.setErrorMessage("警告：特異姿勢が発生します．");
    }
    // 現在の姿勢を復元
    Robot.RestoreOriginalPosture();
  }
  // テキスト情報関係
  // 現在の情報（常に更新）
  void refreshTextInfo() {
    D2T      Ptip;
    // 現在位置
    TextInfo.J1_Label = nf(degrees(Robot.theta1), 0, 1) + " [deg]";
    TextInfo.J2_Label = nf(degrees(Robot.theta2), 0, 1) + " [deg]";
    TextInfo.J3_Label = nf(degrees(Robot.theta3), 0, 1) + " [deg]";
    Ptip = Robot.TipPosition();
    TextInfo.Tx_Label = nf(Ptip.x, 0 , 3) + " [m]";
    TextInfo.Ty_Label = nf(Ptip.y, 0 , 3) + " [m]";
  }
  // 目標角度と位置（ティーチ直後のみ）
  void setTargetInfo() {
    D2T  Tp;
    // 目標位置
    TextInfo.J11_Label = nf(degrees(Robot.RC.theta11), 0, 1) + " [deg]";
    TextInfo.J21_Label = nf(degrees(Robot.RC.theta21), 0, 1) + " [deg]";
    TextInfo.J31_Label = nf(degrees(Robot.RC.theta31), 0, 1) + " [deg]";
    TextInfo.Tx1_Label = nf(Robot.RC.Ptarget.x, 0, 3) + " [m]";
    TextInfo.Ty1_Label = nf(Robot.RC.Ptarget.y, 0, 3) + " [m]";
    // 初期位置
    TextInfo.J10_Label = nf(degrees(Robot.theta10), 0, 1) + " [deg]";
    TextInfo.J20_Label = nf(degrees(Robot.theta20), 0, 1) + " [deg]";
    TextInfo.J30_Label = nf(degrees(Robot.theta30), 0, 1) + " [deg]";
    Tp = Robot.TipPosition0();
    TextInfo.Tx0_Label = nf(Tp.x, 0 , 3) + " [m]";
    TextInfo.Ty0_Label = nf(Tp.y, 0 , 3) + " [m]";
  }
  // ロボットコントローラの更新情報（シミュレーション中のみ）
  void refreshSimulationInfo() {
    TextInfo.TV_Label             = nf(Robot.RC.TipVt.Norm(), 0, 3) + " [m/s]";
    TextInfo.PositionError_Label  = nf(Robot.PositionError(), 0, 3) + "[m]";
    TextInfo.SimulationTime_Label = nf(Sim.RunningTime, 0, 2) + "[s]";
  }
}

void FormKeyDown()
{
  float theta10, theta20, theta30;
  char Key;

  Key = str(key).toUpperCase().charAt(0);
  theta10 = ELROS.Robot.theta1;
  theta20 = ELROS.Robot.theta2;
  theta30 = ELROS.Robot.theta3;
  if (Key == ELROS.Sim.J1P) ELROS.Robot.theta1 += radians(ELROS.Sim.KeyStep);
  if (Key == ELROS.Sim.J1M) ELROS.Robot.theta1 -= radians(ELROS.Sim.KeyStep);
  if (Key == ELROS.Sim.J2P) ELROS.Robot.theta2 += radians(ELROS.Sim.KeyStep);
  if (Key == ELROS.Sim.J2M) ELROS.Robot.theta2 -= radians(ELROS.Sim.KeyStep);
  if (Key == ELROS.Sim.J3P) ELROS.Robot.theta3 += radians(ELROS.Sim.KeyStep);
  if (Key == ELROS.Sim.J3M) ELROS.Robot.theta3 -= radians(ELROS.Sim.KeyStep);
  if (Key == ' ') ELROS.PositionControl_NonCP(false);
  if (Key == 'T') {
    // 軌跡表示のOn/Off
    ELROS.Display.Trajectory = !ELROS.Display.Trajectory;
    // 現在値を記録
    if (ELROS.Display.Trajectory) ELROS.Display.setTrj(ELROS.Robot.TipPosition());
    // オーバーレイ画面の消去
    overlayClear();
  }
  if (Key == 'S') {
    // ストロボ表示のOn/Off
    ELROS.Display.Strobo     = !ELROS.Display.Strobo;
    // オーバーレイ画面の消去
    overlayClear();
  }
  if ((theta10 != ELROS.Robot.theta1) || (theta20 != ELROS.Robot.theta2) || (theta30 != ELROS.Robot.theta3)) {
      ELROS.Sim.TargetTeached = false;
  }
}

void TrajectoryClear_SubMeuClick()
{
  D2T  Ptip = new D2T();

  // 初期姿勢情報の可視化
//  ELROS.VisibleOriginalPosture(true);
  // 初期姿勢の描画
  Ptip = ELROS.Robot.TipPosition();
  ELROS.MoveTo(Ptip.x, Ptip.y);
}

// オーバーレイ画面の消去
void overlayClear()
{
  ELROS.BeginDraw(true);
  ELROS.clrscr();
  ELROS.EndDraw();
}

void mainPaintBoxMouseDown(int X, int Y)
{
  // 目標位置の設定
  if (mouseButton == LEFT) {
    ELROS.TextInfo.resetErrorMessage();
    ELROS.Robot.RC.Ptarget.x = ELROS.World.WinToRealX(X);
    ELROS.Robot.RC.Ptarget.y = ELROS.World.WinToRealY(Y);
    ELROS.Robot.SetOriginalPosture();

    // オーバーレイ画面を消去
    overlayClear();
    // 十文字を描く
    ELROS.TargetBullet();
    // 現在の姿勢を描く
    ELROS.DrawRobotStrobo();

    // モード解除
    ELROS.Sim.TargetTeached = true;
    // 目標姿勢の算出とテキスト情報の更新
    ELROS.CalculateTargetPosture();
  }
}

// onPaint()ハンドラー
void mainPaintBoxRefresh()
{
  // 背景画面
  ELROS.overlayLayer(false);
  ELROS.clrscr();
  ELROS.DrawRobot(false);
  ELROS.refreshTextInfo();
  ELROS.TextInfo.displayTextInfo();

  // オーバーレイ画面
  // 軌跡表示
  if (ELROS.Display.Trajectory == true) {
      ELROS.DrawTrajectory();
  }
  // ストロボ表示
  if (ELROS.Display.Strobo == true) {
    ELROS.DrawRobotStrobo();
  }
  blend(ELROS.PGover, 0, 0, ELROS.World.WindowX, ELROS.World.WindowY, 0, 0, ELROS.World.WindowX, ELROS.World.WindowY, BLEND);
}

