////////////////////////////////////////////////////////////////
// Processingによる古典的な対話式プログラム開発用フレームワーク
//               Crowbar（クロウバー）
//    with 高機能グラフィックスライブラリTomahawk（トマホーク）
//                                  白井　達也（Tatsuya Shirai)
//                       shirai＠mech.suzuka-ct.ac.jp, @tatsuva
//                                       2012/04/20    Ver. 0.9
//                                       2012/05/14    Ver. 1.0
//                                       2012/05/15    Ver. 1.1
//                                       2012/05/24    Ver. 1.2
//                                       2012/05/26    Ver. 1.3
//                          2012/05/27 - 2012/06/01    Ver. 2.0
//                          2012/06/04 - 2012/06/04    Ver. 2.1
//                          2012/06/05 - 2012/06/21    Ver. 3.0
//                          2012/06/21 - 2012/07/02    Ver. 3.1
//                          2012/07/02 - 2012/07/05    Ver. 3.2
//                          2012/07/05 - 2012/07/11    Ver. 3.3
//                          2012/07/11 - 2012/07/16    Ver. 3.4
//                          2012/07/16 - 
////////////////////////////////////////////////////////////////
// 最新版の公開は以下のSourceForge.JPにて行っています．
// http://sourceforge.jp/projects/crowbar/
// ドキュメントは以下のMoodleサイト上のWikiにて公開しています．
// http://www.suzuka-ct.ac.jp/mech/moodle/mod/wiki/view.php?id=1739&page=Crowbar
// 質問などは上記SourceForge.JPあるいはmechMoodle上のフォーラムに
// 投稿するか，あるいはTwitter，またはメールでお問い合わせ下さい．

// グラフィックスライブラリTomahawkを使用しない場合は，タブTomahawkClassをDeleteした上で，以下のコメントアウトされた一行のコードの行頭のコメント記号（//）を外す．
// さらにvoid initCrowbar()中のcrow = startCrowbar.generate(50, 30, 2);が第3引数を持つ場合は削除すること．例） startCrowbar.generate(50, 30);
// class tomahawkClass { tomahawkClass(int mode) {} protected int tomahawkMode = 0; void backupViewports() {} void restoreViewports() {} void refreshView() {} void resetView(boolean flag) {} }

// デバッグ用：P2DとjAVA2Dの切り替え
boolean _JAVA2D = true;

// ----------------------------------------------------------------------------------------
// ユーザが自由なインスタンス名を使えるように，
// crowbarClass.pdeとtomahawkClass.pdeで使用するインスタンスcrowbarのシャロウコピー提供する
// ----------------------------------------------------------------------------------------
crowbarClass crowbar;
class generateCrowbar {
  // Tomahawkを使用しない場合
  crowbarClass generate(int x, int y) {
    return generate(x, y, 0);
  }
  // Tomahawkを使用する場合
  crowbarClass generate(int x, int y, int mode) {
    crowbar = new crowbarClass(x, y, mode);
    return crowbar;
  }
}
generateCrowbar startCrowbar = new generateCrowbar();

// デフォルトのテキスト属性
protected int _FONTSIZE      = 14;
protected int _TEXTCOLOR     = #000002;  // PGraphicsの仕様上の問題点の回避 (#000000や#000001ではtext()で貼った際に色が#000000になって背景と区別がつかない）
protected int _BGCOLOR       = #ffffee;
// デフォルトのスクリーン属性（いまのところ変更予定なし）
protected float _HSTEP       = 1.2;  // 文字送り
protected float _VSTEP       = 1.4;  // 行送り
// タブ表示の初期値
protected float _TABSIZE     = 10.0;
// フォント名
// protected String _FONTNAME = "MS Gothic";
protected String _FONTNAME = "FFScala";
// デバッグ用
protected boolean _DEBUGFLAG = false;

///////////////////////////
// テキストバッファのクラス
// （スクロール対応用）
///////////////////////////
// メモリの確保は単一方向リスト構造とする．
// オーバーラップして折り返す分は次のラインデータに入れる．後日，実行ウィンドウのサイズ変更に
// 対応する際には，一旦，改行されたラインデータを繋げ直して，それからまたオーバーラップし直す．
// その際に行数が減るかも知れない．不要になったラインデータはリスト構造から切り離し，GCに開放させる．
// Ver.1.04.00で文字色の変更に対応するために，write()で出力された単位で一つのclassとして文字情報とした．
// 一応，将来的な拡張を考慮に入れて背景色も考慮する．classとしては他の拡張属性のも対応できるように考えた上で実装する．

// テキスト情報の属性
class textAttrClass {
  protected int  textColor;  // 文字色        ：変更可
  protected int  bgColor;    // 背景色        ：基本的に変えない
  protected int  fontSize;   // フォントサイズ：基本的に変えない
  // コンストラクタ
  textAttrClass () {
    init();
  }
  // システムデフォルト値で初期化
  void init() {
    fontSize   = _FONTSIZE;
    textColor  = _TEXTCOLOR;
    bgColor    = _BGCOLOR;
  }
  // テキスト属性の変更
  void setTextColor(int col) { textColor = col;  }  // ユーザは利用不可
  void setbgColor(int col)   { bgColor   = col;  }  // ユーザは利用不可
  void setFontSize(int size) { fontSize  = size; }  // ユーザは利用不可
  // テキスト属性の複製
  void clone(textAttrClass attr) {
    fontSize  = attr.fontSize;    // フォントサイズ 
    textColor = attr.textColor;   // 文字色
    bgColor   = attr.bgColor;     // 背景色
  }
  // 引数で与えられたテキスト属性と現テキスト属性の比較（等しければtrueを返す）
  boolean checkTextAttr(textAttrClass attr) {
    if (attr.fontSize   != fontSize)  return false;
    if (attr.textColor  != textColor) return false;
    if (attr.bgColor    != bgColor)   return false;
    return true;
  }
}

// テキストの最小単位（センテンス）：write()する度にバッファに追加される
class sentenceClass {
  protected String sentence;   // 同じ文字属性の一連なりの文字列
  textAttrClass    attr;       // 文字属性
  sentenceClass    next;       // 次のセンテンスへのポインタ
  // コンストラクタ
  sentenceClass(String str, textAttrClass attribute) {
    sentence = str;
    attr     = new textAttrClass();
    next     = null;
    attr.clone(attribute);
  }
  // このセンテンスのみを初期化（リスト構造を下流に下らない）
  void clearSentence(textAttrClass attribute) {
    sentence = "";
    attr.clone(attribute);
    next     = null;
  }
  // センテンスのリスト最後部への追加
  void addSentence(String str, textAttrClass attr) {
    if (next == null) next = new sentenceClass(str, attr);  // 終了条件
      else            next.addSentence(str, attr);          // 再帰呼び出し
  }
  // 文字属性を考慮に入れて同じ属性の場合は文字列をマージする
  // 再帰呼び出しではないが，リスト末尾まで「これとその次を比較してこれに次をマージ」を繰り返す．
  void merge() {
    if (next == null) return;  // １個しか無い（あるいは終端を呼ばれた）ならばマージは不要．特に空行のことを考慮にいれた特例
    sentenceClass current = this;
    while (current.next != null) {
      if (current.attr.checkTextAttr(current.next.attr) || (current.sentence.length() == 0)) {
        // もしこれと次のセンテンスのテキスト属性が等しいならば次の文字列をこれの文字列に追加
        current.sentence += current.next.sentence;
        // 文字列数ゼロのセンテンスを後ろの文字列と合体させるために属性を後ろからコピー
        current.attr.clone(current.next.attr);
        // 次のセンテンスを無視してその次と繋げ直す（current.netxt.nextはnullかも知れないが，それは次のwhileの終了条件でチェックされる）
        current.next = current.next.next;
        // ターゲットは次へ移るのではなく，これのままであるのがミソ
      } else {
        // 「これと次が異なる」ならばターゲットを次に移す
        current = current.next;
      }
    }
  }
  // センテンスを指定された長さで分割し，残りを新たに確保した領域にコピーする
  // 分割の必要が無かった場合は文字列数０のセンテンスを返す（扱いは上流に任せる）
  sentenceClass split(int len) {
    sentenceClass newone = new sentenceClass("", attr);  // テキスト属性クローンで新規に作成
    // 文字列の分割コピー
    newone.sentence = sentence.substring(len);
    sentence        = sentence.substring(0, len);
    // リストを挿入
    newone.next = next;
    next        = newone;
    return newone;
  }
  // 文字列の長さ（このリストから先の文字列の長さを合算）
  int length() {
    if (next == null) return sentence.length();                  // 終了条件
      else            return next.length() + sentence.length();  // 再帰呼び出し
  }
   // 属性なしの文字列として返す（このリストから先の全て）
   String getPlainText() {
   if (next == null) return sentence;
     else            return sentence + next.getPlainText();
   }
}

// 一行分のクラス
class lineClass {
  protected int     no;      // 行番号．０から始まる（デバッグ用）
  sentenceClass     line;    // センテンスのリスト構造の先頭（テキストバッファのデータはここに）
  protected boolean flag;    // 入力済みか？（空行を考慮．lineの文字数では判断できないため．さらにGCを使わずリストを再利用するためにも）
  protected boolean overlap; // 前の行の折り返しか（後日，実行ウィンドウのサイズ変更に対応するために）
  lineClass         next;    // 次の行の位置
  // コンストラクタ
  lineClass() {
    clear();
  }
  // クリア
  void clear() {
    no         = 0;
    line       = null;
    flag       = false;
    overlap    = false;
    next       = null;
  }
  // 行単位のデータのセット（オーバーラップ処理は上流で行う．この行がオーバーラップする行ならばoverをtrueにセットしてコール）
  void set(sentenceClass data, boolean over) {
    // 現リストにデータをセット
    // no は初期化時あるいは一つ前のリストにデータをセットされた時に設定済み
    line    = data;    // buffをそのまま使う．したがってbuffはset()した後に再確保が必要
    flag    = true;
    overlap = over;
    if (next == null) next = new lineClass();  // 次の行データの確保（ただし，リスト構造を再利用する場合は確保しない）
    next.no = no + 1;  // 自動加算
  }
  // テキスト属性を考慮に入れて同じテキスト属性を持つ隣接したリストをマージする処理を行単位で実行する
  void merge() { 
    line.merge();
  }
}

///////////////////////////////
// テキストバッファ本体のクラス
class textbufferClass {
  private int           cwidth;            // 実行ウィンドウのテキストの一行の文字数
  private screenClass   screenSetting;     // 画面設定
  int                   fontSize;          // 実行ウィンドウのテキストのフォントサイズ
  private lineClass     topLine;           // 一番最初の行データ
  lineClass             currentWriteLine;  // 現在の書き込み可能な行データ
  textAttrClass         currentAttr;       // 現在の文字属性（主に書き込み用）
  protected int         totalLines;        // 出力済みのテキストバッファの行数を単純に返す（オーバーラップは考慮しない→考慮する）
  protected int         readingLine;       // 現在の読み出しされている行番号
  private sentenceClass buff;              // 現在の未改行のバッファ

  // コンストラクタ
  textbufferClass(screenClass scr, textAttrClass attr) {
    screenSetting    = scr;
    cwidth           = screenSetting.cwidth;  // 一行の文字数
    fontSize         = attr.fontSize;
    topLine          = new lineClass();
    currentWriteLine = topLine;
    currentAttr      = new textAttrClass();  // デフォルト値が設定される
    totalLines       = 0;
    readingLine      = 0;
    buff             = new sentenceClass("", currentAttr);  // 常に最初のセンテンスは空
  }
  // 全テキストバッファのクリア
  // GCで消えるのを期待するのではなく古いリスト構造を再利用する
  void clearTextBuffer() {
    lineClass  temp, next;
    temp = topLine;
    while (true) {
      if (temp.next == null)  break;
      //    if (temp.flag == false) break;  // 安全のためにコメントアウト
      next = temp.next;  // 次のリストのポインタを退避させてから
      temp.clear();      // 現ラインのデータを初期化して
      temp = next;       // 次に移動する
    }
    currentAttr.init();  // システムデフォルト値に初期化
    currentWriteLine = topLine;
    totalLines       = 0;
    readingLine      = 0;
    if (buff != null) buff.clearSentence(currentAttr);  // バッファの初期化（リストを辿らない：GCに期待）
  }
  ///////////////////////
  // オーバーラップ関係
  // 与えられた文字列の幅（文字数では無い）を返す（全角1.0，半角0.5）
  float getStringWidth(String str) {
    float len;
    if (crowbar.tomahawkMode == 2) len = screenSetting.pgText.textWidth(str) / fontSize;
      else                         len = textWidth(str) / fontSize;
    if ((crowbar.proportional == false) && (len < 0.5)) len = 0.5; // 半角文字を0.5に固定
    return len;
  }
  // 指定された文字幅は与えられた文字列の何文字目までなのかを調べる
  int convertStringWidth2Length(String str, int len) {
    int i;
    for (i = len - 1; i < str.length(); i++) {    // len - 1から始めるのは高速化のため
      if (getStringWidth(str.substring(0, i + 1)) > len) break;
    }
    return i;
  }
  // -----------バッファ関連 --------------------
  // バッファにセンテンスを追加する（単純にチェーンを繋ぐ．後で属性をチェックしてマージする必要がある．空文字列でも繋ぐ）
  void appendSentence(String str, textAttrClass attr) {
    if (buff == null) buff = new sentenceClass(str, attr);  // バッファが無い！ことは無いはず．
      else            buff.addSentence(str, attr);
    // 文字表示属性の更新
    currentAttr.clone(attr);
  }
  // データ書き込みの本体（もしこの行がオーバーラップされた行ならばoverlapにtrueをセットしてコールする）
  void _setLine(boolean overlapped) {
    currentWriteLine.set(buff, overlapped);
    currentWriteLine = currentWriteLine.next;
    totalLines++;
  }
  // バッファのデータを行データとして確定する
  // オーバーラップを考慮する必要があるときはoverlapをtrueにセットしてコールする
  // オーバーラップ有効時に折り返し分を新たに行データを作成してそちらに続ける．
  void setLine(boolean overlap) {
    if (!overlap) {
      buff.merge();  // マージを先に行う
      _setLine(false);
    } else {
      normalizeBuffer();
      _setLine(false);
    }
    buff = new sentenceClass("", currentAttr);  // 常に最初のセンテンスは空
  }
  // いまバッファに溜まっているセンテンスが一行の行数を超えているならば超えている分はラインデータとして確定する
  // これはオーバーラップ許可時以外に呼び出してはいけない．
  void normalizeBuffer() {
    if (getStringWidth(buff.getPlainText()) >= cwidth) {
      // オーバーラップ有効時の分割処理
      // lineClassのoverlapはオーバーラップの”発生した”行のみtrueとする
      buff.merge();  // マージを先に行う
      sentenceClass  bufBakPtr, buffPtr;
      textAttrClass  bakAttr;
      float total = 0.0;

      buffPtr = buff;
      while (true) {
        if (buffPtr == null) break;
        // 折り返しが必要な場合
        if (total + getStringWidth(buffPtr.sentence) >= cwidth) {
          // センテンスを分割する
          bufBakPtr = buffPtr.split(convertStringWidth2Length(buffPtr.sentence, int(cwidth - total)));
          buffPtr.next = null;
          // これはオーバーラップした分なので引数true
          _setLine(true);
          // 分割した残りが空の場合
          if (bufBakPtr.sentence.length() == 0) buff = bufBakPtr.next;  // 単に無視
            else                                buff = bufBakPtr;
          buffPtr = buff;
          total   = 0;
        } else {
          total += getStringWidth(buffPtr.sentence);
          if (buffPtr.next == null) {
            // オーバーラップの発生していない行（あるいは残りの部分）
            break;
          } else buffPtr = buffPtr.next;
        }
      }
    }
    return;
  }
  // バッファをフラッシュ（書き出す）
  void flushBuffer(boolean overlap) {
    if (buff.length() > 0) setLine(overlap);
  }
  // 指定された行番号の行データを返す（０行から始める）
  lineClass getTargetLine(int line) {
    lineClass temp = topLine;
    int  num;
    for (num = 0; num < line; num++) {
      if (temp.next == null) break;
      if (temp.flag != true) break;
      temp = temp.next;
    }
    return temp;
  }
  // 表示開始行をダイレクトに指定する
  void setReadingLine(int num) {
    if (num < 0) num = 0;
    if (crowbar.afterRunning && (num >= totalLines - 1)) num = totalLines - 1;
    readingLine = num;
  }
  // 表示開始行数を指定行数分，増加する
  void incrementReadingLine(int num) {
    setReadingLine(readingLine + num);
  }
  // デバッグ用：テキストバッファを全て表示
  void displayAllTextBuffer() {
    int  i;
    lineClass temp;
    temp = topLine;
    sentenceClass sen;
    i = 0;
    println();
    println("No.:(No.):flag:overlap:length[sentence1, sentence2, ...]");
    while (true) {
      if (temp == null) break;
      if (temp.line == null) break;
      print(str(i) + ":" + str(temp.no) + ":" + str(temp.flag) +":" + str(temp.overlap) + ":" + str(temp.line.length()));
      print("[");
      sen = temp.line;
      while (true) {
        print(hex(sen.attr.textColor));
        print("(" + str(sen.sentence.length()) + ")");
        print(",");
        if (sen.next == null) break;
        sen = sen.next;
      } 
      println("]");
      if (temp.next == null) break;
      temp = temp.next;
      i++;
    }
  }
}

// キーボード入力規則のクラス
class keyInputConstraintClass {
  private boolean point;     // [.]キーが押されたか
  private boolean sign;      // [+],[-]キーが押されたか
  private boolean inputted;  // 数字が既に入力済みか
  // コンストラクタ
  keyInputConstraintClass() { reset(); }
  // 初期化
  void reset() { point = sign = inputted = false; }
  // 1) 浮動小数点時の入力規則チェック
  boolean checkFloat(char c) {
    boolean flag = true;
    switch (c) {
      // 符号
    case '+' :
    case '-' :
      if ((sign) || (inputted)) flag = false;
        else                    sign = true;
      break;
      // 小数点
    case '.' :
      if ((point) || (!inputted)) flag  = false;
        else                      point = true;
      break;
    default :
      if (isNumberChar(c)) inputted = true;
        else               flag     = false;
      break;
    }
    return flag;
  }
  // 2) 整数入力時の入力規則チェック（現状は使われていない）
  boolean checkInt(char c) {
    if (c == '.') return false;
    return checkFloat(c);
  }
}

// ログファイル記録用クラス
class loggingClass {
  private String       _defaultFileName = "default.txt";
  protected String     logFileName;    // ファイル名（スケッチフォルダに保存）
  private boolean      loggingNow;     // trueの間はログを記録する
  private PrintWriter  output;         // ログ出力用ファイルポインタ
  // コンストラクタ
  loggingClass() {
    logFileName   = "";
    loggingNow    = false;
    output        = null;
  }
  void setFileName(String filename) {
    if (filename.length() > 0) logFileName = filename;
      else                     logFileName = _defaultFileName;
  }
  boolean isEnableLogging() { return output == null ? false : true; }
  // ---- ログ関係のコマンド -----
  // ユーザーは使用しない
  // ログファイルにのみprint()
  void print2Log(String str) { 
    if (loggingNow) output.print(str);
  }
  // ログファイルにのみprintln()
  void println2Log(String str) { 
    if (loggingNow) output.println(str);
  }
  // ログファイルにのみnewline()
  void newline2Log() { 
    println2Log("");
  }
  // ---- 関数 -------------------------------------------------
  // 実際には一つ上のクラスがI/Fになるのでユーザは使用しないこと
  // ---- 関数 -------------------------------------------------
  String setDefaultFilename() { return _defaultFileName; }
  String setDefaultFilename(String fname) {
    String oldname = _defaultFileName;
    _defaultFileName = fname;
    return oldname;
  }
  // ログ記録開始
  PrintWriter startLogging() {
    if (output != null) stopLogging();
    output     = createWriter(logFileName);
    loggingNow = true;
    println2Log("--- Logging start : " + nowDateTime() + "---");
    return output;
  }
  // ログ記録終了
  void stopLogging() {
    if (output != null) {
      println2Log("--- Logging finished : " + nowDateTime() + "---");
      // ファイルを閉じる
      output.flush();
      output.close();
    }
    // 設定を初期化する
    loggingNow  = false;
    output      = null;
  }
  // ログファイルの保存を一時的に停止する
  void pauseLogging() { 
    loggingNow = false;
  }
  // ログファイルの保存を再開する
  void restartLogging() {
    if (output != null) loggingNow = true;
  }
}

// ------------------------------------
// テキスト画面（実行ウィンドウ）の属性
// ------------------------------------
class screenClass {
  protected int   cwidth, cheight;           // キャラクタの横文字数，行数
  protected int   gwx, gwy;                  // スクリーンサイズ
  protected int   colwidth, rowheight;       // 一文字の文字送り，行間（pixel）
  protected int   frameWidth;                // 額縁の幅（上下左右同一）
  protected float hStep, vStep;              // 文字間隔，行間隔の比率（1.0で密着）
  protected PGraphics pgText;                // テキスト表示専用
  screenClass(int max_x, int max_y, int fsize) {
    cwidth     = max_x;
    cheight    = max_y;
    frameWidth = 10;
    hStep      = _HSTEP;
    vStep      = _VSTEP;
    reCalc(fsize);
  }
  void setPgText(PGraphics pg) { pgText = pg; }
  void reCalc(int fsize) {
    colwidth   = int(fsize * hStep);
    rowheight  = int(fsize * vStep);
    gwx        = cwidth  * colwidth  + frameWidth * 2;
    gwy        = cheight * rowheight + frameWidth * 2;
  }
}

/////////////////////////////////////////
// キーボード入力とコンソール出力のクラス
/////////////////////////////////////////
class consoleClass extends tomahawkClass {
  // 実行ウィンドウ関係の管理用変数
  protected float cx = 0.0, cy = 0.0;        // キャラクタの表示位置（カレントポジション），半角文字は0.5文字
  protected float tabSize = _TABSIZE;        // タブ文字数
  screenClass     screenSetting;
  // テキスト表示の属性の設定（clrscr()で設定される）
  protected textAttrClass  currentAttr;
  // 機能選択のフラグ
  protected boolean nodisplay    = false;   // グラフィックス表示の抑制
  private   boolean _nodisplay;
  protected boolean noTextArea   = false;   // PDEのテキストエリアへの文字列の表示の抑制
  private   boolean _noTextArea;
  protected boolean noOverlap    = false;   // ウィンドウからはみ出す文字列を折り返して次行に表示しない
  private   boolean _noOverlap;
  protected boolean proportional = true;    // 半角英数字の出力幅を0.5に固定しない
  private   boolean _proportional;
  protected boolean degreeBase   = true;    // （主にTomahawkにおける）角度の単位を度とする場合はtrue
  private   boolean _degreeBase;
  // テキスト画面のスクロール表示関係
  textbufferClass      textbuffer;         // バッファ関係のクラス
  private boolean      recordTextBuffer;   // trueの間はテキストバッファに記録する
  protected int        autoScrollLines;    // 自動スクロールする行数（0が指定されている時は完全に消去）
  // キーボード入力規則のチェック
  keyInputConstraintClass  keyCheck;
  // ログファイル関係
  loggingClass         logging;
  private boolean      enableLogging;  // テキストログを記録する（setup()でのみ有効）

  // コンストラクタ（引数は横方向の最大文字数と縦方向の最大行数）
  consoleClass(int max_x, int max_y, int mode) {
    super(mode);
    // スクリーンの設定
    currentAttr   = new textAttrClass();
    screenSetting = new screenClass(max_x, max_y, currentAttr.fontSize);
    // 再設定
    screenReset();
    // デフォルト文字色と背景色の設定
    setColor(currentAttr.textColor, currentAttr.bgColor);
    // スクロール表示用のバッファ関連
    textbuffer       = new textbufferClass(screenSetting, currentAttr);
    recordTextBuffer = true;
    autoScrollLines  = 0;
    // キーボード入力規則チェック
    keyCheck      = new keyInputConstraintClass();
    // ログファイル関係
    logging       = new loggingClass();
    enableLogging = false;
    // 機能スイッチのバックアップの初期化
    backupConfigFlag();
  }
  void backupConfigFlag() {
    _nodisplay    = nodisplay;
    _noTextArea   = noTextArea;
    _noOverlap    = noOverlap;
    _proportional = proportional;
    _degreeBase   = degreeBase;
  }
  void restoreConfigFlag() {
    nodisplay    = _nodisplay;
    noTextArea   = _noTextArea;
    noOverlap    = _noOverlap;
    proportional = _proportional;
    degreeBase   = _degreeBase;
  }

  // Tomahawkが利用可能かどうか？（これは使わないかな）
/*
  boolean isEnableTomahawk() {
    if (screenSetting.pgText == null) return false;
    return true;
  }
*/
  // フォントの設定
  void prepareFont() {
    if (tomahawkMode == 2) {
      // テキスト表示専用レイヤー使用時
      PFont font = createFont(_FONTNAME, currentAttr.fontSize, false);  // フォントを変換
      screenSetting.pgText.beginDraw();
      screenSetting.pgText.textFont(font);                     // フォントを設定
      if (!_JAVA2D) screenSetting.pgText.textMode(SCREEN);     // JAVA2Dならコメントアウトしなくてはいけない．P2Dだと必要．
      screenSetting.pgText.textSize(currentAttr.fontSize);     // フォントサイズを設定（不要？）
      screenSetting.pgText.textAlign(LEFT, TOP);               // 配置
//    screenSetting.pgText.smooth();
      screenSetting.pgText.endDraw();
    } else {
      // テキスト表示専用レイヤー不使用時
      PFont font = createFont(_FONTNAME, currentAttr.fontSize, true);  // フォントを変換
      textFont(font);                     // フォントを設定
//    textMode(SCREEN);                   // これを有効にすると文字が表示されない
      textSize(currentAttr.fontSize);     // フォントサイズを設定（不要？）
      textAlign(LEFT, TOP);               // 配置
    }
  }
  // フォントサイズの初期設定の実体（直接，ユーザは呼び出さない）
  void _setFontSize(int size) {
    if (size <= 0) return;
    currentAttr.fontSize  = size;
    textbuffer.fontSize   = size;
    screenSetting.reCalc(size);
    // 実際にフォントサイズを変更する
    if (tomahawkMode == 2) {
      screenSetting.pgText.beginDraw();
      screenSetting.pgText.textSize(size);
      screenSetting.pgText.endDraw();
    } else {
      textSize(size);
    }
    screenReset();
  }
  // 画面パラメータ変更後のリセット動作
  void screenReset() {
    // スクリーンの設定
    size(screenSetting.gwx, screenSetting.gwy);
    if (tomahawkMode == 2) {
      if (_JAVA2D) screenSetting.pgText = createGraphics(screenSetting.gwx, screenSetting.gwy, JAVA2D);  // 速度が極めて遅い
        else       screenSetting.pgText = createGraphics(screenSetting.gwx, screenSetting.gwy, P2D);     // SCREENを設定する必要がある
    } else         screenSetting.pgText = null;
    _clrscr();  
    // フォント関係
    prepareFont();
  }

  // ----------- 画面出力関係-----------------
  // 文字位置をグラフィックス座標に変換
  private int ptx(float x) { 
    return int(x * screenSetting.colwidth  + screenSetting.frameWidth);
  }
  private int pty(float y) { 
    return int(y * screenSetting.rowheight + screenSetting.frameWidth);
  }

  // 出力位置指定（廃止予定：使っても良いがテキストバッファは非対応）
  void locate(int x, int y) {
    if (x < 0) x = 0;
    if (y < 0) y = 0;
    cx = x;
    cy = y;
  }
  // 行数が最大値に達したか
  private boolean rowCheck() {
    // 達したならば画面消去
    if (cy >= screenSetting.cheight) {
      if (autoScrollLines == 0) {
        // オートスクロール無効（全画面消去）
        _clrscr();
        textbuffer.readingLine += screenSetting.cheight;  // ここだけは強制的にインクリメントする必要がある（まだ出力していない）
      } else {
        // オートスクロール対応
        stopRecordTextBuffer();
        pageLineScroll(autoScrollLines);
        startRecordTextBuffer();
      }
      return true;
    }
    return false;
  }
  // BSキーが押された場合の処理
  void backSpace() {
    if (keyBufferLength() > 0) {
      keyBuffer = keyBuffer.substring(0, keyBuffer.length() - 1);
      writeBS();
    }
  }
  // テキスト文字0.5文字分を削除（パラメータ入力時のみ有効なため）
  private void writeBS() {
    if (cx <= 0) return;
    if (!nodisplay) {
      cx -= 0.5;
      if (tomahawkMode == 2) {
        // テキスト表示レイヤー使用時
        resetAlpha(ptx(cx), pty(cy), screenSetting.colwidth/2, screenSetting.rowheight, true);
      } else {
        // テキスト表示レイヤー不使用時
        noStroke();
        fill(currentAttr.bgColor);    // 背景色
        rect(ptx(cx), pty(cy), screenSetting.colwidth/2, screenSetting.rowheight);
        fill(currentAttr.textColor, 255);          // 塗りつぶし色
      }
    }
    //  if (!noTextArea) print("\b");  // backspaceのエスケープシーケンスは無効の模様．
  }
  // 与えられた文字の大きさを返す．全角ならば１，半角ならば0.5
  private float charLength(char c) {
    float len;
    if (tomahawkMode == 2) len = screenSetting.pgText.textWidth(c) / currentAttr.fontSize;
      else                 len = textWidth(c) / currentAttr.fontSize;
    if ((proportional == false) && (len < 0.5)) len = 0.5; // 半角文字を0.5に固定
    return len;
  }
  // 与えられた文字列の長さ（幅）を返す
/*
  private float getStringWidth(String str) {
    if (tomahawkMode == 2) return screenSetting.pgText.textWidth(str) / currentAttr.fontSize;
    return textWidth(str) / currentAttr.fontSize;
  }
*/
  // α値をリセットする．第５引数がtrueの場合は無条件に#000000で塗りつぶす
  void resetAlpha(int x, int y, int w, int h, boolean clr) {
    PImage pimg;
    int    p, i;
    screenSetting.pgText.beginDraw();
    pimg = screenSetting.pgText.get(x, y, w, h);
    for (i = 0; i < pimg.pixels.length; i++) {
      if (clr) {
        pimg.pixels[i] = color(#000000, 0);
      } else {
        p = pimg.pixels[i] & 0x0ffffff;
        if (p > 0) {
  //        p |= 0x0ff000000;
  //        pimg.pixels[i] = color(p);
        } else {
          pimg.pixels[i] = color(p, 0);
        }
      }
    }
    screenSetting.pgText.set(x, y, pimg);
    screenSetting.pgText.endDraw();
  }
  // 一文字表示（文字列は全てこの関数を介してtext()で表示する
  private void putchar(char c) {
    int  px, py;
    px = ptx(cx);
    py = pty(cy);
    if (c != ' ') {  // 何故かPGraphicsを使ったテキスト表示専用レイヤに対して半角空白をtext()するとゴミが一行上にポツリと表示されるための苦肉の策（バグ？）
                     // https://forum.processing.org/topic/bug-illeagal-pixel-is-dotted-when-space-character-is-displayed-by-using-text-in-pgraphics
      if (tomahawkMode == 2) {
        // テキスト表示専用レイヤー使用時
        screenSetting.pgText.beginDraw();
        screenSetting.pgText.text(c, px, py);
        screenSetting.pgText.endDraw();
        if (!_JAVA2D) resetAlpha(px, py - 2, currentAttr.fontSize * 2, currentAttr.fontSize + 4, false);   // なぜか-2しないと線が残る
      } else {
        // テキスト表示専用レイヤー不使用時
        text(c, px, py);
      }
    }
    cx += charLength(c);
  }

  ///////////////////////
  // テキストバッファ関連
  ///////////////////////
  // テキストバッファへの記録を一時的に停止する
  void stopRecordTextBuffer() { 
    recordTextBuffer = false;
  }
  // テキストバッファへの記録を再開する
  void startRecordTextBuffer() { 
    recordTextBuffer = true;
  }
  // テキストバッファの指定された行から１ページ分を再描画する
  void redrawPage() {
    // 自動スクロール（Run中）のためにカレント属性をバックアップ
    textAttrClass bakAttr = new textAttrClass();
    bakAttr.clone(currentAttr);
    // 再描画
    redrawPage(textbuffer.readingLine);
    // カレント属性のバックアップをリストア
    currentAttr.clone(bakAttr);
  }
  void redrawPage(int line) {
    int       lineCount, dcy;
    lineClass thisline;

    // 表示開始行数のチェック
    textbuffer.incrementReadingLine(0);
    // 初期化
    _clrscr();
    thisline = textbuffer.getTargetLine(line);
    // 再描画
    // スクロール表示時は一時的にオーバーラップを強制的にオフにする
    boolean over_flag = noOverlap;
    disableOverlap();
    for (lineCount = 0; lineCount < screenSetting.cheight; lineCount += dcy) {
      if (!thisline.flag) break; // これ以降，テキストが出力されていないので抜ける
      dcy      = writelnText(thisline.line);
      thisline = thisline.next;
    }
    // 元の状態に戻す
    noOverlap = over_flag;
  }
  // ページ単位のスクロール
  // pageが正で下，負で上にスクロール
  void pageScroll(float page) {
    textbuffer.incrementReadingLine(int(screenSetting.cheight * page));
    redrawPage();
  }
  // 指定した行数だけ上/下にスクロールする
  // num が正で下，負で上にスクロール
  void pageLineScroll(int num) {
    textbuffer.incrementReadingLine(num);
    redrawPage();
  }
  // 最新のページを再描画（カーソル位置が復元されないのは仕方が無い
  void redrawNewestPage() {
    int  linenumber;
    if (textbuffer.totalLines < screenSetting.cheight) {
      textbuffer.setReadingLine(0);
    } else {
      textbuffer.setReadingLine(textbuffer.totalLines - int(screenSetting.cheight / 2.0));
    }
    redrawPage();
  }
  // 新しいテキストバッファ形式で一行分の出力（色つき）
  int writelnText(sentenceClass sen) {
    int num = 1;
    int dy;
    while (true) {
      textColor(sen.attr.textColor);
      // テキストカラーを変更
      if (tomahawkMode == 2) {
        // テキスト表示専用レイヤー使用時
        screenSetting.pgText.beginDraw();
        screenSetting.pgText.fill(sen.attr.textColor, 255f);
        screenSetting.pgText.endDraw();
      } else {
        // テキスト表示専用レイヤー不使用時
        fill(sen.attr.textColor, 255);
      }
      if ((dy = write(sen.sentence)) > 1) num += (dy - 1);
      if (sen.next == null) break;
      sen = sen.next;
    }
    writeln("");
    return num;
  }
  // ------------------------
  // 文字色と背景色の初期設定
  // ------------------------
  void setColor(int tcolor, int bcolor) {
    textColor(tcolor);
    bgColor(bcolor);
  }
  // 文字の色のみを変更する
  // (a) 現在の設定値を返す
  int textColor() { 
    return currentAttr.textColor;
  }
  // (b) 設定（＋変更前の値を返す）
  int textColor(int textcolor) {
    int oldColor;
    oldColor              = currentAttr.textColor;
    if (brightness(textcolor) <= 2.0) textcolor = #000002;  // PGraphicsの仕様上の問題点の回避
    currentAttr.textColor = textcolor;
    if (tomahawkMode == 2) {
      // テキスト表示専用レイヤー使用時
      screenSetting.pgText.beginDraw();
      screenSetting.pgText.fill(currentAttr.textColor, 255);
      screenSetting.pgText.endDraw();
    } else {
      // テキスト表示専用レイヤー不使用時
      fill(currentAttr.textColor);
    }
    return oldColor;
  }
  // 背景の色のみを変更する（あまりお勧めしない）
  // (a) 現在の設定値を返す
  int bgColor() { 
    return currentAttr.bgColor;
  }
  // (b) 設定（＋変更前の値を返す）
  int bgColor(int bgcolor) {
    int oldColor;
    oldColor            = currentAttr.bgColor;
    currentAttr.bgColor = bgcolor;
    return oldColor;
  }
  // タブ文字数の設定
  // (a) 現在の設定値を返す
  float tabSize() { return tabSize; }
  // (b) 設定（＋変更前の値を返す）
  float tabSize(float num) {
    float oldnum = tabSize;
    if ((num < 0.0) || (num >= screenSetting.cwidth)) return oldnum;
    tabSize = num;
    return oldnum;
  }
  // --------
  // タブ出力
  // --------
  String getTabSpace() {
    String space = "";
    float  num;
    int    i;
    num = int((cx + tabSize) % tabSize * 2);
    if (num == 0) return "";
    num = int(tabSize * 2) - num;
    for (i = 0; i < num; i++) space += " ";    
    return space;
  }
  // 引数指定なし（設定値を使用する）
  float tab(boolean flag) {
    String space;
    space = getTabSpace();
    if (flag) write(space); else _write(space);
    return space.length() / 0.5;
  }
  // 引数指定あり（暫定的に指定された値を使用する）
  float tab(float temp, boolean flag) {
    float bak = tabSize;
    float num;
    tabSize = temp;
    num = tab(flag);
    tabSize = bak;
    return num;
  }
  // 引数指定なし（設定値を使用する）
  float tab()  { return tab(true); }
  float _tab() { return tab(false); }
  // 引数指定あり（暫定的に指定された値を使用する）
  float tab(float temp)  { return tab(temp, true); }
  float _tab(float temp) { return tab(temp, false); }

  // 画面消去（画面のみクリア）
  void _clrscr() {
    if (tomahawkMode == 2) {
      // テキスト表示専用レイヤー使用時
      background(currentAttr.bgColor);  // 背景色
      screenSetting.pgText.beginDraw();
      screenSetting.pgText.background(#000000, 0);
      screenSetting.pgText.fill(currentAttr.textColor, 255);      // 塗りつぶし色
      screenSetting.pgText.endDraw();
    } else {
      // テキスト表示専用レイヤー不使用時
      background(_BGCOLOR);
      fill(currentAttr.textColor);      // 塗りつぶし色
    }
    cx = cy = 0;
  }
  // 画面消去（テキストバッファもクリア）
  void clrscr() {
    _clrscr();
    textbuffer.clearTextBuffer();
  }
  // テキスト出力（単純に描画するだけ）: userDraw()中で利用するのに適している
  int _write(String message) {
    int  i, num = 1;
    if (message == null) return 0;
    // テキストの色を復元する
    if (tomahawkMode == 2) {
      // テキスト表示専用レイヤー使用時
      screenSetting.pgText.beginDraw();
      screenSetting.pgText.fill(currentAttr.textColor, 255);
      screenSetting.pgText.endDraw();
    } else {
      // テキスト表示専用レイヤー不使用時
      fill(currentAttr.textColor);
    }
    rowCheck();
    for (i = 0; i < message.length(); i++) {
      putchar(message.charAt(i));
      // 折り返し表示に関する処理
      if (noOverlap != true) {
        if (cx >= screenSetting.cwidth) {
          cx = 0;
          cy++;
          num++;
          rowCheck();
        }
      }
    }
    return num;
  }
  // 座標指定（ログ出力等なし）
  int _write(int x, int y, String message) {
    locate(x, y);
    return _write(message);
  }
  // 改行あり
  int _writeln(String message) {
    int  num;
    num = _write(message);
    cy++;
    cx = 0;
    return num;
  }
  // 座標指定（ログ出力等なし）
  int _writeln(int x, int y, String message) {
    locate(x, y);
    return _writeln(message);
  }
  // テキスト出力（テキストバッファ，ログ記録，バックスペースによる消去に対応）
  // 戻り値は実行ウィンドウに出力した実際の行数（オーバーラップを考慮）
  int write(String message) {
    int i;
    int num = 0;
    if (!nodisplay) {
      rowCheck();
      num = _write(message);
      if (recordTextBuffer) {
        textbuffer.appendSentence(message, currentAttr);
        // もしオーバーラップ指定が有効かつ一行を超えているならば，超えている分を先に確定する．
        if (!noOverlap) textbuffer.normalizeBuffer();
      }
    }
    // コンソールに出力
    if (!noTextArea) {
      if (recordTextBuffer) print(message);
    }
    // ログファイルに記録
    if (enableLogging) print2Log(message);
    return num;
  }
  // 座標指定（ログ出力等対応）
  int write(int x, int y, String message) {
    int     num;
    boolean flag;
    flag = displaySwitch();
    disableDisplay();
    locate(x, y);
    num = write(message);
    displaySwitch(flag);
    return num;
  }
  // テキスト出力（改行）
  int writeln(String message) {
    int num;

    // 以下改行のみ出力
    num = write(message);
    if (!nodisplay) {
      if (recordTextBuffer) textbuffer.setLine(!noOverlap);
      cy++;
      cx = 0;
    }
    // コンソールに出力
    if (!noTextArea) {
      if (recordTextBuffer) println();
    }
    // ログファイルに記録
    if (enableLogging) println2Log("");
    return num;
  }
  // 座標指定（ログ出力等対応）
  int writeln(int x, int y, String message) {
    int     num;
    boolean flag;
    flag = displaySwitch();
    disableDisplay();
    locate(x, y);
    num = writeln(message);
    displaySwitch(flag);
    return num;
  }
  // 改行
  void newline() { 
    writeln("");
  }
  // ----------
  // 設定の変更
  // ----------
  // ウィンドウサイズの変更：　強制的に実行ウィンドウのサイズが変更されるだけでテキスト画面の処理には反映されないため非推奨．
  void windowSize(int x, int y) {
    screenSetting.gwx = x;
    screenSetting.gwy = y;
    size(screenSetting.gwx, screenSetting.gwy);
  }
  // ディスプレイ表示の抑制と許可
  // (a) 現在の値を返す
  boolean displaySwitch() { return nodisplay; }
  // (b) 設定（＋設定前の値を返す）
  boolean displaySwitch(boolean flag) {
    boolean oldflag = nodisplay;
    nodisplay = flag;
    return oldflag;
  }
  void disableDisplay() { 
    nodisplay = true;
  }
  void enableDisplay() { 
    nodisplay = false;
  }
  // テキストエリア表示の抑制と許可
  // (a) 現在の値を返す
  boolean textAreaSwitch() { return noTextArea; }
  // (b) 設定（＋設定前の値を返す）
  boolean textAreaSwitch(boolean flag) { 
    boolean oldflag = noTextArea;
    noTextArea = flag;
    return oldflag;
  }
  void disableTextArea() { 
    noTextArea = true;
  }
  void enableTextArea() { 
    noTextArea = false;
  }
  // オーバーラップ表示（折り返し）の抑制と許可
  // (a) 現在の値を返す
  boolean overlapSwitch() { return noOverlap; }
  // (b) 設定（＋設定前の値を返す）
  boolean overlapSwitch(boolean flag) { 
    boolean oldflag = noOverlap;
    noOverlap = flag;
    return oldflag;
  }
  void disableOverlap() { 
    noOverlap = true;
  }
  void enableOverlap() { 
    noOverlap = false;
  }
  // 半角英数字出力時の文字幅を0.5に固定するか
  // (a) 現在の値を返す
  boolean proportionalFontSwitch() { return proportional; }
  // (b) 設定（＋設定前の値を返す）
  boolean proportionalFontSwitch(boolean flag) {
    boolean oldflag = proportional;
    proportional = flag;
    return oldflag;
  }
  void enableProportionalFont() {
    proportional = true;
  }
  void disableProportionalFont() {
    proportional = false;
  }
  // （主にTomahawkで）角度の単位を度とするか
  // (a) 現在の値を返す
  boolean angleModeSwitch()  { return degreeBase; }
  // (b) 設定（＋設定前の値を返す）
  boolean angleModeSwitch(boolean flag) {
    boolean oldflag = degreeBase;
    degreeBase = flag;
    return oldflag;
  }
  void degreeAngleMode() {
    degreeBase = true;
  }
  void radianAngleMode() {
    degreeBase = false;
  }

  // 【テキスト出力のログファイルへの出力】
  // Main() と userDraw()実行中のcrowbar.write()/writeln()をログに記録
  // ログファイルの記録開始：Options()の中で宣言するのが無難だが，
  // それMain()やuserDraw()の中で宣言しても構わない．
  // ファイルへの追記は仕様上不可能なようなので，現状では未対応
  // 既に記録開始していてクローズされていない場合は，現在記録中のログファイルを閉じて新しく記録を開始する
  // 1) ログファイルのデフォルトファイル名を変更する
  String setDefaultFilename() {
    return logging.setDefaultFilename();
  }
  String setDefaultFilename(String fname) { 
    if (fname.length() == 0) return setDefaultFilename();
    return logging.setDefaultFilename(fname);
  }
  // 2-1) ログファイルのオープン（引数無し：デフォルトのファイル名で保存）
  PrintWriter startLogging() { 
    return startLogging("");
  }
  // 2-2) ログファイルのオープン（ファイル名指定あり）
  PrintWriter startLogging(String fname) {
    logging.setFileName(fname);
    println("ログファイルを作成します：" + logging.logFileName);
    return logging.startLogging();
  }
  // 3) ログファイルのクローズ（明示しないでもcrowbar終了時に閉じてくれる）
  void stopLogging() {
    logging.stopLogging();
  }
  // 4-1) ログ記録の一時中断
  void pauseLogging() { 
    logging.pauseLogging();
  }
  // 4-2) ログ記録の再開
  void restartLogging() { 
    logging.restartLogging();
  }
  // 5) ログファイルにのみprint()
  void print2Log(String str) { 
    logging.print2Log(str);
  }
  // 6) ログファイルにのみprintln()
  void println2Log(String str) { 
    logging.println2Log(str);
  }
  // ログファイルにのみnewline()
  void newline2Log() {
    logging.newline2Log();
  }
  // 以下はユーザが用いてはいけない（システム用）
  void enableLogOutput() { 
    enableLogging = true;
  }   // ログファイル出力を許可
  void disableLogOutput() { 
    enableLogging = false;
  }   // ログファイルの出力を不許可
  boolean isEnableLogging() { 
    return logging.isEnableLogging();  
//  return enableLogging;
  }

  // --------------- キーボード入力関連 -------------------
  // 以下の関数や変数をユーザは利用しないで下さい
  //
  private String keyBuffer = "";
  boolean inputNow = false;    // エコーバックあり
  boolean inputFixed = false;  // ENTERが入力されたらtrue
  // バッファクリア  
  void flushKeyBuffer() {
    keyBuffer = "";
    inputFixed = false;
  }
  // 文字列長を返す
  int keyBufferLength() {
    return keyBuffer.length();
  }
  // コンソールからの入力を開始
  void startInput() {
    inputFixed = false;
    inputNow   = true;
  }
  // キーボード入力完了
  void inputFinish() {
    inputFixed = true;
    inputNow   = false;
  }
  // 文字列の追加
  void strcat(String str) {
    keyBuffer += str;
  }
  // キーバッファーを読み出す
  String getKeyBuffer() {
    String buf;
    buf = keyBuffer;  // インスタンスのクローンが作成されるのなら良いけれど．
    keyBuffer = "";
    inputNow   = false;
    inputFixed = false;
    return buf;  // 文字列データのみを返さなくて良いのか？
  }
}

//////////////////////////////
// パラメータの最小単位のクラス
//////////////////////////////
class arguments {
  boolean ready;
  String  message;
  String  value;
  String  initialValue;   // ""の場合は初期値の指定なしと判断
  String [] selectValues; // nullの場合は選択肢タイプではないと判断（設定時は空白で区切られた文字列）
  char    parameterType;  // 'A': Normal, 'N': 数値のみ, 'S': 選択肢
  // コンストラクタ
  arguments() {
    ready   = false;
    message = value = initialValue = "";
    selectValues  = null;
    parameterType = 'A';
  }
  // 値を設定（標準）
  void set(String str) {
    value = str;
    ready = true;
  }
}

class frontendClass extends consoleClass { 
  arguments [] argv;                    // パラメータ群（ユーザにより対話的に入力される）
  int          argc;                    // パラメータの数
  protected int obtainArgIndex = 0;     // パラメータのコメント宣言時の自動登録用カウンタ
  protected int readArgIndex = 0;       // パラメータ自動読み出し用カウンタ
  protected boolean autoInput = false;  // TABキーを押すとそれ以降のパラメータ入力では初期値を採用

  // コンストラクタ
  frontendClass(int x, int y, int mode) {
    // consoleClassのコンストラクタを呼ぶ
    super(x, y, mode);
    // パラメータの初期化
    argc = 0;
    argv = (arguments [])new arguments[1]; // 常に一つ余分に作成
    argv[0] = new arguments();
  }
  // パラメータの値のセット
  void setValue(String str) {
    argv[obtainArgIndex].set(str);
  }
  // キー入力ルール（数値のみ）
  boolean is_numbersOnly() {
    if (argv[obtainArgIndex].parameterType == 'N') return true;
    return false;
  }
  // キー入力ルール（選択肢）
  // 選択肢型ならば選択肢数（最大１０）を返す
  // 選択肢型ではないならば０を返す
  // 異常時は99を返す
  int is_selectOnly() {
    arguments arg;
    arg = argv[obtainArgIndex];
    if (arg.parameterType == 'S') {
      // チェック１
      if (arg.selectValues == null) {
        println("SystemError: 選択肢型なのに選択肢がセットされていません．");
        return 99;
      }
      // チェック２
      if (arg.selectValues.length > 10) {
        println("SystemError: 選択肢型の選択肢の数が１０個より多い．");
        return 99;
      }
      return arg.selectValues.length;
    }
    return 0;
  }
  // 選択肢型の選択されたパラメータ値を返す
  String get_selectValue(int index) {
    return argv[obtainArgIndex].selectValues[index];
  }
  // 初期値を取得
  String get_initialValue() {
    return argv[obtainArgIndex].initialValue;
  }
  // 全てのパラメータが入力完了して，Main()を実行可能か？
  boolean allReady() {
    int i;
    // パラメータ入力が指定されていない
    if (argv == null) return true;
    // パラメータ入力が指定されている
    for (i = 0; i < argc; i++) {
      if (argv[i].ready != true) return false;
    }
    return true;
  }
  // 全ての入力完了フラグをリセット（再入力時用）
  void allResetArguments() {
    int i;
    for (i = 0; i < argv.length; i++) argv[i].ready = false;
    readArgIndex   = 0;
    obtainArgIndex = 0;
  }
  // パラメータが一つも指定されていない場合にtrue：余計なシステムメッセージの出力を抑制するため
  boolean isNoParameter() {
    if (argc == 0) return true;
    return false;
  }
  // ------------------------------
  // パラメータ説明の取得
  String get_paramMessage(int index) {
    String    message;
    arguments arg;

    // インデックスの範囲のチェック
    if ((index >= argc) || (index < 0)) {
      println();
      print("[警告] パラメータ読み出しのIndexが範囲を超えています！(Index=");
      println(str(index)+")");
      return "";
    }
    // 説明の取得
    arg     = argv[index];
    // 全てのタイプに共通
    message = arg.message;
    // 以下，オプションによってメッセージを追加
    // 初期値が与えられている場合
    if (arg.initialValue.length() > 0) message += "（初期値：" + arg.initialValue + "）";
    // パラメータの種類によって
    if (arg.parameterType == 'N') {
      // 数字のみ
      message += "／数字のみ";
    } else if (arg.parameterType == 'S') {
      // 選択肢型
      int i;
      message += "／（";
      for (i = 0; i < arg.selectValues.length; i++) {
        if (i > 0) message += ",";
        message += str(i) + ":" + arg.selectValues[i];
      }
      message += "）";
    }
    return message;
  }
  // 初期値の自動入力モード（trueで自動入力）
  void automaticInputByDefalutValue(boolean flag) { autoInput = flag; }
  // --------------------------------------------
  // パラメータの宣言（画面に表示するメッセージ）
  // --------------------------------------------
  // パラメータの宣言（標準：数字のみ）
  void define(String str) {
    defineStr(str);
    argv[argc - 1].parameterType = 'N';
  }
  // パラメータの宣言（標準：数字のみ：初期値あり）
  void define(String str1, String str2) {
    defineStr(str1, str2);
    argv[argc - 1].parameterType = 'N';
  }
  // パラメータの宣言（数字のみ：初期値あり，float渡し）
  void define(String str, float f) { 
    define(str, str(f));
  }
  // パラメータの宣言（数字のみ：初期値あり，int渡し）
  void define(String str, int i) { 
    define(str, str(i));
  }

  // パラメータの宣言（入力制限規則なし）：全てのdefine系は最終的にこの関数を呼ぶ
  void defineStr(String str) {
    if (crowbar.status != 0) {
      println("エラー: パラメータの宣言はOptions()かSetup()で行って下さい／" + str);
      if (crowbar.status == 10) {
        crowbar.halt(); // userDraw()の時は安全に終了
        return;
      } else exit();
    }
    argc = argv.length;
    argv = (arguments [])expand(argv, argc + 1);
    argv[argc] = new arguments();
    // 現在の値をセット
    argv[argc - 1].message = str;
  }
  // パラメータの宣言（入力制限規則なし：初期値あり）
  void defineStr(String str1, String str2) {
    defineStr(str1);
    argv[argc - 1].initialValue = str2;
  }
  // パラメータの宣言（入力制限規則なし：初期値あり，float渡し）
  void defineStr(String str, float f) { 
    define(str, str(f));
  }
  // パラメータの宣言（入力制限規則なし：初期値あり，int渡し）
  void defineStr(String str, int i) { 
    define(str, str(i));
  }

  // パラメータの宣言（選択肢型：入力制限規則は0-9のみ）
  void defineSel(String str1, String str2) {
    String list[];
    if (str2.length() <= 0) {
      println("Error: 選択肢型のパラメータ宣言の第２パラメータが空文字列です．");
      crowbar.halt();
      return;
    }
    list = split(str2, ' ');
    if (list.length > 10) {
      println("Error: 選択肢型のパラメータ宣言の選択肢の数は１０個以下です．");
      crowbar.halt();
      return;
    }
    defineStr(str1);
    argv[argc - 1].parameterType = 'S';
    argv[argc - 1].selectValues = list;
  }
  // パラメータの宣言（選択型：初期値あり）
  void defineSel(String str1, String str2, int num) {
    defineSel(str1, str2);
    if (num < 0) {
      println("Error: 選択肢型の初期値は0から9の整数です．");
      crowbar.halt();
      return;
    }
    if (num >= argv[argc - 1].selectValues.length) {
      println("Error: 選択肢型の初期値は0から選択肢の数-1までです．");
      crowbar.halt();
      return;
    }
    argv[argc - 1].initialValue = str(num);
  } 

  //-------------------------
  // パラメータの値を受け取る
  //-------------------------
  // 1) indexを明示的に指定する
  // 1-a) パラメータ名を文字列で返す
  String getStr(int index) {
    if ((index >= argc) || (index < 0)) {
      println();
      print("[警告] パラメータ読み出しのIndexが範囲を超えています！(Index=");
      println(str(index)+")");
      return "";
    }
    // インデックスの自動更新
    readArgIndex = index + 1;
    return argv[index].value;
  }
  // 1-b) パラメータをfloatで返す
  float getFloat(int index) { 
    return float(getStr(index));
  }
  // 1-c) パラメータをintで返す
  int getInt(int index) { 
    return int(getStr(index));
  }

  // 2) indexを明示的に指定しないで順番に自動的に読み出す
  // 2-a) パラメータを文字列で渡す（自動）
  String getStr() { 
    return getStr(readArgIndex);
  }
  // 2-b) パラメータをfloatで返す（自動）
  float getFloat() { 
    return getFloat(readArgIndex);
  }
  // 2-c) パラメータをintで返す（自動）
  int getInt() { 
    return getInt(readArgIndex);
  }
}


//////////////////////////////////////////////////////////////////////////////
// プロセスコントロールのクラス（キーボード入力とコンソール出力のクラスを継承）
//////////////////////////////////////////////////////////////////////////////
class crowbarClass extends frontendClass {
  int          status = 0;          // プロセスの状態を管理
  String []    comment;             // プログラムのコメント．パラメータ入力前に表示される．
  protected boolean halt;           // 強制終了フラグ（halt()でセットされる）
  protected boolean noShowParams  = false;    // 入力済みのパラメータを確認のために表示するか？
  protected boolean _noShowParams = false;    // setup()中でcrowbar.noShowParameters()が指定された場合の対策．再実行時にクリアされてしまうため．
  boolean running                 = false;    // Main()あるいはcrowbar.nonStop()指定時にユーザプログラム実行中はtrue
  boolean afterRunning            = false;    // Main()が終わった後，テキストバッファを再描画可能な間のみtrue;（userDraw()が動いていてもそれは気にしない．今のところ）
  protected boolean nonstop       = false;    // Main()終了後はdraw()の終端に記述したコードを繰り返し実行する．nonStop()でtrue/stop()でfalse
  protected boolean _nonstop      = false;    // setup()中でcrowbar.nonStop()が指定された場合の対策．再実行時にクリアされてしまうため．
  private boolean   userDrawLoop  = true;     // userDraw()のループ実行を停止したり再開したりする際に用いる．trueで繰り返し実行．loop(), noLoop()で設定．
  private boolean   _userDrawLoop = true;     // プログラム再実行時に初期状態に戻すために記憶する．

  // コンストラクタ
  crowbarClass(int x, int y, int mode) {
    super(x, y, mode);
    // プログラムコメントの初期化
    comment = null;
    // 強制終了フラグ
    halt    = false;
    // 動作に関わるスイッチのバックアップの初期化
    backupRunningFlag();
  }
  // 動作に関わるスイッチのバックアップ
  void backupRunningFlag() {
    _nonstop      = nonstop;
    _noShowParams = noShowParams;
    _userDrawLoop = userDrawLoop;
  }
  // 動作に関わるスイッチのリストア
  void restoreRunningFlag() {
    nonstop      = _nonstop;
    noShowParams = _noShowParams;
    userDrawLoop = _userDrawLoop;
  }
  // 以下の関数をユーザは使用しないで下さい
  // 入力済みの全パラメータを表示する
  void showAllParameters() {
    int i, _readArgIndex;
    // インデックスのバックアップ
    _readArgIndex = readArgIndex;
    if (noShowParams) return; // 念のため
    for (i = 0; i < argc; i++) {
      if (!argv[i].ready) break;  // 入力済みの分のみ表示するので．
      write(get_paramMessage(i) + "：[");
      write(getStr(i));
      writeln("]");
    }
    // インデックスのリストア
    readArgIndex = _readArgIndex;
  }
  // 入力済みの全パラメータをログファイルにのみ出力
  void showAllParameters2Log() {
    boolean _display, _textarea;
    // フラグのバックアップ
    _display  = nodisplay;
    _textarea = noTextArea;
    // テキスト出力を一時的に禁止
    disableDisplay();
    disableTextArea();
    // 禁止した上で出力すればログにのみ出力される
    showAllParameters();
    // フラグを戻す
    if (!_display)  enableDisplay();
    if (!_textarea) enableTextArea();
  }
  // -----------------------------------------------
  // パラメータ表示の確認表示の抑制（Options()で指定）
  void noShowParameters() { 
    noShowParams = true;
  }
  void showParameters() { 
    noShowParams = false;
  } // 必要性は無い

  // ----------------------------------------------------------------------------
  // Main()終了後にdraw()終端に記述したコードを繰り返し実行するためのコントロール
  // ----------------------------------------------------------------------------
  void nonStop() { 
    nonstop = true;
  }   // userDraw()関数が有効
  void stop() { 
    nonstop = false;
  }  // userDraw()関数が終了したらプログラム終了

  // ------------------------------------------------------
  // userDraw()のループ実行の停止と再開のためのコントロール
  // ------------------------------------------------------
  void loop() { 
    userDrawLoop = true;
  }
  void noLoop() { 
    userDrawLoop = false;
  }
  
  // ----------------------------------------
  // Crowbarを安全に強制終了する
  // ただし無限ループからの脱出は原理的に無理
  // ----------------------------------------
  void halt() {
    halt   = true;
    status = 99;
  }

  // ----------------------------------------------
  // プログラムの説明（パラメータ入力前に表示される
  // ----------------------------------------------
  // プログラムコメントを表示する
  void outputProgramComments() {
    int i, commNum;
    commNum = comment.length;
    for (i = 0; i < commNum; i++) writeln(comment[i]);
  }
  // プログラムコメントを宣言（複数宣言可能）．setup()内でユーザが宣言する．
  void programComment(String str) {
    String addStr[];
    addStr    = new String[1];
    addStr[0] = str;
    if (comment == null) {
      // 一個目のコメントの場合
      comment = (String [])new String[1];
      comment = addStr;
    } else {
      // 二個目以降のコメントの場合
      comment = concat(comment, addStr);
    }
  }
  // ---------------------------------------------------------------
  // フォントサイズを変更する（Options(), Setup()の中で使用すること）
  // ---------------------------------------------------------------
  // (a) 現在の値を返す
  int fontSize() {
    return currentAttr.fontSize;
  }
  // (b) 設定（＋変更前の値を返す）
  int fontSize(int size) {
    if (crowbar.status != 0) {
      println("エラー: フォントサイズの宣言はOptions()かSetup()で行って下さい／" + int(size));
    }
    int oldSize;
    oldSize = currentAttr.fontSize;
    if (status == 0) _setFontSize(size);  // Main()実行前以外は変更を認めない
    return oldSize;
  }
  // -------------------------------------------------------------------
  // 自動スクロールの設定を変更する：Options(), Setup()の中で使用すること
  // -------------------------------------------------------------------
  // 自動スクロールの行数指定
  void autoScrollLines(int lines) {
    if (status != 0) return;  // Main()実行前以外は変更を認めない
    // 範囲チェック
    if (lines < 0)                     lines = 0;
    if (lines > screenSetting.cheight) lines = 0;
    autoScrollLines = lines;
  }
  void autoScrollPage(float rate) {
    if (status != 0) return;  // Main()実行前以外は変更を認めない
    // 範囲チェック
    if (rate < 0.0) rate = 0.0;
    if (rate > 1.0) rate = 1.0;
    autoScrollLines(int(screenSetting.cheight * rate));
  }
  // ------------------------------------------------------
  // 文字送り，行送りを変更する（Options()の中で使用すること）
  // ------------------------------------------------------
  // (a) 現在の値を返す
  float hStep() {
    return screenSetting.hStep;
  }
  // (b) 設定（＋変更前の値を返す）
  float hStep(float step) {
    float oldStep;
    oldStep = screenSetting.hStep;
    if (step <= 0.0) return oldStep;
    if (status == 0) {
      screenSetting.hStep = step;
      screenSetting.reCalc(fontSize());
      screenReset();
    }
    return oldStep;
  }
  // (a) 現在の値を返す
  float vStep() {
    return screenSetting.vStep;
  }
  // (b) 設定（＋変更前の値を返す）
  float vStep(float step) {
    float oldStep;
    oldStep = screenSetting.vStep;
    if (step <= 0.0) return oldStep;
    if (status == 0) {
      screenSetting.vStep = step;
      screenSetting.reCalc(fontSize());
      screenReset();
    }
    return oldStep;
  }
}

///////////////
// キー入力判定
///////////////
// 数字入力用キーか？
boolean isKeyNumbers(char c) {
  if (isNumberChar(c)) return true;
  switch (c) {
  case '.' :
    //  case ',' :  // お節介過ぎる
  case '+' :
  case '-' :
    return true;
  }
  return false;
}

void keyPressed() {
  if (crowbar.running == false) {
    // ユーザプログラムが動作していない時（パラメータ入力中など）
    if (crowbar.inputNow) {
      // パラメータ入力中
      if (key == ENTER) {
        if (crowbar.is_numbersOnly() || (crowbar.is_selectOnly() != 0)) {
          // 数値入力と選択型（つまり文字列入力時以外）の場合は空入力は拒否
          if ((crowbar.keyBufferLength() == 0) && (crowbar.get_initialValue().length() == 0)) {
            redraw();
            return;
          }
        }
        crowbar.inputFinish();
        crowbar.newline();
      } else if (key == BACKSPACE) {
        if (crowbar.keyBufferLength() > 0) {
          crowbar.backSpace();
        }
      } else if (key == TAB) {
        if (crowbar.get_initialValue().length() > 0) {
          crowbar.automaticInputByDefalutValue(true);
          crowbar.inputFinish();
          crowbar.newline();
        }
      } else {
        // パラメータの入力
        boolean flag = true;
        int     num;
        // キー入力ルール
        // 数値のみの場合
        if (crowbar.is_numbersOnly()) {
          if (!isKeyNumbers(key)) flag = false;
            else                  flag = crowbar.keyCheck.checkFloat(key);
        }
        // 選択肢の場合
        if ((num = crowbar.is_selectOnly()) > 0) {
          if (num == 99) {  // 異常検出
            crowbar.halt();
            return;
          }
          // 既に１文字入力済み（半角空白も含む）
          if (crowbar.keyBufferLength() >= 1) flag = false;
          // 数字キー（0-9）ではない
          if (!isNumberChar(key))             flag = false;
          // 選択肢の数よりも大きな数字
          if (int(str(key)) >= num)           flag = false;
        }
        if (flag) {
          crowbar.strcat(str(key));
          crowbar.write(str(key));
        }
      }
    } else {
      // 強制的に抜けたい場合
      if ((key == 'Q') || (key == 'q')) crowbar.status = 99;
    }
    // スクロール可能な状態（プログラムが一旦終了してキー入力を待っている間のみ）
    if (crowbar.afterRunning) {
      if (key == '-') {
        crowbar.pageScroll(-0.5);
      } else if (key == '+') {
        crowbar.pageScroll(0.5);
      } else if (key == CODED) {
        if (keyCode == UP) {
          crowbar.pageLineScroll(-1);
        } else if (keyCode == DOWN) {
          crowbar.pageLineScroll(1);
        } else if (keyCode == LEFT) {
          crowbar.pageScroll(-0.5);
        } else if (keyCode == RIGHT) {
          crowbar.pageScroll(0.5);
        }
      }
    }
    redraw();
  } else {
    // crowbar.nonStop()指定時のユーザプログラム実行中のキー入力判定
    // コードはuserKeyPressed()に記述すると良い．
    userKeyPressed();
  }
}

///////////////
// プロセス管理
///////////////
void draw() {
  switch (crowbar.status) {
  case 0: // 初期状態
    // Setup()中でcrowbar.nonStop()/noShowParams()が実行された場合の対策(Main()などで変更されている恐れがあるため）
    crowbar.setColor(_TEXTCOLOR, _BGCOLOR);
    crowbar.backupRunningFlag();
    crowbar.backupConfigFlag();
    crowbar.resetView(false);    // 全ビューポートの視点（変換マトリックス）のバックアップ
    // プログラムコメントの表示
    if (crowbar.comment != null) {
      crowbar.writeln("【プログラム説明】");
      crowbar.outputProgramComments();
      crowbar.newline();
    }
    // Tomahawk関係
    if (crowbar.tomahawkMode != 0) crowbar.backupViewports();
    crowbar.status++;
    break;
  case 1: // パラメータ入力開始（再実行時はここにジャンプしてくる）
    // 初期化
    crowbar.allResetArguments();
    crowbar.automaticInputByDefalutValue(false);
    // setup()中でcrowbar.nonStop()/noShowParams()を設定した値がMain()やuserDraw()の実行で変更される恐れがあるので復元する
    crowbar.restoreRunningFlag();
    crowbar.restoreConfigFlag();
    // パラメータの入力要求が一つも無かった場合はメッセージを表示しない
    if (!crowbar.isNoParameter()) {
      crowbar.newline();
      crowbar.writeln("【パラメータの入力】");
    }
    // Tomahawk関係
    if (crowbar.tomahawkMode != 0) crowbar.restoreViewports();
    crowbar.status++;
    break;
  case 2: // パラメータ入力中
    if (!crowbar.isNoParameter()) {
      crowbar.keyCheck.reset();  // キー入力規則のリセット
      crowbar.write(crowbar.get_paramMessage(crowbar.obtainArgIndex) + ":");
      crowbar.flushKeyBuffer();
      crowbar.startInput();
      // 初期設定値の自動入力モード有効の場合はパスする
      if ((crowbar.autoInput == true) && (crowbar.get_initialValue().length() > 0)) {
        crowbar.inputFinish();
        crowbar.newline();
      } else {
        // ループ停止
        noLoop();
      }
    }
    crowbar.status++;
    break;
  case 3: // 入力確定
    if (!crowbar.isNoParameter()) {
      String inputs;
      if (crowbar.inputFixed) {
        boolean flag = true;
        inputs = crowbar.getKeyBuffer();
        // 空かつ初期値が指定されている場合
        if (inputs.length() == 0) {
          inputs = crowbar.get_initialValue();
        }
        // 選択肢型の場合
        int  num;
        if ((num = crowbar.is_selectOnly()) > 0) {
          if (num == 99) { // 異常検出
            crowbar.halt();
            break;
          }
          // 念のためにチェック
          if (((num = int(inputs)) < 0) || (num > 9)) {
            flag = false;
          } else if (num > crowbar.is_selectOnly()) {
            flag = false;
          }
          if (!flag) {
            // あり得ないはず（入力時にチェックしているので）
            println("SystemError: 選択肢型の入力値が異常です（" + str(num) + "）．");
            crowbar.halt();
            break;
          }
          inputs = crowbar.get_selectValue(num);
        }
        crowbar.setValue(inputs);
        crowbar.obtainArgIndex++;
        // ループ再開
        loop();
        crowbar.status++;
      }
    } else crowbar.status++;
    break;
  case 4: // パラメータ入力完了かの判定
    if (!crowbar.isNoParameter()) {
      if (crowbar.allReady()) {
        crowbar.status++;
      } else {
        crowbar.status -= 2;
      }
    } else crowbar.status++;
    break;
  case 5: // 実行か？ 再入力か？
    if (!crowbar.isNoParameter()) {
      crowbar.newline();
      crowbar.writeln("【全てのパラメータの入力が完了】");
      if (!crowbar.noShowParams) crowbar.showAllParameters();
      crowbar.write("プログラムを実行する(Yes) /パラメータを入力し直す(Retry) : ");
      noLoop();
    }
    crowbar.status++;
    break;
  case 6: // キー判定結果
    if (!crowbar.isNoParameter()) {
      if (keyPressed) {
        if ((key == 'y') || (key == 'Y')) {
          crowbar.writeln(str(key));
          crowbar.newline();
          crowbar.status++;
          loop();
        } else if ((key == 'r') || (key == 'R')) {
          crowbar.writeln(str(key));
          crowbar.newline();
          crowbar.status -= 5;
          loop();
        }
      }
    } else crowbar.status++;
    break;
  case 7: // メインプログラム実行前処理（あるならば）
    preMain(); // optionCodeタブ参照
    crowbar.running = true;
    crowbar.status++;
    break;
  case 8: // ログファイルへの出力を許可する:これはもしかしたらstatus==0から許可するかも知れない．余計な文字情報は記録しないという仕様はお節介だったかもという反省．
    crowbar.enableLogOutput();
    if (crowbar.isEnableLogging()) {
      // 全パラメータを出力する
      crowbar.restartLogging();
      crowbar.newline2Log();
      crowbar.showAllParameters2Log();
      crowbar.newline2Log();
    }
    crowbar.status++;
    break;
  case 9: // メインプログラム実行
    Main();
    // ループ再開
    loop();
    crowbar.status++;
    break;
  case 10: // 繰り返し実行コード（オプション）
    if (crowbar.nonstop) {
      // crowbar.nonStop()指定時の繰り返し処理はuserDraw()に記述
      if (crowbar.userDrawLoop) {
        crowbar.resetView(true);  // 全ビューポートのrotate(), scale(), translate()の再設定
        userDraw();
      }
    } else {
      crowbar.status++;
    }
    break;
  case 11: // ログファイル出力を一時的に止める
    crowbar.running = false;
    if (crowbar.isEnableLogging()) {
      crowbar.pauseLogging();
    }
    crowbar.status++;
    break;
  case 12: // メインプログラム実行後処理（あるならば）
    postMain(); // optionCodeタブ参照
    crowbar.status++;
    break;
  case 13: // リトライ確認メッセージ
    crowbar.setColor(_TEXTCOLOR, _BGCOLOR);
    crowbar.newline();
    crowbar.writeln("【プログラム終了】");
    crowbar.write("もう一度，実行しますか？(Yes/No)：");
    // テキストバッファ関連
    crowbar.afterRunning = true;
    crowbar.textbuffer.flushBuffer(!crowbar.noOverlap);
    crowbar.stopRecordTextBuffer();
    crowbar.status++;
    noLoop();
    break;
  case 14: // キー判定結果
    if (keyPressed) {
      if ((key == 'y') || (key == 'Y')) {
        crowbar.writeln(str(key));
        crowbar.newline();
        // テキストバッファ―関連
        crowbar.stopRecordTextBuffer();
        crowbar.redrawNewestPage();
        crowbar.afterRunning = false;
        crowbar.startRecordTextBuffer();
        crowbar.status = 1;
        loop();
      } else if ((key == 'n') || (key == 'N')) {
        crowbar.writeln(str(key));
        // テキストバッファ―関連
        crowbar.stopRecordTextBuffer();
        crowbar.redrawNewestPage();
        crowbar.afterRunning = false;
        crowbar.startRecordTextBuffer();
        crowbar.status++;
        loop();
      }
    }
    break;
  case 15: // 終了
    crowbar.status = 99;
    break;
  case 99: // 終了処理
    println();
    // ログファイルを開いたままなら閉じる
    if (crowbar.isEnableLogging()) {
      println("ファイルを閉じます．");
      crowbar.restartLogging();
      crowbar.newline2Log();
      crowbar.stopLogging();
      crowbar.disableLogOutput();
    }
    println("Crowbarを終了します．");
    exit();
    break;
  }
  crowbar.refreshView();  // Tomahawkのビュー合成
}

void mouseDragged() {
  if (crowbar.running) {
    crowbar.resetView(true);
    userMouseDragged();
  }
}
void mouseMoved() {
  if (crowbar.running) {
    crowbar.resetView(true);
    userMouseMoved();
  }
}
void mousePressed() {
  if (crowbar.running) {
    crowbar.resetView(true);
    userMousePressed();
  }
}
void mouseReleased() {
  if (crowbar.running) {
    crowbar.resetView(true);
    userMouseReleased();
  }
}
void mouseClicked() {
  if (crowbar.running) {
    crowbar.resetView(true);
    userMouseClicked();
  }
}
void keyReleased() {
  if (crowbar.running) {
    crowbar.resetView(true);
    userKeyReleased();
  }
}
void keyTyped() {
  if (crowbar.running) {
    crowbar.resetView(true);
    userKyeTyped();
  }
}

// システム初期設定
void setup()
{
  initCrowbar();
  Options();
  Setup();
}

