////////////////////////////////////////////////////////////////
// Processingによる古典的な対話式プログラム開発用フレームワーク
//               Crowbar（クロウバー）
//    with 高機能グラフィックスライブラリTomahawk（トマホーク）
//                                  白井　達也（Tatsuya Shirai)
//                       shirai＠mech.suzuka-ct.ac.jp, @tatsuva
//                                       2012/04/20    Ver. 0.9
//                                       2012/05/14    Ver. 1.0
//                                       2012/05/15    Ver. 1.1
//                                       2012/05/24    Ver. 1.2
//                                       2012/05/26    Ver. 1.3
//                          2012/05/27 - 2012/06/01    Ver. 2.0
//                          2012/06/04 - 2012/06/04    Ver. 2.1
//                          2012/06/05 - 2012/06/21    Ver. 3.0
//                          2012/06/21 - 2012/07/02    Ver. 3.1
//                          2012/07/02 - 2012/07/05    Ver. 3.2
//                          2012/07/05 - 2012/07/11    Ver. 3.3
//                          2012/07/11 - 2012/07/16    Ver. 3.4
//                          2012/07/16 - 2012/08/02    Ver. 3.5
//                          2012/08/02 - 2012/10/10    Ver. 4.0
//                          2012/10/10 - 
////////////////////////////////////////////////////////////////
// 最新版の公開は以下のSourceForge.JPにて行っています．
// http://sourceforge.jp/projects/crowbar/
// ドキュメントは以下のMoodleサイト上のWikiにて公開しています．
// http://www.suzuka-ct.ac.jp/mech/moodle/mod/wiki/view.php?id=1739&page=Crowbar
// 質問などは上記SourceForge.JPあるいはmechMoodle上のフォーラムに
// 投稿するか，あるいはTwitter，またはメールでお問い合わせ下さい．

// グラフィックスライブラリTomahawkを使用しない場合は，タブTomahawkClassをDeleteした上で，以下のコメントアウトされた一行のコードの行頭のコメント記号（//）を外す．
// さらにvoid initCrowbar()中のcrow = startCrowbar.generate(50, 30, 2);が第3引数を持つ場合は削除すること．例） startCrowbar.generate(50, 30);
// class tomahawkClass { tomahawkClass(int mode) {} protected int tomahawkMode = 0; void backupViewports() {} void restoreViewports() {} void refreshView() {} void resetView(boolean flag) {} }

// デバッグ用：P2DとjAVA2Dの切り替え
boolean _JAVA2D = true;

// ------------------------------------------------------------------------------------------
// ユーザが自由なインスタンス名を使えるように，
// crowbarClass.pdeとtomahawkClass.pdeで使用するインスタンスcrowbarのシャロウコピーを提供する
// ------------------------------------------------------------------------------------------
crowbarClass crowbar;
class generateCrowbar {
  // Tomahawkを使用しない場合
  crowbarClass generate(int x, int y) {
    return generate(x, y, 0);
  }
  // Tomahawkを使用する場合
  crowbarClass generate(int x, int y, int mode) {
    crowbar = new crowbarClass(x, y, mode);
    return crowbar;
  }
}
generateCrowbar startCrowbar = new generateCrowbar();

// デフォルトのテキスト属性
protected int _FONTSIZE      = 14;
protected int _TEXTCOLOR     = #000002;  // PGraphicsの仕様上の問題点の回避 (#000000や#000001ではtext()で貼った際に色が#000000になって背景と区別がつかない）
protected int _BGCOLOR       = #ffffee;
protected int _SYSTEMCOLOR1  = #000002;  // システムメッセージの文字色（キー入力など）
protected int _SYSTEMCOLOR2  = #f04040;  // システムメッセージの文字色（見出し）
protected int _SYSTEMCOLOR3  = #208020;  // システムメッセージの文字色（入力待ち用）
// デフォルトのスクリーン属性（いまのところ変更予定なし）
protected float _HSTEP       = 1.2;  // 文字送り
protected float _VSTEP       = 1.4;  // 行送り
// タブ表示の初期値
protected float _TABSIZE     = 10.0;
// フォント名
// protected String _FONTNAME = "MS Gothic";
// protected String _FONTNAME = "ipam.ttf";  // スケッチのdataフォルダにttfフォントをコピーすれば使用可能
// 以下のコードでフォントファイルのリストを得られる（インストールされているフォントのみ．dataフォルダ内は除外）
// String[] fontList = PFont.list();
// println(fontList);
protected String _FONTNAME = "FFScala";
// ログファイルのデフォルトのファイル名
protected String _DEFAULT_LOG_FILENAME = "default.txt";

// デバッグ用
protected boolean _DEBUGFLAG = false;

///////////////////////////
// テキストバッファのクラス
// （スクロール対応用）
///////////////////////////
// メモリの確保は単一方向リスト構造とする．
// オーバーラップして折り返す分は次のラインデータに入れる．後日，実行ウィンドウのサイズ変更に
// 対応する際には，一旦，改行されたラインデータを繋げ直して，それからまたオーバーラップし直す．
// その際に行数が減るかも知れない．不要になったラインデータはリスト構造から切り離し，GCに開放させる．
// Ver.1.04.00で文字色の変更に対応するために，write()で出力された単位で一つのclassとして文字情報とした．
// 一応，将来的な拡張を考慮に入れて背景色も考慮する．classとしては他の拡張属性のも対応できるように考えた上で実装する．

// テキスト情報の属性
class textAttrClass {
  protected int  textColor;  // 文字色        ：変更可
  protected int  bgColor;    // 背景色        ：基本的に変えない
  protected int  fontSize;   // フォントサイズ：基本的に変えない
  // コンストラクタ
  textAttrClass () {
    init();
  }
  // システムデフォルト値で初期化
  void init() {
    fontSize   = _FONTSIZE;
    textColor  = _TEXTCOLOR;
    bgColor    = _BGCOLOR;
  }
  // テキスト属性の複製
  void clone(textAttrClass attr) {
    fontSize  = attr.fontSize;    // フォントサイズ 
    textColor = attr.textColor;   // 文字色
    bgColor   = attr.bgColor;     // 背景色
  }
  // 引数で与えられたテキスト属性と現テキスト属性の比較（等しければtrueを返す）
  boolean checkTextAttr(textAttrClass attr) {
    if (attr.fontSize   != fontSize)  return false;
    if (attr.textColor  != textColor) return false;
    if (attr.bgColor    != bgColor)   return false;
    return true;
  }
}

// テキストの最小単位（センテンス）：write()する度にバッファに追加される
class sentenceClass {
  protected String sentence;   // 同じ文字属性の一連なりの文字列
  textAttrClass    attr;       // 文字属性
  sentenceClass    next;       // 次のセンテンスへのポインタ
  // コンストラクタ
  sentenceClass(String str, textAttrClass attribute) {
    sentence = str;
    attr     = new textAttrClass();
    next     = null;
    attr.clone(attribute);
  }
  // このセンテンスのみを初期化（リスト構造を下流に下らない）
  void clearSentence(textAttrClass attribute) {
    sentence = "";
    attr.clone(attribute);
    next     = null;
  }
  // センテンスのリスト最後部への追加
  void addSentence(String str, textAttrClass attr) {
    if (next == null) next = new sentenceClass(str, attr);  // 終了条件
      else            next.addSentence(str, attr);          // 再帰呼び出し
  }
  // 文字属性を考慮に入れて同じ属性の場合は文字列をマージする
  // 再帰呼び出しではないが，リスト末尾まで「これとその次を比較してこれに次をマージ」を繰り返す．
  void merge() {
    if (next == null) return;  // １個しか無い（あるいは終端を呼ばれた）ならばマージは不要．特に空行のことを考慮にいれた特例
    sentenceClass current = this;
    while (current.next != null) {
      if (current.attr.checkTextAttr(current.next.attr) || (current.sentence.length() == 0)) {
        // もしこれと次のセンテンスのテキスト属性が等しいならば次の文字列をこれの文字列に追加
        current.sentence += current.next.sentence;
        // 文字列数ゼロのセンテンスを後ろの文字列と合体させるために属性を後ろからコピー
        current.attr.clone(current.next.attr);
        // 次のセンテンスを無視してその次と繋げ直す（current.netxt.nextはnullかも知れないが，それは次のwhileの終了条件でチェックされる）
        current.next = current.next.next;
        // ターゲットは次へ移るのではなく，これのままであるのがミソ
      } else {
        // 「これと次が異なる」ならばターゲットを次に移す
        current = current.next;
      }
    }
  }
  // センテンスを指定された長さで分割し，残りを新たに確保した領域にコピーする
  // 分割の必要が無かった場合は文字列数０のセンテンスを返す（扱いは上流に任せる）
  sentenceClass split(int len) {
    sentenceClass newone = new sentenceClass("", attr);  // テキスト属性クローンで新規に作成
    // 文字列の分割コピー
    newone.sentence = sentence.substring(len);
    sentence        = sentence.substring(0, len);
    // リストを挿入
    newone.next = next;
    next        = newone;
    return newone;
  }
  // 文字列の長さ（このリストから先の文字列の長さを合算）
  int length() {
    if (next == null) return sentence.length();                  // 終了条件
      else            return next.length() + sentence.length();  // 再帰呼び出し
  }
   // 属性なしの文字列として返す（このリストから先の全て）
   String getPlainText() {
   if (next == null) return sentence;
     else            return sentence + next.getPlainText();
   }
}

// 一行分のクラス
class lineClass {
  protected int     no;      // 行番号．０から始まる（デバッグ用）
  sentenceClass     line;    // センテンスのリスト構造の先頭（テキストバッファのデータはここに）
  protected boolean flag;    // 入力済みか？（空行を考慮．lineの文字数では判断できないため．さらにGCを使わずリストを再利用するためにも）
  protected boolean overlap; // 前の行の折り返しか（後日，実行ウィンドウのサイズ変更に対応するために）
  lineClass         next;    // 次の行の位置
  // コンストラクタ
  lineClass() { clear(); }
  // クリア
  void clear() {
    no         = 0;
    line       = null;
    flag       = false;
    overlap    = false;
    next       = null;
  }
  // 行単位のデータのセット（オーバーラップ処理は上流で行う．この行がオーバーラップする行ならばoverをtrueにセットしてコール）
  void set(sentenceClass data, boolean over) {
    // 現リストにデータをセット
    // no は初期化時あるいは一つ前のリストにデータをセットされた時に設定済み
    line    = data;    // buffをそのまま使う．したがってbuffはset()した後に再確保が必要
    flag    = true;
    overlap = over;
    if (next == null) next = new lineClass();  // 次の行データの確保（ただし，リスト構造を再利用する場合は確保しない）
    next.no = no + 1;  // 自動加算
  }
  // テキスト属性を考慮に入れて同じテキスト属性を持つ隣接したリストをマージする処理を行単位で実行する
  void merge() { line.merge(); }
}

///////////////////////////////
// テキストバッファ本体のクラス
class textbufferClass {
  private int           cwidth;            // 実行ウィンドウのテキストの一行の文字数
  private screenClass   screenSetting;     // 画面設定
  int                   fontSize;          // 実行ウィンドウのテキストのフォントサイズ
  private lineClass     topLine;           // 一番最初の行データ
  lineClass             currentWriteLine;  // 現在の書き込み可能な行データ
  textAttrClass         currentAttr;       // 現在の文字属性（主に書き込み用）
  protected int         totalLines;        // 出力済みのテキストバッファの行数を単純に返す（オーバーラップは考慮しない→考慮する）
  protected int         readingLine;       // 現在の読み出しされている行番号
  private sentenceClass buff;              // 現在の未改行のバッファ

  // コンストラクタ
  textbufferClass(screenClass scr, textAttrClass attr) {
    screenSetting    = scr;
    cwidth           = screenSetting.cwidth;  // 一行の文字数
    fontSize         = attr.fontSize;
    topLine          = new lineClass();
    currentWriteLine = topLine;
    currentAttr      = new textAttrClass();  // デフォルト値が設定される
    totalLines       = 0;
    readingLine      = 0;
    buff             = new sentenceClass("", currentAttr);  // 常に最初のセンテンスは空
  }
  // 全テキストバッファのクリア
  // GCで消えるのを期待するのではなく古いリスト構造を再利用する
  void clearTextBuffer() {
    lineClass  temp, next;
    temp = topLine;
    while (true) {
      if (temp.next == null)  break;
      //    if (temp.flag == false) break;  // 安全のためにコメントアウト
      next = temp.next;  // 次のリストのポインタを退避させてから
      temp.clear();      // 現ラインのデータを初期化して
      temp = next;       // 次に移動する
    }
    currentAttr.init();  // システムデフォルト値に初期化
    currentWriteLine = topLine;
    totalLines       = 0;
    readingLine      = 0;
    if (buff != null) buff.clearSentence(currentAttr);  // バッファの初期化（リストを辿らない：GCに期待）
  }
  ///////////////////////
  // オーバーラップ関係
  // 与えられた文字列の幅（文字数では無い）を返す（全角1.0，半角0.5）
  float getStringWidth(String str) {
    float len;
    if (crowbar.tomahawkMode == 2) len = screenSetting.pgText.textWidth(str) / fontSize;
      else                         len = textWidth(str) / fontSize;
    if ((crowbar.config.proportional == false) && (len < 0.5)) len = 0.5; // 半角文字を0.5に固定
    return len;
  }
  // 指定された文字幅は与えられた文字列の何文字目までなのかを調べる
  int convertStringWidth2Length(String str, int len) {
    int i;
    for (i = len - 1; i < str.length(); i++) {    // len - 1から始めるのは高速化のため
      if (getStringWidth(str.substring(0, i + 1)) > len) break;
    }
    return i;
  }
  // -----------バッファ関連 --------------------
  // バッファにセンテンスを追加する（単純にチェーンを繋ぐ．後で属性をチェックしてマージする必要がある．空文字列でも繋ぐ）
  void appendSentence(String str, textAttrClass attr) {
    if (buff == null) buff = new sentenceClass(str, attr);  // バッファが無い！ことは無いはず．
      else            buff.addSentence(str, attr);
    // 文字表示属性の更新
    currentAttr.clone(attr);
  }
  // データ書き込みの本体（もしこの行がオーバーラップされた行ならばoverlapにtrueをセットしてコールする）
  void _setLine(boolean overlapped) {
    currentWriteLine.set(buff, overlapped);
    currentWriteLine = currentWriteLine.next;
    totalLines++;
  }
  // バッファのデータを行データとして確定する
  // オーバーラップを考慮する必要があるときはoverlapをtrueにセットしてコールする
  // オーバーラップ有効時に折り返し分を新たに行データを作成してそちらに続ける．
  void setLine(boolean overlap) {
    if (!overlap) {
      buff.merge();  // マージを先に行う
      _setLine(false);
    } else {
      normalizeBuffer();
      _setLine(false);
    }
    buff = new sentenceClass("", currentAttr);  // 常に最初のセンテンスは空
  }
  // いまバッファに溜まっているセンテンスが一行の行数を超えているならば超えている分はラインデータとして確定する
  // これはオーバーラップ許可時以外に呼び出してはいけない．
  void normalizeBuffer() {
    if (getStringWidth(buff.getPlainText()) >= cwidth) {
      // オーバーラップ有効時の分割処理
      // lineClassのoverlapはオーバーラップの”発生した”行のみtrueとする
      buff.merge();  // マージを先に行う
      sentenceClass  bufBakPtr, buffPtr;
      textAttrClass  bakAttr;
      float total = 0.0;

      buffPtr = buff;
      while (true) {
        if (buffPtr == null) break;
        // 折り返しが必要な場合
        if (total + getStringWidth(buffPtr.sentence) >= cwidth) {
          // センテンスを分割する
          bufBakPtr = buffPtr.split(convertStringWidth2Length(buffPtr.sentence, int(cwidth - total)));
          buffPtr.next = null;
          // これはオーバーラップした分なので引数true
          _setLine(true);
          // 分割した残りが空の場合
          if (bufBakPtr.sentence.length() == 0) buff = bufBakPtr.next;  // 単に無視
            else                                buff = bufBakPtr;
          buffPtr = buff;
          total   = 0;
        } else {
          total += getStringWidth(buffPtr.sentence);
          if (buffPtr.next == null) {
            // オーバーラップの発生していない行（あるいは残りの部分）
            break;
          } else buffPtr = buffPtr.next;
        }
      }
    }
    return;
  }
  // バッファをフラッシュ（書き出す）
  void flushBuffer(boolean overlap) { if (buff.length() > 0) setLine(overlap); }
  // 指定された行番号の行データを返す（０行から始める）
  lineClass getTargetLine(int line) {
    lineClass temp = topLine;
    for (int num = 0; num < line; num++) {
      if (temp.next == null) break;
      if (temp.flag != true) break;
      temp = temp.next;
    }
    return temp;
  }
  // 表示開始行をダイレクトに指定する
  void setReadingLine(int num) {
    if (num < 0) num = 0;
    if (crowbar.afterRunning && (num >= totalLines - 1)) num = totalLines - 1;
    readingLine = num;
  }
  // 表示開始行数を指定行数分，増加する
  void incrementReadingLine(int num) { setReadingLine(readingLine + num); }

  // デバッグ用：テキストバッファを全て表示
  void displayAllTextBuffer() {
    int  i;
    lineClass temp;
    temp = topLine;
    sentenceClass sen;
    i = 0;
    println();
    println("No.:(No.):flag:overlap:length[sentence1, sentence2, ...]");
    while (true) {
      if (temp == null) break;
      if (temp.line == null) break;
      print(str(i) + ":" + str(temp.no) + ":" + str(temp.flag) +":" + str(temp.overlap) + ":" + str(temp.line.length()));
      print("[");
      sen = temp.line;
      while (true) {
        print(hex(sen.attr.textColor));
        print("(" + str(sen.sentence.length()) + ")");
        print(",");
        if (sen.next == null) break;
        sen = sen.next;
      } 
      println("]");
      if (temp.next == null) break;
      temp = temp.next;
      i++;
    }
  }
}

// キーボード入力規則のクラス
class keyInputConstraintClass {
  private boolean point;     // [.]キーが押されたか
  private boolean sign;      // [+],[-]キーが押されたか
  private boolean inputted;  // 数字が既に入力済みか
  // コンストラクタ
  keyInputConstraintClass() { reset(); }
  // 初期化
  void reset() { point = sign = inputted = false; }
  // 1) 浮動小数点時の入力規則チェック
  boolean checkFloat(char c) {
    boolean flag = true;
    switch (c) {
      // 符号
    case '+' :
    case '-' :
      if ((sign) || (inputted)) flag = false;
        else                    sign = true;
      break;
      // 小数点
    case '.' :
      if ((point) || (!inputted)) flag  = false;
        else                      point = true;
      break;
    default :
      if (isNumberChar(c)) inputted = true;
        else               flag     = false;
      break;
    }
    return flag;
  }
  // 2) 整数入力時の入力規則チェック（現状は使われていない）
  boolean checkInteger(char c) {
    if (c == '.') return false;
    return checkFloat(c);
  }
  // 3) 論理型入力時の入力規則チェック
  boolean checkBoolean(char c) {
    if (crowbar.keyBufferLength() >= 1) return false;
    return true;
  }
}

// ログファイル記録用クラス
class loggingClass {
  private String       _defaultFileName = _DEFAULT_LOG_FILENAME;
  protected String     logFileName;    // ファイル名（スケッチフォルダに保存）
  private boolean      loggingNow;     // trueの間はログを記録する
  private PrintWriter  output;         // ログ出力用ファイルポインタ
  // コンストラクタ
  loggingClass() {
    logFileName   = "";
    loggingNow    = false;
    output        = null;
  }
  void setFileName(String filename) {
    if (filename.length() > 0) logFileName = filename;
      else                     logFileName = _defaultFileName;
  }
  boolean isEnableLogging() { return output == null ? false : true; }
  // ---- ログ関係のコマンド -----
  // ユーザーは使用しない
  // ログファイルにのみprint()
  void print2Log(String str)   { if (loggingNow) output.print(str);   }
  // ログファイルにのみprintln()
  void println2Log(String str) { if (loggingNow) output.println(str); }
  // ログファイルにのみnewline()
  void newline2Log() { println2Log(""); }

  // ---- 関数 -------------------------------------------------
  // 実際には一つ上のクラスがI/Fになるのでユーザは使用しないこと
  // ---- 関数 -------------------------------------------------
  String setDefaultFilename() { return _defaultFileName; }
  String setDefaultFilename(String fname) {
    String oldname = _defaultFileName;
    _defaultFileName = fname;
    return oldname;
  }
  // ログ記録開始
  PrintWriter startLogging() {
    if (output != null) stopLogging();
    output     = createWriter(logFileName);
    loggingNow = true;
    println2Log("--- Logging start : " + nowDateTime() + "---");
    return output;
  }
  PrintWriter getLogfile() { return output; }
  // ログ記録終了
  void stopLogging() {
    if (output != null) {
      println2Log("--- Logging finished : " + nowDateTime() + "---");
      // ファイルを閉じる
      output.flush();
      output.close();
    }
    // 設定を初期化する
    loggingNow  = false;
    output      = null;
  }
  // ログファイルの保存を一時的に停止する
  void pauseLogging()   { loggingNow = false;  }
  // ログファイルの保存を再開する
  void restartLogging() { if (output != null) loggingNow = true; }
}

// ------------------------------------
// テキスト画面（実行ウィンドウ）の属性
// ------------------------------------
class screenClass {
  protected int   cwidth, cheight;           // キャラクタの横文字数，行数
  protected int   gwx, gwy;                  // スクリーンサイズ
  protected int   colwidth, rowheight;       // 一文字の文字送り，行間（pixel）
  protected int   frameWidth;                // 額縁の幅（上下左右同一）
  protected float hStep, vStep;              // 文字間隔，行間隔の比率（1.0で密着）
  protected PGraphics pgText;                // テキスト表示専用
  screenClass(int max_x, int max_y, int fsize) {
    cwidth     = max_x;
    cheight    = max_y;
    frameWidth = 10;
    hStep      = _HSTEP;
    vStep      = _VSTEP;
    reCalc(fsize);
  }
  void setPgText(PGraphics pg) { pgText = pg; }
  void reCalc(int fsize) {
    colwidth   = int(fsize * hStep);
    rowheight  = int(fsize * vStep);
    gwx        = cwidth  * colwidth  + frameWidth * 2;
    gwy        = cheight * rowheight + frameWidth * 2;
  }
}

// 機能選択のフラグ
class configFlagClass {
  protected boolean noDisplay;     // グラフィックス表示の抑制
  protected boolean noTextArea;    // PDEのテキストエリアへの文字列の表示の抑制
  protected boolean noOverlap;     // ウィンドウからはみ出す文字列を折り返して次行に表示しない
  protected boolean proportional;  // 半角英数字の出力幅を0.5に固定しない
  protected boolean degreeBase;    // （主にTomahawkにおける）角度の単位を度とする場合はtrue
  configFlagClass() {
    noDisplay = noTextArea = noOverlap  = false;
    proportional = false;
    degreeBase   = true;
  }
  void clone(configFlagClass src) {
    noDisplay    = src.noDisplay;
    noTextArea   = src.noTextArea;
    noOverlap    = src.noOverlap;
    proportional = src.proportional;
    degreeBase   = src.degreeBase;
  }
}

// Draw()におけるリアルタイム文字列入力用のクラス
// ENTERが押されたならばそこまでに入力された一文はbuff2に保管され，gets()されるとこちらが返される．buff2はクリアするが，buffはクリアしない．
// ENTERしていない状態でgets()された場合はbuffを返し，buffをクリアする．
// ENTERされて入力が確定したかどうかを確認した上でgets()するのが基本的な使い方．
// ただし文字列バッファは一つだけ．なお，拡張することはできるがしないだけ．
// エコーバックはどうしよう．バックスペースとかも．
// フラッシュされない読み出しgets()を用意して，自分でリアルタイムに表示して貰えば良いか．
// keyPressed()ではキーコードをチェックしない．片っ端からadd()する．判定はadd()で行なう．
class realtimeInputClass {
  private boolean enable;  // 文字列の格納を許可されている時はtrue，禁止されている時はfalse
  private String [] buff2; // ENTER後に確定した文字列を格納する
  private String buff;     // 現在入力中の文字列（未確定）を一時的に格納する
  private int finished;    // 確定済みの文字列数．未確定ならば0
  private int autoCount;   // カウントダウンしてゼロになったらenableを自動的にfalseに変更する
  // コンストラクタ
  realtimeInputClass() {
    disable();
    flush();
  }
  // enable/disableに切り替える際にバッファをフラッシュするのも手だが，そこまでお節介はしない
  // 文字列の格納を許可
  void enable(int lines) { enable = true; autoCount = lines > 0 ? lines : 0; }  // 0も無期限
  // 文字列の格納を禁止
  void disable()         { enable = false; }
  // 文字列の格納が許可されている時はtrue
  boolean isEnable() { return enable;  }
  // 全バッファをクリア
  void flush()    {
    buff      = "";
    buff2     = (String []) new String[1];
    finished  = 0;
    autoCount = 0;
  }
  // 文字列入力バッファから文字列を取得する
  // FIFOなので確定済みの文字列がバッファ（buff2）に存在するならば最も古い文字列を返す．
  // 未確定状態ならばbuffを返す．
  // flushがtrueならば文字列を返すと共にバッファを一つ削除する
  String gets(boolean flush)   {
    String ret;
    if (finished > 0) {
      // 入力確定済みの文字列がある場合
      ret      = buff2[0];
      if (flush) {
        buff2 = subset(buff2, 1, finished);
        finished--;
      }
    } else {
      // 入力確定済みの文字列が無い場合
      ret  = buff;
      if (flush) buff = "";
    }
    return ret;
  }
  //入力済みの文字列の長さを返す（入力確定済みのバッファ数はfinished()で得られる）
  // 入力確定済みでは無いなら未確定（buff）の文字列を返す
  int length()    {
    if (finished > 0) return buff2[0].length();
    return buff.length();
  }
  // キーボードから入力された文字をバッファに格納する
  // ENTERが押されたならば確定処理を行う
  void add() {
    if (enable == false) return;
    switch (keyCode) {
      case ENTER : 
        buff2[finished++] = buff;
        buff2 = append(buff2, "");
        buff  = "";
        if (autoCount > 0) {
          if (--autoCount <= 0) {
            enable    = false;
            autoCount = 0;
          }
        }
        break;
      case BACKSPACE :
        bs();
        break;
      case UP :
      case DOWN :
      case LEFT :
      case RIGHT :
      case SHIFT :
      case TAB :
      case ESC :
      case DELETE :
        break;
      default :
        buff += key;
    }
  }
  // バックスペースキーが押された場合はバッファ(buff)から一文字削除
  void bs() {
    int len;
    if ((len = buff.length()) > 0) {
      buff = buff.substring(0, len - 1); 
    }
  }
}

/////////////////////////////////////////
// キーボード入力とコンソール出力のクラス
/////////////////////////////////////////
class consoleClass extends tomahawkClass {
  // 実行ウィンドウ関係の管理用変数
  protected float cx = 0.0, cy = 0.0;        // キャラクタの表示位置（カレントポジション），半角文字は0.5文字
  protected float tabSize = _TABSIZE;        // タブ文字数
  screenClass     screenSetting;
  // テキスト表示の属性の設定（clrscr()で設定される）
  protected textAttrClass  currentAttr;
  // 機能選択のフラグ
  configFlagClass config, _config;
  // テキスト画面のスクロール表示関係
  textbufferClass      textbuffer;         // バッファ関係のクラス
  private boolean      recordTextBuffer;   // trueの間はテキストバッファに記録する
  protected int        autoScrollLines;    // 自動スクロールする行数（0が指定されている時は完全に消去）
  // キーボード入力規則のチェック
  keyInputConstraintClass  keyCheck;
  // ログファイル関係
  loggingClass         logging;
  private boolean      enableLogging;  // テキストログを記録する（setup()でのみ有効）
  // Draw()中のリアルタイム文字列入力用バッファ
  realtimeInputClass   input;

  // コンストラクタ（引数は横方向の最大文字数と縦方向の最大行数）
  consoleClass(int max_x, int max_y, int mode) {
    super(mode);
    // スクリーンの設定
    currentAttr   = new textAttrClass();
    screenSetting = new screenClass(max_x, max_y, currentAttr.fontSize);
    // 再設定
    screenReset();
    // デフォルト文字色と背景色の設定
    textColor(currentAttr.textColor, currentAttr.bgColor);
    // スクロール表示用のバッファ関連
    textbuffer       = new textbufferClass(screenSetting, currentAttr);
    recordTextBuffer = true;
    autoScrollLines  = 0;
    // キーボード入力規則チェック
    keyCheck      = new keyInputConstraintClass();
    // ログファイル関係
    logging       = new loggingClass();
    enableLogging = false;
    // 機能スイッチのバックアップの初期化
    config  = new configFlagClass();
    _config = new configFlagClass();
    backupConfigFlag();
    // Draw()中のリアルタイム文字列入力用バッファ
    input = new realtimeInputClass();
  }
  void backupConfigFlag()  { _config.clone(config);  }
  void restoreConfigFlag() {  config.clone(_config); }

  // フォントの設定
  void prepareFont() {
    if (tomahawkMode == 2) {
      // テキスト表示専用レイヤー使用時
      PFont font = createFont(_FONTNAME, currentAttr.fontSize, false);  // フォントを変換
      screenSetting.pgText.beginDraw();
      screenSetting.pgText.textFont(font);                     // フォントを設定
      if (!_JAVA2D) screenSetting.pgText.textMode(SCREEN);     // JAVA2Dならコメントアウトしなくてはいけない．P2Dだと必要．
      screenSetting.pgText.textSize(currentAttr.fontSize);     // フォントサイズを設定（不要？）
      screenSetting.pgText.textAlign(LEFT, TOP);               // 配置
//    screenSetting.pgText.smooth();
      screenSetting.pgText.endDraw();
    } else {
      // テキスト表示専用レイヤー不使用時
      PFont font = createFont(_FONTNAME, currentAttr.fontSize, true);  // フォントを変換
      textFont(font);                     // フォントを設定
//    textMode(SCREEN);                   // これを有効にすると文字が表示されない
      textSize(currentAttr.fontSize);     // フォントサイズを設定（不要？）
      textAlign(LEFT, TOP);               // 配置
    }
  }
  // フォントサイズの初期設定の実体（直接，ユーザは呼び出さない）
  void _setFontSize(int size) {
    if (size <= 0) return;
    currentAttr.fontSize  = size;
    textbuffer.fontSize   = size;
    screenSetting.reCalc(size);
    // 実際にフォントサイズを変更する
    if (tomahawkMode == 2) {
      screenSetting.pgText.beginDraw();
      screenSetting.pgText.textSize(size);
      screenSetting.pgText.endDraw();
    } else {
      textSize(size);
    }
    screenReset();
  }
  // 画面パラメータ変更後のリセット動作
  void screenReset() {
    // スクリーンの設定
    size(screenSetting.gwx, screenSetting.gwy);
    if (tomahawkMode == 2) {
      if (_JAVA2D) screenSetting.pgText = createGraphics(screenSetting.gwx, screenSetting.gwy, JAVA2D);  // 速度が極めて遅い
        else       screenSetting.pgText = createGraphics(screenSetting.gwx, screenSetting.gwy, P2D);     // SCREENを設定する必要がある
    } else         screenSetting.pgText = null;
    _clrscr();  
    // フォント関係
    prepareFont();
  }

  // ----------- 画面出力関係-----------------
  // 文字位置をグラフィックス座標に変換
  private int ptx(float x) { return int(x * screenSetting.colwidth  + screenSetting.frameWidth);  }
  private int pty(float y) { return int(y * screenSetting.rowheight + screenSetting.frameWidth);  }

  // 出力位置指定
  crowbarClass locate(float x, float y) {
    locateX(x);
    locateY(y);
    return crowbar;
  }
  float locateX() { return cx; }
  float locateY() { return cy; }
  float setLocateX(float x) { float oldx = locateX(); if (x < 0.0) x = 0.0; cx = x; return oldx; }
  float setLocateY(float y) { float oldy = locateY(); if (y < 0.0) y = 0.0; cy = y; return oldy; }
  crowbarClass locateX(float x) { setLocateX(x); return crowbar; }
  crowbarClass locateY(float y) { setLocateY(y); return crowbar; }
  // 行数が最大値に達したか
  private boolean rowCheck() {
    // 達したならば画面消去
    if (cy >= screenSetting.cheight) {
      if (autoScrollLines == 0) {
        // オートスクロール無効（全画面消去）
        _clrscr();
        textbuffer.readingLine += screenSetting.cheight;  // ここだけは強制的にインクリメントする必要がある（まだ出力していない）
      } else {
        // オートスクロール対応
        stopRecordTextBuffer();
        pageLineScroll(autoScrollLines);
        startRecordTextBuffer();
      }
      return true;
    }
    return false;
  }
  // BSキーが押された場合の処理
  void backSpace() {
    if (keyBufferLength() > 0) {
      keyBuffer = keyBuffer.substring(0, keyBuffer.length() - 1);
      writeBS();
    }
  }
  // テキスト文字0.5文字分を削除（パラメータ入力時のみ有効なため）
  private void writeBS() {
    if (cx <= 0) return;
    if (!displaySwitch()) {
      cx -= 0.5;
      if (tomahawkMode == 2) {
        // テキスト表示レイヤー使用時
        resetAlpha(ptx(cx), pty(cy), screenSetting.colwidth/2, screenSetting.rowheight, true);
      } else {
        // テキスト表示レイヤー不使用時
        noStroke();
        fill(currentAttr.bgColor);    // 背景色
        rect(ptx(cx), pty(cy), screenSetting.colwidth/2, screenSetting.rowheight);
        fill(currentAttr.textColor, 255);          // 塗りつぶし色
      }
    }
  }
  // 与えられた文字の大きさを返す．全角ならば１，半角ならば0.5
  private float charLength(char c) {
    float len;
    if (tomahawkMode == 2) len = screenSetting.pgText.textWidth(c) / currentAttr.fontSize;
      else                 len = textWidth(c) / currentAttr.fontSize;
    if ((config.proportional == false) && (len < 0.5)) len = 0.5; // 半角文字を0.5に固定
    return len;
  }
  // α値をリセットする．第５引数がtrueの場合は無条件に#000000で塗りつぶす
  void resetAlpha(int x, int y, int w, int h, boolean clr) {
    screenSetting.pgText.beginDraw();
    PImage pimg = screenSetting.pgText.get(x, y, w, h);
    int p;
    for (int i = 0; i < pimg.pixels.length; i++) {
      if (clr) pimg.pixels[i] = color(#000000, 0);
      else {
        p = pimg.pixels[i] & 0x0ffffff;
        if (p > 0) {
  //        p |= 0x0ff000000;
  //        pimg.pixels[i] = color(p);
        } else pimg.pixels[i] = color(p, 0);
      }
    }
    screenSetting.pgText.set(x, y, pimg);
    screenSetting.pgText.endDraw();
  }
  // 一文字表示（文字列は全てこの関数を介してtext()で表示する
  private void putchar(char c) {
    int px = ptx(cx);
    int py = pty(cy);
    if (c != ' ') {  // 何故かPGraphicsを使ったテキスト表示専用レイヤに対して半角空白をtext()するとゴミが一行上にポツリと表示されるための苦肉の策（バグ？）
                     // https://forum.processing.org/topic/bug-illeagal-pixel-is-dotted-when-space-character-is-displayed-by-using-text-in-pgraphics
      if (tomahawkMode == 2) {
        // テキスト表示専用レイヤー使用時
        screenSetting.pgText.beginDraw();
        screenSetting.pgText.text(c, px, py);
        screenSetting.pgText.endDraw();
        if (!_JAVA2D) resetAlpha(px, py - 2, currentAttr.fontSize * 2, currentAttr.fontSize + 4, false);   // なぜか-2しないと線が残る
      } else text(c, px, py); // テキスト表示専用レイヤー不使用時
    }
    cx += charLength(c);
  }

  ///////////////////////
  // テキストバッファ関連
  ///////////////////////
  // テキストバッファへの記録を一時的に停止する
  void stopRecordTextBuffer() {  recordTextBuffer = false; }
  // テキストバッファへの記録を再開する
  void startRecordTextBuffer() { recordTextBuffer = true;  }
  // テキストバッファの指定された行から１ページ分を再描画する
  void redrawPage() {
    // 自動スクロール（Run中）のためにカレント属性をバックアップ
    textAttrClass bakAttr = new textAttrClass();
    bakAttr.clone(currentAttr);
    // 再描画
    redrawPage(textbuffer.readingLine);
    // カレント属性のバックアップをリストア
    currentAttr.clone(bakAttr);
  }
  void redrawPage(int line) {
    int       lineCount, dcy;
    lineClass thisline;

    // 表示開始行数のチェック
    textbuffer.incrementReadingLine(0);
    // 初期化
    _clrscr();
    thisline = textbuffer.getTargetLine(line);
    // 再描画
    // スクロール表示時は一時的にオーバーラップを強制的にオフにする
    boolean over_flag = config.noOverlap;
    disableOverlap();
    for (lineCount = 0; lineCount < screenSetting.cheight; lineCount += dcy) {
      if (!thisline.flag) break; // これ以降，テキストが出力されていないので抜ける
      dcy      = writelnText(thisline.line);
      thisline = thisline.next;
    }
    // 元の状態に戻す
    config.noOverlap = over_flag;
  }
  // ページ単位のスクロール
  // pageが正で下，負で上にスクロール
  void pageScroll(float page) {
    textbuffer.incrementReadingLine(int(screenSetting.cheight * page));
    redrawPage();
  }
  // 指定した行数だけ上/下にスクロールする
  // num が正で下，負で上にスクロール
  void pageLineScroll(int num) {
    textbuffer.incrementReadingLine(num);
    redrawPage();
  }
  // 最新のページを再描画（カーソル位置が復元されないのは仕方が無い
  void redrawNewestPage() {
    int  linenumber;
    if (textbuffer.totalLines < screenSetting.cheight) textbuffer.setReadingLine(0);
    else                                               textbuffer.setReadingLine(textbuffer.totalLines - int(screenSetting.cheight / 2.0));
    redrawPage();
  }
  // 新しいテキストバッファ形式で一行分の出力（色つき）
  int writelnText(sentenceClass sen) {
    int num = 1;
    int dy;
    while (true) {
      textColor(sen.attr.textColor);
      // テキストカラーを変更
      if (tomahawkMode == 2) {
        // テキスト表示専用レイヤー使用時
        screenSetting.pgText.beginDraw();
        screenSetting.pgText.fill(sen.attr.textColor, 255f);
        screenSetting.pgText.endDraw();
      } else {
        // テキスト表示専用レイヤー不使用時
        fill(sen.attr.textColor, 255);
      }
      if ((dy = setWrite(sen.sentence)) > 1) num += (dy - 1);
      if (sen.next == null) break;
      sen = sen.next;
    }
    writeln("");
    return num;
  }
  // ------------------------
  // 文字色と背景色の初期設定
  // ------------------------
  // 文字の色のみを変更する
  // (a) 現在の設定値を返す
  int textColor() { return currentAttr.textColor; }
  // (b) 設定（＋変更前の値を返す）
  int setTextColor(int textcolor) {
    int oldColor = currentAttr.textColor;
    if (brightness(textcolor) <= 2.0) textcolor = #000002;  // PGraphicsの仕様上の問題点の回避
    currentAttr.textColor = textcolor;
    if (tomahawkMode == 2) {
      // テキスト表示専用レイヤー使用時
      screenSetting.pgText.beginDraw();
      screenSetting.pgText.fill(currentAttr.textColor, 255);
      screenSetting.pgText.endDraw();
    } else {
      // テキスト表示専用レイヤー不使用時
      fill(currentAttr.textColor);
    }
    return oldColor;
  }
  // (c) マルチステートメント
  // (c-1) 文字色のみ変更
  crowbarClass textColor(int textcolor) { setTextColor(textcolor); return crowbar; }
  // (c-2) 文字色と背景を同時に変更
  crowbarClass textColor(int tcolor, int bcolor) {
    textColor(tcolor);
    bgColor(bcolor);
    return crowbar;
  }
  // 背景の色のみを変更する（あまりお勧めしない）
  // (a) 現在の設定値を返す
  int bgColor() { return currentAttr.bgColor; }
  // (b) 設定（＋変更前の値を返す）
  int setBgColor(int bgcolor) {
    int oldColor        = currentAttr.bgColor;
    currentAttr.bgColor = bgcolor;
    return oldColor;
  }
  // (c) マルチステートメント
  crowbarClass bgColor(int bgcolor) { setBgColor(bgcolor); return crowbar; }

  // タブ文字数の設定
  // (a) 現在の設定値を返す
  float tabSize() { return tabSize; }
  // (b) 設定（＋変更前の値を返す）
  float setTabSize(float num) {
    float oldnum = tabSize();
    if ((num < 0.0) || (num >= screenSetting.cwidth)) return oldnum;
    tabSize = num;
    return oldnum;
  }
  crowbarClass tabSize(float num) { setTabSize(num); return crowbar; }
  // --------
  // タブ出力
  // --------
  String getTabSpace() {
    String space = "";
    float num = int((cx + tabSize) % tabSize * 2);
//  if (num == 0) return "";
    if (num == 0) num = tabSize;
    num = int(tabSize * 2) - num;
    for (int i = 0; i < num; i++) space += " ";    
    return space;
  }
  // 引数指定なし（設定値を使用する）
  float setTab(boolean flag) {
    String space = getTabSpace();
    if (flag) write(space); else _write(space);
    return space.length() / 0.5;
  }
  // 引数指定あり（暫定的に指定された値を使用する）
  float setTab(float temp, boolean flag) {
    float bak = tabSize();
    tabSize   = temp;
    float num = setTab(flag);
    tabSize   = bak;
    return num;
  }
  // ユーザは以下の関数を使用
  // 引数指定なし（設定値を使用する）
  float setTab()  { return setTab(true); }
  float _setTab() { return setTab(false); }
  // 引数指定あり（暫定的に指定された値を使用する）
  float setTab(float temp)  { return setTab(temp, true); }
  float _setTab(float temp) { return setTab(temp, false); }
  crowbarClass  tab() {  setTab(); return crowbar; }
  crowbarClass _tab() { _setTab(); return crowbar; }
  crowbarClass  tab(float temp) {  setTab(temp); return crowbar; }
  crowbarClass _tab(float temp) { _setTab(temp); return crowbar; }

  // 画面消去（画面のみクリア）
  crowbarClass _clrscr() {
    if (tomahawkMode == 2) {
      // テキスト表示専用レイヤー使用時
      background(currentAttr.bgColor);  // 背景色
      screenSetting.pgText.beginDraw();
      screenSetting.pgText.background(#000000, 0);
      screenSetting.pgText.fill(currentAttr.textColor, 255);      // 塗りつぶし色
      screenSetting.pgText.endDraw();
    } else {
      // テキスト表示専用レイヤー不使用時
//    background(_BGCOLOR);
      background(currentAttr.bgColor);
      fill(currentAttr.textColor);      // 塗りつぶし色
    }
    cx = cy = 0;
    return crowbar;
  }
  // 画面消去（テキストバッファもクリア）
  crowbarClass clrscr()        { _clrscr(); clrTextBuffer(); return crowbar;  }
  // テキストバッファのみクリア）
  crowbarClass clrTextBuffer() { textbuffer.clearTextBuffer(); return crowbar; }

  // テキスト出力（テキストバッファ，ログ記録，バックスペースによる消去に対応）
  // 戻り値は実行ウィンドウに出力した実際の行数（オーバーラップを考慮）
  int writeMain(String message) {
    int num = 0;
    if (!displaySwitch()) {
      rowCheck();
      num = _setWrite(message);
      if (recordTextBuffer) {
        textbuffer.appendSentence(message, currentAttr);
        // もしオーバーラップ指定が有効かつ一行を超えているならば，超えている分を先に確定する．
        if (!config.noOverlap) textbuffer.normalizeBuffer();
      }
    }
    // コンソールに出力
    if ((!config.noTextArea) && (recordTextBuffer)) print(message);
    // ログファイルに記録
    if (enableLogging) print2Log(message);
    return num;
  }
  // ユーザは以下の関数を使用
  // テキスト出力（単純に描画するだけ）: Draw()中で利用するのに適している
  int _setWrite(int value)     { return _setWrite(str(value)); }
  int _setWrite(float value)   { return _setWrite(str(value)); }
  int _setWrite(boolean value) { return _setWrite(str(value)); }
  int _setWrite(String message) {
    int  i, num = 1;
    if (message == null) return 0;
    // テキストの色を復元する
    if (tomahawkMode == 2) {
      // テキスト表示専用レイヤー使用時
      screenSetting.pgText.beginDraw();
      screenSetting.pgText.fill(currentAttr.textColor, 255);
      screenSetting.pgText.endDraw();
    } else {
      // テキスト表示専用レイヤー不使用時
      fill(currentAttr.textColor);
    }
    rowCheck();
    for (i = 0; i < message.length(); i++) {
      if (message.charAt(i) == '\n') {  // 改行（write()は上位で切り分ける）
        cx = 0;
        cy++;
        num++;
        rowCheck();
      } else putchar(message.charAt(i));
      // 折り返し表示に関する処理
      if (config.noOverlap != true) {
        if (cx >= screenSetting.cwidth) {
          // ここでnormalizeBuffer()が必要か？
          cx = 0;
          cy++;
          num++;
          rowCheck();
        }
      }
    }
    return num;
  }
  // 座標指定（ログ出力等なし）
  int _setWrite(float x, float y, int value)     { return _setWrite(x, y, str(value)); }
  int _setWrite(float x, float y, float value)   { return _setWrite(x, y, str(value)); }
  int _setWrite(float x, float y, boolean value) { return _setWrite(x, y, str(value)); }
  int _setWrite(float x, float y, String message) {
    locate(x, y);
    return _setWrite(message);
  }
  crowbarClass _write(int value)      { return _write(str(value)); }
  crowbarClass _write(float value)    { return _write(str(value)); }
  crowbarClass _write(boolean value)  { return _write(str(value)); }
  crowbarClass _write(String message)                   { _setWrite(message);       return crowbar; }
  crowbarClass _write(float x, float y, String message) { _setWrite(x, y, message); return crowbar; }
  // 改行あり
  int _setWriteln(int value)     { return _setWriteln(str(value)); }
  int _setWriteln(float value)   { return _setWriteln(str(value)); }
  int _setWriteln(boolean value) { return _setWriteln(str(value)); }
  int _setWriteln(String message) {
    int  num;
    num = _setWrite(message);
    cy++;
    cx = 0;
    return num;
  }
  // 座標指定（ログ出力等なし）
  int _setWriteln(float x, float y, int value)     { return _setWriteln(x, y, str(value)); }
  int _setWriteln(float x, float y, float value)   { return _setWriteln(x, y, str(value)); }
  int _setWriteln(float x, float y, boolean value) { return _setWriteln(x, y, str(value)); }
  int _setWriteln(float x, float y, String message) {
    locate(x, y);
    return _setWriteln(message);
  }
  crowbarClass _writeln(int value)     { return _writeln(str(value)); }
  crowbarClass _writeln(float value)   { return _writeln(str(value)); }
  crowbarClass _writeln(boolean value) { return _writeln(str(value)); }
  crowbarClass _writeln(String message)                   { _setWriteln(message);       return crowbar; }
  crowbarClass _writeln(float x, float y, int value)      { return _writeln(x, y, str(value)); }
  crowbarClass _writeln(float x, float y, float value)    { return _writeln(x, y, str(value)); }
  crowbarClass _writeln(float x, float y, boolean value)  { return _writeln(x, y, str(value)); }
  crowbarClass _writeln(float x, float y, String message) { _setWriteln(x, y, message); return crowbar; }
  // 文字列表示（改行無し）
  int setWrite(int value)     { return setWrite(str(value)); }
  int setWrite(float value)   { return setWrite(str(value)); }
  int setWrite(boolean value) { return setWrite(str(value)); }
  int setWrite(String message) {
    int num = 0;
    String [] msg = split(message, '\n');  // 改行対応
    for (int i = 0; i < msg.length; i++) {
      if (i < msg.length - 1) num += setWriteln(msg[i]);
        else                  num += writeMain(msg[i]);
    }
    return num;
  }
  // 座標指定（ログ出力等対応）
  int setWrite(float x, float y, int value)      { return setWrite(x, y, str(value)); }
  int setWrite(float x, float y, float value)    { return setWrite(x, y, str(value)); }
  int setWrite(float x, float y, boolean value)  { return setWrite(x, y, str(value)); }
  int setWrite(float x, float y, String message) {
    int     num;
    boolean flag;
    flag = textAreaSwitch();
    disableTextArea();
    locate(x, y);
    num = setWrite(message);
    textAreaSwitch(flag);
    return num;
  }
  crowbarClass write(int value)     { return write(str(value)); }
  crowbarClass write(float value)   { return write(str(value)); }
  crowbarClass write(boolean value) { return write(str(value)); }
  crowbarClass write(String message)                   { setWrite(message);       return crowbar; }
  crowbarClass write(float x, float y, int value)      { return write(x, y, str(value)); }
  crowbarClass write(float x, float y, float value)    { return write(x, y, str(value)); }
  crowbarClass write(float x, float y, boolean value)  { return write(x, y, str(value)); }
  crowbarClass write(float x, float y, String message) { setWrite(x, y, message); return crowbar; }
  // テキスト出力（改行）
  int setWriteln(int value)     { return setWriteln(str(value)); }
  int setWriteln(float value)   { return setWriteln(str(value)); }
  int setWriteln(boolean value) { return setWriteln(str(value)); }
  int setWriteln(String message) {
    int num = setWrite(message); // 以下改行のみ出力
    if (!displaySwitch()) {
      if (recordTextBuffer) textbuffer.setLine(!config.noOverlap);
      cy++;
      cx = 0;
    }
    // コンソールに出力
    if (!config.noTextArea) if (recordTextBuffer) println();
    // ログファイルに記録
    if (enableLogging) println2Log("");
    return num;
  }
  // 座標指定（ログ出力等対応）
  int setWriteln(float x, float y, int value)      { return setWriteln(x, y, str(value)); }
  int setWriteln(float x, float y, float value)    { return setWriteln(x, y, str(value)); }
  int setWriteln(float x, float y, boolean value)  { return setWriteln(x, y, str(value)); }
  int setWriteln(float x, float y, String message) {
    int     num;
    boolean flag;
    flag = textAreaSwitch();
    disableTextArea();
    locate(x, y);
    num = setWriteln(message);
    textAreaSwitch(flag);
    return num;
  }
  crowbarClass writeln(int value)     { return writeln(str(value)); }
  crowbarClass writeln(float value)   { return writeln(str(value)); }
  crowbarClass writeln(boolean value) { return writeln(str(value)); }
  crowbarClass writeln(String message)                   { setWriteln(message);       return crowbar; }
  crowbarClass writeln(float x, float y, int value)      { return writeln(str(value)); }
  crowbarClass writeln(float x, float y, float value)    { return writeln(str(value)); }
  crowbarClass writeln(float x, float y, boolean value)  { return writeln(str(value)); }
  crowbarClass writeln(float x, float y, String message) { setWriteln(x, y, message); return crowbar; }
  // 改行
  crowbarClass newline() { writeln(""); return crowbar; }
  // 行頭へ移動
  crowbarClass cr()      { locate(0.0, locateY()); return crowbar; }
  // ----------
  // 設定の変更
  // ----------
  // ウィンドウサイズの変更：　強制的に実行ウィンドウのサイズが変更されるだけでテキスト画面の処理には反映されないため非推奨．
  crowbarClass windowSize(int x, int y) {
    screenSetting.gwx = x;
    screenSetting.gwy = y;
    size(screenSetting.gwx, screenSetting.gwy);
    return crowbar;
  }
  // ディスプレイ表示の抑制と許可
  // (a) 現在の値を返す
  boolean displaySwitch() { return config.noDisplay; }
  // (b) 設定（＋設定前の値を返す）
  boolean setDisplaySwitch(boolean flag) {
    boolean oldflag = displaySwitch();
    config.noDisplay = flag;
    return oldflag;
  }
  // (c) マルチステートメント
  crowbarClass disableDisplay()            { config.noDisplay = true;  return crowbar; }
  crowbarClass enableDisplay()             { config.noDisplay = false; return crowbar; }
  crowbarClass displaySwitch(boolean flag) { setDisplaySwitch(flag);   return crowbar; }
  // テキストエリア表示の抑制と許可
  // (a) 現在の値を返す
  boolean textAreaSwitch() { return config.noTextArea; }
  // (b) 設定（＋設定前の値を返す）
  boolean setTextAreaSwitch(boolean flag) { 
    boolean oldflag = textAreaSwitch();
    config.noTextArea = flag;
    return oldflag;
  }
  // (c) マルチステートメント
  crowbarClass disableTextArea()            { config.noTextArea = true;  return crowbar; }
  crowbarClass enableTextArea()             { config.noTextArea = false; return crowbar; }
  crowbarClass textAreaSwitch(boolean flag) { setTextAreaSwitch(flag);   return crowbar; }
  // オーバーラップ表示（折り返し）の抑制と許可
  // (a) 現在の値を返す
  boolean overlapSwitch() { return config.noOverlap; }
  // (b) 設定（＋設定前の値を返す）
  boolean setOverlapSwitch(boolean flag) { 
    boolean oldflag = overlapSwitch();
    config.noOverlap = flag;
    return oldflag;
  }
  // (c) マルチステートメント
  crowbarClass disableOverlap()            { config.noOverlap = true;  return crowbar; }
  crowbarClass enableOverlap()             { config.noOverlap = false; return crowbar; }
  crowbarClass overlapSwitch(boolean flag) { setOverlapSwitch(flag);   return crowbar; }
  // 半角英数字出力時の文字幅を0.5に固定するか
  // (a) 現在の値を返す
  boolean proportionalFontSwitch() { return config.proportional; }
  // (b) 設定（＋設定前の値を返す）
  boolean setProportionalFontSwitch(boolean flag) {
    boolean oldflag = proportionalFontSwitch();
    if (!crowbar.checkOptionsSetupOnly("setProportionalFontSwitch()", null, false)) config.proportional = flag;
    return oldflag;
  }
  // (c) マルチステートメント
  crowbarClass enableProportionalFont() {
    if (!crowbar.checkOptionsSetupOnly("enableProportionalFont()", null, false)) config.proportional = true;
    return crowbar;
  }
  crowbarClass disableProportionalFont(){
    if (!crowbar.checkOptionsSetupOnly("disableProportionalFont()", null, false)) config.proportional = false;
    return crowbar;
  }
  crowbarClass proportionalFontSwitch(boolean flag) {
    if (!crowbar.checkOptionsSetupOnly("setProportionalFont()", null, false)) proportionalFontSwitch(flag);
    return crowbar;
  }
  // （主にTomahawkで）角度の単位を度とするか
  // (a) 現在の値を返す
  boolean angleModeSwitch()  { return config.degreeBase; }
  // (b) 設定（＋設定前の値を返す）
  boolean setAngleModeSwitch(boolean flag) {
    boolean oldflag = angleModeSwitch();
    config.degreeBase = flag;
    return oldflag;
  }
  // (c) マルチステートメント
  crowbarClass degreeAngleMode()             { config.degreeBase = true;  return crowbar; }
  crowbarClass radianAngleMode()             { config.degreeBase = false; return crowbar; }
  crowbarClass angleModeSwitch(boolean flag) { setAngleModeSwitch(flag);  return crowbar; }

  // 【テキスト出力のログファイルへの出力】
  // Main() と Draw()実行中のcrowbar.write()/writeln()をログに記録
  // ログファイルの記録開始：Options()の中で宣言するのが無難だが，
  // それMain()やDraw()の中で宣言しても構わない．
  // ファイルへの追記は仕様上不可能なようなので，現状では未対応
  // 既に記録開始していてクローズされていない場合は，現在記録中のログファイルを閉じて新しく記録を開始する
  // 1) ログファイルのデフォルトファイル名を変更する
  String defaultFilename() { return logging.setDefaultFilename();  }
  String setDefaultFilename(String fname) { 
    if (fname.length() == 0) return defaultFilename();
    return logging.setDefaultFilename(fname);
  }
  crowbarClass defaultFilename(String fname) { defaultFilename(fname); return crowbar; }
  // 2-1) ログファイルのオープン（引数無し：デフォルトのファイル名で保存）
  crowbarClass startLogging() { return startLogging(""); }
  // 2-2) ログファイルのオープン（ファイル名指定あり）
  crowbarClass startLogging(String fname) {
    logging.setFileName(fname);
    println("ログファイルを作成します：" + logging.logFileName);
    if (logging.startLogging() == null) crowbar.errorMessage("エラー", "startLogging()にてログファイルの作成に失敗しました．", logging.logFileName, false);
    return crowbar;
  }
  PrintWriter getLogfile() { return logging.getLogfile(); }
  // 3) ログファイルのクローズ（明示しないでもcrowbar終了時に閉じてくれる）
  crowbarClass stopLogging()    { logging.stopLogging();    return crowbar; }
  // 4-1) ログ記録の一時中断
  crowbarClass pauseLogging()   { logging.pauseLogging();   return crowbar; }
  // 4-2) ログ記録の再開
  crowbarClass restartLogging() { logging.restartLogging(); return crowbar; }
  // 5) ログファイルにのみprint()
  crowbarClass print2Log(String str)   { logging.print2Log(str); return crowbar; }
  // 6) ログファイルにのみprintln()
  crowbarClass println2Log(String str) { logging.println2Log(str); return crowbar; }
  // ログファイルにのみnewline()
  crowbarClass newline2Log() { logging.newline2Log(); return crowbar; }

  // 以下はユーザが用いてはいけない（システム用）
  void enableLogOutput()     { enableLogging = true; return; }
  // ログファイル出力を許可
  void disableLogOutput()    { enableLogging = false; return; }
  // ログファイルの出力を不許可
  boolean isEnableLogging()  { return logging.isEnableLogging(); }

  // --------------- キーボード入力関連 -------------------
  // 以下の関数や変数をユーザは利用しないで下さい
  //
  private String keyBuffer = "";
  boolean inputNow = false;    // エコーバックあり
  boolean inputFixed = false;  // ENTERが入力されたらtrue
  // バッファクリア  
  void flushKeyBuffer() {
    keyBuffer = "";
    inputFixed = false;
  }
  // 文字列長を返す
  int keyBufferLength() { return keyBuffer.length(); }
  // コンソールからの入力を開始
  void startInput() {
    inputFixed = false;
    inputNow   = true;
  }
  // キーボード入力完了
  void inputFinish() {
    inputFixed = true;
    inputNow   = false;
  }
  // 文字列の追加
  void strcat(String str) { keyBuffer += str; }
  // キーバッファーを読み出す
  String getKeyBuffer() {
    String buf;
    buf = keyBuffer;  // インスタンスのクローンが作成されるのなら良いけれど．
    keyBuffer = "";
    inputNow   = false;
    inputFixed = false;
    return buf;  // 文字列データのみを返さなくて良いのか？
  }

  // ------------------------------------
  // Draw()中のリアルタイム文字列入力
  // ------------------------------------
  // (1) 有効/無効の切り替え
  // (1-1) 有効化
  crowbarClass inputEnable(int lines)  {
    if (!crowbar.checkOptionsSetupNot("inputEnable()", null, false)) input.enable(lines);
    return crowbar;
  }
  crowbarClass inputEnable() { return inputEnable(1); }
  // (1-2) 無効化
  crowbarClass inputDisable() {
    if (!crowbar.checkOptionsSetupNot("inputEnable()", null, false)) input.disable();
    return crowbar;
  }
  // (2) バッファのクリア
  crowbarClass inputFlush() { input.flush(); return crowbar; }
  // (3) 文字列の取得
  // (3-1) 取得＆バッファクリア
  String gets() { return input.gets(true); }
  // (3-2) 取得のみ
  String echo() { return input.gets(false); }
  // (4) 入力確定（ENTER）？
  int inputFinished() { return input.finished; }
}

//////////////////////////////
// パラメータの最小単位のクラス
//////////////////////////////
class arguments {
  boolean ready;    // 入力済みか
  String  label;    // パラメータ名称．省略時はnull
  String  message;  // テキスト画面に表示するメッセージ
  String  value;    // キーボードから入力されたデータ（文字列）
  String  initialValue;   // ""の場合は初期値の指定なしと判断
  String [] selectValues; // nullの場合は選択肢タイプではないと判断（設定時は空白で区切られた文字列）
  char    parameterType;  // 'A': Normal, 'N': 数値のみ, 'S': 選択肢
  // 配列型用
  int         arrayType; // 0: false, 1:一次元エンドレス, 2:一次元即値, 3:一次元ラベル, 4:二次元（即値，即値）, 5:二次元（即値, ラベル), 6:二次元（ラベル, 即値）, 7:二次元（ラベル, ラベル）
  int         index1;  // エンドレス入力時は初期値0からスタートして入力確定とともにインクリメント．確定してreadyをtrueへ
  int         index2;  // 一次元時は-1のまま
  String [][] arrayValue;
  boolean     endless;
  int         labelIndex1;  // ラベルによる間接指定がされていない場合は -1
  int         labelIndex2;
  // コンストラクタ
  arguments() {
    label   = null;
    message = value = initialValue = "";
    selectValues  = null;
    parameterType = 'A';
    arrayType = 0;
    index1 = index2 = -1;
    endless = false;
    labelIndex1 = labelIndex2 = -1;
    resetArgument();
  }
  void resetArgument() {
    ready   = false;
    if (arrayType == 1) index2 = 0;
    arrayValue = null;
  }
  // 値を設定（標準）
  void set(String str) {
    value = str;
    ready = true;
  }
  // 値を設定（配列）
  boolean setArray(int idx, String str) {
    if ((arrayValue == null) || (arrayType == 0)) return false;
    if (idx > index1) return false;
    arrayValue[0][idx] = str;
    if ((index2 == 0) && (index1 == idx)) ready = true;
    return true;
  }
  boolean setArray(int idx1, int idx2, String str) {
    if ((arrayValue == null) || (arrayType == 0)) return false;
    if ((idx1 == index1) && (idx2 == index2)) ready = true;
    arrayValue[idx1][idx2] = str;
    return true;
  }
  // ラベル名の設定
  void setLabel(String str) { label = str.toUpperCase(); }
}

class frontendClass extends consoleClass { 
  arguments [] argv;                    // パラメータ群（ユーザにより対話的に入力される）
  int          argc;                    // パラメータの数
  protected int obtainArgIndex = 0;     // パラメータのコメント宣言時の自動登録用カウンタ
  protected int readArgIndex = 0;       // パラメータ自動読み出し用カウンタ
  protected boolean autoInput = false;  // TABキーを押すとそれ以降のパラメータ入力では初期値を採用
  protected int obtainArrayIndex1, obtainArrayIndex2;  // 配列入力時のインデックス

  // コンストラクタ
  frontendClass(int x, int y, int mode) {
    // consoleClassのコンストラクタを呼ぶ
    super(x, y, mode);
    // パラメータの初期化
    argc = 0;
    argv = (arguments [])new arguments[1]; // 常に一つ余分に作成
    argv[0] = new arguments();
    resetArrayIndex();
  }
  // パラメータの値のセット
  void setValue(String str) { argv[obtainArgIndex].set(str); }
  // キー入力ルール（数値のみ）
  boolean is_numbersOnly() {
    if (argv[obtainArgIndex].parameterType == 'N') return true;
    return false;
  }
  boolean is_integerOnly() {
    if (argv[obtainArgIndex].parameterType == 'I') return true;
    return false;
  }
  boolean is_booleanOnly() {
    if (argv[obtainArgIndex].parameterType == 'B') return true;
    return false;
  }
  // キー入力ルール（選択肢）
  // 選択肢型ならば選択肢数（最大１０）を返す
  // 選択肢型ではないならば０を返す
  // 異常時は99を返す
  int is_selectOnly() {
    arguments arg;
    arg = argv[obtainArgIndex];
    if (arg.parameterType == 'S') {
      // チェック１
      if (arg.selectValues == null) {
        crowbar.errorMessage("エラー", "選択肢型なのに選択肢がセットされていません", arg.message, true);
        return 99;
      }
      // チェック２
      if (arg.selectValues.length > 10) {
        crowbar.errorMessage("エラー", "選択肢型の選択肢の数が１０個より多い", arg.message, true);
        return 99;
      }
      return arg.selectValues.length;
    }
    return 0;
  }
  // 選択肢型の選択されたパラメータ値を返す
  String get_selectValue(int index) { return argv[obtainArgIndex].selectValues[index]; }
  // 初期値を取得
  String get_initialValue()         { return argv[obtainArgIndex].initialValue;        }
  ///////////////
  // キー入力判定
  // 数字入力用キーか？
  boolean isKeyIntegers(char c) {
    if (isNumberChar(c)) return true;
    switch (c) {
    case '+' :
    case '-' :
      return true;
    }
    return false;
  }
  boolean isKeyNumbers(char c) {
    if (isKeyIntegers(c)) return true;
    if (c == '.')         return true;
    return false;
  }
  boolean isKeyBoolean(char c) {
    switch (c) {
      case '0':
      case '1':
      case 'Y':
      case 'y':
      case 'N':
      case 'n':
      case 'T':
      case 't':
      case 'F':
      case 'f': return true;
    }
    return false;
  }
  // 全てのパラメータが入力完了して，Main()を実行可能か？
  boolean allReady() {
    int i;
    // パラメータ入力が指定されていない
    if (argv == null) return true;
    // パラメータ入力が指定されている
    for (i = 0; i < argc; i++) {
      if (argv[i].ready != true) return false;
    }
    return true;
  }
  // 全ての入力完了フラグをリセット（再入力時用）
  void allResetArguments() {
    int i;
    for (i = 0; i < argv.length; i++) argv[i].resetArgument(); // argv[i].ready = false;
    readArgIndex   = 0;
    obtainArgIndex = 0;
    resetArrayIndex();
  }
  // パラメータが一つも指定されていない場合にtrue：余計なシステムメッセージの出力を抑制するため
  boolean isNoParameter() {
    if (argc == 0) return true;
    return false;
  }
  // 配列入力関係
  // arrayType; // 0: false, 1:一次元エンドレス, 2:一次元即値, 3:一次元ラベル, 4:二次元（即値，即値）, 5:二次元（即値, ラベル), 6:二次元（ラベル, 即値）, 7:二次元（ラベル, ラベル）
  void resetArrayIndex() { obtainArrayIndex1 = obtainArrayIndex2 = 0; }
  boolean setArrayValue(String data) {
    arguments param = argv[obtainArgIndex];
    if (param.arrayType == 1) {
      // 空の場合は入力完了
      if (data.length() == 0){
        param.ready = true;
        return true;
      }
      // エンドレス時は先に拡張（expand()は二次元に対応しているのか不明なので←調べろよ）
      String [][] newArray = new String [1][++param.index2];
      for (int i = 0; i < param.index2 - 1; i++) newArray[0][i] = param.arrayValue[0][i];
      param.arrayValue = newArray;
    }
    param.arrayValue[obtainArrayIndex1][obtainArrayIndex2] = data;
    if (param.arrayType == 1) {
      obtainArrayIndex2++;  // エンドレス時
    } else {
      if (++obtainArrayIndex2 >= param.index2) {
        obtainArrayIndex2 = 0;
        if (++obtainArrayIndex1 >= param.index1) {
          obtainArrayIndex1 = 0;
          param.ready = true;
        }
      }
    }
    return true;
  }
  void initArrayInput() {
    arguments param = argv[obtainArgIndex];
    if ((param.index1 >= 0) && (param.index2 >= 0)) return;
    // ラベル指定時の値の読み出し
    switch (param.arrayType) {
      case 0: return; // 配列では無い
      case 1: return; // エンドレス時は自動で拡張
      case 2:
      case 4: break;  // ラベルなし
      case 3:
      case 6:
      case 7:
        if ((param.index1 = getInt(param.labelIndex1)) < 0) {
          crowbar.errorMessage("実行時エラー", "配列の大きさとして入力された値が0以下です", "array()", true);
          return;
        }
        if ((param.arrayType == 3) || (param.arrayType == 6)) break;
      case 5:
        if ((param.index2 = getInt(param.labelIndex2)) < 0) {
          crowbar.errorMessage("実行時エラー", "配列の大きさとして入力された値が0以下です", "array()", true);
          return;
        }
    }
    param.arrayValue = new String [param.index1][param.index2];
  }

  // ------------------------------
  // パラメータ説明の取得
  // detail : 詳細な情報（型指定）を表示する場合はtrue
  String get_paramMessage(int index, boolean detail) {
    String    message;
    arguments arg;

    // インデックスの範囲のチェック
    if ((index >= argc) || (index < 0)) {
      crowbar.errorMessage("エラー", "get系命令でパラメータ読み出しのIndexが範囲を超えています", "Index=" + str(index), true);
      return "";
    }
    // 説明の取得
    arg     = argv[index];
    // 全てのタイプに共通
    message = arg.message;
    // 配列の場合
    // arrayType; // 0: false, 1:一次元エンドレス, 2:一次元即値, 3:一次元ラベル, 4:二次元（即値，即値）, 5:二次元（即値, ラベル), 6:二次元（ラベル, 即値）, 7:二次元（ラベル, ラベル）
    switch (arg.arrayType) {
      case 1: case 2: case 3:
        message += "(" + str(crowbar.obtainArrayIndex2) + ")"; break;
      case 4: case 5: case 6: case 7:
        message += "(" + str(crowbar.obtainArrayIndex1) + "," + str(crowbar.obtainArrayIndex2) + ")"; break;
    }
    if (!detail) return message;
    // 以下，オプションによってメッセージを追加
    // 初期値が与えられている場合
    if (arg.initialValue.length() > 0) {
      if (!crowbar.run.silentMode) message += "（" + arg.initialValue + "）";
        else                       message += "（初期値：" + arg.initialValue + "）";
    }
    // パラメータの種類によって
    if (arg.parameterType == 'N') {
      // 数字のみ
      if (!crowbar.run.silentMode) message += "／数字のみ";
    } else if (arg.parameterType == 'S') {
      // 選択肢型
      int i;
      message += "／（";
      for (i = 0; i < arg.selectValues.length; i++) {
        if (i > 0) message += ",";
        message += str(i) + ":" + arg.selectValues[i];
      }
      message += "）";
    }
    return message;
  }
  // 初期値の自動入力モード（trueで自動入力）
  void automaticInputByDefalutValue(boolean flag) { autoInput = flag; }
  // --------------------------------------------
  // パラメータの宣言（画面に表示するメッセージ）
  // --------------------------------------------
  // Ver.4 から
  // パラメータの宣言
  crowbarClass param(String str) {
    if (!crowbar.checkOptionsSetupOnly("defineStr()", str, true)) {
      argc = argv.length;
      argv = (arguments [])expand(argv, argc + 1);
      argv[argc] = new arguments();
      // 現在の値をセット
      argv[argc - 1].message = str;
    }
    return crowbar;
  }
  // (1) 型指定
  // (1a) 浮動小数点型
  crowbarClass setFloat() {
    if (crowbar.checkOptionsSetupOnly("setFloat()", null, true)) return crowbar;
    if (argc <= 0) crowbar.errorMessage("エラー", "パラメータの宣言（param()）が行なわれる前に型指定を使用しないで下さい", "setFloat()", true);
    else argv[argc - 1].parameterType = 'N';
    return crowbar;
  }
  crowbarClass setFloat(float init) {
    setFloat();
    if (argc <= 0) return crowbar;
    argv[argc - 1].initialValue = str(init);
    return crowbar;
  }
  crowbarClass setFloat(int init) { return setFloat(float(init)); }
  // (1b) 整数型
  crowbarClass setInt() {
    if (crowbar.checkOptionsSetupOnly("setInt()", null, true)) return crowbar;
    if (argc <= 0) crowbar.errorMessage("エラー", "パラメータの宣言（param()）が行なわれる前に型指定を使用しないで下さい", "setInt()", true);
    else argv[argc - 1].parameterType = 'I';
    return crowbar;
  }
  crowbarClass setInt(int init) {
    setInt();
    if (argc <= 0) return crowbar;
    argv[argc - 1].initialValue = str(init);
    return crowbar;
  }
  crowbarClass setInt(float init) { return setInt(int(init)); }
  // (1c) 論理型
  crowbarClass setBoolean() {
    if (crowbar.checkOptionsSetupOnly("setBoolean()", null, true)) return crowbar;
    if (argc <= 0) crowbar.errorMessage("エラー", "パラメータの宣言（param()）が行なわれる前に型指定を使用しないで下さい", "setBoolean()", true);
    else argv[argc - 1].parameterType = 'B';
    return crowbar;
  }
  crowbarClass setBoolean(boolean init) {
    setBoolean();
    if (argc <= 0) return crowbar;
    argv[argc - 1].initialValue = str(init);
    return crowbar;
  }
  crowbarClass setBoolean(int init)    { return setBoolean(boolean(init)); }
  crowbarClass setBoolean(float init)  { return setBoolean(int(init)); }
  crowbarClass setBoolean(String init) {
    int val = -1;
    init = init.toUpperCase();
    if (init.equals("TRUE"))       val = 1;
    else if (init.equals("FALSE")) val = 0;
    else if (init.length() == 1) {
      switch (init.charAt(0)) {
        case '1':
        case 'T':
        case 'Y': val = 1; break;
        case '0':
        case 'F':
        case 'N': val = 0; break;
      }
    }
    if (val < 0) crowbar.errorMessage("エラー", "論理型(setBoolean()）の初期値の指定が正しくありません", init, true);
    else {
      if (val == 1) setBoolean(true);
      else          setBoolean(false);
    }
    return crowbar;
  }
  // (1d) 多肢選択型
  crowbarClass setSelect(String str) {
    if (crowbar.checkOptionsSetupOnly("setSelect()", null, true)) return crowbar;
    if (argc <= 0) crowbar.errorMessage("エラー", "パラメータの宣言（param()）が行なわれる前に型指定を使用しないで下さい", "setSelect()", true);
    String list[];
    if (str.length() <= 0) {
      crowbar.errorMessage("エラー", "setSelect()で選択肢型のパラメータ宣言の第２パラメータが空文字列です", str, true);
      return crowbar;
    }
    list = split(str, ' ');
    if (list.length > 10) {
      crowbar.errorMessage("エラー", "setSelect()で選択肢型のパラメータ宣言の選択肢の数が１０個を超えています", str, true);
      return crowbar;
    }
    argv[argc - 1].parameterType = 'S';
    argv[argc - 1].selectValues = list;
    return crowbar;
  }
  crowbarClass setSelect(String str, int num) {
    setSelect(str);
    if ((num < 0) || (num > 9)) {
      crowbar.errorMessage("エラー", "setSelect()で選択肢型の初期値が0から9の整数ではありません", str, true);
      return crowbar;
    }
    if (num >= argv[argc - 1].selectValues.length) {
      crowbar.errorMessage("エラー", "setSelect()で選択肢型の初期値が0から選択肢の数-1の整数ではありません", str, true);
      return crowbar;
    }
    argv[argc - 1].initialValue = str(num);
    return crowbar;
  }
  // (1e) 文字列型
  crowbarClass setStr() {
    if (crowbar.checkOptionsSetupOnly("setStr()", null, true)) return crowbar;
    if (argc <= 0) crowbar.errorMessage("エラー", "パラメータの宣言（param()）が行なわれる前に型指定を使用しないで下さい", "setStr()", true);
    else argv[argc - 1].parameterType = 'S';
    return crowbar;
  }
  crowbarClass setStr(String init) {
    setStr();
    if (argc <= 0) return crowbar;
    argv[argc - 1].initialValue = init;
    return crowbar;
  }
  crowbarClass setStr(float init) { return setStr(str(init)); }
  crowbarClass setStr(int init)   { return setStr(str(init)); }

  // (2) ラベル名の追加
  crowbarClass label(String str) {
    if (crowbar.checkOptionsSetupOnly("label()", str, true)) return crowbar;
    if (argc - 1 < 0) {
      crowbar.errorMessage("エラー", "パラメータの宣言（param()）が行なわれる前にlabel()を使用しないで下さい", str, true);
      return crowbar;
    }
    if (checkLabel(str) >= 0) {
      crowbar.errorMessage("エラー", "label()において指定されたラベルと同じパラメータが既に作成済みです（重複）", str, true);
      return crowbar;
    }
    argv[argc - 1].setLabel(str);
    return crowbar;
  }
  // ラベル名をインデックス指定で読み出す(インデックスの自動更新はしない）
  String getLabel(int index) {
    if ((index >= argc) || (index < 0)) return null;
    return argv[index].label;
  }

  // (3) 配列型の追加
  //  int  arrayType; // 0: false, 1:一次元エンドレス, 2:一次元即値, 3:一次元ラベル
  // (3a) 一次元配列（エンドレス入力）
  crowbarClass array() {
    arguments param = argv[argc - 1];
    if (crowbar.checkOptionsSetupOnly("array()", null, true)) return crowbar;
    if (argc <= 0) {
      crowbar.errorMessage("エラー", "パラメータの宣言（param()）が行なわれる前に配列型指定を使用しないで下さい", "array()", true);
      return crowbar;
    }
    param.arrayType = 1;
    param.endless = true;
    param.index2 = 0;
    param.arrayValue = new String[1][1];
    return crowbar;
  }
  // (3b) 一次元配列（即値指定）
  crowbarClass array(int num) {
    arguments param = argv[argc - 1];
    array();
    if (num <= 0) {
      crowbar.errorMessage("エラー", "配列数は0より大きな数字を指定して下さい", "array()", true);
      return crowbar;
    }
    param.arrayType = 2;
    param.endless = false;
    param.index1 = 1;
    param.index2 = num;
    return crowbar;
  }
  // (3c) 一次元配列（ラベル指定）
  crowbarClass array(String label) {
    arguments param = argv[argc - 1];
    array();
    if ((param.labelIndex1 = checkLabel(label)) < 0) {
      crowbar.errorMessage("エラー", "array()において指定されたラベルが宣言されていません", label, true);
      return crowbar;
    }
    param.arrayType = 3;
    param.index1 = 1;
    param.endless = false;
    return crowbar;
  }
  //  int  arrayType; // 4:二次元（即値，即値）, 5:二次元（即値, ラベル), 6:二次元（ラベル, 即値）, 7:二次元（ラベル, ラベル）
  // (3d) 二次元配列（即値, 即値）
  crowbarClass array(int num1, int num2) {
    arguments param = argv[argc - 1];
    array();
    if ((num1 <= 0) || (num2 <= 0)) {
      crowbar.errorMessage("エラー", "配列数は0より大きな数字を指定して下さい", "array()", true);
      return crowbar;
    }
    param.arrayType = 4;
    param.endless = false;
    param.index1 = num1;
    param.index2 = num2;
    return crowbar;
  }
  // (3e) 二次元配列（即値, ラベル）
  crowbarClass array(int num, String label) {
    arguments param = argv[argc - 1];
    array();
    if (num <= 0) {
      crowbar.errorMessage("エラー", "配列数は0より大きな数字を指定して下さい", "array()", true);
      return crowbar;
    } else if ((param.labelIndex2 = checkLabel(label)) < 0) {
      crowbar.errorMessage("エラー", "array()において指定されたラベルが宣言されていません", label, true);
      return crowbar;
    }
    param.arrayType = 5;
    param.endless = false;
    param.index1 = num;
    return crowbar;
  }
  // (3f) 二次元配列（ラベル, 即値）
  crowbarClass array(String label, int num) {
    arguments param = argv[argc - 1];
    array();
    if ((param.labelIndex1 = checkLabel(label)) < 0) {
      crowbar.errorMessage("エラー", "array()において指定されたラベルが宣言されていません", label, true);
      return crowbar;
    } else if (num <= 0) {
      crowbar.errorMessage("エラー", "配列数は0より大きな数字を指定して下さい", "array()", true);
      return crowbar;
    }
    param.arrayType = 6;
    param.endless = false;
    param.index2 = num;
    return crowbar;
  }
  // (3e) 二次元配列（ラベル, ラベル）
  crowbarClass array(String label1, String label2) {
    arguments param = argv[argc - 1];
    array();
    if ((param.labelIndex1 = checkLabel(label1)) < 0) {
      crowbar.errorMessage("エラー", "array()において指定されたラベルが宣言されていません", label1, true);
      return crowbar;
    } else if ((param.labelIndex2 = checkLabel(label2)) < 0) {
      crowbar.errorMessage("エラー", "array()において指定されたラベルが宣言されていません", label2, true);
      return crowbar;
    }
    param.arrayType = 7;
    param.endless = false;
    return crowbar;
  }

  // Ver.3までの互換（削除予定）
  // (1) 数字のみ（浮動小数点）
  // (1a) 標準
  crowbarClass define(String str) {
    if (!crowbar.checkOptionsSetupOnly("define()", str, true)) {
      defineStr(str);
      argv[argc - 1].parameterType = 'N';
    }
    return crowbar;
  }
  // (1b) 初期値あり
  // (1b-1) 文字列
  crowbarClass define(String str1, String str2) {
    if (!crowbar.checkOptionsSetupOnly("define()", str1, true)) {
      defineStr(str1, str2);
      argv[argc - 1].parameterType = 'N';
    }
    return crowbar;
  }
  // (1b-2) 浮動小数点
  crowbarClass define(String str, float f) { return define(str, str(f)); }
  // (1b-3) 整数
  crowbarClass define(String str, int i)   { return define(str, str(i)); }

  // (2) 入力制限規則なし ：全てのdefine系は最終的にこの関数を呼ぶ
  crowbarClass defineStr(String str) {
    if (!crowbar.checkOptionsSetupOnly("defineStr()", str, true)) {
      argc = argv.length;
      argv = (arguments [])expand(argv, argc + 1);
      argv[argc] = new arguments();
      // 現在の値をセット
      argv[argc - 1].message = str;
    }
    return crowbar;
  }
  // パラメータの宣言（入力制限規則なし：初期値あり）
  crowbarClass defineStr(String str1, String str2) {
    defineStr(str1);
    argv[argc - 1].initialValue = str2;
    return crowbar;
  }
  // パラメータの宣言（入力制限規則なし：初期値あり，float渡し）
  crowbarClass defineStr(String str, float f) { return define(str, str(f)); }
  // パラメータの宣言（入力制限規則なし：初期値あり，int渡し）
  crowbarClass defineStr(String str, int i)   { return define(str, str(i)); }
  // パラメータの宣言（選択肢型：入力制限規則は0-9のみ）
  crowbarClass defineSel(String str1, String str2) {
    if (crowbar.checkOptionsSetupOnly("defineSel()", str1, true)) return crowbar;
    String list[];
    if (str2.length() <= 0) {
      crowbar.errorMessage("エラー", "defineSel()で選択肢型のパラメータ宣言の第２パラメータが空文字列です", str1, true);
      return crowbar;
    }
    list = split(str2, ' ');
    if (list.length > 10) {
      crowbar.errorMessage("エラー", "defineSel()で選択肢型のパラメータ宣言の選択肢の数が１０個を超えています", str1, true);
      return crowbar;
    }
    defineStr(str1);
    argv[argc - 1].parameterType = 'S';
    argv[argc - 1].selectValues = list;
    return crowbar;
  }
  // パラメータの宣言（選択型：初期値あり）
  crowbarClass defineSel(String str1, String str2, int num) {
    defineSel(str1, str2);
    if (num < 0) {
      crowbar.errorMessage("エラー", "defineSel()で選択肢型の初期値が0から9の整数ではありません", str1, true);
      return crowbar;
    }
    if (num >= argv[argc - 1].selectValues.length) {
      crowbar.errorMessage("エラー", "defineSel()で選択肢型の初期値が0から選択肢の数-1の整数ではありません", str1, true);
      return crowbar;
    }
    argv[argc - 1].initialValue = str(num);
    return crowbar;
  }

  //-------------------------
  // パラメータの値を受け取る
  //-------------------------
  // 1) indexを明示的に指定する
  // 1-a) パラメータを文字列で返す
  String getStr(int index) {
    if (argv[index].arrayType != 0) {
      crowbar.errorMessage("エラー", "配列型パラメータを読み出す時はArray型のget命令を使用して下さい", "", true);
      return "";
    }
    if ((index >= argc) || (index < 0)) {
      crowbar.errorMessage("エラー", "get系命令においてパラメータ読み出しのIndexが範囲を超えています", "Index=" + str(index), true);
      return "";
    }
    // インデックスの自動更新
    readArgIndex = index + 1;
    return argv[index].value;
  }
  // arrayType; // 0: false, 1:一次元エンドレス, 2:一次元即値, 3:一次元ラベル, 4:二次元（即値，即値）, 5:二次元（即値, ラベル), 6:二次元（ラベル, 即値）, 7:二次元（ラベル, ラベル）
  // 一次元配列
  String [] getStrArray(int index) {
    if ((index >= argc) || (index < 0)) {
      crowbar.errorMessage("エラー", "get系命令においてパラメータ読み出しのIndexが範囲を超えています", "Index=" + str(index), true);
      return null;
    }
    // 配列型のチェック
    switch (argv[index].arrayType) {
      case 1: case 2: case 3: break;
      default :
      crowbar.errorMessage("エラー", "指定された配列は一次元配列ではありません", "Index=" + str(index), true);
      return null;
    }
    // インデックスの自動更新
    readArgIndex = index + 1;
    return argv[index].arrayValue[0];
  }
  // 二次元配列
  String [][] getStrArray2(int index) {
    if ((index >= argc) || (index < 0)) {
      crowbar.errorMessage("エラー", "get系命令においてパラメータ読み出しのIndexが範囲を超えています", "Index=" + str(index), true);
      return null;
    }
    // 配列型のチェック
    switch (argv[index].arrayType) {
      case 4: case 5: case 6: case 7: break;
      default :
      crowbar.errorMessage("エラー", "指定された配列は二次元配列ではありません", "Index=" + str(index), true);
      return null;
    }
    // インデックスの自動更新
    readArgIndex = index + 1;
    return argv[index].arrayValue;
  }
  // 1-b) パラメータをfloatで返す
  float getFloat(int index)     { return float(getStr(index));   }
  float [] getFloatArray(int index) {
    String [] ptr = getStrArray(index);
    if (ptr == null) return null;
    float [] ret = new float [argv[index].index2];
    for (int i = 0; i < ptr.length; i++) {
      ret[i] = float(ptr[i]);
    }
    return ret;
  }
  float [][] getFloatArray2(int index) {
    String [][] ptr = getStrArray2(index);
    if (ptr == null) return null;
    float [][] ret = new float [argv[index].index1][argv[index].index2];
    for (int i = 0; i < ptr.length; i++) {
      for (int j = 0; j < ptr[0].length; j++) {
        ret[i][j] = float(ptr[i][j]);
      }
    }
    return ret;
  }
  // 1-c) パラメータをintで返す
  int getInt(int index)         { return int(getStr(index));     }
  int [] getIntArray(int index) {
    String [] ptr = getStrArray(index);
    if (ptr == null) return null;
    int [] ret = new int [argv[index].index2];
    for (int i = 0; i < ptr.length; i++) {
      ret[i] = int(ptr[i]);
    }
    return ret;
  }
  int [][] getIntArray2(int index) {
    String [][] ptr = getStrArray2(index);
    if (ptr == null) return null;
    int [][] ret = new int [argv[index].index1][argv[index].index2];
    for (int i = 0; i < ptr.length; i++) {
      for (int j = 0; j < ptr[0].length; j++) {
        ret[i][j] = int(ptr[i][j]);
      }
    }
    return ret;
  }
  // 1-d) パラメータをbooleanで返す
  boolean getBoolean(int index) { return boolean(getStr(index)); }
  boolean [] getBooleanArray(int index) {
    String [] ptr = getStrArray(index);
    if (ptr == null) return null;
    boolean [] ret = new boolean [argv[index].index2];
    for (int i = 0; i < ptr.length; i++) {
      ret[i] = boolean(ptr[i]);
    }
    return ret;
  }
  boolean [][] getBooleanArray2(int index) {
    String [][] ptr = getStrArray2(index);
    if (ptr == null) return null;
    boolean [][] ret = new boolean [argv[index].index1][argv[index].index2];
    for (int i = 0; i < ptr.length; i++) {
      for (int j = 0; j < ptr[0].length; j++) {
        ret[i][j] = boolean(ptr[i][j]);
      }
    }
    return ret;
  }

  // 2) indexを明示的に指定しないで順番に自動的に読み出す
  // 2-a) パラメータを文字列で渡す（自動）
  String getStr()            { return getStr(readArgIndex); }
  String [] getStrArray()    { return getStrArray(readArgIndex); }
  String [][] getStrArray2() { return getStrArray2(readArgIndex); }
  // 2-b) パラメータをfloatで返す（自動）
  float getFloat()            { return getFloat(readArgIndex); }
  float [] getFloatArray()    { return getFloatArray(readArgIndex); }
  float [][] getFloatArray2() { return getFloatArray2(readArgIndex); }
  // 2-c) パラメータをintで返す（自動）
  int getInt()            { return getInt(readArgIndex); }
  int [] getIntArray()    { return getIntArray(readArgIndex); }
  int [][] getIntArray2() { return getIntArray2(readArgIndex); }
  // 2-d) パラメータをbooleanで返す（自動）
  boolean getBoolean()            { return getBoolean(readArgIndex); }
  boolean [] getBooleanArray()    { return getBooleanArray(readArgIndex); }
  boolean [][] getBooleanArray2() { return getBooleanArray2(readArgIndex); }
  
  // 3) ラベル名を明示的に指定する
  // ラベル名からインデックス番号を得る（存在しない場合は -1）
  int checkLabel(String str) {
    String label = str.toUpperCase();
    for (int i = 0; i < argc; i++) {
      if (argv[i].label == null) continue;
      if (argv[i].label.equals(label) == true) return i;
    }
    return -1;
  }
  int checkLabel4get(String label) {
    int index;
    if ((index = checkLabel(label)) < 0) crowbar.errorMessage("エラー", "get系命令において指定されたラベルを持つパラメータ宣言が存在しません", label, true);
    return index;
  }
  // 3-a) パラメータを文字列で返す
  String getStr(String label) {
    int index;
    if ((index = checkLabel4get(label)) < 0) return "";
    return getStr(index);
  }
  String [] getStrArray(String label) {
    int index;
    if ((index = checkLabel4get(label)) < 0) return null;
    return getStrArray(index);
  }
  String [][] getStrArray2(String label) {
    int index;
    if ((index = checkLabel4get(label)) < 0) return null;
    return getStrArray2(index);
  }
  // 3-b) パラメータをfloatで返す
  float getFloat(String label)     { return float(getStr(label));   }
  float [] getFloatArray(String label) {
    int index;
    if ((index = checkLabel4get(label)) < 0) return null;
    return getFloatArray(index);
  }
  float [][] getFloatArray2(String label) {
    int index;
    if ((index = checkLabel4get(label)) < 0) return null;
    return getFloatArray2(index);
  }
  // 3-c) パラメータをintで返す
  int getInt(String label)         { return int(getStr(label));     }
  int [] getIntArray(String label) {
    int index;
    if ((index = checkLabel4get(label)) < 0) return null;
    return getIntArray(index);
  }
  int [][] getIntArray2(String label) {
    int index;
    if ((index = checkLabel4get(label)) < 0) return null;
    return getIntArray2(index);
  }
  // 3-d) パラメータをbooleanで返す
  boolean getBoolean(String label) { return boolean(getStr(label)); }
  boolean [] getBooleanArray(String label) {
    int index;
    if ((index = checkLabel4get(label)) < 0) return null;
    return getBooleanArray(index);
  }
  boolean [][] getBooleanArray2(String label) {
    int index;
    if ((index = checkLabel4get(label)) < 0) return null;
    return getBooleanArray2(index);
  }
}

// Options()で設定したパラメータのバックアップと復元
// 別にOptions()以外でも変更可能なので，二回目以降の実行時にOptions()で設定した値を戻さないと挙動が変わってしまうため
class runningParameterClass {
  protected boolean noShowParams;    // 入力済みのパラメータを確認のために表示するか？
  protected boolean nonStop;         // Main()終了後はdraw()の終端に記述したコードを繰り返し実行する．nonStop()でtrue/stop()でfalse
  private boolean   userDrawLoop;    // Draw()のループ実行を停止したり再開したりする際に用いる．trueで繰り返し実行．loop(), noLoop()で設定．
  private int       nonStopCounter;  // Drawの実行回数（-1で無制限，0で実行しない）
  protected boolean silentMode;      // システムメッセージ表示を最小限に抑える
  // コンストラクタ
  runningParameterClass() {
    noShowParams   = nonStop = false;
    userDrawLoop   = true;
    nonStopCounter = -1;
    silentMode     = false;
  }
  void clone(runningParameterClass src) {
    noShowParams   = src.noShowParams;
    nonStop        = src.nonStop;
    userDrawLoop   = src.userDrawLoop;
    nonStopCounter = src.nonStopCounter;
    silentMode     = src.silentMode;
  }
}

//////////////////////////////////////////////////////////////////////////////
// プロセスコントロールのクラス（キーボード入力とコンソール出力のクラスを継承）
//////////////////////////////////////////////////////////////////////////////
class crowbarClass extends frontendClass {
  int                   status = -1;              // プロセスの状態を管理
  String []             comment;                  // プログラムのコメント．パラメータ入力前に表示される．
  protected boolean     halt;                     // 強制終了フラグ（halt()でセットされる）
  boolean               running       = false;    // Main()あるいはcrowbar.nonStop()指定時にユーザプログラム実行中はtrue
  boolean               afterRunning  = false;    // Main()が終わった後，テキストバッファを再描画可能な間のみtrue;（Draw()が動いていてもそれは気にしない．今のところ）
  runningParameterClass run, _run;
  // コンストラクタ
  crowbarClass(int x, int y, int mode) {
    super(x, y, mode);
    // プログラムコメントの初期化
    comment = null;
    // 強制終了フラグ
    halt    = false;
    // 実行状態制御用フラグ類の確保（バックアップ含む）
    run  = new runningParameterClass();
    _run = new runningParameterClass();
    // 動作に関わるスイッチのバックアップの初期化
    backupRunningFlag();
  }
  // Options()かSetup()以外では指定してはいけない命令をチェックしエラーを表示する 
  // エラー時にtrue
  boolean checkOptionsSetupOnly(String command, String a, boolean halt) {
    if (crowbar.status > 0) {
      errorMessage("エラー", command + "はOptions()かSetup()で使用して下さい", a, halt);
      return true;
    }
    return false;
  }
  // Options()やSetup()で指定してはいけない命令をチェックしエラーを表示する 
  // エラー時にtrue
  boolean checkOptionsSetupNot(String command, String a, boolean halt) {
    if (crowbar.status <= 0) {
      errorMessage("エラー", command + "はOptions()やSetup()で使用してはいけません", a, halt);
      return true;
    }
    return false;
  }
  // エラーメッセージを出力し，必要ならばhaltのフラグを立てる
  void errorMessage(String head, String body, String tail, boolean halt) {
    print("[" + head + "] " + body);
    if ((tail != null) && (tail.length() > 0)) print("（" + tail + "）");
    println("．");
    if (halt) crowbar.halt();
    return;
  }

  // 動作に関わるスイッチのバックアップ
  void backupRunningFlag()  { _run.clone(run); }
  // 動作に関わるスイッチのリストア
  void restoreRunningFlag() { run.clone(_run); }

  // 以下の関数をユーザは使用しないで下さい
  // 入力済みの全パラメータを表示する
  crowbarClass showAllParameters() {
    int i, _readArgIndex;
    // インデックスのバックアップ
    _readArgIndex = readArgIndex;
    if (run.noShowParams) return this; // 念のため
    for (i = 0; i < argc; i++) {
      if (!argv[i].ready) break;  // 入力済みの分のみ表示するので．
/*
      write(get_paramMessage(i, false) + "：");
      if (getLabel(i) != null) write("(" + getLabel(i) + ") = ");
      writeln("[" + getStr(i) + "]");
*/
      // 配列に配慮
      // arrayType; // 0: false, 1:一次元エンドレス, 2:一次元即値, 3:一次元ラベル, 4:二次元（即値，即値）, 5:二次元（即値, ラベル), 6:二次元（ラベル, 即値）, 7:二次元（ラベル, ラベル）
      switch (argv[i].arrayType) {
        case 0: writeln(get_paramMessage(i, false) + " = " + getStr(i));  break;
        case 1: case 2: case 3:
          String [] str1 = getStrArray(i);
          for (obtainArrayIndex2 = 0; obtainArrayIndex2 < str1.length; obtainArrayIndex2++) writeln(get_paramMessage(i, false) + " = " + str1[obtainArrayIndex2]);
          break; 
        case 4: case 5: case 6: case 7:
          String [][] str2 = getStrArray2(i);
          for (obtainArrayIndex1 = 0; obtainArrayIndex1 < str2.length; obtainArrayIndex1++) {
            for (obtainArrayIndex2 = 0; obtainArrayIndex2 < str2[0].length; obtainArrayIndex2++) writeln(get_paramMessage(i, false) + " = " + str2[obtainArrayIndex1][obtainArrayIndex2]);
          }
          break;
      }
    }
    // インデックスのリストア
    readArgIndex = _readArgIndex;
    return this;
  }
  // 入力済みの全パラメータをログファイルにのみ出力
  crowbarClass showAllParameters2Log() {
    boolean _display, _textarea;
    // フラグのバックアップ
    _display  = displaySwitch();
    _textarea = textAreaSwitch();
    // テキスト出力を一時的に禁止
    disableDisplay();
    disableTextArea();
    // 禁止した上で出力すればログにのみ出力される
    showAllParameters();
    // フラグを戻す
    if (!_display)  enableDisplay();
    if (!_textarea) enableTextArea();
    return this;
  }

  // -----------------------------------------------
  // パラメータ表示の確認表示の抑制（Options()で指定）
  crowbarClass noShowParameters() {
    if (!crowbar.checkOptionsSetupOnly("noShowParameters()", null, false)) run.noShowParams = true;
    return this;
  }
  crowbarClass showParameters()   {  // 必要性は無い
    if (!crowbar.checkOptionsSetupOnly("showParameters()", null, false))   run.noShowParams = false;
    return this;
  }
  // ----------------------------------------
  // システムメッセージ表示を最小限に抑制する
  crowbarClass silentMode()       {
    if (!checkOptionsSetupOnly("silentMode()", null, false)) run.silentMode = true;
    return this;
  }
  // システムメッセージの文字色を変更する
  crowbarClass systemColor(int color1, int color2, int color3) {
    if (checkOptionsSetupOnly("systemColor()", null, false)) return this;
    systemColor1(color1);
    systemColor2(color2);
    systemColor3(color3);
    return this;
  }
  // システム文字色を変更する
  crowbarClass systemColor1(int textcolor) {
    if (!crowbar.checkOptionsSetupOnly("systemColor1()", null, false)) _SYSTEMCOLOR1 = textcolor;
    return crowbar;
  }
  crowbarClass systemColor2(int textcolor) {
    if (!crowbar.checkOptionsSetupOnly("systemColor2()", null, false)) _SYSTEMCOLOR2 = textcolor;
    return crowbar;
  }
  crowbarClass systemColor3(int textcolor) {
    if (!crowbar.checkOptionsSetupOnly("systemColor3()", null, false)) _SYSTEMCOLOR3 = textcolor;
    return crowbar;
  }

  // ----------------------------------------------------------------------------
  // Main()終了後にdraw()終端に記述したコードを繰り返し実行するためのコントロール
  // ----------------------------------------------------------------------------
  crowbarClass nonStop() { run.nonStop = true; return this; }  // Draw()関数が有効
  crowbarClass stop()    {                         // Draw()関数が終了したらプログラム終了
    run.nonStop        = false; 
    run.nonStopCounter = -1;
    return this;
  }  
  crowbarClass nonStop(int count) {
    nonStop();
    run.nonStopCounter = count;
    return this;
  }
  // nonStopCounterを-1してカウントアップしたらtrue
  boolean checkNonStopCounter() {
    if (run.nonStopCounter == -1) return false;  // 設定されていない
    if (--run.nonStopCounter > 0) return false;  // まだまだ
    run.nonStopCounter = -1;
    return true;
  }

  // ------------------------------------------------------
  // Draw()のループ実行の停止と再開のためのコントロール
  // ------------------------------------------------------
  crowbarClass loop()   { run.userDrawLoop = true;  return this; }
  crowbarClass noLoop() { run.userDrawLoop = false; return this; }
  
  // ----------------------------------------
  // Crowbarを安全に強制終了する
  // ただし無限ループからの脱出は原理的に無理
  // ----------------------------------------
  crowbarClass halt() {
    halt   = true;
//  status = 90;
    return this;
  }

  // ----------------------------------------------
  // プログラムの説明（パラメータ入力前に表示される
  // ----------------------------------------------
  // プログラムコメントを表示する
  void outputProgramComments() {
    int i, commNum;
    commNum = comment.length;
    for (i = 0; i < commNum; i++) writeln(comment[i]);
  }
  // プログラムコメントを宣言（複数宣言可能）．setup()内でユーザが宣言する．
  crowbarClass programComment(String str) {
    checkOptionsSetupOnly("programComment()", null, false);
    String addStr[];
    addStr    = new String[1];
    addStr[0] = str;
    if (comment == null) {
      // 一個目のコメントの場合
      comment = (String [])new String[1];
      comment = addStr;
    } else {
      // 二個目以降のコメントの場合
      comment = concat(comment, addStr);
    }
    return this;
  }
  // ---------------------------------------------------------------
  // フォントサイズを変更する（Options(), Setup()の中で使用すること）
  // ---------------------------------------------------------------
  // (a) 現在の値を返す
  int fontSize() { return currentAttr.fontSize; }
  // (b) 設定（＋変更前の値を返す）
  int setFontSize(int size) {
    if (checkOptionsSetupOnly("setFontSize()", str(size), false)) return this.fontSize();
    int oldSize;
    oldSize = currentAttr.fontSize;
    _setFontSize(size);
    return oldSize;
  }
  // (c) マルチステートメント
  crowbarClass fontSize(int size) {
    if (!checkOptionsSetupOnly("fontSize()", str(size), false)) this.setFontSize(size);
    return this;
  }
  // -------------------------------------------------------------------
  // 自動スクロールの設定を変更する：Options(), Setup()の中で使用すること
  // -------------------------------------------------------------------
  // 自動スクロールの行数指定
  int autoScrollLines() { return autoScrollLines; }
  int setAutoScrollLines(int lines) {
    int oldlines = autoScrollLines();
//  if (status > 0) return oldlines;  // Main()実行前以外は変更を認めない（認めても良いかな）
    // 範囲チェック
    if (lines < 0)                     lines = 0;
    if (lines > screenSetting.cheight) lines = 0;
    autoScrollLines = lines;
    return oldlines;
  }
  crowbarClass autoScrollLines(int lines) { setAutoScrollLines(lines); return this; }
  float autoScrollPage() { return (autoScrollLines / screenSetting.cheight); }
  float setAutoScrollPage(float rate) {
    float oldrate = autoScrollPage();
//  if (status > 0) return oldrate;  // Main()実行前以外は変更を認めない（認めても良いかな）
    // 範囲チェック
    if (rate < 0.0) rate = 0.0;
    if (rate > 1.0) rate = 1.0;
    autoScrollLines(int(screenSetting.cheight * rate));
    return oldrate;
  }
  crowbarClass autoScrollPage(float rate) { setAutoScrollPage(rate); return this; }
  // ------------------------------------------------------
  // 文字送り，行送りを変更する（Options()の中で使用すること）
  // ------------------------------------------------------
  // (a) 現在の値を返す
  float hStep() { return screenSetting.hStep; }
  // (b) 設定（＋変更前の値を返す）
  float setHStep(float step) {
    float oldStep = hStep();
    if (checkOptionsSetupOnly("setHStep()", str(step), false)) return oldStep;
    if (step <= 0.0) return oldStep;
    screenSetting.hStep = step;
    screenSetting.reCalc(fontSize());
    screenReset();
    return oldStep;
  }
  // (c) マルチステートメント
  crowbarClass hStep(float step) {
    if (!checkOptionsSetupOnly("hStep()", str(step), false)) setHStep(step);
    return this; 
  }
  // (a) 現在の値を返す
  float vStep() { return screenSetting.vStep; }
  // (b) 設定（＋変更前の値を返す）
  float setVStep(float step) {
    float oldStep = vStep();
    if (checkOptionsSetupOnly("setVStep()", str(step), false)) return oldStep;
    if (step <= 0.0) return oldStep;
    screenSetting.vStep = step;
    screenSetting.reCalc(fontSize());
    screenReset();
    return oldStep;
  }
  crowbarClass vStep(float step) {
    if (!checkOptionsSetupOnly("vStep()", str(step), false)) setVStep(step);
    return this;
  }
  // プロセスコントローラ内での手動スクロール許可
  void enableManualScroll() {
    stopRecordTextBuffer();
    textbuffer.flushBuffer(!crowbar.overlapSwitch());
    afterRunning = true;
  }
  // プロセスコントローラ内での手動スクロールの禁止
  void disableManualScroll() {
    stopRecordTextBuffer();
    redrawNewestPage();
    afterRunning = false;
    startRecordTextBuffer();
  }
}

void keyPressed() {
  if (crowbar.running == false) {
    // ユーザプログラムが動作していない時（パラメータ入力中など）
    if (crowbar.inputNow) {
      // パラメータ入力中
      if (key == ENTER) {
        if (crowbar.is_numbersOnly() || (crowbar.is_selectOnly() != 0)) {
          // 数値入力と選択型（つまり文字列入力時以外）の場合は空入力は拒否
          if ((crowbar.keyBufferLength() == 0) && (crowbar.get_initialValue().length() == 0)) {
            redraw();
            return;
          }
        }
        crowbar.inputFinish();
        crowbar.newline();
      } else if (key == BACKSPACE) {
        if (crowbar.keyBufferLength() > 0) crowbar.backSpace();
      } else if (key == TAB) {
        if (crowbar.get_initialValue().length() > 0) {
          crowbar.automaticInputByDefalutValue(true);
          crowbar.inputFinish();
          crowbar.newline();
        }
      } else {
        // パラメータの入力
        boolean flag = true;
        int     num;
        // キー入力ルール
        // 数値のみの場合
        if (crowbar.is_numbersOnly()) {
          if (!crowbar.isKeyNumbers(key)) flag = false;
            else                          flag = crowbar.keyCheck.checkFloat(key);
        }
        // 整数のみの場合
        if (crowbar.is_integerOnly()) {
          if (!crowbar.isKeyIntegers(key)) flag = false;
            else                           flag = crowbar.keyCheck.checkInteger(key);
        }
        // 論理型のみの場合
        if (crowbar.is_booleanOnly()) {
          if (!crowbar.isKeyBoolean(key)) flag = false;
            else                          flag = crowbar.keyCheck.checkBoolean(key);
        }
        // 選択肢の場合
        if ((num = crowbar.is_selectOnly()) > 0) {
          if (num == 99) {  // 異常検出
            crowbar.halt();
            return;
          }
          if (crowbar.keyBufferLength() >= 1) flag = false; // 既に１文字入力済み（半角空白も含む）
          if (!isNumberChar(key))             flag = false; // 数字キー（0-9）ではない
          if (int(str(key)) >= num)           flag = false; // 選択肢の数よりも大きな数字
        }
        if (flag) {
          crowbar.strcat(str(key));
          crowbar.write(str(key));
        }
      }
    } else {
      // 強制的に抜けたい場合
      if (getKEY() == 'Q') crowbar.status = 99;
    }
    // スクロール可能な状態（プログラムが一旦終了してキー入力を待っている間のみ）
    if (crowbar.afterRunning) {
      if (key == '-')      crowbar.pageScroll(-0.5);
      else if (key == '+') crowbar.pageScroll(0.5);
      else if (key == CODED) {
        if (keyCode == UP)         crowbar.pageLineScroll(-1);
        else if (keyCode == DOWN)  crowbar.pageLineScroll(1);
        else if (keyCode == LEFT)  crowbar.pageScroll(-0.5);
        else if (keyCode == RIGHT) crowbar.pageScroll(0.5);
      }
    }
    redraw();
  } else {
    // Draw()実行時のリアルタイム文字列入力
    if (crowbar.input.isEnable()) crowbar.input.add();
    // crowbar.nonStop()指定時のユーザプログラム実行中のキー入力判定
    // コードはKeyPressed()に記述すると良い．
    KeyPressed();
  }
}

///////////////
// プロセス管理
///////////////
void draw() {
  // halt()発生時の処理（Ver.4より改善）
  if (crowbar.halt) {
    crowbar.status = 90;
    crowbar.halt = false;
  }
  // 状態遷移の管理
  switch (crowbar.status) {
  case -1: return;    // crowbar未定義
  case 0: // 初期状態
    // Setup()中でcrowbar.nonStop()/noShowParams()が実行された場合の対策(Main()などで変更されている恐れがあるため）
//  crowbar.setColor(_SYSTEMCOLOR1, _BGCOLOR);
    crowbar.textColor(_SYSTEMCOLOR1);
    crowbar.backupRunningFlag();
    crowbar.backupConfigFlag();
    crowbar.resetView(false);    // 全ビューポートの視点（変換マトリックス）のバックアップ
    // プログラムコメントの表示
    if (crowbar.comment != null) {
      if (!crowbar.run.silentMode) crowbar.textColor(_SYSTEMCOLOR2).writeln("【プログラム説明】").textColor(_SYSTEMCOLOR1);
      crowbar.outputProgramComments();
      crowbar.newline();
    }
    // Tomahawk関係
    if (crowbar.tomahawkMode != 0) crowbar.backupViewports();
    crowbar.status++;
    break;
  case 1: // パラメータ入力開始（再実行時はここにジャンプしてくる）
    // 初期化
    crowbar.allResetArguments();
    crowbar.automaticInputByDefalutValue(false);
    // setup()中でcrowbar.nonStop()/noShowParams()を設定した値がMain()やDraw()の実行で変更される恐れがあるので復元する
    crowbar.restoreRunningFlag();
    crowbar.restoreConfigFlag();
    // パラメータの入力要求が一つも無かった場合はメッセージを表示しない
    if (!crowbar.isNoParameter()) {
      if (!crowbar.run.silentMode) crowbar.newline().textColor(_SYSTEMCOLOR2).writeln("【パラメータの入力】").textColor(_SYSTEMCOLOR1);
    }
    // Tomahawk関係
    if (crowbar.tomahawkMode != 0) crowbar.restoreViewports();
    // リアルタイム文字列入力の無効化
    crowbar.inputDisable();
    crowbar.status++;
    break;
  case 2: // パラメータ入力開始
    if (!crowbar.isNoParameter()) {
      crowbar.keyCheck.reset();  // キー入力規則のリセット
      crowbar.initArrayInput();  // 配列型の入力前処理
      crowbar.write(crowbar.get_paramMessage(crowbar.obtainArgIndex, true) + ":");
      crowbar.flushKeyBuffer();
      crowbar.startInput();
      // 初期設定値の自動入力モード有効の場合はパスする
      if ((crowbar.autoInput == true) && (crowbar.get_initialValue().length() > 0)) {
        crowbar.inputFinish();
        crowbar.newline();
      } else noLoop();  // ループ停止
    }
    crowbar.status++;
    break;
  case 3: // 入力中
    if (!crowbar.isNoParameter()) {
      String inputs;
      if (crowbar.inputFixed) {
        // 入力確定
        boolean flag = true;
        inputs = crowbar.getKeyBuffer();
        // 空かつ初期値が指定されている場合
        if (inputs.length() == 0) inputs = crowbar.get_initialValue();
        // 整数型の場合
        if (crowbar.is_integerOnly()) {
          if (inputs.length() != 0) inputs = str(int(inputs));
        }
        // 選択肢型の場合
        int  num;
        if ((num = crowbar.is_selectOnly()) > 0) {
          if (num == 99) { // 異常検出
            crowbar.halt();
            break;
          }
          // 念のためにチェック
          if (((num = int(inputs)) < 0) || (num > 9)) flag = false;
          else if (num > crowbar.is_selectOnly())     flag = false;
          if (!flag) {
            // あり得ないはず（入力時にチェックしているので）
            crowbar.errorMessage("実行時エラー", "選択肢型の入力値が異常です", str(num), true);
            break;
          }
          inputs = crowbar.get_selectValue(num);
        }
        // 論理型の場合
        if (crowbar.is_booleanOnly()) {
          switch(inputs.charAt(0)) {
            case '1':
            case 'T':
            case 't':
            case 'Y':
            case 'y': inputs = str(true); break;
            case '0':
            case 'F':
            case 'f':
            case 'N':
            case 'n': inputs = str(false); break;
            default:
              crowbar.errorMessage("実行時エラー", "論理型の入力値が異常です", inputs, true);
              break;
          }
        }
        // 配列に配慮
        if (crowbar.argv[crowbar.obtainArgIndex].arrayType == 0) {
          crowbar.setValue(inputs);
          crowbar.obtainArgIndex++;
        } else {
          crowbar.setArrayValue(inputs);
          if (crowbar.argv[crowbar.obtainArgIndex].ready) {
            crowbar.obtainArgIndex++;
            crowbar.resetArrayIndex();
          }
        }
        // ループ再開
        loop();
        crowbar.status++;
      }
    } else crowbar.status++;
    break;
  case 4: // パラメータ入力完了かの判定
    if (!crowbar.isNoParameter()) {
      if (crowbar.allReady()) crowbar.status++;
      else                    crowbar.status -= 2;
    } else crowbar.status++;
    break;
  case 5: // 実行か？ 再入力か？
    if (!crowbar.isNoParameter()) crowbar.newline();
    if (!crowbar.isNoParameter() && !crowbar.run.silentMode) {
      crowbar.textColor(_SYSTEMCOLOR2).writeln("【全てのパラメータの入力が完了】").textColor(_SYSTEMCOLOR1);
      if (!crowbar.run.noShowParams) crowbar.showAllParameters();
      crowbar.textColor(_SYSTEMCOLOR3).write("プログラムを実行する(Yes) /パラメータを入力し直す(Retry) : ").textColor(_SYSTEMCOLOR1);
      // 手動スクロールの許可
      crowbar.enableManualScroll();
      noLoop();
    }
    crowbar.status++;
    break;
  case 6: // キー判定結果
    if (!crowbar.isNoParameter() && !crowbar.run.silentMode) {
      if (keyPressed) {
        if (getKEY() == 'Y') {
          crowbar.disableManualScroll(); // 手動スクロールの禁止
          crowbar.writeln(str(key)).newline();
          crowbar.status++;
          loop();
        } else if (getKEY() == 'R') {
          crowbar.disableManualScroll(); // 手動スクロールの禁止
          crowbar.writeln(str(key)).newline();
          crowbar.status -= 5;
          loop();
        }
      }
    } else crowbar.status++;
    break;
  case 7: // メインプログラム実行前処理（あるならば）
    preMain(); // optionCodeタブ参照
    if (crowbar.halt) break;
    crowbar.running = true;
    crowbar.status++;
    break;
  case 8: // ログファイルへの出力を許可する:これはもしかしたらstatus==0から許可するかも知れない．余計な文字情報は記録しないという仕様はお節介だったかもという反省．
    crowbar.enableLogOutput();
    if (crowbar.isEnableLogging()) {
      // 全パラメータを出力する
      crowbar.restartLogging().newline2Log().showAllParameters2Log().newline2Log();
    }
    crowbar.status++;
    break;
  case 9: // メインプログラム実行
    Main();
    loop();  // ループ再開
    if (crowbar.halt) break;
    // リアルタイム文字列入力の初期化
    crowbar.inputFlush();
    crowbar.status++;
    break;
  case 10: // 繰り返し実行コード（オプション）
    if (crowbar.run.nonStop) {
      // crowbar.nonStop()指定時の繰り返し処理はDraw()に記述
      if (crowbar.run.userDrawLoop) {
        crowbar.resetView(true);  // 全ビューポートのrotate(), scale(), translate()の再設定
        Draw();
        if (crowbar.checkNonStopCounter()) {  // もしnonStopCounterが-1ではなく，-1してカウントアップしたならば終了
          crowbar.stop();
          crowbar.status++;
        } 
      }
      break;
    }
    // リアルタイム文字列入力の停止と初期化
    crowbar.inputFlush();
    crowbar.inputDisable();
    crowbar.status++;
    break;
  case 11: // ログファイル出力を一時的に止める
    crowbar.running = false;
    if (crowbar.isEnableLogging()) crowbar.pauseLogging();
    crowbar.status++;
    break;
  case 12: // メインプログラム実行後処理（あるならば）
    postMain(); // optionCodeタブ参照
    crowbar.status++;
    break;
  case 13: // リトライ確認メッセージ
    // テキストバッファ関連
//  crowbar.textbuffer.flushBuffer(!crowbar.overlapSwitch());
//  crowbar.stopRecordTextBuffer();
    crowbar.textbuffer.flushBuffer(!crowbar.overlapSwitch());
//  crowbar.setColor(_SYSTEMCOLOR2, _BGCOLOR).newline();
    crowbar.textColor(_SYSTEMCOLOR2).newline();
    if (!crowbar.run.silentMode) crowbar.writeln("【プログラム終了】").textColor(_SYSTEMCOLOR3).write("もう一度，実行しますか？(Yes/No)：");
    else                         crowbar.textColor(_SYSTEMCOLOR3).write("再実行(Y/N)：");
    crowbar.textColor(_SYSTEMCOLOR1);
    crowbar.enableManualScroll();
    crowbar.status++;
    noLoop();
    break;
  case 14: // キー判定結果
    if (keyPressed) {
      int K = getKEY();
      if ((K == 'Y') || (K == 'N')) {
        crowbar.writeln(str(key)).newline();
        // テキストバッファ―関連
        crowbar.disableManualScroll();
        if (K == 'Y') crowbar.status = 1;
        else          crowbar.status++;
        loop();
      }
    }
    break;
  case 15: // 終了
    crowbar.status = 99;
    break;
  case 90 : // crowbar.halt()で飛んできた
    crowbar.running = false;
    crowbar.afterRunning = true;
    crowbar.newline().textColor(_SYSTEMCOLOR2).writeln("【重大なエラーが発生しました】：コンソール領域のエラーメッセージをよく確認して下さい．");
    crowbar.textColor(_SYSTEMCOLOR3).writeln("[Q]キーを押すとCrowbarを終了します：");
    crowbar.stopRecordTextBuffer();
    crowbar.status++;
    noLoop();
    break;
  case 91 : // キー入力待ち
    if (keyPressed) {
      if (getKEY() == 'Q') {
        crowbar.status = 99;
        loop();
      }
    }
    break;
  case 99: // 終了処理
    println();
    // ログファイルを開いたままなら閉じる
    if (crowbar.isEnableLogging()) {
      println("ファイルを閉じます．");
      crowbar.restartLogging().newline2Log().stopLogging();
      crowbar.disableLogOutput();
    }
    println("Crowbarを終了します．");
    exit();
    break;
  default:
    if ((crowbar.status > 90) && (crowbar.halt)) crowbar.status = 99;  // crowbar.halt()で戻って来たのに crowbar.statusが++された場合などにキャッチ．90じゃなくて15でも良いのだが．
  }
  crowbar.refreshView();  // Tomahawkのビュー合成
}

void mouseDragged() {
  if (crowbar.running) {
    crowbar.resetView(true);
    MouseDragged();
  }
}
void mouseMoved() {
  if (crowbar.running) {
    crowbar.resetView(true);
    MouseMoved();
  }
}
void mousePressed() {
  if (crowbar.running) {
    crowbar.resetView(true);
    MousePressed();
  }
}
void mouseReleased() {
  if (crowbar.running) {
    crowbar.resetView(true);
    MouseReleased();
  }
}
void mouseClicked() {
  if (crowbar.running) {
    crowbar.resetView(true);
    MouseClicked();
  }
}
void keyReleased() {
  if (crowbar.running) {
    crowbar.resetView(true);
    KeyReleased();
  }
}
void keyTyped() {
  if (crowbar.running) {
    crowbar.resetView(true);
    KeyTyped();
  }
}

// システム初期設定
void setup()
{
  initCrowbar();
  Options();
  Setup();
  crowbar.clrscr();
  crowbar.status = 0;  // 準備完了 （この手続きは不要なのだが気持ちの問題）
}

