(*
 * Copyright (C) 2002,2004 Jun Kikuchi <kikuchi@bonnou.com>
 *
 * This file is part of ÍobO.
 *
 * ÍobO is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * ÍobO is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 *)
unit KeyUnit;

interface

uses
  Classes, gcryptUnit, PacketUnit;

const
  ECHECK_NG    = 'L[̓e`FbNŃG[܂';
  EFILE_HEADER = 'Íꂽt@C̏񂪎擾o܂ł';
  EKEYID       = 'L[Ⴂ܂';
  EDECRYPT     = 'Íf[^̕Ɏs܂';

type
  TLiteralPacketHeader = class;

  TSecKey = class
  private
    FSecKey: TSecKeyPacket;
    FUserID: TUserIDPacket;
    FCertSign: TCertSignPacket;
    FSecSubkey: TSecSubkeyPacket;
    FSubkeySign: TSubkeySignPacket;
    FOnProgress: TNotifyEvent;
  protected
    function CompKeyID(A, B: TKeyID): Boolean;
  public
    constructor Create;
    destructor Destroy; override;
    procedure GenNewKey(KeyLen: Integer; UserID, PW: string);
    function Lock(PW: string): Boolean;
    function Unlock(PW: string): Boolean;
    function IsLocked: Boolean;
    function CheckKey: Boolean;
    procedure SavePubKeyToStream(S: TStream);
    procedure SaveToStream(S: TStream);
    procedure LoadFromStream(S: TStream);
    procedure DecodeFileHeader(I: TStream; LH: TLiteralPacketHeader);
    procedure DecodeFile(I, O: TStream; LH: TLiteralPacketHeader);
    property SecKey: TSecKeyPacket read FSecKey;
    property UserID: TUserIDPacket read FUserID;
    property CertSign: TCertSignPacket read FCertSign;
    property SecSubKey: TSecSubkeyPacket read FSecSubkey;
    property SubkeySign: TSubkeySignPacket read FSubkeySign;
    property OnProgress: TNotifyEvent read FOnProgress write FOnProgress;
  end;

  TPubKey = class
  private
    FPubKey: TPubKeyPacket;
    FUserID: TUserIDPacket;
    FCertSign: TCertSignPacket;
    FPubSubkey: TPubSubkeyPacket;
    FSubkeySign: TSubkeySignPacket;
    FOnProgress: TNotifyEvent;
  public
    constructor Create;
    destructor Destroy; override;
    function CheckKey: Boolean;
    procedure SaveToStream(S: TStream);
    procedure LoadFromStream(S: TStream);
    procedure EncodeFile(I, O: TStream; LH: TLiteralPacketHeader);
    property PubKey: TPubKeyPacket read FPubKey;
    property UserID: TUserIDPacket read FUserID;
    property CertSign: TCertSignPacket read FCertSign;
    property PubSubkey: TPubSubkeyPacket read FPubSubkey;
    property SubkeySign: TSubkeySignPacket read FSubkeySign;
    property OnProgress: TNotifyEvent read FOnProgress write FOnProgress;
  end;

  FILE_MODES = (
    FILE_MODE_BIN  = $62,
    FILE_MODE_TEXT = $74
  );

  TLiteralPacketHeader = class
  private
    FMode: FILE_MODES;
    FFileName: string;
    FDate: TDateTime;
  public
    property Mode: FILE_MODES read FMode write FMode;
    property FileName: string read FFileName write FFileName;
    property Date: TDateTime read FDate write FDate;
  end;

implementation

uses
  PacketIOUnit, PacketPipeUnit;
  
function TSecKey.CompKeyID(A, B: TKeyID): Boolean;
var
  I: Integer;
begin
  Result := False;

  if High(A) <> High(B) then
    Exit;

  for I := 0 to High(A) do
  begin
    if A[I] <> B[I] then
      Exit;
  end;

  Result := True;
end;

constructor TSecKey.Create;
begin
  inherited;
  FSecKey := TSecKeyPacket.Create;
  FUserID := TUserIDPacket.Create;
  FCertSign := TCertSignPacket.Create;
  FSecSubkey := TSecSubkeyPacket.Create;
  FSubkeySign := TSubkeySignPacket.Create;
  FOnProgress := nil;
end;

destructor TSecKey.Destroy;
begin
  FSubkeySign.Free;
  FSecSubkey.Free;
  FCertSign.Free;
  FUserID.Free;
  FSecKey.Free;
  inherited;
end;

procedure TSecKey.GenNewKey(KeyLen: Integer; UserID, PW: string);
var
  RetCode: Boolean;
begin
  FSecKey.PubKeyAlgo := GCRY_PK_DSA;
  FSecKey.DEK.CipherAlgo := GCRY_CIPHER_CAST5;
  FSecKey.DEK.S2K.HashAlgo := GCRY_MD_SHA1;
  FSecKey.GenNewKey(KeyLen, PW);
  RetCode := FSecKey.Unlock(PW);
  Assert(RetCode = True);

  FUserID.UID := UserID;

  FCertSign.HashAlgo := GCRY_MD_SHA1;
  SetLength(FCertSign.SymKeyAlgoList, 3);
  FCertSign.SymKeyAlgoList[0] := GCRY_CIPHER_AES;
  FCertSign.SymKeyAlgoList[1] := GCRY_CIPHER_CAST5;
  FCertSign.SymKeyAlgoList[2] := GCRY_CIPHER_3DES;
  SetLength(FCertSign.HashAlgoList, 2);
  FCertSign.HashAlgoList[0] := GCRY_MD_SHA1;
  FCertSign.HashAlgoList[1] := GCRY_MD_RMD160;
  SetLength(FCertSign.CompAlgoList, 2);
  FCertSign.CompAlgoList[0] := COMP_ZLIB;
  FCertSign.CompAlgoList[1] := COMP_ZIP;
  SetLength(FCertSign.FeatureList, 1);
  FCertSign.FeatureList[0] := FEAT_MODIF_DETECT;

  FSecKey.Sign(FCertSign, FUserID);

  FSecSubkey.PubKeyAlgo := GCRY_PK_ELG_E;
  FSecSubkey.DEK.CipherAlgo := GCRY_CIPHER_CAST5;
  FSecSubkey.DEK.S2K.HashAlgo := GCRY_MD_SHA1;
  FSecSubkey.GenNewKey(KeyLen, PW);
  RetCode := FSecSubkey.Unlock(PW);
  Assert(RetCode = True);
  FSecSubkey.Lock(PW);

  FSubkeySign.HashAlgo := GCRY_MD_SHA1;

  FSecKey.Sign(FSubkeySign, TPubSubkeyPacket(FSecSubkey));
  FSecKey.Lock(PW);
end;

function TSecKey.Lock(PW: string): Boolean;
begin
  Result := False;
  if FSecKey.Lock(PW) then
    Result := FSecSubkey.Lock(PW);
end;

function TSecKey.Unlock(PW: string): Boolean;
begin
  Result := False;
  if FSecKey.Unlock(PW) then
    Result := FSecSubkey.Unlock(PW);
end;

function TSecKey.IsLocked: Boolean;
begin
  Result := FSecKey.IsLocked and FSecSubkey.IsLocked;
end;

function TSecKey.CheckKey: Boolean;
begin
  Result := FSecKey.Verify(FCertSign, FUserID);
  if Result then
    Result := FSecKey.Verify(FSubkeySign, TPubSubkeyPacket(FSecSubkey));
end;

procedure TSecKey.SaveToStream(S: TStream);
var
  PIO: TPacketIO;
begin
  PIO := TPacketIO.Create;
  try
    PIO.Save(S, FSecKey);
    PIO.Save(S, FUserID);
    PIO.Save(S, FCertSign);
    PIO.Save(S, FSecSubkey);
    PIO.Save(S, FSubkeySign);
  finally
    PIO.Free;
  end;
end;

procedure TSecKey.SavePubKeyToStream(S: TStream);
var
  PIO: TPacketIO;
begin
  PIO := TPacketIO.Create;
  try
    PIO.Save(S, TPubKeyPacket(FSecKey));
    PIO.Save(S, FUserID);
    PIO.Save(S, FCertSign);
    PIO.Save(S, TPubSubkeyPacket(FSecSubkey));
    PIO.Save(S, FSubkeySign);
  finally
    PIO.Free;
  end;
end;

procedure TSecKey.LoadFromStream(S: TStream);
var
  PIO: TPacketIO;
begin
  PIO := TPacketIO.Create;
  try
    PIO.Load(S, FSecKey);
    PIO.Load(S, FUserID);
    PIO.Load(S, FCertSign);
    PIO.Load(S, FSecSubkey);
    PIO.Load(S, FSubkeySign);

    if not CheckKey then
      raise EPacketRead.Create(ECHECK_NG);
  finally
    PIO.Free;
  end;
end;

procedure TSecKey.DecodeFileHeader(I: TStream; LH: TLiteralPacketHeader);
var
  SSK: TSessionKey;
  ESK: TPubKeyEncSessKeyPacket;
  PIO: TPacketIO;
  PD1, PD2, PD3: TPartialPacketDecodePipe;
  EPD: TEncryptPacketsDecodePipe;
  CPD: TCompressPacketDecodePipe;
  LPD: TLiteralPacketDecodePipe;
begin
  Assert(not FSecSubkey.IsLocked);

  SSK := TSessionKey.Create;
  try
    ESK := TPubKeyEncSessKeyPacket.Create;
    try
      PIO := TPacketIO.Create;
      try
        PIO.Load(I, ESK);
      finally
        PIO.Free;
      end;
      if not CompKeyID(FSecSubKey.KeyID, ESK.KeyID) then
        raise EPacketRead.Create(EKEYID);
        
      if not FSecSubkey.Decrypt(SSK, ESK) then
        raise EPacketRead.Create(EDECRYPT);
    finally
      ESK.Free;
    end;

    PD1 := TPartialPacketDecodePipe.Create(I, PACKET_ENCMDC);
    try
      EPD := TEncryptPacketsDecodePipe.Create(PD1, SSK);
      try
        PD2 := TPartialPacketDecodePipe.Create(EPD, PACKET_COMPRESS);
        try
          CPD := TCompressPacketDecodePipe.Create(PD2);
          try
            PD3 := TPartialPacketDecodePipe.Create(CPD, PACKET_LITERAL);
            try
              LPD := TLiteralPacketDecodePipe.Create(PD3, LH);
              try
                while not LPD.ReadHeader do
                begin
                  LPD.DecodeToNull(64);
                  if (I.Position >= I.Size) and (not LPD.ReadHeader) then
                    raise EPacketRead.Create(EFILE_HEADER);
                end;
              finally
                LPD.Free;
              end;
            finally
              PD3.Free;
            end;
          finally
            CPD.Free;
          end;
        finally
          PD2.Free;
        end;
      finally
        EPD.Free;
      end;
    finally
      PD1.Free;
    end;
  finally
    SSK.Free;
  end;
end;

procedure TSecKey.DecodeFile(I, O: TStream; LH: TLiteralPacketHeader);
var
  SSK: TSessionKey;
  ESK: TPubKeyEncSessKeyPacket;
  PIO: TPacketIO;
  PD1, PD2, PD3: TPartialPacketDecodePipe;
  EPD: TEncryptPacketsDecodePipe;
  CPD: TCompressPacketDecodePipe;
  LPD: TLiteralPacketDecodePipe;
begin
  Assert(not FSecSubkey.IsLocked);

  SSK := TSessionKey.Create;
  try
    ESK := TPubKeyEncSessKeyPacket.Create;
    try
      PIO := TPacketIO.Create;
      try
        PIO.Load(I, ESK);
      finally
        PIO.Free;
      end;
      if not CompKeyID(FSecSubKey.KeyID, ESK.KeyID) then
        raise EPacketRead.Create(EKEYID);

      FSecSubkey.Decrypt(SSK, ESK);
    finally
      ESK.Free;
    end;

    PD1 := TPartialPacketDecodePipe.Create(I, PACKET_ENCMDC);
    PD1.OnProgress := OnProgress;
    try
      EPD := TEncryptPacketsDecodePipe.Create(PD1, SSK);
      try
        PD2 := TPartialPacketDecodePipe.Create(EPD, PACKET_COMPRESS);
        try
          CPD := TCompressPacketDecodePipe.Create(PD2);
          try
            PD3 := TPartialPacketDecodePipe.Create(CPD, PACKET_LITERAL);
            try
              LPD := TLiteralPacketDecodePipe.Create(PD3, LH);
              try
                LPD.DecodeTo(O);
              finally
                LPD.Free;
              end;
            finally
              PD3.Free;
            end;
          finally
            CPD.Free;
          end;
        finally
          PD2.Free;
        end;
      finally
        EPD.Free;
      end;
    finally
      PD1.Free;
    end;
  finally
    SSK.Free;
  end;
end;

constructor TPubKey.Create;
begin
  inherited;
  FPubKey := TPubKeyPacket.Create;
  FUserID := TUserIDPacket.Create;
  FCertSign := TCertSignPacket.Create;
  FPubSubkey := TPubSubkeyPacket.Create;
  FSubkeySign := TSubkeySignPacket.Create;
  FOnProgress := nil;
end;

destructor TPubKey.Destroy;
begin
  FSubkeySign.Free;
  FPubSubkey.Free;
  FCertSign.Free;
  FUserID.Free;
  FPubKey.Free;
  inherited;
end;

function TPubKey.CheckKey: Boolean;
begin
  Result := FPubKey.Verify(FCertSign, FUserID);
  if Result then
    Result := FPubKey.Verify(FSubkeySign, FPubSubkey);
end;

procedure TPubKey.SaveToStream(S: TStream);
var
  PIO: TPacketIO;
begin
  PIO := TPacketIO.Create;
  try
    PIO.Save(S, FPubKey);
    PIO.Save(S, FUserID);
    PIO.Save(S, FCertSign);
    PIO.Save(S, FPubSubkey);
    PIO.Save(S, FSubkeySign);
  finally
    PIO.Free;
  end;
end;

procedure TPubKey.LoadFromStream(S: TStream);
var
  PIO: TPacketIO;
begin
  PIO := TPacketIO.Create;
  try
    PIO.Load(S, FPubKey);
    PIO.Load(S, FUserID);
    PIO.Load(S, FCertSign);
    PIO.Load(S, FPubSubkey);
    PIO.Load(S, FSubkeySign);

    if not CheckKey then
      raise EPacketRead.Create(ECHECK_NG);
  finally
    PIO.Free;
  end;
end;

procedure TPubKey.EncodeFile(I, O: TStream; LH: TLiteralPacketHeader);
var
  SSK: TSessionKey;
  ESK: TPubKeyEncSessKeyPacket;
  PIO: TPacketIO;
  PE1, PE2, PE3: TPartialPacketEncodePipe;
  EPE: TEncryptPacketsEncodePipe;
  CPE: TCompressPacketEncodePipe;
  LPE: TLiteralPacketEncodePipe;
begin
  SSK := TSessionKey.Create;
  try
    SSK.CipherAlgo := FCertSign.SymKeyAlgoList[0];
    SSK.GenNewSessionKey;

    ESK := TPubKeyEncSessKeyPacket.Create;
    try
      FPubSubkey.Encrypt(ESK, SSK);

      PIO := TPacketIO.Create;
      try
        PIO.Save(O, ESK);
      finally
        PIO.Free;
      end;
    finally
      ESK.Free;
    end;

    PE1 := TPartialPacketEncodePipe.Create(O, PACKET_ENCMDC);
    try
      EPE := TEncryptPacketsEncodePipe.Create(PE1, SSK);
      try
        PE2 := TPartialPacketEncodePipe.Create(EPE, PACKET_COMPRESS);
        try
          CPE := TCompressPacketEncodePipe.Create(PE2, FCertSign.CompAlgoList[0]);
          try
            PE3 := TPartialPacketEncodePipe.Create(CPE, PACKET_LITERAL);
            try
              LPE := TLiteralPacketEncodePipe.Create(PE3, LH);
              try
                LPE.OnProgress := OnProgress;
                LPE.EncodeFrom(I);
              finally
                LPE.Free;
              end;
            finally
              PE3.Free;
            end;
          finally
            CPE.Free;
          end;
        finally
          PE2.Free;
        end;
      finally
        EPE.Free;
      end;
    finally
      PE1.Free;
    end;

  finally
    SSK.Free;
  end;
end;

end.
