(*
 * Copyright (C) 2002,2004 Jun Kikuchi <kikuchi@bonnou.com>
 *
 * This file is part of ÍobO.
 *
 * ÍobO is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * ÍobO is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 *)
unit LockKeyDataUnit;

interface

uses
  SysUtils, gcryptUnit, KeyDataUnit;

type
  TSalt = array of Byte;

  TS2K = class
  private
    FHashAlgo: GCRY_MD_ALGOS;
    FSalt: TSalt;
    FCount: Integer;    
  public
    constructor Create;
    procedure RandomizeSalt(Level: GCRY_RANDOME_LEVEL = GCRY_STRONG_RANDOM);
    procedure HashPassphrase(var RetKey: array of Byte; PW: string);
    property HashAlgo: GCRY_MD_ALGOS read FHashAlgo write FHashAlgo;
    property Salt: TSalt read FSalt write FSalt;
    property Count: Integer read FCount write FCount;
  end;

  TIV = array of Byte;

  TDEK = class
  private
    FCipherAlgo: GCRY_CIPHER_ALGOS;
    FIV: TIV;
    FS2K: TS2K;
  protected
    procedure SetCipherAlgo(Value: GCRY_CIPHER_ALGOS);
  public
    constructor Create;
    destructor Destroy; override;
    procedure RandomizeIV(Level: GCRY_RANDOME_LEVEL = GCRY_STRONG_RANDOM);
    procedure Lock(KD: TKeyData; PW: string);
    function Unlock(KD: TKeyData; PW: string): Boolean;
    property CipherAlgo: GCRY_CIPHER_ALGOS read FCipherAlgo write SetCipherAlgo;
    property IV: TIV read FIV write FIV;
    property S2K: TS2K read FS2K;
  end;

implementation

constructor TS2K.Create;
var
  I: Integer;
begin
  inherited;
  FHashAlgo := GCRY_MD_SHA1;

  SetLength(FSalt, 8);
  for I := 0 to High(FSalt) do
    FSalt[I] := 0;

  FCount := 96; (* 65536 *)
end;

procedure TS2K.RandomizeSalt(Level: GCRY_RANDOME_LEVEL = GCRY_STRONG_RANDOM);
begin
  gcry_randomize(PByte(FSalt), Length(FSalt), Level);
end;

procedure TS2K.HashPassphrase(var RetKey: array of Byte; PW: string);
var
  PreloadZero: Byte;
  I, J, K,
  HashDataLen, HashCount, HashCount1, HashCount2,
  PreloadCount, GenKeyLen, RetKeyLen, DigestLen: Integer;
  HashData: array of Byte;
  MD: GCRY_MD_HD;
  Digest: PByteArray;
  Count: Integer;
begin
  HashDataLen := Length(FSalt) + Length(PW); 
  SetLength(HashData, HashDataLen);

  I := 0;
  for J := 0 to High(FSalt) do
  begin
    HashData[I] := FSalt[J];
    Inc(I);
  end;
  for J := 1 to Length(PW) do
  begin
    HashData[I] := Byte(PW[J]);
    Inc(I);
  end;

  Count := (Word(16) + (FCount and 15)) shl ((FCount shr 4) + 6);
  if Count < HashDataLen then
    HashCount := HashDataLen
  else
    HashCount := Count;

  HashCount1 := HashCount div HashDataLen;
  HashCount2 := HashCount mod HashDataLen;

  MD := gcry_md_open(HashAlgo, GCRY_MD_FLAG_SECURE);
  Assert(MD <> nil);

  PreloadCount := 0;
  GenKeyLen    := 0;
  RetKeyLen    := Length(RetKey);
  DigestLen    := gcry_md_get_algo_dlen(HashAlgo);
  Assert(DigestLen <> 0);

  PreloadZero := 0;

  while GenKeyLen < RetKeyLen do
  begin
    if PreloadCount <> 0 then
    begin
      gcry_md_reset(MD);
      for J := 1 to PreloadCount do
        gcry_md_write(MD, @PreloadZero, 1);
    end;

    for K := 1 to HashCount1 do
    begin
      gcry_md_write(MD, PByte(HashData), HashDataLen);
    end;
    gcry_md_write(MD, PByte(HashData), HashCount2);

    gcry_md_final(MD);

    Digest := gcry_md_read(MD, HashAlgo);
    K := 0;
    while (K < DigestLen) and (GenKeyLen < RetKeyLen) do
    begin
      RetKey[GenKeyLen] := Digest[K];
      Inc(GenKeyLen);
      Inc(K);
    end;
    
    Inc(PreloadCount);
  end;
  gcry_md_close(MD);
end;

procedure TDEK.SetCipherAlgo(Value: GCRY_CIPHER_ALGOS);
var
  BlockSize, I: Integer;
begin
  FCipherAlgo := Value;

  BlockSize := gcry_cipher_get_algo_blklen(FCipherAlgo);
  Assert(BlockSize <> -1);

  SetLength(FIV, BlockSize);
  for I := 0 to High(FIV) do
    FIV[I] := 0;
end;

constructor TDEK.Create;
begin
  inherited;
  CipherAlgo := GCRY_CIPHER_CAST5;
  FS2K := TS2K.Create;
end;

destructor TDEK.Destroy;
begin
  FS2K.Free;
  inherited;
end;

procedure TDEK.RandomizeIV(Level: GCRY_RANDOME_LEVEL = GCRY_STRONG_RANDOM);
begin
  gcry_randomize(PByte(FIV), Length(FIV), Level);
end;

procedure TDEK.Lock(KD: TKeyData; PW: string);
var
  CipherKeyLen: Integer;
  CipherKey: array of Byte;

  CH: GCRY_CIPHER_HD;
  SrcData, DestData, CheckData: array of Byte;
  DataLen: Integer;

  MD: GCRY_MD_HD;
  HashData: PByteArray;

  I, J, K: Integer;

  RC: Integer;
begin
  CipherKeyLen := gcry_cipher_get_algo_keylen(FCipherAlgo);
  Assert(CipherKeyLen <> -1);
  SetLength(CipherKey, CipherKeyLen);
  S2K.HashPassphrase(CipherKey, PW);

  DataLen := 0;
  for I := 0 to KD.NumKey -1 do
    DataLen := DataLen + KD.KeyLen[I];
    
  SetLength(CheckData, 20);
  DataLen := DataLen + 20;

  SetLength(SrcData, DataLen);
  SetLength(DestData, DataLen);

  K := 0;
  for I := 0 to KD.NumKey -1 do
  begin
    for J := 0 to KD.KeyLen[I] -1 do
    begin
      SrcData[K] := KD[I, J];
      Inc(K);
    end;
  end;

  MD := gcry_md_open(GCRY_MD_SHA1, GCRY_MD_FLAG_SECURE);
  Assert(MD <> nil);
  gcry_md_write(MD, PByte(SrcData), Length(SrcData) -20);
  gcry_md_final(MD);
  HashData := gcry_md_read(MD, GCRY_MD_SHA1);
  for I := 0 to High(CheckData) do
    CheckData[I] := HashData[I];
  gcry_md_close(MD);

  J := Length(SrcData) - Length(CheckData);
  for I := 0 to High(CheckData) do
  begin
    SrcData[J] := CheckData[I];
    Inc(J);
  end;

  CH := gcry_cipher_open(FCipherAlgo, GCRY_CIPHER_MODE_CFB, GCRY_CIPHER_ENABLE_SYNC);
  Assert(CH <> nil);
  RC := gcry_cipher_setkey(CH, PByte(CipherKey), Length(CipherKey));
  Assert(RC = 0);
  RC := gcry_cipher_setiv(CH, PByte(FIV), Length(FIV));
  Assert(RC = 0);
  RC := gcry_cipher_encrypt(CH, PByte(DestData), Length(DestData), PByte(SrcData), Length(SrcData));
  Assert(RC = 0);
  gcry_cipher_close(CH);

  for I := 0 to KD.NumKey -1 do
    KD.KeyLen[I] := 0;

  KD.KeyLen[0] := Length(DestData);
  for I := 0 to High(DestData) do
    KD[0, I] := DestData[I];
end;

function TDEK.Unlock(KD: TKeyData; PW: string): Boolean;
var
  CipherKeyLen: Integer;
  CipherKey: array of Byte;

  CH: GCRY_CIPHER_HD;
  SrcData, DestData, CheckData: array of Byte;

  IsCheckOK: Boolean;
  MD: GCRY_MD_HD;
  HashData: PByteArray;

  Bits: Word;
  Bytes: Cardinal;

  I, J, K: Integer;

  RC: Integer;
begin
  CipherKeyLen := gcry_cipher_get_algo_keylen(FCipherAlgo);
  Assert(CipherKeyLen <> -1);
  SetLength(CipherKey, CipherKeyLen);
  S2K.HashPassphrase(CipherKey, PW);

  SetLength(CheckData, 20);

  SetLength(SrcData, KD.KeyLen[0]);
  SetLength(DestData, KD.KeyLen[0]);

  for I := 0 to High(SrcData) do
    SrcData[I] := KD[0, I];

  CH := gcry_cipher_open(FCipherAlgo, GCRY_CIPHER_MODE_CFB, GCRY_CIPHER_ENABLE_SYNC);
  Assert(CH <> nil);
  RC := gcry_cipher_setkey(CH, PByte(CipherKey), Length(CipherKey));
  Assert(RC = 0);
  RC := gcry_cipher_setiv(CH, PByte(FIV), Length(FIV));
  Assert(RC = 0);
  RC := gcry_cipher_decrypt(CH, PByte(DestData), Length(DestData), PByte(SrcData), Length(SrcData));
  Assert(RC = 0);
  gcry_cipher_close(CH);

  J := Length(DestData) - Length(CheckData);
  for I := 0 to High(CheckData) do
  begin
    CheckData[I] := DestData[J];
    Inc(J);
  end;

  MD := gcry_md_open(GCRY_MD_SHA1, GCRY_MD_FLAG_SECURE);
  Assert(MD <> nil);
  gcry_md_write(MD, PByte(DestData), Length(DestData) -20);
  gcry_md_final(MD);
  HashData := gcry_md_read(MD, GCRY_MD_SHA1);

  IsCheckOK := True;
  for I := 0 to High(CheckData) do
  begin
    if CheckData[I] <> HashData[I] then
    begin
      IsCheckOK := False;
      break;
    end;
  end;

  gcry_md_close(MD);

  if IsCheckOK then
  begin
    J := 0;
    for I := 0 to KD.NumKey -1 do
    begin
      Bits := DestData[J];
      Bits := Bits shl 8;
      Bits := Bits or DestData[J+1];

      Bytes := (Bits + 7) div 8;
      KD.KeyLen[I] := Bytes + 2;
      for K := 0 to KD.KeyLen[I] -1 do
      begin
        KD[I, K] := DestData[J];
        Inc(J);
      end;
    end;
  end;

  Result := IsCheckOK;
end;

end.
