(*
 * Copyright (C) 2002,2004 Jun Kikuchi <kikuchi@bonnou.com>
 *
 * This file is part of ÍobO.
 *
 * ÍobO is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * ÍobO is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 *)
unit PacketUnit;

interface

uses
  Types, Classes, SysUtils, gcryptUnit, KeyDataUnit, LockKeyDataUnit;

type
  TFingerprint = array of Byte;
  TKeyID       = array of Byte;
  TMPIList     = array of GCRY_MPI;
  TDigest      = array of Byte;

  TKeyPacket = class
  private
    FVer: Byte;
    FDate: TDateTime;
    FPubKeyAlgo: GCRY_PK_ALGOS;
    FKeyData: TKeyData;
    FKeyID: TKeyID;
  protected
    procedure SetPubKeyAlgo(Value: GCRY_PK_ALGOS); virtual;
    procedure SexpToKeyData(KeyData: TKeyData; Sexp: GCRY_SEXP; TopName, Tokens: string);
    function KeyDataToMPIList: TMPIList;
    procedure ReleaseMPIList(MPIS: TMPIList);
    function HashStream(HashAlgo: GCRY_MD_ALGOS; S: TStream): TDigest;
    function GetStrKeyID: string;
  public
    constructor Create;
    destructor Destroy; override;
    property Ver: Byte read FVer write FVer;
    property Date: TDateTime read FDate write FDate;
    property PubKeyAlgo: GCRY_PK_ALGOS read FPubKeyAlgo write SetPubKeyAlgo;
    property KeyData: TKeyData read FKeyData;
    property KeyID: TKeyID read FKeyID write FKeyID;
    property StrKeyID: string read GetStrKeyID;
  end;

  TPubSubkeyPacket        = class;
  TUserIDPacket           = class;
  TSignPacket             = class;
  TCertSignPacket         = class;
  TSubkeySignPacket       = class;
  TPubKeyEncSessKeyPacket = class;
  TSessionKey             = class;

  TPKCS1 = array of Byte;

  TPubKeyPacket = class(TKeyPacket)
  protected
    procedure SetPubKeyAlgo(Value: GCRY_PK_ALGOS); override;
    function KeyDataToSexp: GCRY_SEXP;
    function HashPackets(UID: TUserIDPacket; Sign: TCertSignPacket): TDigest; overload;
    function HashPackets(Subkey: TPubSubkeyPacket; Sign: TSubkeySignPacket): TDigest; overload;
    function PKCS1Encode(SesKey: TSessionKey): TPKCS1;
  public
    constructor Create;
    function CalcFingerprint: TFingerprint;
    function CalcKeyID: TKeyID;
    function Verify(Sign: TCertSignPacket; UID: TUserIDPacket): Boolean; overload;
    function Verify(Sign: TSubkeySignPacket; Subkey: TPubSubkeyPacket): Boolean; overload;
    procedure Encrypt(ESK: TPubKeyEncSessKeyPacket; SesKey: TSessionKey);
  end;

  TPubSubkeyPacket = class(TPubKeyPacket)
  end;

  TSecKeyPacket = class(TPubKeyPacket)
  private
    FDEK: TDEK;
    FSecKeyData: TKeyData;
    FIsLocked: Boolean;
  protected
    procedure SetPubKeyAlgo(Value: GCRY_PK_ALGOS); override;
    function KeyDataToMPIList: TMPIList;
    function KeyDataToSexp: GCRY_SEXP;
    procedure SignDigest(KeyData: TKeyData; Digest: TDigest);
    function PKCS1Decode(SesKey: TSessionKey; PKCS1: TPKCS1): Boolean;
  public
    constructor Create;
    destructor Destroy; override;
    function Lock(PW: string): Boolean;
    function Unlock(PW: string): Boolean;
    procedure Sign(Sign: TCertSignPacket; UID: TUserIDPacket); overload;
    procedure Sign(Sign: TSubkeySignPacket; Subkey: TPubSubkeyPacket); overload;
    function Decrypt(SesKey: TSessionKey; ESK: TPubKeyEncSessKeyPacket): Boolean;
    procedure GenNewKey(KeyLen: Integer; PW: string);
    property DEK: TDEK read FDEK;
    property SecKeyData: TKeyData read FSecKeyData;
    property IsLocked: Boolean read FIsLocked;
  end;

  TSecSubkeyPacket = class(TSecKeyPacket)
  end;

  TUserIDPacket = class
  private
    FUID: string;
  public
    property UID: string read FUID write FUID;
  end;

  SIGN_TYPES = (
    SIGN_CERT   = $13,
    SIGN_SUBKEY = $18
  );

  SIGN_SUB_TYPES = (
    SIGN_SUB_PREF_SYMM = $11,
    SIGN_SUB_PREF_HASH = $21,
    SIGN_SUB_PREF_COMP = $22,
    SIGN_SUB_KEYSERV   = $23,
    SIGN_SUB_FEATURE   = $30
  );

  THashLeft2 = array of Byte;

  COMPRESS_ALGOS = (
    COMP_NONE = 0,
    COMP_ZIP  = 1,
    COMP_ZLIB = 2
  );

  FEATURES = (
    FEAT_MODIF_DETECT = $01
  );

  TSymKeyAlgoList = array of GCRY_CIPHER_ALGOS;
  THashAlgoList   = array of GCRY_MD_ALGOS;
  TCompAlgoList   = array of COMPRESS_ALGOS;
  TFeatureList    = array of FEATURES;

  TSignPacket = class(TKeyPacket)
  private
    FSignType: SIGN_TYPES;
    FHashAlgo: GCRY_MD_ALGOS;
    FHashLeft2: THashLeft2;
  protected
    procedure SetPubKeyAlgo(Value: GCRY_PK_ALGOS); override;
    function KeyDataToSexp: GCRY_SEXP;
  public
    constructor Create;
    property SignType: SIGN_TYPES read FSignType write FSignType;
    property HashAlgo: GCRY_MD_ALGOS read FHashAlgo write FHashAlgo;
    property HashLeft2: THashLeft2 read FHashLeft2 write FHashLeft2;
  end;

  TCertSignPacket = class(TSignPacket)
  public
    SymKeyAlgoList: TSymKeyAlgoList;
    HashAlgoList: THashAlgoList;
    CompAlgoList: TCompAlgoList;
    FeatureList: TFeatureList;
    constructor Create;
  end;

  TSubkeySignPacket = class(TSignPacket)
  public
    constructor Create;
  end;

  TPubKeyEncSessKeyPacket = class(TKeyPacket)
  protected
    procedure SetPubKeyAlgo(Value: GCRY_PK_ALGOS); override;
    function KeyDataToSexp: GCRY_SEXP;
  public
    constructor Create;
  end;

  TSessionKeyData = array of Byte;

  TSessionKey = class
  private
    FSessionKeyData: TSessionKeyData;
    FCipherAlgo: GCRY_CIPHER_ALGOS;
  protected
    procedure SetCipherAlgo(Algo: GCRY_CIPHER_ALGOS);
  public
    constructor Create;
    function GenNewSessionKey(Level: GCRY_RANDOME_LEVEL = GCRY_STRONG_RANDOM): Boolean;
    function CalcCheckSum: Word;
    property CipherAlgo: GCRY_CIPHER_ALGOS read FCipherAlgo write SetCipherAlgo;
    property KeyData: TSessionKeyData read FSessionKeyData write FSessionKeyData;
  end;

implementation

uses
  PacketIOUnit;

procedure TKeyPacket.SetPubKeyAlgo(Value: GCRY_PK_ALGOS);
begin
  FPubKeyAlgo := Value;
end;

procedure TKeyPacket.SexpToKeyData(KeyData: TKeyData; Sexp: GCRY_SEXP; TopName, Tokens: string);
var
  I, J, K: Integer;
  Key, Token: GCRY_SEXP;
  MPI: GCRY_MPI;
  BinKey, P: PByte;
  BinKeyLen: Cardinal;
  RC: Integer;
begin
  Key := gcry_sexp_find_token(Sexp, PChar(TopName), 0);
  Assert(Key <> nil);

  KeyData.NumKey := Length(Tokens);

  for I := 1 to Length(Tokens) do
  begin
    Token := gcry_sexp_find_token(Key, @Tokens[I], 1);
    Assert(Token <> nil);
    
    MPI := gcry_sexp_nth_mpi(Token, 1, GCRYMPI_FMT_USG);
    Assert(MPI <> nil);
    
    gcry_sexp_release(Token);

    RC := gcry_mpi_aprint(GCRYMPI_FMT_PGP, @BinKey, @BinKeyLen, MPI);
    if RC <> 0 then raise EGCrypt.Create(RC);
    gcry_mpi_release(MPI);

    J := I - 1;
    P := BinKey;
    KeyData.KeyLen[J] := BinKeyLen;
    for K := 0 to BinKeyLen -1 do
    begin
      KeyData[J, K] := P^;
      Inc(P);
    end;

    gcry_free(BinKey);
  end;
  gcry_sexp_release(Key);
end;

function TKeyPacket.KeyDataToMPIList: TMPIList;
var
  I, RC: Integer;
  NByte: Cardinal;
begin
  SetLength(Result, FKeyData.NumKey);

  for I := 0 to FKeyData.NumKey -1 do
  begin
    NByte := FKeyData.KeyLen[I];
    RC := gcry_mpi_scan(@Result[I], GCRYMPI_FMT_PGP, PByte(FKeyData.Key[I]), @NByte);
    if RC <> 0 then raise EGCrypt.Create(RC);
    Assert(Integer(Nbyte) = FKeyData.KeyLen[I]);
  end;
end;

procedure TKeyPacket.ReleaseMPIList(MPIS: TMPIList);
var
  I: Integer;
begin
  for I := 0 to High(MPIS) do
    gcry_mpi_release(MPIS[I]);
end;

function TKeyPacket.HashStream(HashAlgo: GCRY_MD_ALGOS; S: TStream): TDigest;
var
  MD: GCRY_MD_HD;
  B: Byte;
  PDigestData: PByteArray;
  DigestLen, I: Integer;
begin
  MD := gcry_md_open(HashAlgo, GCRY_MD_FLAG_SECURE);
  Assert(MD <> nil);

  S.Position := 0;
  while S.Read(B, 1) = 1 do
    gcry_md_write(MD, @B, 1);

  gcry_md_final(MD);
  PDigestData := gcry_md_read(MD, HashAlgo);

  DigestLen := gcry_md_get_algo_dlen(HashAlgo);
  Assert(DigestLen <> -1);
  SetLength(Result, DigestLen);
  for I := 0 to High(Result) do
    Result[I] := PDigestData[I];

  gcry_md_close(MD);
end;

function TKeyPacket.GetStrKeyID: string;
var
  I: Integer;
  Hex: PChar;
  Val: Byte;
begin
  GetMem(Hex, 3);
  try
    Hex[0] := #00;
    Hex[1] := #00;
    Hex[2] := #00;

    Result := '';
    for I := 0 to High(FKeyID) do
    begin
      Val := FKeyID[I];
      BinToHex(PChar(@Val), Hex, 1);
      Result := Result + Hex;
    end;
  finally
    FreeMem(Hex);
  end;
end;

constructor TKeyPacket.Create;
var
  I: Integer;
begin
  inherited;
  FVer := 0;
  FDate := 0;
  FPubKeyAlgo := GCRY_PK_NONE;
  FKeyData := TKeyData.Create;
  SetLength(FKeyID, 8);
  for I := 0 to High(FKeyID) do
    FKeyID[I] := 0;
end;

destructor TKeyPacket.Destroy;
begin
  FKeyData.Free;
  inherited;
end;

procedure TPubKeyPacket.SetPubKeyAlgo(Value: GCRY_PK_ALGOS);
var
  NKey: Integer;
begin
  inherited SetPubKeyAlgo(Value);
  NKey := gcry_pk_algo_info(PubKeyAlgo, GCRYCTL_GET_ALGO_NPKEY, nil, nil);
  Assert(NKey <> -1);
  FKeyData.NumKey := NKey;
end;

function TPubKeyPacket.KeyDataToSexp: GCRY_SEXP;
var
  Buf: string;
  RC: Integer;
  Keys: TMPIList;
begin
  SetLength(Keys, 0);
  case FPubKeyAlgo of
  GCRY_PK_DSA:
    begin
      Keys := KeyDataToMPIList;
      Assert(Length(Keys) = 4);
      Buf := '(public-key(openpgp-dsa((p%m)(q%m)(g%m)(y%m))))';
      RC := gcry_sexp_build(@Result, nil, PChar(Buf), Keys[0], Keys[1], Keys[2], Keys[3]);
      if RC <> 0 then raise EGCrypt.Create(RC);
      ReleaseMPIList(Keys);
    end;
  GCRY_PK_ELG_E:
    begin
      Keys := KeyDataToMPIList;
      Assert(Length(Keys) = 3);
      Buf := '(public-key(openpgp-elg((p%m)(g%m)(y%m))))';
      RC := gcry_sexp_build(@Result, nil, PChar(Buf), Keys[0], Keys[1], Keys[2]);
      if RC <> 0 then raise EGCrypt.Create(RC);
      ReleaseMPIList(Keys);
    end;
  else
    Assert(False);
  end;
end;

function TPubKeyPacket.HashPackets(UID: TUserIDPacket; Sign: TCertSignPacket): TDigest;
var
  MS, MS2: TMemoryStream;
  PIO: TPacketIO;
  B: Byte;
  I: Integer;
  Len: Cardinal;
  UTF8: UTF8String;
begin
  MS := TMemoryStream.Create;
  try
    PIO := TPacketIO.Create;
    try
      B := $99;
      MS.WriteBuffer(B, 1);
      MS2 := TMemoryStream.Create;
      try
        PIO.WritePacketBody(MS2, TPubKeyPacket(Self));
        PIO.WriteWord(MS, MS2.Size);
        MS2.SaveToStream(MS);
      finally
        MS2.Free;
      end;

      B := $B4;
      MS.Write(B, 1);
      UTF8 := UTF8Encode(UID.UID);
      PIO.WriteLongword(MS, Length(UTF8));
      for I := 1 to Length(UTF8) do
      begin
        B := Byte(UTF8[I]);
        MS.Write(B, 1);
      end;

      Len := PIO.WriteSigHashedBody(MS, Sign);

      B := Sign.Ver;
      MS.WriteBuffer(B, 1);
      B := $FF;
      MS.WriteBuffer(B, 1);
      
      PIO.WriteLongword(MS, Len);
    finally
      PIO.Free;
    end;
        
    Result := HashStream(Sign.HashAlgo, MS);
  finally
    MS.Free;
  end;
end;

function TPubKeyPacket.HashPackets(Subkey: TPubSubkeyPacket; Sign: TSubkeySignPacket): TDigest;
var
  MS, MS2: TMemoryStream;
  PIO: TPacketIO;
  B: Byte;
  Len: Cardinal;
begin
  MS := TMemoryStream.Create;
  try
    PIO := TPacketIO.Create;
    try
      B := $99;
      MS.WriteBuffer(B, 1);
      MS2 := TMemoryStream.Create;
      try
        PIO.WritePacketBody(MS2, TPubKeyPacket(Self));
        PIO.WriteWord(MS, MS2.Size);
        MS2.SaveToStream(MS);
      finally
        MS2.Free;
      end;

      B := $99;
      MS.WriteBuffer(B, 1);
      MS2 := TMemoryStream.Create;
      try
        PIO.WritePacketBody(MS2, TPubKeyPacket(Subkey));
        PIO.WriteWord(MS, MS2.Size);
        MS2.SaveToStream(MS);
      finally
        MS2.Free;
      end;

      Len := PIO.WriteSigHashedBody(MS, Sign);

      B := Sign.Ver;
      MS.WriteBuffer(B, 1);
      B := $FF;
      MS.WriteBuffer(B, 1);
      
      PIO.WriteLongword(MS, Len);
    finally
      PIO.Free;
    end;

    Result := HashStream(Sign.HashAlgo, MS);
  finally
    MS.Free;
  end;
end;

function TPubKeyPacket.PKCS1Encode(SesKey: TSessionKey): TPKCS1;
var
  NBits, NBytes: Integer;
  SexpKey: GCRY_SEXP;
  PS, PSWK: array of Byte;
  PSLen, PSWKLen: Integer;
  I, J: Integer;
  CS: Word;
begin
  SexpKey := KeyDataToSexp;
  NBits := gcry_pk_get_nbits(SexpKey);
  Assert(NBits <> 0);
  NBytes := (NBits + 7) div 8;
  SetLength(Result, NBytes);

  PSLen := NBytes - Length(SesKey.KeyData) - 5;
  SetLength(PS, PSLen);
  gcry_randomize(PByte(PS), PSLen, GCRY_STRONG_RANDOM);
  while True do
  begin
    PSWKLen := 0;
    for I := 0 to High(PS) do
    begin
      if PS[I] = 0 then
        Inc(PSWKLen);
    end;

    if PSWKLen = 0 then
      Break;

    SetLength(PSWK, PSWKLen);
    gcry_randomize(PByte(PSWK), PSWKLen, GCRY_STRONG_RANDOM);
    J := 0;
    for I := 0 to High(PS) do
    begin
      if PS[I] = 0 then
      begin
        PS[I] := PSWK[J];
        Inc(J);
      end;
    end;
  end;

  I := 0;

  Result[I] := $02;
  Inc(I);

  for J := 0 to High(PS) do
  begin
    Result[I] := PS[J];
    Inc(I);
  end;

  Result[I] := $00;
  Inc(I);

  Result[I] := Byte(SesKey.CipherAlgo);
  Inc(I);

  for J := 0 to High(SesKey.KeyData) do
  begin
    Result[I] := SesKey.KeyData[J];
    Inc(I);
  end;

  CS := SesKey.CalcCheckSum;
  Result[I] := (CS shr 8) and $FF;
  Inc(I);
  Result[I] := CS and $FF;
end;

constructor TPubKeyPacket.Create;
begin
  inherited;
  Ver := 4;
end;

function TPubKeyPacket.CalcFingerprint: TFingerprint;
var
  MS, MS2: TMemoryStream;
  PIO: TPacketIO;
  B: Byte;
begin
  MS := TMemoryStream.Create;
  try
    MS2 := TMemoryStream.Create;
    try
      PIO := TPacketIO.Create;
      try
        PIO.WritePacketBody(MS2, Self);
        
        B := $99;
        MS.WriteBuffer(B, 1);

        PIO.WriteWord(MS, MS2.Size);
      finally
        PIO.Free;
      end;

      MS2.SaveToStream(MS);
    finally
      MS2.Free;
    end;

    Result := TFingerprint(HashStream(GCRY_MD_SHA1, MS));
  finally
    MS.Free;
  end;
end;

function TPubKeyPacket.CalcKeyID: TKeyID;
var
  FP: TFingerprint;
  I: Integer;
begin
  FP := CalcFingerprint;
  SetLength(Result, 8);
  for I := 0 to High(Result) do
    Result[I] := FP[12 + I];
end;

function TPubKeyPacket.Verify(Sign: TCertSignPacket; UID: TUserIDPacket): Boolean;
var
  SexpSign, SexpData, SexpKey: GCRY_SEXP;
  Digest: TDigest;
  NByte: Cardinal;
  MPI: GCRY_MPI;
  RC: Integer;
begin
  SexpSign := Sign.KeyDataToSexp;
  SexpKey  := KeyDataToSexp;

  Digest   := HashPackets(UID, Sign);
  NByte := Length(Digest);
  RC := gcry_mpi_scan(@MPI, GCRYMPI_FMT_USG, PByte(Digest), @NByte);
  if RC <> 0 then raise EGCrypt.Create(RC);
  Assert(Integer(NByte) = Length(Digest));

  RC := gcry_sexp_build(@SexpData, nil, '%m', MPI);
  if RC <> 0 then raise EGCrypt.Create(RC);

  RC := gcry_pk_verify(SexpSign, SexpData, SexpKey);
  Result := (RC = 0);

  gcry_mpi_release(MPI);
  gcry_sexp_release(SexpKey);
  gcry_sexp_release(SexpData);
  gcry_sexp_release(SexpSign);
end;

function TPubKeyPacket.Verify(Sign: TSubkeySignPacket; Subkey: TPubSubkeyPacket): Boolean;
var
  SexpSign, SexpData, SexpKey: GCRY_SEXP;
  Digest: TDigest;
  NByte: Cardinal;
  MPI: GCRY_MPI;
  RC: Integer;
begin
  SexpSign := Sign.KeyDataToSexp;
  SexpKey  := KeyDataToSexp;

  Digest   := HashPackets(Subkey, Sign);
  NByte := Length(Digest);
  RC := gcry_mpi_scan(@MPI, GCRYMPI_FMT_USG, PByte(Digest), @NByte);
  if RC <> 0 then raise EGCrypt.Create(RC);
  Assert(Integer(NByte) = Length(Digest));

  RC := gcry_sexp_build(@SexpData, nil, '%m', MPI);
  if RC <> 0 then raise EGCrypt.Create(RC);

  RC := gcry_pk_verify(SexpSign, SexpData, SexpKey);
  Result := (RC = 0);

  gcry_mpi_release(MPI);
  gcry_sexp_release(SexpKey);
  gcry_sexp_release(SexpData);
  gcry_sexp_release(SexpSign);
end;

procedure TPubKeyPacket.Encrypt(ESK: TPubKeyEncSessKeyPacket; SesKey: TSessionKey);
var
  SexpESK, SexpData, SexpKey: GCRY_SEXP;
  PKCS1: TPKCS1;
  MPI: GCRY_MPI;
  NByte: Cardinal;
  RC: Integer;
begin
  SexpKey := KeyDataToSexp;

  PKCS1 := PKCS1Encode(SesKey);
  Nbyte := Length(PKCS1);
  RC := gcry_mpi_scan(@MPI, GCRYMPI_FMT_USG, PByte(PKCS1), @NByte);
  if RC <> 0 then raise EGCrypt.Create(RC);
  Assert(Integer(NByte) = Length(PKCS1));

  RC := gcry_sexp_build(@SexpData, nil, '%m', MPI);
  if RC <> 0 then raise EGCrypt.Create(RC);

  RC := gcry_pk_encrypt(@SexpESK, SexpData, SexpKey);
  if RC <> 0 then raise EGCrypt.Create(RC);

  ESK.KeyID := CalcKeyID;
  ESK.PubKeyAlgo := PubKeyAlgo;

  case FPubKeyAlgo of
  GCRY_PK_ELG_E:
    SexpToKeyData(ESK.KeyData, SexpESK, 'openpgp-elg', 'ab');
  else
    Assert(False);
  end;

  gcry_mpi_release(MPI);
  gcry_sexp_release(SexpKey);
  gcry_sexp_release(SexpData);
  gcry_sexp_release(SexpESK);
end;

procedure TSecKeyPacket.SetPubKeyAlgo(Value: GCRY_PK_ALGOS);
var
  NKey: Integer;
begin
  inherited SetPubKeyAlgo(Value);
  NKey := gcry_pk_algo_info(PubKeyAlgo, GCRYCTL_GET_ALGO_NSKEY, nil, nil);
  Assert(NKey <> -1);
  FSecKeyData.NumKey := NKey - KeyData.NumKey;
end;

function TSecKeyPacket.KeyDataToMPIList: TMPIList;
var
  I, J, RC: Integer;
  NByte: Cardinal;
begin
  Result := inherited KeyDataToMPIList;
  SetLength(Result, FKeyData.NumKey + FSecKeyData.NumKey);

  I := FKeyData.NumKey;
  for J := 0 to FSecKeyData.NumKey -1 do
  begin
    NByte := FSecKeyData.KeyLen[J];
    RC := gcry_mpi_scan(@Result[I], GCRYMPI_FMT_PGP, PByte(FSecKeyData.Key[J]), @NByte);
    if RC <> 0 then raise EGCrypt.Create(RC);
    Assert(Integer(Nbyte) = FSecKeyData.KeyLen[J]);
    Inc(I);
  end;
end;

function TSecKeyPacket.KeyDataToSexp: GCRY_SEXP;
var
  Buf: string;
  RC: Integer;
  Keys: TMPIList;
begin
  Assert(FIsLocked = False);

  SetLength(Keys, 0);
  case FPubKeyAlgo of
  GCRY_PK_DSA:
    begin
      Keys := KeyDataToMPIList;
      Assert(Length(Keys) = 5);
      Buf := '(private-key(openpgp-dsa((p%m)(q%m)(g%m)(y%m)(x%m))))';
      RC := gcry_sexp_build(@Result, nil, PChar(Buf), Keys[0], Keys[1], Keys[2], Keys[3], Keys[4]);
      if RC <> 0 then raise EGCrypt.Create(RC);
      ReleaseMPIList(Keys);
    end;
  GCRY_PK_ELG_E:
    begin
      Keys := KeyDataToMPIList;
      Assert(Length(Keys) = 4);
      Buf := '(private-key(openpgp-elg((p%m)(g%m)(y%m)(x%m))))';
      RC := gcry_sexp_build(@Result, nil, PChar(Buf), Keys[0], Keys[1], Keys[2], Keys[3]);
      if RC <> 0 then raise EGCrypt.Create(RC);
      ReleaseMPIList(Keys);
    end;
  else
    Assert(False);
  end;
end;

procedure TSecKeyPacket.SignDigest(KeyData: TKeyData; Digest: TDigest);
var
  Nbyte: Cardinal;
  MPI: GCRY_MPI;
  RC: Integer;
  SexpSign, SexpData, SexpSKey: GCRY_SEXP;
begin
  NByte := Length(Digest);
  RC := gcry_mpi_scan(@MPI, GCRYMPI_FMT_USG, PByte(Digest), @NByte);
  if RC <> 0 then raise EGCrypt.Create(RC);
  Assert(Integer(NByte) = Length(Digest));

  RC := gcry_sexp_build(@SexpData, nil, '%m', MPI);
  if RC <> 0 then raise EGCrypt.Create(RC);

  SexpSKey := KeyDataToSexp;

  RC := gcry_pk_sign(@SexpSign, SexpData, SexpSKey);
  if RC <> 0 then raise EGCrypt.Create(RC);

  case FPubKeyAlgo of
  GCRY_PK_DSA:
    SexpToKeyData(KeyData, SexpSign, 'openpgp-dsa', 'rs');
  else
    Assert(False);
  end;

  gcry_mpi_release(MPI);
  gcry_sexp_release(SexpSign);
  gcry_sexp_release(SexpSKey);
  gcry_sexp_release(SexpData);
end;

function TSecKeyPacket.PKCS1Decode(SesKey: TSessionKey; PKCS1: TPKCS1): Boolean;
var
  I, J: Integer;
  CS: Word;
begin
  Result := False;

  if PKCS1[0] <> $02 then
    Exit;

  J := 0;
  for I := 1 to High(PKCS1) do
  begin
    if PKCS1[I] = 0 then
    begin
      J := I;
      Break;
    end;
  end;

  if J = 0 then
    Exit;

  Inc(J);
  if J >= High(PKCS1) then
    Exit;

  SesKey.CipherAlgo := GCRY_CIPHER_ALGOS(PKCS1[J]);
  Inc(J);

  if J + Length(SesKey.KeyData) + 2 <> Length(PKCS1) then
    Exit;

  for I := 0 to High(SesKey.KeyData) do
  begin
    SesKey.KeyData[I] := PKCS1[J];
    Inc(J);
  end;

  CS := PKCS1[J] shl 8;
  Inc(J);
  CS := CS or (PKCS1[J] and $FF);

  if CS <> SesKey.CalcCheckSum then
    Exit;

  Result := True;
end;

constructor TSecKeyPacket.Create;
begin
  inherited;
  FDEK := TDEK.Create;
  FSecKeyData := TKeyData.Create;
  FIsLocked := True;
end;

destructor TSecKeyPacket.Destroy;
begin
  FDEK.Free;
  FSecKeyData.Free;
  inherited;
end;

function TSecKeyPacket.Lock(PW: string): Boolean;
begin
  if not FIsLocked then
  begin
    FDEK.Lock(FSecKeyData, PW);
    FIsLocked := True;
    Result := True;
  end
  else
    Result := False;
end;

function TSecKeyPacket.Unlock(PW: string): Boolean;
begin
  if FIsLocked then
  begin
    Result := FDEK.Unlock(FSecKeyData, PW);
    if Result then
      FIsLocked := False;
  end
  else
    Result := False;
end;

procedure TSecKeyPacket.Sign(Sign: TCertSignPacket; UID: TUserIDPacket);
var
  Digest: TDigest;
begin
  Sign.Date := now;
  Sign.PubKeyAlgo := FPubKeyAlgo;
  Sign.KeyID := CalcKeyID;

  Digest := HashPackets(UID, Sign);

  Assert(Length(Sign.HashLeft2) = 2);
  Sign.HashLeft2[0] := Digest[0];
  Sign.HashLeft2[1] := Digest[1];

  SignDigest(Sign.KeyData, Digest);
end;

procedure TSecKeyPacket.Sign(Sign: TSubkeySignPacket; Subkey: TPubSubkeyPacket);
var
  Digest: TDigest;
begin
  Sign.Date := now;
  Sign.PubKeyAlgo := FPubKeyAlgo;
  Sign.KeyID := CalcKeyID;

  Digest := HashPackets(Subkey, Sign);

  Assert(Length(Sign.HashLeft2) = 2);
  Sign.HashLeft2[0] := Digest[0];
  Sign.HashLeft2[1] := Digest[1];
  
  SignDigest(Sign.KeyData, Digest);
end;

function TSecKeyPacket.Decrypt(SesKey: TSessionKey; ESK: TPubKeyEncSessKeyPacket): Boolean;
var
  SexpESK, SexpData, SexpKey: GCRY_SEXP;
  MPI: GCRY_MPI;
  PBuf: PByteArray;
  BufLen: Cardinal;
  PKCS1: TPKCS1;
  RC, I: Integer;
  RCB: Boolean;
begin
  Result := False;

  SexpKey := nil;
  SexpESK := nil;
  try
    SexpKey := KeyDataToSexp;
    SexpESK := ESK.KeyDataToSexp;
    RC := gcry_pk_decrypt(@SexpData, SexpESK, SexpKey);
    if RC <> 0 then
      Exit;

    MPI := gcry_sexp_nth_mpi(SexpData, 0, GCRYMPI_FMT_USG);
    if MPI = nil then
      Exit;

    RC := gcry_mpi_aprint(GCRYMPI_FMT_USG, @PBuf, @BufLen, MPI);
    if RC <> 0 then
      Exit;

    SetLength(PKCS1, BufLen);
    for I := 0 to High(PKCS1) do
      PKCS1[I] := PBuf[I];

    gcry_free(PBuf);

    RCB := PKCS1Decode(SesKey, PKCS1);
    if not RCB then
      Exit;

    Result := True;
  finally
    gcry_sexp_release(SexpData);
    gcry_sexp_release(SexpESK);
    gcry_sexp_release(SexpKey);
  end;
end;

procedure TSecKeyPacket.GenNewKey(KeyLen: Integer; PW: string);
var
  RC: Integer;
  KeyName: string;
  KeyGenParam: string;
  Param, Key: GCRY_SEXP;
begin
  case FPubKeyAlgo of
  GCRY_PK_DSA:
    KeyName := 'openpgp-dsa';
  GCRY_PK_ELG_E:
    KeyName := 'openpgp-elg';
  else
    Assert(False);
  end;
  
  KeyGenParam := '(genkey(' + KeyName + '(nbits %d)))';
  RC := gcry_sexp_build(@Param, nil, PChar(KeyGenParam), KeyLen);
  if RC <> 0 then raise EGCrypt.Create(RC);

  RC := gcry_pk_genkey(@Key, Param);
  if RC <> 0 then raise EGCrypt.Create(RC);

  Date := Now;

  case FPubKeyAlgo of
  GCRY_PK_DSA:
    begin
      SexpToKeyData(KeyData, Key, 'public-key', 'pqgy');
      SexpToKeyData(SecKeyData, Key, 'private-key', 'x');
    end;
  GCRY_PK_ELG_E:
    begin
      SexpToKeyData(KeyData, Key, 'public-key', 'pgy');
      SexpToKeyData(SecKeyData, Key, 'private-key', 'x');
    end;
  else
    Assert(False);
  end;

  FIsLocked := False;

  FDEK.RandomizeIV;
  FDEK.S2K.RandomizeSalt;

  Lock(PW);

  gcry_sexp_release(Key);
  gcry_sexp_release(Param);
end;

procedure TSignPacket.SetPubKeyAlgo(Value: GCRY_PK_ALGOS);
var
  NKey: Integer;
begin
  inherited SetPubKeyAlgo(Value);
  NKey := gcry_pk_algo_info(PubKeyAlgo, GCRYCTL_GET_ALGO_NSIGN, nil, nil);
  Assert(NKey <> -1);
  FKeyData.NumKey := NKey;
end;

function TSignPacket.KeyDataToSexp: GCRY_SEXP;
var
  Keys: TMPIList;
  Buf: string;
  RC: Integer;
begin
  SetLength(Keys, 0);
  case FPubKeyAlgo of
  GCRY_PK_DSA:
   begin
     Keys := KeyDataToMPIList;
     Assert(Length(Keys) = 2);
     Buf := '(sig-val(openpgp-dsa((r%m)(s%m))))';
     RC := gcry_sexp_build(@Result, nil, PChar(Buf), Keys[0], Keys[1]);
     if RC <> 0 then raise EGCrypt.Create(RC);
     ReleaseMPIList(Keys);
   end;
  else
    Assert(False);
  end;
end;

constructor TSignPacket.Create;
begin
  inherited;
  Ver := 4;
  FHashAlgo := GCRY_MD_SHA1;
  SetLength(FHashLeft2, 2);
  FHashLeft2[0] := 0;
  FHashLeft2[1] := 0;
end;

constructor TCertSignPacket.Create;
begin
  inherited;
  SignType := SIGN_CERT;
end;

constructor TSubkeySignPacket.Create;
begin
  inherited;
  SignType := SIGN_SUBKEY;
end;

procedure TPubKeyEncSessKeyPacket.SetPubKeyAlgo(Value: GCRY_PK_ALGOS);
var
  NKey: Integer;
begin
  inherited SetPubKeyAlgo(Value);
  NKey := gcry_pk_algo_info(PubKeyAlgo, GCRYCTL_GET_ALGO_NENCR, nil, nil);
  Assert(NKey <> -1);
  KeyData.NumKey := NKey;
end;

function TPubKeyEncSessKeyPacket.KeyDataToSexp: GCRY_SEXP;
var
  Keys: TMPIList;
  Buf: string;
  RC: Integer;
begin
  SetLength(Keys, 0);
  case FPubKeyAlgo of
  GCRY_PK_ELG_E:
    begin
      Keys := KeyDataToMPIList;
      Assert(Length(Keys) = 2);
      Buf := '(enc-val(openpgp-elg((a%m)(b%m))))';
      RC := gcry_sexp_build(@Result, nil, PChar(Buf), Keys[0], Keys[1]);
      if RC <> 0 then raise EGCrypt.Create(RC);
      ReleaseMPIList(Keys);
    end;
  else
    Assert(False);
  end;
end;

constructor TPubKeyEncSessKeyPacket.Create;
begin
  inherited;
  Ver := 3;
end;

procedure TSessionKey.SetCipherAlgo(Algo: GCRY_CIPHER_ALGOS);
var
  KeyLen: Integer;
begin
  FCipherAlgo := Algo;
  KeyLen := gcry_cipher_get_algo_keylen(FCipherAlgo);
  Assert(KeyLen <> -1);
  SetLength(FSessionKeyData, KeyLen);
end;

constructor TSessionKey.Create;
begin
  inherited;
  SetLength(FSessionKeyData, 0);
  FCipherAlgo := GCRY_CIPHER_NONE;
end;

function TSessionKey.GenNewSessionKey(Level: GCRY_RANDOME_LEVEL = GCRY_STRONG_RANDOM): Boolean;
var
  CH: GCRY_CIPHER_HD;
  I, RC: Integer;
begin
  Result := False;

  CH := gcry_cipher_open(FCipherAlgo, GCRY_CIPHER_MODE_CFB, GCRY_CIPHER_SECURE);
  Assert(CH <> nil);
  for I := 0 to 16 do
  begin
    gcry_randomize(PByte(FSessionKeyData), Length(FSessionKeyData), Level);
    RC := gcry_cipher_setkey(CH, PByte(FSessionKeyData), Length(FSessionKeyData));
    if RC = 0 then
    begin
      Result := True;
      Break;
    end;
  end;
  gcry_cipher_close(CH);
end;

function TSessionKey.CalcCheckSum: Word;
var
  I: Integer;
begin
  Result := 0;
  for I := 0 to High(FSessionKeyData) do
    Result := Result + FSessionKeyData[I];
end;

end.
