/*
NewsHeader.cs : The class of the "header part" in the news article.
 The header information is stored in the "Hashtable".
 
Copyright (c) 2005 CSamura
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions
are met:
1. Redistributions of source code must retain the above copyright
   notice, this list of conditions and the following disclaimer.
2. Redistributions in binary form must reproduce the above copyright
   notice, this list of conditions and the following disclaimer in the
   documentation and/or other materials provided with the distribution.
3. The name of the author may not be used to endorse or promote products
   derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

using System;
using System.Collections;
using System.Globalization;
using System.Text;
using System.Text.RegularExpressions;

namespace CSamura.NNTP
{
	/// <summary>
	/// j[XL̃wb_̃NX
	/// </summary>
	[Serializable]
	public class NewsHeader
	{
		/// <summary>
		/// wb_i[nbVe[u
		/// </summary>
		private Hashtable headers;

		/// <summary>
		/// j[XL̑Mꂽ
		/// </summary>
		private DateTime Date;

		/// <summary>
		/// fR[hꂽ From wb_̒l
		/// </summary>
		private string From;

		/// <summary>
		/// fR[hꂽ Subject wb_̒l
		/// </summary>
		private string Subject;

		/// <summary>
		/// NewsHeader ̃ftHgRXgN^
		/// </summary>
		public NewsHeader()
		{
			headers = new Hashtable();
		}

		/// <summary>
		/// NewsHeader ̃RXgN^B
		/// ML̃wb_쐬B
		/// </summary>
		/// <param name="name">M҂̖O</param>
		/// <param name="address">M҂̃[AhX</param>
		/// <param name="group">M̃j[XO[v</param>
		/// <param name="subject">j[XL̃^Cg</param>
		public NewsHeader(string name, string address, string group, string subject)
		{
			headers = new Hashtable();

			// From wb_̏
			string from = this.EncodeFrom(name, address);
			headers.Add("From", from);

			// Subject wb_̏
			Subject = subject;
			headers.Add("Subject", this.EncodeSubject(subject));

			// Date wb_̏
			this.SetNowDate();

			// Newsgroups wb_̏
			headers.Add("Newsgroups", group);

			// ̑̃wb_̏
			headers.Add("Content-Type", "text/plain; charset=\"iso-2022-jp\"");
			headers.Add("Content-Transfer-Encoding", "7bit");
			headers.Add("MIME-Version", "1.0");
			headers.Add("X-Newsreader", NNTPClient.xreader);
		}

		/// <summary>
		/// NewsHeader ̃RXgN^B
		/// ɎMwb_̕
		/// </summary>
		/// <param name="h">Mwb_̕</param>
		public NewsHeader(string h)
		{
			headers = new Hashtable();

			// wb_̒lsɂ킽ꍇ̏
			string h2 = Regex.Replace(h, "\r\n[\\s]+", "");

			// wb_̃L[ƒlo߂̐K\
			Regex r = new Regex("(?<key>[-\\w]+): (?<value>[^\r]+)\r\n");
			
			Match m = r.Match(h2);
			while(m.Success)
			{
				string hkey = m.Groups["key"].Value.ToString();
				string hvalue = m.Groups["value"].Value.ToString();
				headers.Add(hkey, hvalue);
				m = m.NextMatch();
			}
			if(this.ContainsKey("Subject"))
			{
				Subject = this.GetSubjectDecode(headers["Subject"].ToString());
			}
			if(this.ContainsKey("From"))
			{
				From = this.GetSubjectDecode(headers["From"].ToString());
			}
			if(this.ContainsKey("Date"))
			{
				this.SetDate(headers["Date"].ToString());
			}
		}

		/// <summary>
		/// L[ɑΉlo\bh
		/// </summary>
		/// <param name="key">L[̕</param>
		/// <returns>L[ɑΉl̕</returns>
		public string GetHeaderValue(string key)
		{
			if(this.ContainsKey(key))
			{
				return headers[key].ToString();
			}
			else
			{
				return "̃L[͊܂܂Ă܂B";
			}
		}

		/// <summary>
		/// wb_̃L[ƒlׂĎo\bh
		/// </summary>
		/// <returns>L̃wb_ׂ̂ẴL[ƒl</returns>
		public string GetHeaders()
		{
			IDictionaryEnumerator ide = headers.GetEnumerator();
			string res ="";
			while(ide.MoveNext())
			{
				res += ide.Key.ToString() + ": " + ide.Value.ToString() + "\r\n";
			}
			return res;
		}

		/// <summary>
		/// ̖Õwb_ZbgĂ邩𒲂ׂ郁\bh
		/// </summary>
		/// <param name="key">wb_̃L[</param>
		/// <returns>܂܂Ă true, łȂ false</returns>
		public bool ContainsKey(string key)
		{
			return headers.ContainsKey(key);
		}

		/// <summary>
		/// Subject wb_ =?iso-2022-jp?B?...?= ƂfR[h
		/// </summary>
		/// <param name="src">fR[h镶</param>
		/// <returns>fR[h</returns>
		public string GetSubjectDecode(string src)
		{
			Regex subRegex;
			Match m;

			// ̕ (Ƃ "=?iso-2022-jp?B?...?=" ) 3̕ɕ鐳K\
			subRegex = new Regex(@"=[?](?<enc>[-\w\d^?]+)[?](?<form>[BQ])[?](?<src>[^?]+)[?]=");
			string res = "";
			m = subRegex.Match(src);

			// ̏̕ǂ܂Ői񂾂 index ̒lŎ
			int index = 0;
			while(m.Success)
			{
				// fR[hȂXLbv
				res += src.Substring(index, m.Index - index);

				// "iso-2022-jp" 
				string encd = m.Groups["enc"].ToString();
				// "B" ܂ "Q"
				string formd = m.Groups["form"].ToString();
				// fR[h镶񕔕
				string srcd = m.Groups["src"].ToString();

				// "B" ܂ "Q" GR[fBOŃfR[h
				if(formd == "B")
				{
					byte[] b = Convert.FromBase64String(srcd);
					res += Encoding.GetEncoding(encd).GetString(b);
				}
				else if(formd == "Q")
				{
					byte[] b = CSamura.NNTP.QuotedPrintable.GetBytes(srcd);
					res += Encoding.GetEncoding(encd).GetString(b);
				}
				index = m.Index + m.Length;
				m = m.NextMatch();
			}
			// fR[hc̕
			if(index < src.Length)
			{
				res += src.Substring(index, src.Length - index);
			}
			return res;
		}

		/// <summary>
		/// j[XL̃^CgA=?iso-2022-jp?B?....?= ̌`ɃGR[h
		/// </summary>
		/// <param name="src">GR[h镶</param>
		/// <returns>GR[hꂽ</returns>
		public string GetSubjectEncode(string src)
		{
			byte[] b = Encoding.GetEncoding("iso-2022-jp").GetBytes(src);
			string br = Convert.ToBase64String(b);
			string res = "=?iso-2022-jp?B?" + br + "?=";

			return res;
		}

		/// <summary>
		/// Subject wb_쐬
		/// </summary>
		/// <param name="subject">^Cg̕</param>
		/// <returns> Subject wb_̒l</returns>
		public string EncodeSubject(string subject)
		{
			string res;
			
			if(TextCheck.Is2ByteString(subject)) res = subject;
			else res = this.GetSubjectEncode(subject);

			return res;
		}

		/// <summary>
		/// From wb_쐬
		/// </summary>
		/// <param name="name">M҂̖O</param>
		/// <param name="address">M҂̃[AhX</param>
		/// <returns>From wb_̒l</returns>
		public string EncodeFrom(string name, string address)
		{
			string name2 = "\"" + name + "\"";

			// From ̒lB
			From = name2 + " <" + address + ">";

			if(!TextCheck.Is2ByteString(name))
			{
				name2 = this.GetSubjectEncode(name2);
			}
			string res = name2 + " <" + address + ">";
			return res;
		}

		/// <summary>
		/// j[XL Date wb_̒l DateTime IuWFNgɕϊ
		/// </summary>
		/// <param name="date">j[XL Date wb_̕</param>
		/// <returns>true: ϊ̐, false: Date wb_̏Ⴄ</returns>
		private bool SetDate(string date)
		{
			// Date wb_̒l DateTime.ParseExact \bhň悤ɕϊB
			// ܂A+0900  +09:00 ɕϊB
			Regex r = new Regex(@"(?<forward>\w{3}, \d{1,2} \w{3} \d{4} \d{2}:\d{2}:\d{2} [+-])(?<h>\d{2})(?<m>\d{2})");
			Match m = r.Match(date);
			if(m.Success)
			{
				string forward = m.Groups["forward"].Value.ToString();
				string hour = m.Groups["h"].Value.ToString();
				string min = m.Groups["m"].Value.ToString();

				string d = forward + hour + ":" + min;

				// Date tB[hɊi[
				Date = DateTime.ParseExact(d, "ddd, d MMM yyyy HH':'mm':'ss zzz", DateTimeFormatInfo.InvariantInfo);
				
				return true;
			}
			else return false;
		}

		/// <summary>
		/// ݎԂ Date wb_ƂĊi[
		/// </summary>
		private void SetNowDate()
		{
			Date = DateTime.Now;

			string date = Date.ToString("ddd, d MMM yyyy HH':'mm':'ss zzz", DateTimeFormatInfo.InvariantInfo);

			// +09:00  +0900 ɂ
			date = date.Substring(0, date.Length-3) + date.Substring(date.Length-2, 2);
			headers.Add("Date", date);
		}

		/// <summary>
		/// Date tB[h̒l𓾂
		/// </summary>
		/// <returns>Date tB[h̒l𕶎ŕԂ</returns>
		public string GetDate()
		{
			try
			{
				// Date tB[h̒l𕶎ɕϊ
				string date = Date.ToString("yyyy'N'MM''dd'' HH''mm''ss'b'", DateTimeFormatInfo.CurrentInfo);
				return date;
			}
			// Date tB[h̏ꍇ
			catch(NullReferenceException)
			{
				return "w肳Ă܂B";
			}
		}

		/// <summary>
		/// wb_̒lB
		/// L[܂݂Ȃ΁AL[B
		/// </summary>
		/// <param name="hkey">wb_̃L[̕</param>
		/// <param name="hvalue">wb_̒l̕</param>
		public void SetHeader(string hkey, string hvalue)
		{
			// L[łɑ݂邩ǂŏꍇ킯
			if(this.ContainsKey(hkey))
			{
				headers[hkey] = hvalue;
			}
			else headers.Add(hkey, hvalue);
		}

		/// <summary>
		/// j[XL̃^Cg̃vpeB
		/// </summary>
		public string ArticleSubject
		{
			get
			{
				return Subject;
			}
		}
		public string ArticleFrom
		{
			get
			{
				return From;
			}
		}
	}
}
