/*
	SHARP MZ-700 Emulator 'EmuZ-700'
	SHARP MZ-800 Emulator 'EmuZ-800'
	SHARP MZ-1500 Emulator 'EmuZ-1500'

	Author : Takeda.Toshiya
	Date   : 2008.06.05 -

	[ virtual machine ]
*/

#include "mz700.h"
#include "../../emu.h"
#include "../device.h"
#include "../event.h"

#include "../and.h"
#include "../datarec.h"
#include "../i8253.h"
#include "../i8255.h"
#include "../io.h"
#include "../pcm1bit.h"
#include "../z80.h"

#ifdef USE_DEBUGGER
#include "../debugger.h"
#endif

//#include "cmos.h"
#include "emm.h"
#include "kanji.h"
#include "keyboard.h"
#include "memory.h"
#include "ramfile.h"

#if defined(_MZ800) || defined(_MZ1500)
#include "../disk.h"
#include "../mb8877.h"
#include "../not.h"
#include "../sn76489an.h"
#include "../z80pio.h"
#include "../z80sio.h"
#include "floppy.h"
#if defined(_MZ1500)
#include "../mz1p17.h"
#include "../prnfile.h"
#include "psg.h"
#endif
#include "quickdisk.h"
#endif

// ----------------------------------------------------------------------------
// initialize
// ----------------------------------------------------------------------------

VM::VM(EMU* parent_emu) : emu(parent_emu)
{
#if defined(_MZ800)
	boot_mode = config.boot_mode;
#endif
	
	// create devices
	first_device = last_device = NULL;
	dummy = new DEVICE(this, emu);	// must be 1st device
	event = new EVENT(this, emu);	// must be 2nd device
#if defined(_USE_QT)
	dummy->set_device_name(_T("1st Dummy"));
	event->set_device_name(_T("EVENT"));
#endif	
	and_int = new AND(this, emu);
	drec = new DATAREC(this, emu);
	pit = new I8253(this, emu);
	pio = new I8255(this, emu);
	io = new IO(this, emu);
	pcm = new PCM1BIT(this, emu);
	cpu = new Z80(this, emu);
#if defined(_USE_QT)
	and_int->set_device_name(_T("AND GATE(INTERRUPT)"));
	cpu->set_device_name(_T("CPU(Z80)"));
#endif	
	
//	cmos = new CMOS(this, emu);
	emm = new EMM(this, emu);
	kanji = new KANJI(this, emu);
	keyboard = new KEYBOARD(this, emu);
	memory = new MEMORY(this, emu);
	ramfile = new RAMFILE(this, emu);
#if defined(_USE_QT)
	emm->set_device_name(_T("EMM"));
	kanji->set_device_name(_T("KANJI ROM"));
	keyboard->set_device_name(_T("KEYBOARD I/F"));
	memory->set_device_name(_T("MEMORY"));
	ramfile->set_device_name(_T("RAM FILES"));
#endif	
#if defined(_MZ800) || defined(_MZ1500)
	and_snd = new AND(this, emu);
  #if defined(_USE_QT)
	and_snd->set_device_name(_T("AND GATE(SOUND)"));
  #endif
	fdc = new MB8877(this, emu);	// mb8876
#if defined(_MZ800)
	not_pit = new NOT(this, emu);
  #if defined(_USE_QT)
	not_pit->set_device_name(_T("NOT GATE(PIT)"));
  #endif
	psg = new SN76489AN(this, emu);
#elif defined(_MZ1500)
	if(config.printer_device_type == 0) {
		printer = new PRNFILE(this, emu);
	} else if(config.printer_device_type == 1) {
		printer = new MZ1P17(this, emu);
	} else {
		printer = dummy;
	}
	not_reset = new NOT(this, emu);
	not_strobe = new NOT(this, emu);
	psg_l = new SN76489AN(this, emu);
	psg_r = new SN76489AN(this, emu);
  #if defined(_USE_QT)
	not_reset->set_device_name(_T("NOT GATE(RESET)"));
	not_strobe->set_device_name(_T("NOT GATE(PRINTER STROBE)"));
	psg_l->set_device_name(_T("SN76489AN PSG(LEFT)"));
	psg_r->set_device_name(_T("SN76489AN PSG(RIGHT)"));
  #endif
#endif
	pio_int = new Z80PIO(this, emu);
	sio_rs = new Z80SIO(this, emu);
	sio_qd = new Z80SIO(this, emu);
	
	floppy = new FLOPPY(this, emu);
#if defined(_USE_QT)
	floppy->set_device_name(_T("FLOPPY I/F"));
#endif
#if defined(_MZ1500)
	psg = new PSG(this, emu);
#if defined(_USE_QT)
	psg->set_device_name(_T("PSG I/F"));
#endif
#endif
	qd = new QUICKDISK(this, emu);
#if defined(_USE_QT)
	qd->set_device_name(_T("QUICKDISK I/F"));
#endif
#endif
	
	// set contexts
	event->set_context_cpu(cpu);
	event->set_context_sound(pcm);
#if defined(_MZ800)
	event->set_context_sound(psg);
#elif defined(_MZ1500)
	event->set_context_sound(psg_l);
	event->set_context_sound(psg_r);
#endif
	event->set_context_sound(drec);

#if defined(USE_SOUND_FILES)
#if defined(_MZ800) || defined(_MZ1500)
	if(fdc->load_sound_data(MB8877_SND_TYPE_SEEK, _T("FDDSEEK.WAV"))) {
		event->set_context_sound(fdc);
	}
	//if(qd->load_sound_data(MB8877_SND_TYPE_SEEK, _T("QD_SEEK.WAV"))) {
	//	event->set_context_sound(qd);
	//}
#endif
	drec->load_sound_data(DATAREC_SNDFILE_EJECT, _T("CMTEJECT.WAV"));
	//drec->load_sound_data(DATAREC_SNDFILE_PLAY, _T("CMTPLAY.WAV"));
	//drec->load_sound_data(DATAREC_SNDFILE_STOP, _T("CMTSTOP.WAV"));
#if defined(_MZ1500)
	// Is MZ-821?
	drec->load_sound_data(DATAREC_SNDFILE_RELAY_ON, _T("RELAY_ON.WAV"));
	drec->load_sound_data(DATAREC_SNDFILE_RELAY_OFF, _T("RELAYOFF.WAV"));
#else
	drec->load_sound_data(DATAREC_SNDFILE_RELAY_ON, _T("CMTPLAY.WAV"));
	drec->load_sound_data(DATAREC_SNDFILE_RELAY_OFF, _T("CMTSTOP.WAV"));
#endif
#endif
	
	// VRAM/PCG wait
	memory->set_context_cpu(cpu);
	
	// memory mapped I/O
	memory->set_context_pio(pio);
	memory->set_context_pit(pit);
	
#if defined(_MZ1500)
	// psg mixer
	psg->set_context_psg_l(psg_l);
	psg->set_context_psg_r(psg_r);
#endif
	
#if defined(_MZ800)
	// 8253:CLK#0 <- 1.10MHz
	pit->set_constant_clock(0, 1100000);
#else
	// 8253:CLK#0 <- 895KHz
	pit->set_constant_clock(0, CPU_CLOCKS / 4);
#endif
	
#if defined(_MZ800) || defined(_MZ1500)
	// 8253:OUT#0 AND 8255:PC0 -> SPEAKER
	pit->set_context_ch0(and_snd, SIG_AND_BIT_0, 1);
	pio->set_context_port_c(and_snd, SIG_AND_BIT_1, 1, 0);
	and_snd->set_context_out(pcm, SIG_PCM1BIT_SIGNAL, 1);
	and_snd->set_mask(SIG_AND_BIT_0 | SIG_AND_BIT_1);
#else
	// 8253:OUT#0 -> SPEAKER
	pit->set_context_ch0(pcm, SIG_PCM1BIT_SIGNAL, 1);
#endif
#if defined(_MZ800)
	// 8253:OUT#0 -> NOT -> Z80PIO:PA4
	pit->set_context_ch0(not_pit, SIG_NOT_INPUT, 1);
	not_pit->set_context_out(pio_int, SIG_Z80PIO_PORT_A, 0x10);
#elif defined(_MZ1500)
	// 8253:OUT#0 -> Z80PIO:PA4
	pit->set_context_ch0(pio_int, SIG_Z80PIO_PORT_A, 0x10);
#endif
	
	// 8253:CLK#1 <- 15.7KHz
	pit->set_constant_clock(1, CPU_CLOCKS / 228);
	
	// 8253:OUT#1 -> 8253:CLK#2
	pit->set_context_ch1(pit, SIG_I8253_CLOCK_2, 1);
	
	// 8253:OUT#2 (N)AND 8255:PC2 -> Z80:INT
	pit->set_context_ch2(and_int, SIG_AND_BIT_0, 1);
	pio->set_context_port_c(and_int, SIG_AND_BIT_1, 4, 0);
	and_int->set_context_out(cpu, SIG_CPU_IRQ, 1);
	and_int->set_mask(SIG_AND_BIT_0 | SIG_AND_BIT_1);
	
#if defined(_MZ1500)
	// 8253:OUT#2 -> Z80PIO:PA5
	pit->set_context_ch2(pio_int, SIG_Z80PIO_PORT_A, 0x20);
#endif
	
	// 8255:PA0-3 -> KEYBOARD:STROBE
	pio->set_context_port_a(keyboard, SIG_KEYBOARD_COLUMN, 0x0f, 0);
#if defined(_MZ800)
	// 8255:PA4 -> JOYSTICK #1
	// 8255:PA5 -> JOYSTICK #2
#endif
	// 8255:PA7 -> 556 RESET
	
	// 8255:PB0-7 <- KEYBOARD:DATA
	keyboard->set_context_pio(pio);
	
	// 8255:PC0 -> AND -> SPEAKER
	// 8255:PC1 -> DATA RECORDER:WRITE DATA
	pio->set_context_port_c(drec, SIG_DATAREC_MIC, 0x02, 0);
	// 8255:PC2 -> (N)AND -> Z80:INT
	// 8255:PC3 -> DATA RECORDER:MOTOR ON/OFF
	pio->set_context_port_c(drec, SIG_DATAREC_TRIG, 0x08, 0);
	// 8255:PC4 <- DATA RECORDER:MOTOR REMOTE
	drec->set_context_remote(pio, SIG_I8255_PORT_C, 0x10);
	// 8255:PC5 <- DATA RECORDER:READ DATA
	drec->set_context_ear(pio, SIG_I8255_PORT_C, 0x20);
	// 8255:PC6 <- MEMORY:556 OUT (1.5KHz)
	// 8255:PC7 <- MEMORY:VBLANK
	
#if defined(_MZ800) || defined(_MZ1500)
	// Z80PIO:PA2 <- GND
	// Z80PIO:PA3 <- GND
#if defined(_MZ800)
	// Z80PIO:PA4 <- NOT <- 8253:OUT#0
	// Z80PIO:PA5 <- HBLANK
	memory->set_context_pio_int(pio_int);
#elif defined(_MZ1500)
	// Z80PIO:PA0 <- PRINTER:RDA (BUSY)
	// Z80PIO:PA1 <- PRINTER:STA (PE)
	if(config.printer_device_type == 0) {
		PRNFILE *prnfile = (PRNFILE *)printer;
		prnfile->set_context_busy(pio_int, SIG_Z80PIO_PORT_A, 0x01);
	} else if(config.printer_device_type == 1) {
		MZ1P17 *mz1p17 = (MZ1P17 *)printer;
		mz1p17->mode = MZ1P17_MODE_MZ2;
		mz1p17->set_context_busy(pio_int, SIG_Z80PIO_PORT_A, 0x01);
	}
	// Z80PIO:PA4 <- 8253:OUT#0
	// Z80PIO:PA5 <- 8253:OUT#2
	// Z80PIO:PA6 -> NOT -> PRINTER:IRT (RESET)
	// Z80PIO:PA7 -> NOT -> PRINTER:RDP (STROBE)
	// Z80PIO:PB  -> PRINTER:DATA
	pio_int->set_context_port_a(not_reset, SIG_NOT_INPUT, 0x40, 0);
	not_reset->set_context_out(printer, SIG_PRINTER_RESET, 0x01);
	pio_int->set_context_port_a(not_strobe, SIG_NOT_INPUT, 0x80, 0);
	not_strobe->set_context_out(printer, SIG_PRINTER_STROBE, 0x01);
	pio_int->set_context_port_b(printer, SIG_PRINTER_DATA, 0xff, 0);
#endif
#endif
	
#if defined(_MZ800) || defined(_MZ1500)
	// Z80SIO:RTSA -> QD:WRGA
	sio_qd->set_context_rts(0, qd, QUICKDISK_SIO_RTSA, 1);
	// Z80SIO:DTRB -> QD:MTON
	sio_qd->set_context_dtr(1, qd, QUICKDISK_SIO_DTRB, 1);
	// Z80SIO:SENDA -> QD:RECV
	sio_qd->set_context_sync(0, qd, QUICKDISK_SIO_SYNC, 1);
	sio_qd->set_context_rxdone(0, qd, QUICKDISK_SIO_RXDONE, 1);
	sio_qd->set_context_send(0, qd, QUICKDISK_SIO_DATA);
	sio_qd->set_context_break(0, qd, QUICKDISK_SIO_BREAK, 1);
	// Z80SIO:CTSA <- QD:PROTECT
	// Z80SIO:DCDA <- QD:INSERT
	// Z80SIO:DCDB <- QD:HOE
	qd->set_context_sio(sio_qd);
	
	sio_rs->set_tx_clock(0, 1200 * 16);	// 1200 baud
	sio_rs->set_rx_clock(0, 1200 * 16);	// baud-rate can be changed by jumper pin
	sio_rs->set_tx_clock(1, 1200 * 16);
	sio_rs->set_rx_clock(1, 1200 * 16);
	
	sio_qd->set_tx_clock(0, 101562.5);
	sio_qd->set_rx_clock(0, 101562.5);
	sio_qd->set_tx_clock(1, 101562.5);
	sio_qd->set_rx_clock(1, 101562.5);
	
	// floppy drives
	floppy->set_context_cpu(cpu);
	floppy->set_context_fdc(fdc);
	fdc->set_context_drq(floppy, SIG_FLOPPY_DRQ, 1);
#endif
	
	// cpu bus
	cpu->set_context_mem(memory);
	cpu->set_context_io(io);
#if defined(_MZ800) || defined(_MZ1500)
	cpu->set_context_intr(pio_int);
	// z80 family daisy chain
	// 0=8253:OUT2
	pio_int->set_context_intr(cpu, 1);
	pio_int->set_context_child(sio_rs);
	sio_rs->set_context_intr(cpu, 2);
	sio_rs->set_context_child(sio_qd);
	sio_qd->set_context_intr(cpu, 3);
#else
	cpu->set_context_intr(dummy);
#endif
#ifdef USE_DEBUGGER
	cpu->set_context_debugger(new DEBUGGER(this, emu));
#endif
	
	// emm
	io->set_iomap_range_rw(0x00, 0x03, emm);
	// kanji
	io->set_iomap_range_rw(0xb8, 0xb9, kanji);
	// ramfile
	io->set_iomap_range_rw(0xea, 0xeb, ramfile);
	// cmos
//	io->set_iomap_range_rw(0xf8, 0xfa, cmos);
	
#if defined(_MZ800)
	// 8255/8253
	io->set_iomap_range_rw(0xd0, 0xd3, pio);
	io->set_iomap_range_rw(0xd4, 0xd7, pit);
#endif
	
#if defined(_MZ800) || defined(_MZ1500)
	// floppy drives
	io->set_iomap_range_rw(0xd8, 0xdb, fdc);
	io->set_iomap_range_w(0xdc, 0xdf, floppy);
#endif
	
	// memory mapper
#if defined(_MZ800)
	io->set_iomap_range_r(0xe0, 0xe1, memory);
	io->set_iomap_range_w(0xe0, 0xe6, memory);
#elif defined(_MZ1500)
	io->set_iomap_range_w(0xe0, 0xe6, memory);
#else
	io->set_iomap_range_w(0xe0, 0xe4, memory);
#endif
	
#if defined(_MZ800)
	// crtc
	io->set_iomap_range_w(0xcc, 0xcf, memory);
	io->set_iomap_single_r(0xce, memory);
	// palette
	io->set_iomap_single_w(0xf0, memory);
#elif defined(_MZ1500)
	// palette
	io->set_iomap_range_w(0xf0, 0xf1, memory);
#endif
	
#if defined(_MZ800)
	// joystick
//	io->set_iomap_range_r(0xf0, 0xf1, joystick);
#endif
	
	// psg
#if defined(_MZ800)
	io->set_iomap_single_w(0xf2, psg);
#elif defined(_MZ1500)
	io->set_iomap_single_w(0xe9, psg);
	io->set_iomap_single_w(0xf2, psg_l);
	io->set_iomap_single_w(0xf3, psg_r);
#endif
	
#if defined(_MZ800) || defined(_MZ1500)
	// z80pio/sio
	// z80pio and z80sio*2
	static const int z80_sio_addr[4] = {0, 2, 1, 3};
	static const int z80_pio_addr[4] = {1, 3, 0, 2};
	for(int i = 0; i < 4; i++) {
		io->set_iomap_alias_rw(0xb0 + i, sio_rs, z80_sio_addr[i]);
		io->set_iomap_alias_rw(0xf4 + i, sio_qd, z80_sio_addr[i]);
		io->set_iomap_alias_rw(0xfc + i, pio_int, z80_pio_addr[i]);
	}
#else
	// printer
	io->set_iovalue_single_r(0xfe, 0xc0);
#endif

	// initialize all devices
	for(DEVICE* device = first_device; device; device = device->next_device) {
		device->initialize();
	}
#if defined(_MZ800) || defined(_MZ1500)
	for(int i = 0; i < MAX_DRIVE; i++) {
		fdc->set_drive_type(i, DRIVE_TYPE_2DD);
	}
#endif
}

VM::~VM()
{
	// delete all devices
	for(DEVICE* device = first_device; device;) {
		DEVICE *next_device = device->next_device;
		device->release();
		delete device;
		device = next_device;
	}
}

DEVICE* VM::get_device(int id)
{
	for(DEVICE* device = first_device; device; device = device->next_device) {
		if(device->this_device_id == id) {
			return device;
		}
	}
	return NULL;
}

// ----------------------------------------------------------------------------
// drive virtual machine
// ----------------------------------------------------------------------------

void VM::reset()
{
	// reset all devices
	for(DEVICE* device = first_device; device; device = device->next_device) {
		device->reset();
	}
	and_int->write_signal(SIG_AND_BIT_0, 0, 1);	// CLOCK = L
	and_int->write_signal(SIG_AND_BIT_1, 1, 1);	// INTMASK = H
#if defined(_MZ800) || defined(_MZ1500)
	and_snd->write_signal(SIG_AND_BIT_1, 1, 1);	// SNDMASK = H
#endif
#if defined(_MZ1500)
	pio_int->write_signal(SIG_Z80PIO_PORT_A, 0x02, 0x03);	// BUSY = L, PE = H
#endif
}

void VM::run()
{
	event->drive();
}

// ----------------------------------------------------------------------------
// debugger
// ----------------------------------------------------------------------------

#ifdef USE_DEBUGGER
DEVICE *VM::get_cpu(int index)
{
	if(index == 0) {
		return cpu;
	}
	return NULL;
}
#endif

// ----------------------------------------------------------------------------
// draw screen
// ----------------------------------------------------------------------------

void VM::draw_screen()
{
	memory->draw_screen();
}

#if defined(_MZ800) || defined(_MZ1500)
uint32_t VM::get_access_lamp_status()
{
	uint32_t status = fdc->read_signal(0) | qd->read_signal(0);
	return (status & (1 | 4)) ? 1 : (status & (2 | 8)) ? 2 : 0;
}
#endif

// ----------------------------------------------------------------------------
// soud manager
// ----------------------------------------------------------------------------

void VM::initialize_sound(int rate, int samples)
{
	// init sound manager
	event->initialize_sound(rate, samples);
	
	// init sound gen
	pcm->initialize_sound(rate, 8000);
#if defined(_MZ800)
	psg->initialize_sound(rate, 3579545, 8000);
#elif defined(_MZ1500)
	psg_l->initialize_sound(rate, 3579545, 8000);
	psg_r->initialize_sound(rate, 3579545, 8000);
#endif
}

uint16_t* VM::create_sound(int* extra_frames)
{
	return event->create_sound(extra_frames);
}

int VM::get_sound_buffer_ptr()
{
	return event->get_sound_buffer_ptr();
}

#ifdef USE_SOUND_VOLUME
void VM::set_sound_device_volume(int ch, int decibel_l, int decibel_r)
{
#if defined(_MZ800)
	if(ch-- == 0) {
		psg->set_volume(0, decibel_l, decibel_r);
	} else
#elif defined(_MZ1500)
	if(ch-- == 0) {
		psg_l->set_volume(0, decibel_l, decibel_r);
	} else if(ch-- == 0) {
		psg_r->set_volume(0, decibel_l, decibel_r);
	} else
#endif
	if(ch-- == 0) {
		pcm->set_volume(0, decibel_l, decibel_r);
	} else if(ch-- == 0) {
		drec->set_volume(0, decibel_l, decibel_r);
	}
#if defined(USE_SOUND_FILES)
#if defined(_MZ1500) || defined(_MZ800)
	else if(ch-- == 0) {
		fdc->set_volume(MB8877_SND_TYPE_SEEK, decibel_l, decibel_r);
	}
#endif
	else if(ch-- == 0) {
		drec->set_volume(2 + DATAREC_SNDFILE_RELAY_ON, decibel_l, decibel_r);
		drec->set_volume(2 + DATAREC_SNDFILE_RELAY_OFF, decibel_l, decibel_r);
		drec->set_volume(2 + DATAREC_SNDFILE_EJECT, decibel_l, decibel_r);
	}		
#endif
}
#endif

// ----------------------------------------------------------------------------
// user interface
// ----------------------------------------------------------------------------

void VM::play_tape(const _TCHAR* file_path)
{
	drec->play_tape(file_path);
	drec->set_remote(true);
}

void VM::rec_tape(const _TCHAR* file_path)
{
	drec->rec_tape(file_path);
	drec->set_remote(true);
}

void VM::close_tape()
{
#if defined(USE_SOUND_FILES)
	drec->write_signal(SIG_SOUNDER_ADD + DATAREC_SNDFILE_EJECT, 1, 1);
#endif
	emu->lock_vm();
	drec->close_tape();
	emu->unlock_vm();

	drec->set_remote(false);
}

bool VM::is_tape_inserted()
{
	return drec->is_tape_inserted();
}

bool VM::is_tape_playing()
{
	return drec->is_tape_playing();
}

bool VM::is_tape_recording()
{
	return drec->is_tape_recording();
}

int VM::get_tape_position()
{
	return drec->get_tape_position();
}

void VM::push_play()
{
	drec->set_ff_rew(0);
	drec->set_remote(true);
}

void VM::push_stop()
{
	drec->set_remote(false);
}

void VM::push_fast_forward()
{
	drec->set_ff_rew(1);
	drec->set_remote(true);
}

void VM::push_fast_rewind()
{
	drec->set_ff_rew(-1);
	drec->set_remote(true);
}

#if defined(_MZ800) || defined(_MZ1500)
void VM::open_quick_disk(int drv, const _TCHAR* file_path)
{
	if(drv == 0) {
		qd->open_disk(file_path);
	}
}

void VM::close_quick_disk(int drv)
{
	if(drv == 0) {
		qd->close_disk();
	}
}

bool VM::is_quick_disk_inserted(int drv)
{
	if(drv == 0) {
		return qd->is_disk_inserted();
	} else {
		return false;
	}
}

void VM::open_floppy_disk(int drv, const _TCHAR* file_path, int bank)
{
	fdc->open_disk(drv, file_path, bank);
}

void VM::close_floppy_disk(int drv)
{
	fdc->close_disk(drv);
}

bool VM::is_floppy_disk_inserted(int drv)
{
	return fdc->is_disk_inserted(drv);
}

void VM::is_floppy_disk_protected(int drv, bool value)
{
	fdc->is_disk_protected(drv, value);
}

bool VM::is_floppy_disk_protected(int drv)
{
	return fdc->is_disk_protected(drv);
}
#endif

bool VM::is_frame_skippable()
{
	return event->is_frame_skippable();
}

void VM::update_config()
{
#if defined(_MZ800)
	if(boot_mode != config.boot_mode) {
		// boot mode is changed !!!
		boot_mode = config.boot_mode;
		reset();
	} else {
#endif
		for(DEVICE* device = first_device; device; device = device->next_device) {
			device->update_config();
		}
#if defined(_MZ800)
	}
#endif
}

#define STATE_VERSION	1

void VM::save_state(FILEIO* state_fio)
{
	state_fio->FputUint32(STATE_VERSION);
	
	for(DEVICE* device = first_device; device; device = device->next_device) {
		device->save_state(state_fio);
	}
#if defined(_MZ800)
	state_fio->FputInt32(boot_mode);
#endif
}

bool VM::load_state(FILEIO* state_fio)
{
	if(state_fio->FgetUint32() != STATE_VERSION) {
		return false;
	}
	for(DEVICE* device = first_device; device; device = device->next_device) {
		if(!device->load_state(state_fio)) {
			return false;
		}
	}
#if defined(_MZ800)
	boot_mode = state_fio->FgetInt32();
#endif
	return true;
}

