/*
 * Cueplot: a GUI front-end to gnuplot
 * Copyright (C) 2007, 2008 Muneyuki Noguchi
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation, 
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
#include "ConfigDialog.h"
#include "FileLineEdit.h"
#include "Process.h"

#include <QtCore/QCoreApplication>
#include <QtCore/QSettings>
#if QT_VERSION >= 0x040200
# include <QtGui/QDialogButtonBox>
#endif
#include <QtGui/QFileDialog>
#include <QtGui/QGridLayout>
#include <QtGui/QGroupBox>
#include <QtGui/QLabel>
#include <QtGui/QLineEdit>
#include <QtGui/QMessageBox>
#include <QtGui/QPushButton>
#include <QtGui/QRadioButton>

ConfigDialog *ConfigDialog::instance_ = 0;

ConfigDialog &ConfigDialog::instance(QWidget *parent)
{
	if (!instance_) {
		instance_ = new ConfigDialog(parent);
	}
	return *instance_;
}

void ConfigDialog::destroyInstance()
{
	if (instance_) {
		delete instance_;
		instance_ = 0;
	}
}

ConfigDialog::ConfigDialog(QWidget *parent)
: QDialog(parent)
{
	setWindowTitle(
#ifdef Q_OS_DARWIN
			tr("Preferences")
#else
			tr("Configure")
#endif
			+ " - "
			+ QCoreApplication::applicationName());

	defaultRadioButton = new QRadioButton;
	defaultRadioButton->setText(tr("%1 &which is in %2")
			.arg(Process::FILE_NAME).arg(
#ifdef Q_OS_WIN32
				"%PATH%"
#else
				"$PATH"
#endif
				));

	pathRadioButton = new QRadioButton;
	pathRadioButton->setText(tr("&Path:"));

	pathLineEdit = new FileLineEdit;
	connect(pathRadioButton, SIGNAL(toggled(bool)),
			pathLineEdit, SLOT(setEnabled(bool)));

	pathPushButton = new QPushButton;
	pathPushButton->setText(tr("Op&en..."));
	pathPushButton->setAutoDefault(false);
	pathPushButton->setDefault(false);
	connect(pathRadioButton, SIGNAL(toggled(bool)),
			pathPushButton, SLOT(setEnabled(bool)));
	connect(pathPushButton, SIGNAL(clicked()),
			this, SLOT(openFile()));

	QGridLayout *pathLayout = new QGridLayout;
	pathLayout->addWidget(defaultRadioButton, 0, 0, 1, 3);
	pathLayout->addWidget(pathRadioButton, 1, 0);
	pathLayout->addWidget(pathLineEdit, 1, 1);
	pathLayout->addWidget(pathPushButton, 1, 2);

	QGroupBox *pathGroupBox = new QGroupBox(tr("Path to %1")
			.arg(Process::FILE_NAME));
	pathGroupBox->setLayout(pathLayout);

	QPushButton *acceptPushButton = new QPushButton;
	acceptPushButton->setText(tr("&OK"));
	acceptPushButton->setAutoDefault(true);
	acceptPushButton->setDefault(true);
	connect(acceptPushButton, SIGNAL(clicked()), this, SLOT(accept()));

	QPushButton *applyPushButton = new QPushButton;
	applyPushButton->setText(tr("&Apply"));
	connect(applyPushButton, SIGNAL(clicked()), this, SLOT(apply()));

	QPushButton *cancelPushButton = new QPushButton;
	cancelPushButton->setText(tr("&Cancel"));
	connect(cancelPushButton, SIGNAL(clicked()), this, SLOT(reject()));

	// ------------------------ create layouts
	// ----------- assemble layouts and widgets

#if QT_VERSION >= 0x040200
	QDialogButtonBox *dialogButtonBox = new QDialogButtonBox;
	dialogButtonBox->addButton(acceptPushButton, 
			QDialogButtonBox::AcceptRole);
	dialogButtonBox->addButton(applyPushButton,
			QDialogButtonBox::ApplyRole);
	dialogButtonBox->addButton(cancelPushButton,
			QDialogButtonBox::RejectRole);
#else
	QHBoxLayout *buttonLayout = new QHBoxLayout;
	buttonLayout->addStretch();
	buttonLayout->addWidget(acceptPushButton);
	buttonLayout->addWidget(applyPushButton);
	buttonLayout->addWidget(cancelPushButton);
#endif

	QVBoxLayout *mainLayout = new QVBoxLayout;
	mainLayout->addWidget(pathGroupBox);
	mainLayout->addStretch();
#if QT_VERSION >= 0x040200
	mainLayout->addWidget(dialogButtonBox);
#else
	mainLayout->addLayout(buttonLayout);
#endif

	setLayout(mainLayout);
}

void ConfigDialog::setProcess(Process &process)
{
	process_ = &process;

	QSettings settings;
	settings.beginGroup("Process");

	QString path = settings.value("path").toString();
	if (path == Process::FILE_NAME) {
		defaultRadioButton->setChecked(true);
		pathLineEdit->setEnabled(false);
		pathPushButton->setEnabled(false);
	} else {
		pathRadioButton->setChecked(true);
		pathLineEdit->setFile(path);
	}

	settings.endGroup();
}

void ConfigDialog::accept()
{
	apply();
	QDialog::accept();
}

void ConfigDialog::apply()
{
	QSettings settings;
	settings.beginGroup("Process");
	QString path;
	if (defaultRadioButton->isChecked()) {
		path = Process::FILE_NAME;
	} else {
		path = pathLineEdit->text();
	}
	if (path != settings.value("path").toString()) {
		settings.setValue("path", path);
		QMessageBox::information(this, tr("Restart Required") + " - " 
				+ QCoreApplication::applicationName(), 
				tr("You will have to restart %1 for the change "
					"to take effect.")
				.arg(QCoreApplication::applicationName()));
	}
	settings.endGroup();
}

void ConfigDialog::openFile()
{
	QString path = QFileDialog::getOpenFileName(0,
			tr("Open gnuplot") + " - "
			+ QCoreApplication::applicationName(), pathLineEdit->text(),
			"gnuplot (" + Process::FILE_NAME + ")");
	if (!path.isNull()) {
		pathLineEdit->setFile(path);

	}
}
