/*
 * Cueplot: a GUI front-end to gnuplot
 * Copyright (C) 2007 Muneyuki Noguchi
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation, 
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
#include "FitDialog.h"
#include "FitBasicWidget.h"
#include "FitModifierWidget.h"
#include "FitOthersWidget.h"
#include "FitRangeWidget.h"
#include "Fit.h"

#include <QtCore/QCoreApplication>
#if QT_VERSION >= 0x040200
# include <QtGui/QDialogButtonBox>
#endif
#include <QtGui/QListWidget>
#include <QtGui/QHBoxLayout>
#include <QtGui/QPushButton>
#include <QtGui/QSplitter>
#include <QtGui/QStackedWidget>


FitDialog *FitDialog::instance_ = 0;

FitDialog &FitDialog::instance(QWidget *parent)
{
	if (!instance_) {
		instance_ = new FitDialog(parent);
		instance_->init();
	}
	return *instance_;
}

void FitDialog::destroyInstance()
{
	if (instance_) {
		delete instance_;
		instance_ = 0;
	}
}

FitDialog::FitDialog(QWidget *parent)
: QDialog(parent)
{
	setWindowTitle(tr("Fittings") + " - " 
			+ QCoreApplication::applicationName());

	fitBasicWidget = new FitBasicWidget;
	connect(fitBasicWidget, SIGNAL(optionChanged()),
			this, SLOT(setPushButton()));
	fitRangeWidget = new FitRangeWidget;
	fitModifierWidget = new FitModifierWidget;
	fitOthersWidget = new FitOthersWidget;

	pagesStackedWidget = new QStackedWidget;
	pagesStackedWidget->addWidget(fitBasicWidget);
	pagesStackedWidget->addWidget(fitRangeWidget);
	pagesStackedWidget->addWidget(fitModifierWidget);
	pagesStackedWidget->addWidget(fitOthersWidget);

	contentsListWidget = new QListWidget;
	for (int i = 0, j = pagesStackedWidget->count(); i < j; i++) {
		QWidget *widget = pagesStackedWidget->widget(i);
		contentsListWidget->addItem(widget->windowTitle());
		contentsListWidget->item(i)->setWhatsThis(widget->whatsThis());
	}
	contentsListWidget->setCurrentRow(0);
	connect(contentsListWidget,
			SIGNAL(currentItemChanged(QListWidgetItem *, QListWidgetItem *)),
			this, SLOT(changePage(QListWidgetItem *, QListWidgetItem *)));

	QSplitter *centerSplitter = new QSplitter;
	centerSplitter->addWidget(contentsListWidget);
	centerSplitter->addWidget(pagesStackedWidget);

	fitPushButton = new QPushButton;
	connect(fitPushButton, SIGNAL(clicked()), SLOT(fit()));
	fitPushButton->setText(tr("&Fit"));

	QPushButton *closePushButton = new QPushButton;
	closePushButton->setText(tr("&Close"));
	connect(closePushButton, SIGNAL(clicked()), this, SLOT(accept()));

	// create layouts
#if QT_VERSION >= 0x040200
	QDialogButtonBox *dialogButtonBox = new QDialogButtonBox;
	dialogButtonBox->addButton(fitPushButton,
			QDialogButtonBox::ActionRole);
	dialogButtonBox->addButton(closePushButton,
			QDialogButtonBox::RejectRole);
#else
	QHBoxLayout *buttonLayout = new QHBoxLayout;
	buttonLayout->addStretch();
	buttonLayout->addWidget(fitPushButton);
	buttonLayout->addWidget(closePushButton);
#endif

	QVBoxLayout *mainLayout = new QVBoxLayout;
	mainLayout->addWidget(centerSplitter);
#if QT_VERSION >= 0x040200
	mainLayout->addWidget(dialogButtonBox);
#else
	mainLayout->addLayout(buttonLayout);
#endif

	setLayout(mainLayout);
}

void FitDialog::init()
{
	Fit fit;
	fitBasicWidget->setFitFunctionName(fit.functionName());
	fitBasicWidget->setFitFunction(fit.function());
	fitBasicWidget->setFitDataFile(fit.dataFile());
	fitBasicWidget->setFitParameterFile(fit.isParameterFile());
	fitBasicWidget->setFitParameterFile(fit.parameterFile());
	fitBasicWidget->setFitVariable(fit.variable());

	fitModifierWidget->setFitModifier(fit.dataFileModifier());

	fitOthersWidget->setFitLimit(fit.fitLimit());
	fitOthersWidget->setFitMaxIter(fit.fitMaxIter());

	setPushButton();
}

void FitDialog::fit()
{
	Fit fit;

	fit.setFunctionName(fitBasicWidget->fitFunctionName());
	fit.setFunction(fitBasicWidget->fitFunction());
	fit.setDataFile(fitBasicWidget->fitDataFile());
	fit.setParameterFile(fitBasicWidget->isFitParameterFile());
	fit.setParameterFile(fitBasicWidget->fitParameterFile());
	fit.setVariable(fitBasicWidget->fitVariable());

	fit.setDataFileModifier(fitModifierWidget->fitModifier());

	fit.setFitLimit(fitOthersWidget->fitLimit());
	fit.setFitMaxIter(fitOthersWidget->fitMaxIter());

	emit executed(fit.command());
}

void FitDialog::setPushButton()
{
	fitPushButton->setEnabled(fitBasicWidget->canFit());
}

void FitDialog::changePage(QListWidgetItem *current,
		QListWidgetItem *previous)
{
	if (!current) {
		current = previous;
	}
	pagesStackedWidget->setCurrentIndex(contentsListWidget->row(current));
}
