/*
 * Cueplot: a GUI front-end to gnuplot
 * Copyright (C) 2007, 2009 Muneyuki Noguchi <nogu@users.sourceforge.jp>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation, 
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
#ifndef FIT_INCLUDED
#define FIT_INCLUDED

#include <QtCore/QString>
#include <QtCore/QStringList>

#include "abstractcommand.h"
#include "fitrange.h"
#include "fitmodifier.h"

//! fit コマンド
class Fit : public AbstractCommand
{
	public:
		Fit();

		/*!
		 * \brief フィッティングする範囲を設定する
		 * \param range フィッティングする範囲
		 */
		void setRange(const FitRange &range);
		/*!
		 * \brief フィッティングする範囲を得る
		 * \return フィッティングする範囲
		 */
		const FitRange &range() const;

		/*!
		 * \brief フィッティングに用いる関数の名前を設定する
		 * \param functionName フィッティングに用いる関数の名前
		 */
		void setFunctionName(const QString &functionName);
		/*!
		 * \brief フィッティングに用いる関数の名前を得る
		 * \return フィッティングに用いる関数の名前
		 */
		const QString &functionName() const;

		/*!
		 * \brief フィッティングに用いる関数の定義を設定する
		 * \param function フィッティングに用いる関数の定義
		 */
		void setFunction(const QString &function);
		/*!
		 * \brief フィッティングに用いる関数の定義を得る
		 * \return フィッティングに用いる関数の定義
		 */
		const QString &function() const;

		/*!
		 * \brief フィッティングに用いるデータファイルの名前を設定する
		 * \param file フィッティングに用いるデータファイルの名前
		 */
		void setDataFile(const QString &file);
		/*!
		 * \brief フィッティングに用いるデータファイルの名前を得る
		 * \return フィッティングに用いるデータファイルの名前
		 */
		const QString &dataFile() const;

		/*!
		 * \brief データファイルの修飾子を設定する
		 * \param mod 設定するデータファイルの修飾子
		 */
		void setDataFileModifier(const FitModifier &mod);
		/*!
		 * \brief データファイルの修飾子を得る
		 * \return データファイルの修飾子
		 */
		const FitModifier &dataFileModifier() const;

		/*!
		 * \brief ファイルでパラメータを指定するかどうかを設定する
		 * \param isFile ファイルでパラメータを指定するかどうか
		 */
		void setParameterFile(bool isFile);
		/*!
		 * \brief ファイルでパラメータを指定するかどうかを返す
		 * \return ファイルでパラメータを指定するならば true
		 */
		bool isParameterFile() const;

		/*!
		 * \brief パラメータファイルの名前を設定する
		 * \param file 設定するパラメータファイルの名前
		 */
		void setParameterFile(const QString &file);
		/*!
		 * \brief パラメータファイルの名前を得る
		 * \return パラメータファイルの名前
		 */
		const QString &parameterFile() const;

		/*!
		 * \brief パラメータ変数の名前を設定する
		 * \return 設定するパラメータ変数の名前
		 */
		void setVariable(const QStringList &var);
		/*!
		 * \brief パラメータ変数の名前を設定する
		 * \return パラメータ変数の名前
		 */
		const QStringList &variable() const;

		/*!
		 * \brief フィッティングの限界を設定する
		 * \param limit 設定するフィッティングの限界
		 */
		void setFitLimit(double limit);
		/*!
		 * \brief フィッティングの限界を得る
		 * \return フィッティングの限界
		 */
		double fitLimit() const;

		/*!
		 * \brief フィッティングの最大繰り返し数を指定する
		 * \param iter 設定するフィッティングの最大繰り返し数
		 */
		void setFitMaxIter(int iter);
		/*!
		 * \brief フィッティングの最大繰り返し数を得る
		 * \return フィッティングの最大繰り返し数
		 */
		int fitMaxIter() const;

		QString command() const;

	private:
		static const double DEFAULT_FIT_LIMIT;
		static const int DEFAULT_FIT_MAX_ITER = 0;
		FitRange range_;
		QString functionName_;
		QString function_;
		QString dataFile_;
		FitModifier dataFileModifier_;
		bool isParameterFile_;
		QString parameterFile_;
		QStringList variable_;
		double fitLimit_;
		int fitMaxIter_;
};

#endif // FIT_INCLUDED
