/*
 * Cueplot: a GUI front-end to gnuplot
 * Copyright (C) 2007, 2009 Muneyuki Noguchi <nogu@users.sourceforge.jp>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation, 
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
#ifndef PLOT_INCLUDED
#define PLOT_INCLUDED

#include <QtCore/QString>

#include "abstractcommand.h"
#include "plotrange.h"
#include "plotmodifier.h"
#include "plottitle.h"
#include "plotstyle.h"

//! plot コマンド
class Plot : public AbstractCommand
{
	public:
		Plot();

		/*!
		 * \brief プロットする範囲を設定する
		 * \param range プロットする範囲
		 */
		void setRange(const PlotRange &range);
		/*!
		 * \brief プロットする範囲を得る
		 * \return プロットする範囲
		 */
		const PlotRange &range() const;

		/*!
		 * \brief 関数をプロットするかどうかを設定する
		 * \param isFunction データではなく関数をプロットするかどうか
		 */
		void setFunction(bool isFunction);
		/*!
		 * \brief 関数をプロットするかどうかを返す
		 * \return データではなく関数をプロットするならば true
		 */
		bool isFunction() const;

		/*!
		 * \brief プロットする関数を設定する
		 * \param name プロットする関数
		 */
		void setFunction(const QString &name);
		/*!
		 * \brief プロットする関数を得る
		 * \return プロットする関数
		 */
		const QString &function() const;

		/*!
		 * \brief プロットするデータファイルの名前を設定する
		 * \param name プロットするデータファイルの名前
		 */
		void setDataFile(const QString &name);
		/*!
		 * \brief プロットするデータファイルの名前を得る
		 * \return プロットするデータファイルの名前
		 */
		const QString &dataFile() const;

		/*!
		 * \brief データファイルの修飾子を設定する
		 * \param mod 設定するデータファイルの修飾子
		 */
		void setDataFileModifier(const PlotModifier &mod);
		/*!
		 * \brief データファイルの修飾子を得る
		 * \return データファイルの修飾子
		 */
		const PlotModifier &dataFileModifier() const;

		/*!
		 * \brief 軸を設定する
		 * \param axis 設定する軸
		 * \note 初期設定にするときは引数に負の数を指定する
		 * \sa Plot::AXIS
		 */
		void setAxis(int axis);
		/*!
		 * \brief 軸を得る
		 * \return 軸
		 * \note 初期設定のときは負の数を返す
		 * \sa Plot::AXIS
		 */
		int axis() const;

		/*!
		 * \brief タイトルの仕様を設定する
		 * \param title 設定するタイトルの仕様
		 */
		void setTitleSpecification(const PlotTitle &title);
		/*!
		 * \brief タイトルの仕様を得る
		 * \return タイトルの仕様
		 */
		const PlotTitle &titleSpecification() const;

		/*!
		 * \brief スタイルを設定する
		 * \param style 設定するタイトル
		 */
		void setStyle(const PlotStyle &style);
		/*!
		 * \brief スタイルを得る
		 * \return スタイル
		 */
		const PlotStyle &style() const;

		QString command() const;

		//! 軸の組み合わせ
		static const QString AXIS[];
	private:
		PlotRange range_;
		bool isFunction_;
		QString function_;
		QString dataFile_;
		PlotModifier dataFileModifier_;
		int axis_;
		PlotTitle titleSpecification_;
		PlotStyle style_;
};
#endif // PLOT_INCLUDED
