/*
 * Cueplot: a GUI front-end to gnuplot
 * Copyright (C) 2007, 2008, 2009 Muneyuki Noguchi <nogu@users.sourceforge.jp>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation, 
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
#ifndef PLOT_MODIFIER_INCLUDED
#define PLOT_MODIFIER_INCLUDED

#include <QtCore/QString>
#include <QtCore/QStringList>

#include "abstractcommandoption.h"

//! plot コマンドのオプション
class PlotModifier : public AbstractCommandOption
{
	public:
		PlotModifier();

		/*!
		 * \brief 最初のインデックスを設定する
		 * \param m 最初のインデックス
		 * \note 初期設定にするときは引数に負の数を指定する
		 */
		void setM(int m);
		/*!
		 * \brief 最初のインデックスを得る
		 * \return 最初のインデックス
		 * \note 初期設定のときは負の数を返す
		 */
		int m() const;

		/*!
		 * \brief 最後のインデックスを設定する
		 * \param n 最後のインデックス
		 * \note 初期設定にするときは引数に負の数を指定する
		 */
		void setN(int n);
		/*!
		 * \brief 最後のインデックスを得る
		 * \return 最後のインデックス
		 * \note 初期設定のときは負の数を返す
		 */
		int n() const;

		/*!
		 * \brief インデックスの差分を設定する
		 * \param p インデックスの差分
		 * \note 初期設定にするときは引数に負の数を指定する
		 */
		void setP(int p);
		/*!
		 * \brief インデックスの差分を得る
		 * \return インデックスの差分
		 * \note 初期設定のときは負の数を返す
		 */
		int p() const;

		/*!
		 * \brief 行の差分を設定する
		 * \param pointIncr 行の差分
		 * \note 初期設定にするときは引数に負の数を指定する
		 */
		void setPointIncrement(int pointIncr);
		/*!
		 * \brief 行の差分を得る
		 * \return 行の差分
		 * \note 初期設定のときは負の数を返す
		 */
		int pointIncrement() const;

		/*!
		 * \brief ブロックの差分を設定する
		 * \param blockIncr ブロックの差分
		 * \note 初期設定にするときは引数に負の数を指定する
		 */
		void setBlockIncrement(int blockIncr);
		/*!
		 * \brief ブロックの差分を得る
		 * \return ブロックの差分
		 * \note 初期設定のときは負の数を返す
		 */
		int blockIncrement() const;

		/*!
		 * \brief 開始行を設定する
		 * \param startPoint 開始行
		 * \note 初期設定にするときは引数に負の数を指定する
		 */
		void setStartPoint(int startPoint);
		/*!
		 * \brief 開始行を得る
		 * \return 開始行
		 * \note 初期設定のときは負の数を返す
		 */
		int startPoint() const;

		/*!
		 * \brief 開始ブロックを設定する
		 * \param startBlock 開始ブロック
		 * \note 初期設定にするときは引数に負の数を指定する
		 */
		void setStartBlock(int startBlock);
		/*!
		 * \brief 開始ブロックを得る
		 * \return 開始ブロック
		 * \note 初期設定のときは負の数を返す
		 */
		int startBlock() const;

		/*!
		 * \brief 終了行を設定する
		 * \param endPoint 終了行
		 * \note 初期設定にするときは引数に負の数を指定する
		 */
		void setEndPoint(int endPoint);
		/*!
		 * \brief 終了行を得る
		 * \return 終了行
		 * \note 初期設定のときは負の数を返す
		 */
		int endPoint() const;

		/*!
		 * \brief 終了ブロックを設定する
		 * \param endBlock 終了ブロック
		 * \note 初期設定にするときは引数に負の数を指定する
		 */
		void setEndBlock(int endBlock);
		/*!
		 * \brief 終了ブロックを得る
		 * \return 終了ブロック
		 * \note 初期設定のときは負の数を返す
		 */
		int endBlock() const;

		/*!
		 * \brief エントリを設定する
		 * \param axis 設定する軸
		 * \param entry 設定するエントリ
		 */
		void setEntry(int axis, const QString &entry);
		/*!
		 * \brief エントリを得る
		 * \param axis 軸
		 * \return エントリ
		 */
		const QString &entry(int axis) const;

		/*!
		 * \brief 書式を設定する
		 * \param format 設定する書式
		 */
		void setFormat(const QString &format);
		/*!
		 * \brief 書式を得る
		 * \return 書式
		 */
		const QString &format() const;

		/*!
		 * \brief 平滑化する方法を設定する
		 * \param smooth 設定する平滑化する方法
		 * \sa PlotModifier::SMOOTH
		 */
		void setSmooth(int smooth);
		/*!
		 * \brief 平滑化する方法を得る
		 * \return 平滑化する方法
		 * \sa PlotModifier::SMOOTH
		 */
		int smooth() const;

		QString option() const;

		//! 平滑化する方法
		static const QString SMOOTH[];
	private:
		static const int EVERY_NUM = 6;
		static const int AXIS_NUM = 3;
		int m_;
		int n_;
		int p_;
		int everyArray[EVERY_NUM];
		QStringList entry_;
		QString format_;
		int smooth_;
};
#endif // PLOT_MODIFIER_INCLUDED
