/*
 * Cueplot: a GUI front-end to gnuplot
 * Copyright (C) 2007, 2008 Muneyuki Noguchi <nogu@users.sourceforge.jp>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation, 
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
#include "fitmodifierwidget.h"

#include <limits> // max()

#include <QtGui/QCheckBox>
#include <QtGui/QLineEdit>
#include <QtGui/QGridLayout>
#include <QtGui/QSpinBox>
#include <QtGui/QTabWidget>
#include <QtGui/QVBoxLayout>

#include "fitmodifier.h"

FitModifierWidget::FitModifierWidget(QWidget *parent)
: QWidget(parent)
{
	setWindowTitle(tr("Data File"));
	setWhatsThis(tr("fit {index &lt;index list&gt;}<br>"
				"{every &lt;every list&gt;}<br>"
				"{using &lt;using list&gt;}"));

	const QString AXIS[] = { "&x", "&y", "&z" };
	for (int i = 0; i < AXIS_NUM; i++) {
		columnCheckBox[i] = new QCheckBox;
		columnCheckBox[i]->setText(tr("Specify a %1 column:").arg(AXIS[i]));

		columnLineEdit[i] = new QLineEdit;
		connect(columnCheckBox[i], SIGNAL(toggled(bool)),
				columnLineEdit[i], SLOT(setEnabled(bool)));
	}

	formatCheckBox = new QCheckBox;
	formatCheckBox->setText(tr("Specify a fo&rmat:"));

	formatLineEdit = new QLineEdit;
	connect(formatCheckBox, SIGNAL(toggled(bool)),
			formatLineEdit, SLOT(setEnabled(bool)));

	const QString INDEX_TEXT[] = {
		tr("Specify a s&tarting index:"),
		tr("Specify a &ending index:"),
		tr("Specify i&ncrement of index:")
	};
	const int INDEX_NUM = sizeof(INDEX_TEXT) / sizeof(INDEX_TEXT[0]);
	for (int i = 0; i < INDEX_NUM; i++) {
		QCheckBox *checkBox = new QCheckBox;
		checkBox->setText(INDEX_TEXT[i]);
		connect(checkBox, SIGNAL(toggled(bool)),
				this, SLOT(setIndex()));
		indexCheckBoxList.append(checkBox);

		QSpinBox *spinBox = new QSpinBox;
		spinBox->setMaximum(std::numeric_limits<int>::max());
		connect(checkBox, SIGNAL(toggled(bool)),
				spinBox, SLOT(setEnabled(bool)));
		indexSpinBoxList.append(spinBox);
	}

	const QString SAMPLING_TEXT[] = {
		tr("Specify &point increment:"),
		tr("Specify a s&tart point:"),
		tr("Specify a &end point:"),
		tr("Specify &block increment:"),
		tr("Specify a st&art block:"),
		tr("Specify a e&nd block:")
	};
	const int SAMPLING_NUM = sizeof(SAMPLING_TEXT) / sizeof(SAMPLING_TEXT[0]);
	for (int i = 0; i < SAMPLING_NUM; i++) {
		QCheckBox *checkBox = new QCheckBox;
		checkBox->setText(SAMPLING_TEXT[i]);
		samplingCheckBoxList.append(checkBox);

		QSpinBox *spinBox = new QSpinBox;
		spinBox->setMaximum(std::numeric_limits<int>::max());
		connect(checkBox, SIGNAL(toggled(bool)),
				spinBox, SLOT(setEnabled(bool)));
		samplingSpinBoxList.append(spinBox);
	}

	// assemble layouts and widgets

	QGridLayout *dataSetLayout = new QGridLayout;
	for (int i = 0; i < INDEX_NUM; i++) {
		dataSetLayout->addWidget(indexCheckBoxList[i], i, 0);
		dataSetLayout->addWidget(indexSpinBoxList[i], i, 1);
	}
	dataSetLayout->addItem(new QSpacerItem(0, 0, 
				QSizePolicy::Expanding, QSizePolicy::Minimum), 0, 2);
	dataSetLayout->addItem(new QSpacerItem(0, 0,
				QSizePolicy::Minimum, QSizePolicy::Expanding), INDEX_NUM, 0);

	QWidget *dataSetWidget = new QWidget;
	dataSetWidget->setWhatsThis(tr("fit {index &lt;index list&gt;}"));
	dataSetWidget->setLayout(dataSetLayout);

	QGridLayout *periodLayout = new QGridLayout;
	for (int i = 0; i < SAMPLING_NUM; i++) {
		periodLayout->addWidget(samplingCheckBoxList[i], i, 0);
		periodLayout->addWidget(samplingSpinBoxList[i], i, 1);
	}
	periodLayout->addItem(new QSpacerItem(0, 0,
				QSizePolicy::Expanding, QSizePolicy::Minimum), 0, 2);
	periodLayout->addItem(new QSpacerItem(0, 0,
				QSizePolicy::Minimum, QSizePolicy::Expanding), SAMPLING_NUM, 0);

	QWidget *periodWidget = new QWidget;
	periodWidget->setWhatsThis(tr("fit {every &lt;every list&gt;}"));
	periodWidget->setLayout(periodLayout);

	QGridLayout *colFormatLayout = new QGridLayout;
	for (int i = 0; i < AXIS_NUM; i++) {
		colFormatLayout->addWidget(columnCheckBox[i], i, 0);
		colFormatLayout->addWidget(columnLineEdit[i], i, 1);
	}
	colFormatLayout->addItem(new QSpacerItem(0, 0,
				QSizePolicy::Expanding, QSizePolicy::Minimum), 0, 2);
	colFormatLayout->addWidget(formatCheckBox, 3, 0);
	colFormatLayout->addWidget(formatLineEdit, 3, 1, 1, 2);
	colFormatLayout->addItem(new QSpacerItem(0, 0,
				QSizePolicy::Minimum, QSizePolicy::Expanding), 4, 0);

	QWidget *colFormatWidget = new QWidget;
	colFormatWidget->setWhatsThis(tr("fit {using &lt;using list&gt;}"));
	colFormatWidget->setLayout(colFormatLayout);

	QTabWidget *modTabWidget = new QTabWidget;
	modTabWidget->addTab(dataSetWidget, tr("&Indexes"));
	modTabWidget->addTab(periodWidget, tr("&Sampling"));
	modTabWidget->addTab(colFormatWidget, tr("C&olumns"));

	QVBoxLayout *mainLayout = new QVBoxLayout;
	mainLayout->addWidget(modTabWidget);

	setLayout(mainLayout);
}

FitModifierWidget::~FitModifierWidget()
{
	while (!indexCheckBoxList.isEmpty()) {
		delete indexCheckBoxList.takeFirst();
		delete indexSpinBoxList.takeFirst();
	}
	while (!samplingCheckBoxList.isEmpty()) {
		delete samplingCheckBoxList.takeFirst();
		delete samplingSpinBoxList.takeFirst();
	}
}

void FitModifierWidget::setIndex()
{
	bool flag = true;
	for (int i = 1, j = indexCheckBoxList.size(); i < j; i++) {
		flag = flag && indexCheckBoxList[i - 1]->isChecked();
		indexCheckBoxList[i]->setEnabled(flag);
		indexSpinBoxList[i]->setEnabled(
				flag && indexCheckBoxList[i]->isChecked());
	}
}

void FitModifierWidget::setModifier(const FitModifier &fitModifier)
{
	bool compFlag = true;
	for (int i = 0, j = indexCheckBoxList.size(); i < j; i++) {
		int num = -1;
		switch (i) {
		case 0:
			num = fitModifier.m();
			break;
		case 1:
			num = fitModifier.n();
			break;
		case 2:
			num = fitModifier.p();
		}
		bool flag =  (num >= 0);
		if (flag) {
			indexSpinBoxList[i]->setValue(num);
		}
		indexCheckBoxList[i]->setEnabled((i == 0) ? true : compFlag);
		compFlag = compFlag && flag;
		indexCheckBoxList[i]->setChecked(compFlag);
		indexSpinBoxList[i]->setEnabled(compFlag);
	}

	for (int i = 0, j = samplingCheckBoxList.size(); i < j; i++) {
		int num = -1;
		switch (i) {
		case 0:
			num = fitModifier.pointIncrement();
			break;
		case 1:
			num = fitModifier.startPoint();
			break;
		case 2:
			num = fitModifier.endPoint();
			break;
		case 3:
			num = fitModifier.blockIncrement();
			break;
		case 4:
			num = fitModifier.startBlock();
			break;
		case 5:
			num = fitModifier.endBlock();
		}
		bool flag = (num >= 0);
		if (flag) {
			samplingSpinBoxList[i]->setValue(num);
		}
		samplingCheckBoxList[i]->setChecked(flag);
		samplingSpinBoxList[i]->setEnabled(flag);
	}

	for (int i = 0; i < AXIS_NUM; i++) {
		QString column = fitModifier.entry(i);
		bool hasColumn = !column.isEmpty();
		if (hasColumn) {
			columnLineEdit[i]->setText(column);
		}
		columnCheckBox[i]->setChecked(hasColumn);
		columnLineEdit[i]->setEnabled(hasColumn);
	}
	QString format = fitModifier.format();
	bool hasFormat = !format.isEmpty();
	if (hasFormat) {
		formatLineEdit->setText(format);
	}
	formatCheckBox->setChecked(hasFormat);
	formatLineEdit->setEnabled(hasFormat);
}

FitModifier FitModifierWidget::modifier() const
{
	FitModifier fitModifier;

	QList<int> indexList;
	for (int i = 0, j = indexCheckBoxList.size(); i < j; i++) {
		indexList.append(indexCheckBoxList[i]->isChecked() ?  
				indexSpinBoxList[i]->value() : -1);
	}
	fitModifier.setM(indexList.at(0));
	fitModifier.setN(indexList.at(1));
	fitModifier.setP(indexList.at(2));

	QList<int> samplingList;
	for (int i = 0, j = samplingCheckBoxList.size(); i < j; i++) {
		samplingList.append(samplingCheckBoxList[i]->isChecked() 
				? samplingSpinBoxList[i]->value() : -1);
	}
	fitModifier.setPointIncrement(samplingList.at(0));
	fitModifier.setBlockIncrement(samplingList.at(1));
	fitModifier.setStartPoint(samplingList.at(2));
	fitModifier.setStartBlock(samplingList.at(3));
	fitModifier.setEndPoint(samplingList.at(4));
	fitModifier.setEndBlock(samplingList.at(5));

	for (int i = 0; i < AXIS_NUM; i++) {
		fitModifier.setEntry(i, columnCheckBox[i]->isChecked()
				? columnLineEdit[i]->text() : "");
	}
	fitModifier.setFormat(formatCheckBox->isChecked() 
			? formatLineEdit->text() : "");

	return fitModifier;
}
