/*
 * Cueplot: a GUI front-end to gnuplot
 * Copyright (C) 2007, 2008, 2009 Muneyuki Noguchi <nogu@users.sourceforge.jp>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation, 
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
#include "plotdatawidget.h"

#include <QtCore/QMimeData>
#include <QtGui/QApplication>
#include <QtGui/QClipboard>
#include <QtGui/QGridLayout>
#include <QtGui/QPushButton>

#include "plot.h"
#include "plotlistmodel.h"
#include "plotlistview.h"
#include "plotoptiondialog.h"

PlotDataWidget::PlotDataWidget(QWidget *parent)
: QWidget(parent)
{
	addPushButton = new QPushButton;
	addPushButton->setText(tr("&Add"));
	addPushButton->setEnabled(false);
	addPushButton->setIcon(QIcon(":/images/add.png"));
	connect(addPushButton, SIGNAL(clicked()),
			this, SIGNAL(dataAdded()));

	optionPushButton = new QPushButton;
	optionPushButton->setText(tr("O&ption..."));
	optionPushButton->setEnabled(false);
	optionPushButton->setSizePolicy(
			QSizePolicy(QSizePolicy::Fixed, QSizePolicy::Fixed));
	connect(optionPushButton, SIGNAL(clicked()),
			this, SLOT(setPlotOptions()));

	upPushButton = new QPushButton;
	upPushButton->setText(tr("&Up"));
	upPushButton->setEnabled(false);
	upPushButton->setIcon(QIcon(":/images/up.png"));
	connect(upPushButton, SIGNAL(clicked()),
			this, SLOT(setDataUp()));

	downPushButton = new QPushButton;
	downPushButton->setText(tr("&Down"));
	downPushButton->setEnabled(false);
	downPushButton->setIcon(QIcon(":/images/down.png"));
	connect(downPushButton, SIGNAL(clicked()),
			this, SLOT(setDataDown()));

	deletePushButton = new QPushButton;
	deletePushButton->setText(tr("D&elete"));
	deletePushButton->setEnabled(false);
	deletePushButton->setIcon(QIcon(":/images/remove.png"));
	deletePushButton->setSizePolicy(
			QSizePolicy(QSizePolicy::Fixed, QSizePolicy::Fixed));
	connect(deletePushButton, SIGNAL(clicked()),
			this, SLOT(deleteData()));

	plotListModel = new PlotListModel;
	connect(plotListModel, 
			SIGNAL(dataChanged(const QModelIndex &, const QModelIndex &)),
			this, SLOT(changeDialog(const QModelIndex &, const QModelIndex &)));
	connect(plotListModel, SIGNAL(rowsInserted(const QModelIndex &, int, int)),
			this, SLOT(insertDialog(const QModelIndex &, int, int)));
	connect(plotListModel, SIGNAL(rowsRemoved(const QModelIndex &, int, int)),
			this, SLOT(removeDialog(const QModelIndex &, int, int)));
	connect(plotListModel, SIGNAL(rowsMoved(int, int)),
			this, SLOT(moveDialog(int, int)));

	plotListView = new PlotListView;
	plotListView->setModel(plotListModel);
	// Task Tracker at trolltech.com
	// #182440 items are removed when using InternalMove drag drop mode
#if !defined(Q_OS_DARWIN) || QT_VERSION >= 0x040400
	plotListView->setDragEnabled(true);
#endif
	plotListView->setAcceptDrops(true);
	plotListView->setDropIndicatorShown(true);
	plotListView->setSelectionMode(QAbstractItemView::ExtendedSelection);
	connect(plotListView, SIGNAL(cutRequested()),
			this, SLOT(cut()));
	connect(plotListView, SIGNAL(copyRequested()),
			this, SLOT(copy()));
	connect(plotListView, SIGNAL(pasteRequested()),
			this, SLOT(paste()));
	connect(plotListView, SIGNAL(selectAllRequested()),
			this, SLOT(selectAll()));

	selectionModel = plotListView->selectionModel();
	connect(selectionModel,
			SIGNAL(selectionChanged(const QItemSelection &,
					const QItemSelection &)), this, SLOT(setPushButton()));

	// ----------- assemble layouts and widgets
	// ------------------------ create layouts

	QGridLayout *mainLayout = new QGridLayout;
	mainLayout->setMargin(0);
	mainLayout->addWidget(addPushButton, 0, 0);
	mainLayout->addWidget(optionPushButton, 1, 0);
	mainLayout->addWidget(upPushButton, 2, 0);
	mainLayout->addWidget(downPushButton, 3, 0);
	// ウィンドウを最大化したときのことを考えて、QSpacerItem を入れる
	mainLayout->addItem(new QSpacerItem(0, 0, 
				QSizePolicy::Minimum, QSizePolicy::Expanding), 4, 0);
	mainLayout->addItem(new QSpacerItem(0, 0, 
				QSizePolicy::Expanding, QSizePolicy::Minimum), 5, 0, 1, 2);
	mainLayout->addWidget(deletePushButton, 5, 2);
	mainLayout->addWidget(plotListView, 0, 1, 5, 2);

	setLayout(mainLayout);
}

void PlotDataWidget::addData(const Plot &plot)
{
	// 挿入する場所
	QModelIndexList list = selectionModel->selectedIndexes();
	int row = list.size() ? list.last().row() + 1 : plotListModel->rowCount();

	// 項目を挿入する
	plotListModel->insertRow(row);

	// 項目を設定する
	plotListModel->setPlot(plotListModel->index(row), plot);
}

void PlotDataWidget::setDataUp()
{
	moveData(true);
}

void PlotDataWidget::setDataDown()
{
	moveData(false);
}

void PlotDataWidget::moveData(bool up)
{
	QModelIndexList list = selectionModel->selectedIndexes();
	int size = list.size();
	if (!size) {
		return;
	}
	qSort(list);
	if (up) {
		for (int i = 0; i < size; i++) {
			int row = list[i].row();
			// 一番上にある項目はこれ以上、上に動かせない
			if (row > 0) {
				plotListModel->moveRow(row, row - 1);
			}
		}
	} else {
		for (int i = size - 1; i >= 0; i--) {
			int row = list[i].row();
			// 一番下にある項目はこれ以上、下に動かせない
			if (row < plotListModel->rowCount() - 1) {
				plotListModel->moveRow(row, row + 1);
			}
		}

	}
}

void PlotDataWidget::deleteData()
{
	for (int i = 0, j = selectionModel->selectedIndexes().size(); i < j; i++) {
		// 項目が削除されるとselectionModel->selectedIndexes()の中身が変わる
		plotListModel->removeRow(
				selectionModel->selectedIndexes().first().row());
	}
}

void PlotDataWidget::setPushButton()
{
	// 項目が一つでも選択されていたら、
	// オプションボタンと削除ボタンを有効にする
	QModelIndexList list = selectionModel->selectedIndexes();
	bool isSelected = list.size();
	optionPushButton->setEnabled(isSelected);
	deletePushButton->setEnabled(isSelected);

	// 選択された項目があり、複数の項目がリスト中にあれば、
	// 上ボタンと下ボタンを有効にする
	int count = plotListModel->rowCount();
	bool isMulti = isSelected && (count > 1);
	// 一番上の項目
	int min = count;
	// 一番下の項目
	int max = -1;
	if (isMulti) {
		for (int i = 0, j = list.size(); i < j; i++) {
			int row = list.at(i).row();
			if (min > row) {
				min = row;
			}
			if (max < row) {
				max = row;
			}
		}
	}
	// 選択された項目のうち一番上の項目が一番上にあるときは上ボタンを無効にする
	upPushButton->setEnabled(isMulti && min != 0);
	// 選択された項目のうち一番下の項目が一番下にあるときは下ボタンを無効にする
	downPushButton->setEnabled(isMulti && max != count - 1);
}

void PlotDataWidget::changeOption(const PlotOptionDialog &dialog,
		const QList<Plot> &list)
{
	int idx = 0;
	// オプションを変えたダイアログをリストの中から探す
	for (int i = 0, j = dialogList.size(); i < j; i++) {
		const PlotOptionDialog *d = dialogList.at(i);
		if (d == &dialog) {
			plotListModel->setPlot(plotListModel->index(i), list[idx]);
			idx++;
		}
	}
}

void PlotDataWidget::setAdditionEnabled(bool enable)
{
	addPushButton->setEnabled(enable);
}

void PlotDataWidget::setPlotOptions()
{
	QModelIndexList list = selectionModel->selectedIndexes();
	int size = list.size();
	if (!size) {
		return;
	}
	PlotOptionDialog *plotOptionDialog = dialogList.at(list[0].row());
	// 新しいダイアログウィンドウを開くかどうかを示すフラグ
	// i) 対応ダイアログを開いたことのない項目がある
	// ii) 既存のダイアログの対応項目の数と選択項目の数が一致しない
	bool flag = (!plotOptionDialog
			|| (size != dialogList.count(plotOptionDialog)));
	if (!flag) {
		for (int i = 0; i < size; i++) {
			// iii) 既存のダイアログの対応項目と選択項目が一致しない
			PlotOptionDialog *d = dialogList.at(list[i].row());
			if (d != plotOptionDialog) {
				flag = true;
				break;
			}
		}
	}
	if (flag) {
		// 選択項目に対応する既存のダイアログウィンドウをすべて閉じる
		for (int i = 0; i < size; i++) {
			PlotOptionDialog *dialog = dialogList.at(list[i].row());
			if (!dialog) {
				continue;
			}
			if (dialog->canClose()) {
				deleteDialog(*dialog);
			} else {
				return;
			}
		}

		plotOptionDialog = new PlotOptionDialog;
		QList<Plot> plot;
		for (int i = 0; i < size; i++) {
			plot.append(plotListModel->plot(list[i]));
			dialogList[list[i].row()] = plotOptionDialog;
		}
		plotOptionDialog->setPlot(plot);
		connect(plotOptionDialog, 
				SIGNAL(optionChanged(const PlotOptionDialog &,
						const QList<Plot> &)),
				this,
				SLOT(changeOption(const PlotOptionDialog &,
						const QList<Plot> &)));
	}
	plotOptionDialog->show();
	plotOptionDialog->raise();
	plotOptionDialog->activateWindow();
}

void PlotDataWidget::changeDialog(const QModelIndex &topLeft, 
		const QModelIndex &bottomRight)
{
	for (int i = topLeft.row(), j = bottomRight.row(); i <= j; i++) {
		PlotOptionDialog *plotOptionDialog = dialogList.at(i);
		if (plotOptionDialog) {
			QList<Plot> list;
			int idx = 0;
			while ((idx = dialogList.indexOf(plotOptionDialog, idx)) >= 0) {
				list.append(plotListModel->plot(plotListModel->index(idx)));
				idx++;
			}
			plotOptionDialog->setPlot(list);
		}
	}
	setAction();
}

void PlotDataWidget::insertDialog(const QModelIndex &/*parent*/,
		int start, int end)
{
	selectionModel->clear();
	for (int i = start; i <= end; i++) {
		dialogList.insert(i, 0);
		selectionModel->select(plotListModel->index(i),
				QItemSelectionModel::Select);
	}
	selectionModel->setCurrentIndex(plotListModel->index(end), 
			QItemSelectionModel::Select);
	setPushButton();
}

void PlotDataWidget::removeDialog(const QModelIndex &/*parent*/,
		int start, int end)
{
	for (int i = start; i <= end; i++) {
		deleteDialog(*dialogList.takeAt(start));
	}
	setPushButton();
	setAction();
}

void PlotDataWidget::moveDialog(int from, int to)
{
	dialogList.move(from, to);

	selectionModel->select(plotListModel->index(from), 
			QItemSelectionModel::Deselect);
	// 動かした項目を選択状態にする
	QModelIndex modelIndex = plotListModel->index(to); 
	selectionModel->select(modelIndex, QItemSelectionModel::Select);
	selectionModel->setCurrentIndex(modelIndex, 
			QItemSelectionModel::Select);
}

void PlotDataWidget::setAction()
{
	// チェックされている項目があるか調べる
	bool isChecked = false;
	for (int i = 0, j = plotListModel->rowCount(); i < j; i++) {
		if (plotListModel->data(plotListModel->index(i),
					Qt::CheckStateRole) == Qt::Checked) {
			isChecked = true;
			break;
		}
	}
	emit setPlotEnabled(isChecked);
}

bool PlotDataWidget::hasItem() const
{
	return plotListModel->rowCount() > 0;
}

QStringList PlotDataWidget::commandList() const
{
	QStringList commandList;
	for (int i = 0, j = plotListModel->rowCount(); i < j; i++) {
		QModelIndex modelIndex = plotListModel->index(i);
		if (plotListModel->data(modelIndex, 
					Qt::CheckStateRole) == Qt::Checked) {
			commandList << plotListModel->plot(modelIndex).command();
		}
	}
	return commandList;
}

void PlotDataWidget::closeDialog()
{
	while (!dialogList.isEmpty()) {
		deleteDialog(*dialogList.takeFirst());
	}
}

void PlotDataWidget::deleteDialog(PlotOptionDialog &plotOptionDialog)
{
	if (!(&plotOptionDialog)) {
		return;
	}
	// 二重deleteを防ぐ
	int idx = 0;
	while ((idx = dialogList.indexOf(&plotOptionDialog, idx)) >= 0) {
		dialogList[idx] = 0;
		idx++;
	}
	delete &plotOptionDialog;
}

void PlotDataWidget::cut()
{
	if (QWidget::focusWidget() != plotListView) {
		return;
	}
	copy();
	deleteData();
}

void PlotDataWidget::copy()
{
	if (QWidget::focusWidget() != plotListView) {
		return;
	}
	QApplication::clipboard()->setMimeData(plotListModel->mimeData(
				selectionModel->selectedIndexes()));
}

void PlotDataWidget::paste()
{
	const QMimeData *data = QApplication::clipboard()->mimeData();
	if (!data) {
		return;
	}
	QWidget *widget = QWidget::focusWidget();
	if (widget != plotListView) {
		return;
	}
	QModelIndexList list = selectionModel->selectedIndexes();
	int row = list.size() ? list.last().row() + 1 : plotListModel->rowCount();
	if (data->hasFormat("application/vnd.row.list")) {
		plotListModel->decodePlot(data, row);
	} else if (data->hasUrls()) {
		plotListModel->decodeUrl(data, row);
	} else if (data->hasText()) {
		plotListModel->decodeText(data, row);
	}
}

void PlotDataWidget::selectAll()
{
	QWidget *widget = QWidget::focusWidget();
	if (widget != plotListView) {
		return;
	}
	for (int i = 0, j = plotListModel->rowCount(); i < j; i++) {
		selectionModel->select(plotListModel->index(i), 
				QItemSelectionModel::Select);
	}
}
