/*
 * Cueplot: a GUI front-end to gnuplot
 * Copyright (C) 2008 Muneyuki Noguchi <nogu@users.sourceforge.jp>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation, 
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
#ifndef PLOT_LIST_MODEL_INCLUDED
#define PLOT_LIST_MODEL_INCLUDED

#include <QtCore/QAbstractListModel>

#include "plot.h"

//! プロットする項目を格納する List Model
class PlotListModel : public QAbstractListModel
{
	Q_OBJECT

	public:
		explicit PlotListModel(QObject *parent = 0);

		QVariant data(const QModelIndex &index, 
				int role = Qt::DisplayRole) const;
		bool setData(const QModelIndex &index, const QVariant &value, 
				int role = Qt::EditRole);

		bool dropMimeData(const QMimeData *data, Qt::DropAction action,
				int row, int column, const QModelIndex &parent);
		QStringList mimeTypes() const;
		QMimeData *mimeData(const QModelIndexList &indexes) const;

		QVariant headerData(int section, Qt::Orientation orientation,
				int role = Qt::DisplayRole) const;
		Qt::ItemFlags flags(const QModelIndex &index) const;

		bool insertRows(int row, int count, 
				const QModelIndex &parent = QModelIndex());
		bool removeRows(int row, int count, 
				const QModelIndex &parent = QModelIndex());

		int rowCount(const QModelIndex &parent = QModelIndex()) const;
		Qt::DropActions supportedDropActions() const;
		/*!
		 * \brief 項目を移動させる
		 * \param from 項目の移動前の位置
		 * \param to 項目の移動後の位置
		 * \param parent 親項目
		 */
		bool moveRow(int from, int to, 
				const QModelIndex &parent = QModelIndex());
		/*!
		 * \brief 指定したインデックスに対するオプションを得る
		 * \param index インデックス
		 */
		Plot plot(const QModelIndex &index) const;
		/*!
		 * \brief 指定したインデックスに対するオプションを設定する
		 * \param index インデックス
		 * \param plot オプション
		 */
		bool setPlot(const QModelIndex &index, const Plot &plot);

		//! インポートされた URL データをデコードする
		bool decodeUrl(const QMimeData *data, int beginRow);
		//! インポートされたテキストデータをデコードする
		bool decodeText(const QMimeData *data, int beginRow);
		//! インポートされたPlotクラスのオブジェクトをデコードする
		bool decodePlot(const QMimeData *data, int beginRow);


	Q_SIGNALS:
		/*!
		 * \brief このシグナルは項目を移動させたときに発信される
		 * \param from 項目の移動前の位置
		 * \param to 項目の移動後の位置
		 */
		void rowsMoved(int from, int to);

	private:
		// don't implement these
		PlotListModel(const PlotListModel &);
		PlotListModel &operator=(const PlotListModel &);

		//! インポートされた位置データをデコードする
		bool decodeRow(const QMimeData *data, int beginRow);

		//! オプションを格納するリスト
		QList<Plot> plotList;
		//! チェックボックスの状態を格納するリスト
		QList<Qt::CheckState> checkList;
};
#endif // PLOT_LIST_MODEL_INCLUDED
