/*
 * Cueplot: a GUI front-end to gnuplot
 * Copyright (C) 2007, 2008 Muneyuki Noguchi <nogu@users.sourceforge.jp>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation, 
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
#include "terminaloptionwidget.h"

#include <QtGui/QComboBox>
#include <QtGui/QLabel>
#include <QtGui/QHBoxLayout>
#include <QtGui/QPushButton>
#include <QtGui/QScrollArea>
#include <QtGui/QStackedWidget>
#include <QtGui/QVBoxLayout>

#include "terminal.h"
#include "terminalemtexoptionwidget.h"
#include "terminallatexoptionwidget.h"
#include "terminalpbmoptionwidget.h"
#include "terminalpostscriptoptionwidget.h"

TerminalOptionWidget::TerminalOptionWidget(QWidget *parent)
: AbstractOptionWidget(parent), terminal(0)
{
	setWindowTitle(tr("Terminal"));
}

TerminalOptionWidget::~TerminalOptionWidget()
{
	delete terminal;
}

void TerminalOptionWidget::construct()
{
	setWhatsThis("set terminal");

	typeComboBox = new QComboBox;
	typeComboBox->setEditable(true);
	for (int i = 0; !Terminal::TERMINALS[i].isEmpty(); i++) {
		typeComboBox->addItem(Terminal::TERMINALS[i]);
	}
	connect(typeComboBox, SIGNAL(editTextChanged(const QString &)),
			this, SIGNAL(modified()));
	connect(typeComboBox, SIGNAL(activated(int)),
			this, SIGNAL(modified()));
	connect(typeComboBox, SIGNAL(activated(int)),
			this, SLOT(changeTerminal(int)));

	QLabel *typeLabel = new QLabel;
	typeLabel->setText(tr("&Terminal Type:"));
	typeLabel->setSizePolicy(QSizePolicy::Fixed, QSizePolicy::Fixed);
	typeLabel->setBuddy(typeComboBox);

	typeStackedWidget = new QStackedWidget;
	typeStackedWidget->addWidget(new TerminalEmtexOptionWidget);
	typeStackedWidget->addWidget(new TerminalLatexOptionWidget);
	typeStackedWidget->addWidget(new TerminalPbmOptionWidget);
	typeStackedWidget->addWidget(new TerminalPostScriptOptionWidget);

	initBitArray.resize(typeStackedWidget->count());

	initPushButton = new QPushButton;
	initPushButton->setText(tr("&Initialize Terminal Options"));
	connect(initPushButton, SIGNAL(clicked()),
			this, SLOT(initOption()));
	int index = typeComboBox->findText(Terminal::TERMINALS_WITH_OPTIONS[0]);
	if (index >= 0) {
		changeTerminal(index);
	}

	QScrollArea *scrollArea = new QScrollArea;
	scrollArea->setWidget(typeStackedWidget);
	scrollArea->setWidgetResizable(true);
	scrollArea->setFrameShadow(QFrame::Plain);
	scrollArea->setFrameShape(QFrame::NoFrame);

	QHBoxLayout *buttonLayout = new QHBoxLayout;
	buttonLayout->addStretch();
	buttonLayout->addWidget(initPushButton);

	QHBoxLayout *listLayout = new QHBoxLayout;
	listLayout->addWidget(typeLabel);
	listLayout->addWidget(typeComboBox);

	QVBoxLayout *mainLayout = new QVBoxLayout;
	mainLayout->addLayout(listLayout);
	mainLayout->addWidget(scrollArea);
	mainLayout->addLayout(buttonLayout);

	setLayout(mainLayout);
}

void TerminalOptionWidget::init()
{
	delete terminal;
	terminal = new Terminal;
	setWidget();
}

void TerminalOptionWidget::initOption()
{
	qobject_cast<AbstractOptionWidget *>(typeStackedWidget->currentWidget())
		->init();
}

void TerminalOptionWidget::apply()
{
	terminal->setTerminalType(typeComboBox->currentText());

	if (hasOption(terminal->terminalType())) {
		qobject_cast<AbstractOptionWidget *>(typeStackedWidget->currentWidget())
			->apply();
	} else {
		emit executed(terminal->command());
	}
}

void TerminalOptionWidget::setWidget()
{
	QString type = terminal->terminalType();
	int index = typeComboBox->findText(type);
	bool flag = false;
	if (index >= 0) {
		typeComboBox->setCurrentIndex(index);
		changeTerminal(index);
		flag = hasOption(type);
	}
	typeStackedWidget->setEnabled(flag);
	initPushButton->setEnabled(flag);
}

bool TerminalOptionWidget::hasOption(const QString &type)
{
	for (int i = 0; !Terminal::TERMINALS_WITH_OPTIONS[i].isEmpty(); i++) {
		if (type == Terminal::TERMINALS_WITH_OPTIONS[i]) {
			return true;
		}
	}
	return false;
}

void TerminalOptionWidget::changeTerminal(int index)
{
	QString type = typeComboBox->itemText(index);
	for (int i = 0; !Terminal::TERMINALS_WITH_OPTIONS[i].isEmpty(); i++) {
		if (type == Terminal::TERMINALS_WITH_OPTIONS[i]) {
			typeStackedWidget->setEnabled(true);
			initPushButton->setEnabled(true);
			if (!initBitArray.at(i)) {
				AbstractOptionWidget *optionWidget =
					qobject_cast<AbstractOptionWidget *>(typeStackedWidget->widget(i));
				optionWidget->construct();
				optionWidget->init();
				connect(optionWidget, SIGNAL(executed(const QString &)),
						this, SIGNAL(executed(const QString &)));
				connect(optionWidget, SIGNAL(modified()),
						this, SIGNAL(modified()));
				initBitArray.setBit(i);
			}
			typeStackedWidget->setCurrentIndex(i);
			return;
		}
	}
	typeStackedWidget->setEnabled(false);
	initPushButton->setEnabled(false);
}
