/*
 * Cueplot: a GUI front-end to gnuplot
 * Copyright (C) 2007, 2008, 2009 Muneyuki Noguchi <nogu@users.sourceforge.jp>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation, 
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
#include "plotmodifierwidget.h"

#include <limits> // max()

#include <QtGui/QCheckBox>
#include <QtGui/QComboBox>
#include <QtGui/QLineEdit>
#include <QtGui/QGridLayout>
#include <QtGui/QSpinBox>
#include <QtGui/QTabWidget>
#include <QtGui/QVBoxLayout>

#include "plotmodifier.h"

PlotModifierWidget::PlotModifierWidget(QWidget *parent)
: AbstractPlotOptionWidget(parent)
{
	setWindowTitle(tr("Data File"));
	setWhatsThis(tr("plot {index &lt;index list&gt;}<br>"
				"{every &lt;every list&gt;}<br>"
				"{using &lt;using list&gt;}<br>"
				"{smooth &lt;option&gt;}"));

	const QString AXIS[] = { "&x", "&y", "&z" };
	for (int i = 0; i < AXIS_NUM; i++) {
		columnCheckBox[i] = new QCheckBox;
		columnCheckBox[i]->setText(tr("Specify a %1 column:").arg(AXIS[i]));
		connect(columnCheckBox[i], SIGNAL(toggled(bool)),
				this, SIGNAL(modified()));

		columnLineEdit[i] = new QLineEdit;
		connect(columnCheckBox[i], SIGNAL(toggled(bool)),
				columnLineEdit[i], SLOT(setEnabled(bool)));
		connect(columnLineEdit[i], SIGNAL(textChanged(const QString &)),
				this, SIGNAL(modified()));
	}

	formatCheckBox = new QCheckBox;
	formatCheckBox->setText(tr("Specify a &format:"));
	connect(formatCheckBox, SIGNAL(toggled(bool)),
			this, SIGNAL(modified()));

	formatLineEdit = new QLineEdit;
	connect(formatCheckBox, SIGNAL(toggled(bool)),
			formatLineEdit, SLOT(setEnabled(bool)));
	connect(formatLineEdit, SIGNAL(textChanged(const QString &)),
			this, SIGNAL(modified()));

	const QString INDEX_TEXT[] = {
		tr("Specify a s&tarting index:"),
		tr("Specify a &ending index:"),
		tr("Specify i&ncrement of index:")
	};
	const int INDEX_NUM = sizeof(INDEX_TEXT) / sizeof(INDEX_TEXT[0]);
	for (int i = 0; i < INDEX_NUM; i++) {
		QCheckBox *checkBox = new QCheckBox;
		checkBox->setText(INDEX_TEXT[i]);
		connect(checkBox, SIGNAL(toggled(bool)),
				this, SLOT(setIndex()));
		connect(checkBox, SIGNAL(toggled(bool)),
				this, SIGNAL(modified()));
		indexCheckBoxList.append(checkBox);

		QSpinBox *spinBox = new QSpinBox;
		spinBox->setMaximum(std::numeric_limits<int>::max());
		connect(checkBox, SIGNAL(toggled(bool)),
				spinBox, SLOT(setEnabled(bool)));
		connect(spinBox, SIGNAL(valueChanged(int)),
				this, SIGNAL(modified()));
		indexSpinBoxList.append(spinBox);
	}

	const QString SAMPLING_TEXT[] = {
		tr("Specify poi&nt increment:"),
		tr("Specify a s&tart point:"),
		tr("Specify a &end point:"),
		tr("Specify &block increment:"),
		tr("Specify a sta&rt block:"),
		tr("Specify a en&d block:")
	};
	const int SAMPLING_NUM = sizeof(SAMPLING_TEXT) / sizeof(SAMPLING_TEXT[0]);
	for (int i = 0; i < SAMPLING_NUM; i++) {
		QCheckBox *checkBox = new QCheckBox;
		checkBox->setText(SAMPLING_TEXT[i]);
		connect(checkBox, SIGNAL(toggled(bool)),
				this, SIGNAL(modified()));
		samplingCheckBoxList.append(checkBox);

		QSpinBox *spinBox = new QSpinBox;
		spinBox->setMaximum(std::numeric_limits<int>::max());
		connect(checkBox, SIGNAL(toggled(bool)),
				spinBox, SLOT(setEnabled(bool)));
		connect(spinBox, SIGNAL(valueChanged(int)),
				this, SIGNAL(modified()));
		samplingSpinBoxList.append(spinBox);
	}

	interpCheckBox = new QCheckBox;
	interpCheckBox->setText(tr("Sp&ecify:"));
	connect(interpCheckBox, SIGNAL(toggled(bool)),
			this, SIGNAL(modified()));

	interpComboBox = new QComboBox;
	for (int i = 0; !PlotModifier::SMOOTH[i].isEmpty(); i++) {
		interpComboBox->addItem(PlotModifier::SMOOTH[i]);
	}
	connect(interpCheckBox, SIGNAL(toggled(bool)),
			interpComboBox, SLOT(setEnabled(bool)));
	connect(interpComboBox, SIGNAL(activated(int)),
			this, SIGNAL(modified()));

	// create layouts
	// assemble layouts and widgets

	QGridLayout *dataSetLayout = new QGridLayout;
	for (int i = 0; i < INDEX_NUM; i++) {
		dataSetLayout->addWidget(indexCheckBoxList[i], i, 0);
		dataSetLayout->addWidget(indexSpinBoxList[i], i, 1);
	}
	dataSetLayout->addItem(new QSpacerItem(0, 0, 
				QSizePolicy::Expanding, QSizePolicy::Minimum), 0, 2);
	dataSetLayout->addItem(new QSpacerItem(0, 0,
				QSizePolicy::Minimum, QSizePolicy::Expanding), INDEX_NUM, 0);

	QWidget *dataSetWidget = new QWidget;
	dataSetWidget->setWhatsThis(tr("plot {index &lt;index list&gt;}"));
	dataSetWidget->setLayout(dataSetLayout);

	QGridLayout *periodLayout = new QGridLayout;
	for (int i = 0; i < SAMPLING_NUM; i++) {
		periodLayout->addWidget(samplingCheckBoxList[i], i, 0);
		periodLayout->addWidget(samplingSpinBoxList[i], i, 1);
	}
	periodLayout->addItem(new QSpacerItem(0, 0,
				QSizePolicy::Expanding, QSizePolicy::Minimum), 0, 2);
	periodLayout->addItem(new QSpacerItem(0, 0,
				QSizePolicy::Minimum, QSizePolicy::Expanding), SAMPLING_NUM, 0);

	QWidget *periodWidget = new QWidget;
	periodWidget->setWhatsThis(tr("plot {every &lt;every list&gt;}"));
	periodWidget->setLayout(periodLayout);

	QGridLayout *colFormatLayout = new QGridLayout;
	for (int i = 0; i < AXIS_NUM; i++) {
		colFormatLayout->addWidget(columnCheckBox[i], i, 0);
		colFormatLayout->addWidget(columnLineEdit[i], i, 1);
	}
	colFormatLayout->addItem(new QSpacerItem(0, 0,
				QSizePolicy::Expanding, QSizePolicy::Minimum), 0, 2);
	colFormatLayout->addWidget(formatCheckBox, 3, 0);
	colFormatLayout->addWidget(formatLineEdit, 3, 1, 1, 2);
	colFormatLayout->addItem(new QSpacerItem(0, 0,
				QSizePolicy::Minimum, QSizePolicy::Expanding), 4, 0);

	QWidget *colFormatWidget = new QWidget;
	colFormatWidget->setWhatsThis(tr("plot {using &lt;using list&gt;}"));
	colFormatWidget->setLayout(colFormatLayout);

	QGridLayout *interpLayout = new QGridLayout;
	interpLayout->addWidget(interpCheckBox, 0, 0);
	interpLayout->addWidget(interpComboBox, 0, 1);
	interpLayout->addItem(new QSpacerItem(0, 0,
				QSizePolicy::Expanding, QSizePolicy::Minimum), 0, 2);
	interpLayout->addItem(new QSpacerItem(0, 0,
				QSizePolicy::Minimum, QSizePolicy::Expanding), 1, 0);

	QWidget *interpWidget = new QWidget;
	interpWidget->setWhatsThis(tr("plot {smooth &lt;option&gt;}"));
	interpWidget->setLayout(interpLayout);

	QTabWidget *modTabWidget = new QTabWidget;
	modTabWidget->addTab(dataSetWidget, tr("&Indexes"));
	modTabWidget->addTab(periodWidget, tr("&Sampling"));
	modTabWidget->addTab(colFormatWidget, tr("Co&lumns"));
	modTabWidget->addTab(interpWidget, tr("S&moothing"));

	QVBoxLayout *mainLayout = new QVBoxLayout;
	mainLayout->addWidget(modTabWidget);

	setLayout(mainLayout);
}

PlotModifierWidget::~PlotModifierWidget()
{
	while (!indexCheckBoxList.isEmpty()) {
		delete indexCheckBoxList.takeFirst();
		delete indexSpinBoxList.takeFirst();
	}
	while (!samplingCheckBoxList.isEmpty()) {
		delete samplingCheckBoxList.takeFirst();
		delete samplingSpinBoxList.takeFirst();
	}
}

void PlotModifierWidget::setIndex()
{
	bool flag = true;
	for (int i = 1, j = indexCheckBoxList.size(); i < j; i++) {
		flag = flag && indexCheckBoxList[i - 1]->isChecked();
		indexCheckBoxList[i]->setEnabled(flag);
		indexSpinBoxList[i]->setEnabled(
				flag && indexCheckBoxList[i]->isChecked());
	}
}

void PlotModifierWidget::setModifier(const QList<PlotModifier> &list)
{
	plotModifierList = list;
	setWidget();
}

const QList<PlotModifier> &PlotModifierWidget::modifier() const
{
	return plotModifierList;
}

void PlotModifierWidget::apply()
{
	int indexArray[3];
	bool hasIndexArray[3];
	for (int i = 0; i < 3; i++) {
		bool isChecked = indexCheckBoxList[i]->isChecked();
		int index = isChecked ? indexSpinBoxList[i]->value() : -1;
		indexArray[i] = index;
		hasIndexArray[i] = (!isChecked || index >= 0);
	}
	int samplingArray[6];
	bool hasSamplingArray[6];
	for (int i = 0; i < 6; i++) {
		bool isChecked = samplingCheckBoxList[i]->isChecked();
		int sampling = isChecked ? samplingSpinBoxList[i]->value() : -1;
		samplingArray[i] = sampling;
		hasSamplingArray[i] = (!isChecked || sampling >= 0);
	}
	bool isChecked = interpCheckBox->isChecked();
	int smooth = isChecked ? interpComboBox->currentIndex() : -1;
	bool hasSmooth = (!isChecked || smooth >= 0);

	QString entryArray[AXIS_NUM];
	bool hasEntryArray[AXIS_NUM];
	for (int i = 0; i < AXIS_NUM; i++) {
		bool isChecked = columnCheckBox[i]->isChecked();
		QString entry = isChecked ? columnLineEdit[i]->text() : "";
		entryArray[i] = entry;
		hasEntryArray[i] = (!isChecked || !entry.isEmpty());
	}

	isChecked = formatCheckBox->isChecked();
	QString format = isChecked ? formatLineEdit->text() : "";
	bool hasFormat = (!isChecked || !format.isEmpty());
	for (int i = 0, j = plotModifierList.size(); i < j; i++) {
		if (hasIndexArray[0]) {
			plotModifierList[i].setM(indexArray[0]);
		}
		if (hasIndexArray[1]) {
			plotModifierList[i].setN(indexArray[1]);
		}
		if (hasIndexArray[2]) {
			plotModifierList[i].setP(indexArray[2]);
		}

		if (hasSamplingArray[0]) {
			plotModifierList[i].setPointIncrement(samplingArray[0]);
		}
		if (hasSamplingArray[1]) {
			plotModifierList[i].setStartPoint(samplingArray[1]);
		}
		if (hasSamplingArray[2]) {
			plotModifierList[i].setEndPoint(samplingArray[2]);
		}
		if (hasSamplingArray[3]) {
			plotModifierList[i].setBlockIncrement(samplingArray[3]);
		}
		if (hasSamplingArray[4]) {
			plotModifierList[i].setStartBlock(samplingArray[4]);
		}
		if (hasSamplingArray[5]) {
			plotModifierList[i].setEndBlock(samplingArray[5]);
		}

		if (hasSmooth) {
			plotModifierList[i].setSmooth(smooth);
		}

		for (int k = 0; k < AXIS_NUM; k++) {
			if (hasEntryArray[k]) {
				plotModifierList[i].setEntry(k, entryArray[k]);
			}
		}
		if (hasFormat) {
			plotModifierList[i].setFormat(format);
		}
	}
}

void PlotModifierWidget::setWidget()
{
	// 選択している項目に関数があるとき
	if (plotModifierList.isEmpty()) {
		return;
	}
	setIndexWidget();
	setSamplingWidget();
	setColumnWidget();
	setSmoothingWidget();
}

void PlotModifierWidget::setIndexWidget()
{
	PlotModifier plotModifier = plotModifierList[0];
	for (int i = 0, j = indexCheckBoxList.size(); i < j; i++) {
		bool isSame = true;
		int num = index(plotModifier, i);
		for (int k = 1, l = plotModifierList.size(); k < l; k++) {
			PlotModifier pm = plotModifierList[k];
			int n = index(pm, i);
			if (n != num) {
				isSame = false;
				indexSpinBoxList[i]->setMinimum(-1);
				indexSpinBoxList[i]->setSpecialValueText(" ");
				indexCheckBoxList[i]->setChecked(true);
				indexSpinBoxList[i]->setEnabled(true);
				indexSpinBoxList[i]->setValue(-1);
				break;
			}
		}
		if (!isSame) {
			continue;
		}
		indexSpinBoxList[i]->setMinimum(0);
		indexSpinBoxList[i]->setSpecialValueText("");
		bool flag = (num >= 0);
		indexCheckBoxList[i]->setChecked(flag);
		indexSpinBoxList[i]->setEnabled(flag);
		if (flag) {
			indexSpinBoxList[i]->setValue(num);
		}
	}
	setIndex();
}

void PlotModifierWidget::setSamplingWidget()
{
	PlotModifier plotModifier = plotModifierList[0];
	for (int i = 0, j = samplingCheckBoxList.size(); i < j; i++) {
		bool isSame = true;
		int num = sampling(plotModifier, i);
		for (int k = 1, l = plotModifierList.size(); k < l; k++) {
			PlotModifier pm = plotModifierList[k];
			int n = sampling(pm, i);
			if (num != n) {
				isSame = false;
				samplingSpinBoxList[i]->setMinimum(-1);
				samplingSpinBoxList[i]->setSpecialValueText(" ");
				samplingCheckBoxList[i]->setChecked(true);
				samplingSpinBoxList[i]->setEnabled(true);
				samplingSpinBoxList[i]->setValue(-1);
				break;
			}
		}
		if (!isSame) {
			continue;
		}
		samplingSpinBoxList[i]->setMinimum(0);
		samplingSpinBoxList[i]->setSpecialValueText("");
		bool flag = (num >= 0);
		samplingCheckBoxList[i]->setChecked(flag);
		samplingSpinBoxList[i]->setEnabled(flag);
		if (flag) {
			samplingSpinBoxList[i]->setValue(num);
		}
	}
}

void PlotModifierWidget::setColumnWidget()
{
	QString entry[AXIS_NUM];
	for (int i = 0; i < AXIS_NUM; i++) {
		entry[i] = plotModifierList[0].entry(i);
	}
	for (int i = 0; i < AXIS_NUM; i++) {
		bool isSame = true;
		QString column = entry[i];
		for (int k = 1, l = plotModifierList.size(); k < l; k++) {
			QString c = plotModifierList[k].entry(i);
			if (c != column) {
				isSame = false;
				columnCheckBox[i]->setChecked(true);
				columnLineEdit[i]->setEnabled(true);
				columnLineEdit[i]->setText("");
				break;
			}
		}
		if (!isSame) {
			continue;
		}
		bool hasColumn = !column.isEmpty();
		if (hasColumn) {
			columnLineEdit[i]->setText(column);
		}
		columnCheckBox[i]->setChecked(hasColumn);
		columnLineEdit[i]->setEnabled(hasColumn);
	}

	QString format = plotModifierList[0].format();
	for (int i = 1, j = plotModifierList.size(); i < j; i++) {
		if (plotModifierList[i].format() != format) {
			formatCheckBox->setChecked(true);
			formatLineEdit->setEnabled(true);
			formatLineEdit->setText("");
			return;
		}
	}
	bool hasFormat = !format.isEmpty();
	if (hasFormat) {
		formatLineEdit->setText(format);
	}
	formatCheckBox->setChecked(hasFormat);
	formatLineEdit->setEnabled(hasFormat);
}

void PlotModifierWidget::setSmoothingWidget()
{
	int smooth = plotModifierList[0].smooth();
	for (int i = 0, j = plotModifierList.size(); i < j; i++) {
		if (plotModifierList[i].smooth() != smooth) {
			interpCheckBox->setChecked(true);
			interpComboBox->setEnabled(true);
			interpComboBox->setCurrentIndex(-1);
			return;
		}
	}
	bool isPos = (smooth >= 0);
	if (isPos) {
		interpComboBox->setCurrentIndex(smooth);
	}
	interpCheckBox->setChecked(isPos);
	interpComboBox->setEnabled(isPos);
}

int PlotModifierWidget::index(const PlotModifier &plotModifier, int num) const
{
	switch (num) {
	case 0:
		return plotModifier.m();
	case 1:
		return plotModifier.n();
	case 2:
		return plotModifier.p();
	}
	return -1;
}

int PlotModifierWidget::sampling(const PlotModifier &plotModifier, int num) 
const
{
	switch (num) {
	case 0:
		return plotModifier.pointIncrement();
	case 1:
		return plotModifier.startPoint();
	case 2:
		return plotModifier.endPoint();
	case 3:
		return plotModifier.blockIncrement();
	case 4:
		return plotModifier.startBlock();
	case 5:
		return plotModifier.endBlock();
	}
	return -1;
}
