/*
 * Cueplot: a GUI front-end to gnuplot
 * Copyright (C) 2007, 2008, 2009 Muneyuki Noguchi <nogu@users.sourceforge.jp>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation, 
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
#include "plotstylewidget.h"

#include <QtGui/QCheckBox>
#include <QtGui/QComboBox>
#include <QtGui/QDoubleSpinBox>
#include <QtGui/QGridLayout>
#include <QtGui/QSpinBox>

PlotStyleWidget::PlotStyleWidget(QWidget *parent)
: AbstractPlotOptionWidget(parent)
{
	setWindowTitle(tr("Style"));
	setWhatsThis(tr("plot {with &lt;style&gt;}"));

	styleCheckBox = new QCheckBox;
	styleCheckBox->setText(tr("&Specify:"));
	connect(styleCheckBox, SIGNAL(toggled(bool)),
			this, SIGNAL(modified()));

	styleComboBox = new QComboBox;
	for (int i = 0; !PlotStyle::STYLE[i].isEmpty(); i++) {
		styleComboBox->addItem(PlotStyle::STYLE[i]);
	}
	connect(styleCheckBox, SIGNAL(toggled(bool)),
			styleComboBox, SLOT(setEnabled(bool)));
	connect(styleComboBox, SIGNAL(activated(int)),
			this, SIGNAL(modified()));

	lineTypeCheckBox = new QCheckBox;
	lineTypeCheckBox->setText(tr("Specify a &line type:"));
	connect(lineTypeCheckBox, SIGNAL(toggled(bool)),
			this, SIGNAL(modified()));

	lineTypeSpinBox = new QSpinBox;
	connect(lineTypeCheckBox, SIGNAL(toggled(bool)),
			lineTypeSpinBox, SLOT(setEnabled(bool)));
	connect(lineTypeSpinBox, SIGNAL(valueChanged(int)),
			this, SIGNAL(modified()));

	lineWidthCheckBox = new QCheckBox;
	lineWidthCheckBox->setText(tr("Specify l&ine width:"));
	connect(lineWidthCheckBox, SIGNAL(toggled(bool)),
			this, SIGNAL(modified()));

	lineWidthDoubleSpinBox = new QDoubleSpinBox;
	connect(lineWidthCheckBox, SIGNAL(toggled(bool)),
			lineWidthDoubleSpinBox, SLOT(setEnabled(bool)));
	connect(lineWidthDoubleSpinBox, SIGNAL(valueChanged(double)),
			this, SIGNAL(modified()));

	pointTypeCheckBox = new QCheckBox;
	pointTypeCheckBox->setText(tr("Specify a poi&nt type:"));
	connect(pointTypeCheckBox, SIGNAL(toggled(bool)),
			this, SIGNAL(modified()));

	pointTypeSpinBox = new QSpinBox;
	connect(pointTypeCheckBox, SIGNAL(toggled(bool)),
			pointTypeSpinBox, SLOT(setEnabled(bool)));
	connect(pointTypeSpinBox, SIGNAL(valueChanged(int)),
			this, SIGNAL(modified()));

	pointSizeCheckBox = new QCheckBox;
	pointSizeCheckBox->setText(tr("Specify poin&t size:"));
	connect(pointSizeCheckBox, SIGNAL(toggled(bool)),
			this, SIGNAL(modified()));

	pointSizeDoubleSpinBox = new QDoubleSpinBox;
	connect(pointSizeCheckBox, SIGNAL(toggled(bool)),
			pointSizeDoubleSpinBox, SLOT(setEnabled(bool)));
	connect(pointSizeDoubleSpinBox, SIGNAL(valueChanged(double)),
			this, SIGNAL(modified()));

	QGridLayout *mainLayout = new QGridLayout;
	mainLayout->addWidget(styleCheckBox, 0, 0);
	mainLayout->addWidget(styleComboBox, 0, 1);
	mainLayout->addItem(new QSpacerItem(0, 0, 
				QSizePolicy::Expanding, QSizePolicy::Minimum), 0, 2);
	mainLayout->addWidget(lineTypeCheckBox, 1, 0);
	mainLayout->addWidget(lineTypeSpinBox, 1, 1);
	mainLayout->addWidget(lineWidthCheckBox, 2, 0);
	mainLayout->addWidget(lineWidthDoubleSpinBox, 2, 1);
	mainLayout->addWidget(pointTypeCheckBox, 3, 0);
	mainLayout->addWidget(pointTypeSpinBox, 3, 1);
	mainLayout->addWidget(pointSizeCheckBox, 4, 0);
	mainLayout->addWidget(pointSizeDoubleSpinBox, 4, 1);
	mainLayout->addItem(new QSpacerItem(0, 0,
				QSizePolicy::Minimum, QSizePolicy::Expanding), 5, 0);

	setLayout(mainLayout);
}

void PlotStyleWidget::setStyle(const QList<PlotStyle> &list)
{
	plotStyleList = list;
	setWidget();
}

const QList<PlotStyle> &PlotStyleWidget::style() const
{
	return plotStyleList;
}

void PlotStyleWidget::apply()
{
	bool isChecked = styleCheckBox->isChecked();
	int style = isChecked ? styleComboBox->currentIndex() : -1;
	bool hasStyle = (!isChecked || style >= 0);

	isChecked = lineTypeCheckBox->isChecked();
	int lineType = isChecked ? lineTypeSpinBox->value() : -1;
	bool hasLineType = (!isChecked || lineType >= 0);

	isChecked = lineWidthCheckBox->isChecked();
	double lineWidth = isChecked ? lineWidthDoubleSpinBox->value() : -1;
	bool hasLineWidth = (!isChecked || lineWidth >= 0);

	isChecked = pointTypeCheckBox->isChecked();
	int pointType = isChecked ? pointTypeSpinBox->value() : -1;
	bool hasPointType = (!isChecked || pointType >= 0);

	isChecked = pointSizeCheckBox->isChecked();
	double pointSize = isChecked ? pointSizeDoubleSpinBox->value() : -1;
	bool hasPointSize = (!isChecked || pointSize >= 0);
	for (int i = 0, j = plotStyleList.size(); i < j; i++) {
		if (hasStyle) {
			plotStyleList[i].setStyle(style);
		}
		if (hasLineType) {
			plotStyleList[i].setLineType(lineType);
		}
		if (hasLineWidth) {
			plotStyleList[i].setLineWidth(lineWidth);
		}
		if (hasPointType) {
			plotStyleList[i].setPointType(pointType);
		}
		if (hasPointSize) {
			plotStyleList[i].setPointSize(pointSize);
		}
	}
}

void PlotStyleWidget::setWidget()
{
	setStyleWidget();
	setLineTypeWidget();
	setLineWidthWidget();
	setPointTypeWidget();
	setPointSizeWidget();
}

void PlotStyleWidget::setStyleWidget()
{
	int style = plotStyleList[0].style();
	for (int i = 1, j = plotStyleList.size(); i < j; i++) {
		if (plotStyleList[i].style() != style) {
			styleCheckBox->setChecked(true);
			styleComboBox->setEnabled(true);
			styleComboBox->setCurrentIndex(-1);
			return;
		}
	}
	bool hasStyle = (style >= 0);
	styleCheckBox->setChecked(hasStyle);
	styleComboBox->setEnabled(hasStyle);
	if (hasStyle) {
		styleComboBox->setCurrentIndex(style);
	}
}

void PlotStyleWidget::setLineTypeWidget()
{
	int lineType = plotStyleList[0].lineType();
	for (int i = 1, j = plotStyleList.size(); i < j; i++) {
		if (lineType != plotStyleList[i].lineType()) {
			lineTypeSpinBox->setMinimum(-1);
			lineTypeSpinBox->setSpecialValueText(" ");
			lineTypeCheckBox->setChecked(true);
			lineTypeSpinBox->setEnabled(true);
			lineTypeSpinBox->setValue(-1);
			return;
		}
	}
	lineTypeSpinBox->setMinimum(0);
	lineTypeSpinBox->setSpecialValueText("");
	bool hasLineType = (lineType >= 0);
	lineTypeCheckBox->setChecked(hasLineType);
	lineTypeSpinBox->setEnabled(hasLineType);
	if (hasLineType) {
		lineTypeSpinBox->setValue(lineType);
	}
}

void PlotStyleWidget::setLineWidthWidget()
{
	double lineWidth = plotStyleList[0].lineWidth();
	for (int i = 1, j = plotStyleList.size(); i < j; i++) {
		if (lineWidth != plotStyleList[i].lineWidth()) {
			lineWidthDoubleSpinBox->setMinimum(-1);
			lineWidthDoubleSpinBox->setSpecialValueText(" ");
			lineWidthCheckBox->setChecked(true);
			lineWidthDoubleSpinBox->setEnabled(true);
			lineWidthDoubleSpinBox->setValue(-1);
			return;
		}
	}
	lineWidthDoubleSpinBox->setMinimum(0);
	lineWidthDoubleSpinBox->setSpecialValueText("");
	bool hasLineWidth = (lineWidth >= 0);
	lineWidthCheckBox->setChecked(hasLineWidth);
	lineWidthDoubleSpinBox->setEnabled(hasLineWidth);
	if (hasLineWidth) {
		lineWidthDoubleSpinBox->setValue(lineWidth);
	}
}

void PlotStyleWidget::setPointTypeWidget()
{
	int pointType = plotStyleList[0].pointType();
	for (int i = 1, j = plotStyleList.size(); i < j; i++) {
		if (pointType != plotStyleList[i].pointType()) {
			pointTypeSpinBox->setMinimum(-1);
			pointTypeSpinBox->setSpecialValueText(" ");
			pointTypeCheckBox->setChecked(true);
			pointTypeSpinBox->setEnabled(true);
			pointTypeSpinBox->setValue(-1);
			return;
		}
	}
	pointTypeSpinBox->setMinimum(0);
	pointTypeSpinBox->setSpecialValueText("");
	bool hasPointType = (pointType >= 0);
	pointTypeCheckBox->setChecked(hasPointType);
	pointTypeSpinBox->setEnabled(hasPointType);
	if (hasPointType) {
		pointTypeSpinBox->setValue(pointType);
	}
}


void PlotStyleWidget::setPointSizeWidget()
{
	double pointSize = plotStyleList[0].pointSize();
	for (int i = 1, j = plotStyleList.size(); i < j; i++) {
		if (pointSize != plotStyleList[i].pointSize()) {
			pointSizeDoubleSpinBox->setMinimum(-1);
			pointSizeDoubleSpinBox->setSpecialValueText(" ");
			pointSizeCheckBox->setChecked(true);
			pointSizeDoubleSpinBox->setEnabled(true);
			pointSizeDoubleSpinBox->setValue(-1);
			return;
		}
	}
	pointSizeDoubleSpinBox->setMinimum(0);
	pointSizeDoubleSpinBox->setSpecialValueText("");
	bool hasPointSize = (pointSize >= 0);
	pointSizeCheckBox->setChecked(hasPointSize);
	pointSizeDoubleSpinBox->setEnabled(hasPointSize);
	if (hasPointSize) {
		pointSizeDoubleSpinBox->setValue(pointSize);
	}
}
