/*
 * Cueplot: a GUI front-end to gnuplot
 * Copyright (C) 2007, 2008, 2009 Muneyuki Noguchi <nogu@users.sourceforge.jp>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation, 
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
#include "plottitlewidget.h"

#include <QtGui/QLineEdit>
#include <QtGui/QGridLayout>
#include <QtGui/QRadioButton>

PlotTitleWidget::PlotTitleWidget(QWidget *parent)
: AbstractPlotOptionWidget(parent)
{
	setWindowTitle(tr("Title"));
	setWhatsThis(tr("plot {&lt;title-spec&gt;}"));

	defaultTitleRadioButton = new QRadioButton;
	defaultTitleRadioButton->setText(tr("&Defaults"));
	defaultTitleRadioButton->setChecked(true);
	connect(defaultTitleRadioButton, SIGNAL(toggled(bool)),
			this, SIGNAL(modified()));

	noTitleRadioButton = new QRadioButton;
	noTitleRadioButton->setText(tr("&No title"));
	connect(noTitleRadioButton, SIGNAL(toggled(bool)),
			this, SIGNAL(modified()));

	customTitleRadioButton = new QRadioButton;
	customTitleRadioButton->setText(tr("C&ustom title:"));
	connect(customTitleRadioButton, SIGNAL(toggled(bool)),
			this, SIGNAL(modified()));

	titleLineEdit = new QLineEdit;
	connect(titleLineEdit, SIGNAL(textChanged(const QString &)),
			this, SIGNAL(modified()));
	connect(customTitleRadioButton, SIGNAL(toggled(bool)),
			titleLineEdit, SLOT(setEnabled(bool)));

	QGridLayout *mainLayout = new QGridLayout;
	mainLayout->addWidget(defaultTitleRadioButton, 0, 0, 1, 2);
	mainLayout->addWidget(noTitleRadioButton, 1, 0, 1, 2);
	mainLayout->addWidget(customTitleRadioButton, 2, 0, 1, 2);
	mainLayout->addItem(new QSpacerItem(20, 20,
				QSizePolicy::Fixed, QSizePolicy::Minimum), 3, 0);
	mainLayout->addWidget(titleLineEdit, 3, 1);
	mainLayout->addItem(new QSpacerItem(0, 0,
				QSizePolicy::Minimum, QSizePolicy::Expanding), 4, 0);

	setLayout(mainLayout);
}

void PlotTitleWidget::setTitle(const QList<PlotTitle> &list)
{
	plotTitleList = list;
	setWidget();
}

const QList<PlotTitle> &PlotTitleWidget::title() const
{
	return plotTitleList;
}

void PlotTitleWidget::apply()
{
	bool isNotitle = noTitleRadioButton->isChecked();
	bool isCustom = customTitleRadioButton->isChecked();
	bool isSame = isNotitle || isCustom || defaultTitleRadioButton->isChecked();
	QString title = isCustom ? titleLineEdit->text() : "";
	for (int i = 0, j = plotTitleList.size(); i < j; i++) {
		if (!isSame) {
			continue;
		}
		plotTitleList[i].setNoTitle(isNotitle);
		plotTitleList[i].setTitle(title);
	}
}

void PlotTitleWidget::setWidget()
{
	PlotTitle plotTitle = plotTitleList[0];
	bool isNoTitle = plotTitle.isNoTitle();
	QString title = plotTitle.title();
	bool isDefault = title.isEmpty();
	for (int i = 1, j = plotTitleList.size(); i < j; i++) {
		QString t = plotTitleList[i].title();
		if (plotTitleList[i].isNoTitle() != isNoTitle
				|| t.isEmpty() != isDefault
				|| (!isDefault && t != title)) {
			// 何も選択されていない状態にするためautoExclusive属性を
			// 一旦すべて無効にする
			noTitleRadioButton->setAutoExclusive(false);
			defaultTitleRadioButton->setAutoExclusive(false);
			customTitleRadioButton->setAutoExclusive(false);

			noTitleRadioButton->setChecked(false);
			defaultTitleRadioButton->setChecked(false);
			customTitleRadioButton->setChecked(false);

			noTitleRadioButton->setAutoExclusive(true);
			defaultTitleRadioButton->setAutoExclusive(true);
			customTitleRadioButton->setAutoExclusive(true);

			titleLineEdit->setEnabled(false);
			return;
		}
	}
	bool isCustom;
	if (isNoTitle) {
		noTitleRadioButton->setChecked(true);
		isCustom = false;
	} else if (isDefault) {
		defaultTitleRadioButton->setChecked(true);
		isCustom = false;
	} else {
		customTitleRadioButton->setChecked(true);
		titleLineEdit->setText(title);
		isCustom = true;
	}
	titleLineEdit->setEnabled(isCustom);
}
