/*
 * Cueplot: a GUI front-end to gnuplot
 * Copyright (C) 2007, 2008, 2009 Muneyuki Noguchi <nogu@users.sourceforge.jp>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation, 
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
#include "plotoptiondialog.h"

#include <QtCore/QCoreApplication>
#include <QtCore/QDir>
#include <QtGui/QCloseEvent>
#if QT_VERSION >= 0x040200
# include <QtGui/QDialogButtonBox>
#endif
#include <QtGui/QListWidget>
#include <QtGui/QMessageBox>
#include <QtGui/QPushButton>
#include <QtGui/QSplitter>
#include <QtGui/QStackedWidget>
#include <QtGui/QVBoxLayout>

#include "plotmodifierwidget.h"
#include "plototherswidget.h"
#include "plotstylewidget.h"
#include "plottitlewidget.h"
#if QT_VERSION >= 0x040200
# include "misc.h"
#endif

/*!
 * \brief リストに含まれる Plot クラスのオプジェクトがすべて
 * データファイルを表すのかどうかを返す
 * \param list Plot クラスのオブジェクトのリスト
 * \return すべてデータファイルなら true
 */
static bool isDataFile(const QList<Plot> &list)
{
	for (int i = 0, j = list.size(); i < j; i++) {
		if (list[i].isFunction()) {
			return false;
		}
	}
	return true;
}

PlotOptionDialog::PlotOptionDialog(QWidget *parent)
: QDialog(parent)
{
	othersWidget = new PlotOthersWidget;
	titleWidget = new PlotTitleWidget;
	styleWidget = new PlotStyleWidget;
	modifierWidget = new PlotModifierWidget;

	pagesStackedWidget = new QStackedWidget;
	pagesStackedWidget->addWidget(modifierWidget);
	pagesStackedWidget->addWidget(titleWidget);
	pagesStackedWidget->addWidget(styleWidget);
	pagesStackedWidget->addWidget(othersWidget);

	contentsListWidget = new QListWidget;
	for (int i = 0, j = pagesStackedWidget->count(); i < j; i++) {
		QWidget *widget = pagesStackedWidget->widget(i);
		contentsListWidget->addItem(widget->windowTitle());
		contentsListWidget->item(i)->setWhatsThis(widget->whatsThis());
	}
	contentsListWidget->setCurrentRow(0);
	connect(contentsListWidget,
			SIGNAL(currentItemChanged(QListWidgetItem *, QListWidgetItem *)),
			this, SLOT(changePage(QListWidgetItem *, QListWidgetItem *)));

	QSplitter *centerSplitter = new QSplitter;
	centerSplitter->addWidget(contentsListWidget);
	centerSplitter->addWidget(pagesStackedWidget);

#if QT_VERSION >= 0x040200
	openPushButton = new QPushButton;
	openPushButton->setText(tr("O&pen"));
	openPushButton->setAutoDefault(false);
	openPushButton->setDefault(false);
	connect(openPushButton, SIGNAL(clicked()), this, SLOT(open()));
#endif

	QPushButton *acceptPushButton = new QPushButton;
	acceptPushButton->setText(tr("&OK"));
	acceptPushButton->setAutoDefault(true);
	acceptPushButton->setDefault(true);
	connect(acceptPushButton, SIGNAL(clicked()), this, SLOT(accept()));

	QPushButton *applyPushButton = new QPushButton;
	applyPushButton->setText(tr("&Apply"));
	connect(applyPushButton, SIGNAL(clicked()), this, SLOT(apply()));

	QPushButton *cancelPushButton = new QPushButton;
	cancelPushButton->setText(tr("&Cancel"));
	connect(cancelPushButton, SIGNAL(clicked()), this, SLOT(reject()));

	// ------------------------ create layouts
	// ----------- assemble layouts and widgets

#if QT_VERSION >= 0x040200
	QDialogButtonBox *dialogButtonBox = new QDialogButtonBox;
	dialogButtonBox->addButton(openPushButton, 
			QDialogButtonBox::ActionRole);
	dialogButtonBox->addButton(acceptPushButton, 
			QDialogButtonBox::AcceptRole);
	dialogButtonBox->addButton(applyPushButton,
			QDialogButtonBox::ApplyRole);
	dialogButtonBox->addButton(cancelPushButton,
			QDialogButtonBox::RejectRole);
#else
	QHBoxLayout *buttonLayout = new QHBoxLayout;
	buttonLayout->addStretch();
	buttonLayout->addWidget(acceptPushButton);
	buttonLayout->addWidget(applyPushButton);
	buttonLayout->addWidget(cancelPushButton);
#endif

	QVBoxLayout *mainLayout = new QVBoxLayout;
	mainLayout->addWidget(centerSplitter);
#if QT_VERSION >= 0x040200
	mainLayout->addWidget(dialogButtonBox);
#else
	mainLayout->addLayout(buttonLayout);
#endif

	setLayout(mainLayout);

	adjustSize();
	// 黄金比に近い整数比
	int h = height();
	int w = h * 8 / 5;
	if (w < width()) {
		resize(w, h);
	}
}

void PlotOptionDialog::setPlot(const QList<Plot> &list)
{
	plotList = list;
	if (list.size() == 1) {
		bool isFunction = list[0].isFunction();
		QString text;
		if (isFunction) {
			text = list[0].function();
		} else {
			text = list[0].dataFile();
			// ホームディレクトリを"~"で表示する
			text.replace(
				QRegExp('^' + QRegExp::escape(
#if QT_VERSION >= 0x040200
					QDir::toNativeSeparators(QDir::homePath())
#else
					QDir::convertSeparators(QDir::homePath())
#endif
				)), "~");
		}
		setWindowTitle(tr("Options of %1 %2")
				.arg(isFunction ? tr("Function") : tr("Data File"))
				.arg(text)
				+ "[*] - " + QCoreApplication::applicationName());
	} else {
		setWindowTitle(tr("Options of %1 Items").arg(list.size())
				+ "[*] - " + QCoreApplication::applicationName());
	}
	QList<int> axisList;
	QList<PlotTitle> titleList;
	QList<PlotStyle> styleList;
	for (int i = 0, j = list.size(); i < j; i++) {
		axisList.append(list[i].axis());
		titleList.append(list[i].titleSpecification());
		styleList.append(list[i].style());
	}
	othersWidget->setAxis(axisList);
	titleWidget->setTitle(titleList);
	styleWidget->setStyle(styleList);
	bool isFile = isDataFile(list);
#if QT_VERSION >= 0x040200
	openPushButton->setEnabled(isFile);
#endif
	if (!isFile) {
		modifierWidget->setEnabled(false);
		QString title = modifierWidget->windowTitle();
		for (int i = 0, j = pagesStackedWidget->count(); i < j; i++) {
			QListWidgetItem *listWidgetItem = contentsListWidget->item(i);
			if (listWidgetItem->text() == title) {
				listWidgetItem->setFlags(listWidgetItem->flags() 
						& ~Qt::ItemIsEnabled);
				break;
			}
		}
	} else {
		QList<PlotModifier> modifierList;
		for (int i = 0, j = list.size(); i < j; i++) {
			modifierList.append(list[i].dataFileModifier());
		}
		modifierWidget->setModifier(modifierList);
	}
	for (int i = 0, j = pagesStackedWidget->count(); i < j; i++) {
		AbstractPlotOptionWidget *widget
				= qobject_cast<AbstractPlotOptionWidget *>(
						pagesStackedWidget->widget(i));
		connect(widget, SIGNAL(modified()), this, SLOT(setModified()));
	}
}

bool PlotOptionDialog::canClose()
{
	if (!isWindowModified()) {
		return true;
	}
	switch (QMessageBox::warning(this, tr("Unsaved Changes") + " - "
				+ QCoreApplication::applicationName(),
				tr("There are unsaved changes in this option.\n"
					"Do you want to apply the changes?"),
				tr("&Apply"), tr("&Discard"), tr("&Cancel"), 0, 2)) {
	case 0:
		apply();
		return true;
	case 1:
		discard();
		return true;
	case 2:
		return false;
	}
	return false;
}

#if QT_VERSION >= 0x040200
void PlotOptionDialog::open()
{
	for (int i = 0, j = plotList.size(); i < j; i++) {
		if (!plotList[i].isFunction()) {
			::openDataFile(plotList[i].dataFile(), *this);
		}
	}
}
#endif

void PlotOptionDialog::accept()
{
	apply();
	QDialog::accept();
}

void PlotOptionDialog::apply()
{
	for (int i = 0, j = pagesStackedWidget->count(); i < j; i++) {
		qobject_cast<AbstractPlotOptionWidget *>(pagesStackedWidget->widget(i))
			->apply();
	}
	QList<Plot> list = plotList;
	QList<int> axisList = othersWidget->axis();
	QList<PlotTitle> titleList = titleWidget->title(); 
	QList<PlotStyle> styleList = styleWidget->style();
	for (int i = 0, j = list.size(); i < j; i++) {
		list[i].setAxis(axisList[i]);
		list[i].setTitleSpecification(titleList[i]);
		list[i].setStyle(styleList[i]);
	}
	if (isDataFile(list)) {
		QList<PlotModifier> modifierList = modifierWidget->modifier();
		for (int i = 0, j = list.size(); i < j; i++) {
			list[i].setDataFileModifier(modifierList[i]);
		}
	}
	emit optionChanged(*this, list);
	setWindowModified(false);
}

void PlotOptionDialog::reject()
{
	discard();
	QDialog::reject();
}

void PlotOptionDialog::changePage(QListWidgetItem *current,
		QListWidgetItem *previous)
{
	if (!current) {
		current = previous;
	}
	pagesStackedWidget->setCurrentIndex(contentsListWidget->row(current));
}

void PlotOptionDialog::setModified()
{
	setWindowModified(true);
}

void PlotOptionDialog::closeEvent(QCloseEvent *e)
{
	discard();
	e->accept();
}

void PlotOptionDialog::discard()
{
	for (int i = 0, j = pagesStackedWidget->count(); i < j; i++) {
		qobject_cast<AbstractPlotOptionWidget *>(pagesStackedWidget->widget(i))
			->discard();
	}
	setWindowModified(false);
}
