/*
 * Cueplot: a GUI front-end to gnuplot
 * Copyright (C) 2007, 2009 Muneyuki Noguchi <nogu@users.sourceforge.jp>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation, 
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
#include "axisrange.h"

#include <QtCore/QStringList>

const QString AxisRange::AXIS[] = {
	"x", "y", "z", "x2", "y2", "cb", "r", "t", "u", "v", ""
};

AxisRange::AxisRange()
{
	for (int i = 0; !AXIS[i].isEmpty(); i++) {
		minList.append("*");
		maxList.append("*");
		reverseList.append(false);
		writeBackList.append(false);
		restoreList.append(false);
	}
}

void AxisRange::setMinimum(int axis, const QString &min)
{
	Q_ASSERT_X(axis >= 0
		&& axis < static_cast<int>(sizeof(AXIS) / sizeof(AXIS[0]) - 1),
		"AxisRange::setMinimum", "index out of range");
	minList[axis] = min;
}

const QString &AxisRange::minimum(int axis) const
{
	Q_ASSERT_X(axis >= 0
		&& axis < static_cast<int>(sizeof(AXIS) / sizeof(AXIS[0]) - 1),
		"AxisRange::minimum", "index out of range");
	return minList.at(axis);
}

void AxisRange::setMaximum(int axis, const QString &max)
{
	Q_ASSERT_X(axis >= 0
		&& axis < static_cast<int>(sizeof(AXIS) / sizeof(AXIS[0]) - 1),
		"AxisRange::setMaximum", "index out of range");
	maxList[axis] = max;
}

const QString &AxisRange::maximum(int axis) const
{
	Q_ASSERT_X(axis >= 0
		&& axis < static_cast<int>(sizeof(AXIS) / sizeof(AXIS[0]) - 1),
		"AxisRange::maximum", "index out of range");
	return maxList.at(axis);
}

void AxisRange::setReverse(int axis, bool reverse)
{
	Q_ASSERT_X(axis >= 0
		&& axis < static_cast<int>(sizeof(AXIS) / sizeof(AXIS[0]) - 1),
		"AxisRange::setReverse", "index out of range");
	reverseList[axis] = reverse;
}

bool AxisRange::isReverse(int axis) const
{
	Q_ASSERT_X(axis >= 0
		&& axis < static_cast<int>(sizeof(AXIS) / sizeof(AXIS[0]) - 1),
		"AxisRange::isReverse", "index out of range");
	return reverseList.at(axis);
}

void AxisRange::setWrittenBack(int axis, bool writeBack)
{
	Q_ASSERT_X(axis >= 0
		&& axis < static_cast<int>(sizeof(AXIS) / sizeof(AXIS[0]) - 1),
		"AxisRange::setWrittenBack", "index out of range");
	writeBackList[axis] = writeBack;
}

bool AxisRange::writeBacks(int axis) const
{
	Q_ASSERT_X(axis >= 0
		&& axis < static_cast<int>(sizeof(AXIS) / sizeof(AXIS[0]) - 1),
		"AxisRange::writeBacks", "index out of range");
	return writeBackList.at(axis);
}

void AxisRange::setRestored(int axis, bool restore)
{
	Q_ASSERT_X(axis >= 0
		&& axis < static_cast<int>(sizeof(AXIS) / sizeof(AXIS[0]) - 1),
		"AxisRange::setRestored", "index out of range");
	restoreList[axis] = restore;
}

bool AxisRange::restores(int axis) const
{
	Q_ASSERT_X(axis >= 0
		&& axis < static_cast<int>(sizeof(AXIS) / sizeof(AXIS[0]) - 1),
		"AxisRange::restores", "index out of range");
	return restoreList.at(axis);
}

QString AxisRange::command() const
{
	QStringList commandList;
	for (int i = 0; !AXIS[i].isEmpty(); i++) {
		QString str = "set " + AXIS[i] + "range ";
		if (restoreList.at(i)) {
			str += "restore";
			commandList << str;
		} else {
			str += "[ " + minList.at(i) + " : " + 
				maxList.at(i) + " ] ";
			str += reverseList.at(i) ? "" : "no";
			str += "reverse ";
			str += writeBackList.at(i) ? "" : "no";
			str += "writeback";
			commandList << str;
		}
	}
	return commandList.join("\n");
}
