/*
 * Cueplot: a GUI front-end to gnuplot
 * Copyright (C) 2009 Muneyuki Noguchi <nogu@users.sourceforge.jp>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation, 
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
#include <cstdlib> // exit()

#include <QtCore/QString>

#include "fit.h"

static void test_init(Fit &fit)
{
	if (fit.functionName() != "f(x)") {
		exit(1);
	}
	if (!fit.isParameterFile()) {
		exit(1);
	}
	fit.setFunction("a*x+b");
	fit.setDataFile("foo");
	fit.setParameterFile(QString("bar"));
	if (fit.command() != "f(x)=a*x+b\nfit f(x) 'foo' via 'bar'") {
		exit(1);
	}
}

static void test_range(Fit &fit)
{
	FitRange fitRange;
	fitRange.setMinimum(0, "0");
	fitRange.setMaximum(0, "10");
	fit.setRange(fitRange);
	if (fit.command() != "f(x)=a*x+b\nfit [0:10] f(x) 'foo' via 'bar'") {
		exit(1);
	}
	fitRange.setMinimum(1, "-pi");
	fit.setRange(fitRange);
	if (fit.command() != "f(x)=a*x+b\n"
			"fit [0:10] [-pi:] f(x) 'foo' via 'bar'") {
		exit(1);
	}
	fitRange.setMaximum(1, "pi");
	fit.setRange(fitRange);
	if (fit.command() != "f(x)=a*x+b\n"
			"fit [0:10] [-pi:pi] f(x) 'foo' via 'bar'") {
		exit(1);
	}
	fitRange.setMinimum(0, "");
	fitRange.setMaximum(0, "");
	fit.setRange(fitRange);
	if (fit.command() != "f(x)=a*x+b\nfit [ ] [-pi:pi] f(x) 'foo' via 'bar'") {
		exit(1);
	}
	fitRange.setMinimum(0, "0");
	fitRange.setMaximum(0, "10");
	fitRange.setDummyVariable(0, "y");
	fit.setRange(fitRange);
	if (fit.command() != "f(x)=a*x+b\n"
			"fit [y=0:10] [-pi:pi] f(x) 'foo' via 'bar'") {
		exit(1);
	}
	fitRange.setDummyVariable(1, "z");
	fit.setRange(fitRange);
	if (fit.command() != "f(x)=a*x+b\n"
			"fit [y=0:10] [-pi:pi] f(x) 'foo' via 'bar'") {
		exit(1);
	}
	fit.setRange(FitRange());
}

static void test_modifier(Fit &fit)
{
	FitModifier fitModifier;
	fitModifier.setN(2);
	fit.setDataFileModifier(fitModifier);
	if (fit.command() != "f(x)=a*x+b\nfit f(x) 'foo' via 'bar'") {
		exit(1);
	}
	fitModifier.setM(1);
	fit.setDataFileModifier(fitModifier);
	if (fit.command() != "f(x)=a*x+b\nfit f(x) 'foo' index 1:2 via 'bar'") {
		exit(1);
	}
	fitModifier.setP(1);
	fit.setDataFileModifier(fitModifier);
	if (fit.command() != "f(x)=a*x+b\nfit f(x) 'foo' index 1:2:1 via 'bar'") {
		exit(1);
	}
	fitModifier.setM(-1);
	fitModifier.setN(-1);
	fitModifier.setP(-1);
	fit.setDataFileModifier(fitModifier);
	if (fit.command() != "f(x)=a*x+b\nfit f(x) 'foo' via 'bar'") {
		exit(1);
	}
	fitModifier.setStartPoint(3);
	fit.setDataFileModifier(fitModifier);
	if (fit.command() != "f(x)=a*x+b\nfit f(x) 'foo' every ::3 via 'bar'") {
		exit(1);
	}
	fitModifier.setPointIncrement(2);
	fit.setDataFileModifier(fitModifier);
	if (fit.command() != "f(x)=a*x+b\nfit f(x) 'foo' every 2::3 via 'bar'") {
		exit(1);
	}
	fitModifier.setEndPoint(10);
	fit.setDataFileModifier(fitModifier);
	if (fit.command() != "f(x)=a*x+b\nfit f(x) 'foo' every 2::3::10"
			" via 'bar'") {
		exit(1);
	}
	fitModifier.setEndBlock(12);
	fit.setDataFileModifier(fitModifier);
	if (fit.command() != "f(x)=a*x+b\nfit f(x) 'foo' every 2::3::10:12"
			" via 'bar'") {
		exit(1);
	}
	fitModifier.setStartBlock(4);
	fit.setDataFileModifier(fitModifier);
	if (fit.command() != "f(x)=a*x+b\nfit f(x) 'foo' every 2::3:4:10:12"
			" via 'bar'") {
		exit(1);
	}
	fitModifier.setBlockIncrement(1);
	fit.setDataFileModifier(fitModifier);
	if (fit.command() != "f(x)=a*x+b\nfit f(x) 'foo' every 2:1:3:4:10:12"
			" via 'bar'") {
		exit(1);
	}
	fitModifier.setPointIncrement(-1);
	fitModifier.setBlockIncrement(-1);
	fitModifier.setStartPoint(-1);
	fitModifier.setStartBlock(-1);
	fitModifier.setEndPoint(-1);
	fitModifier.setEndBlock(-1);
	fit.setDataFileModifier(fitModifier);
	if (fit.command() != "f(x)=a*x+b\nfit f(x) 'foo' via 'bar'") {
		exit(1);
	}
	fitModifier.setEntry(1, "($2*3)");
	fit.setDataFileModifier(fitModifier);
	if (fit.command() != "f(x)=a*x+b\nfit f(x) 'foo' via 'bar'") {
		exit(1);
	}
	fitModifier.setEntry(0, "($1/3)");
	fit.setDataFileModifier(fitModifier);
	if (fit.command() != "f(x)=a*x+b\nfit f(x) 'foo' using ($1/3):($2*3)"
			" via 'bar'") {
		exit(1);
	}
	fitModifier.setEntry(2, "3");
	fit.setDataFileModifier(fitModifier);
	if (fit.command() != "f(x)=a*x+b\nfit f(x) 'foo' using ($1/3):($2*3):3"
			" via 'bar'") {
		exit(1);
	}
	fitModifier.setFormat("%lf");
	fit.setDataFileModifier(fitModifier);
	if (fit.command() != "f(x)=a*x+b\nfit f(x) 'foo' using ($1/3):($2*3):3 "
			"'%lf' via 'bar'") {
		exit(1);
	}
	fit.setDataFileModifier(FitModifier());
}

static void test_variable(Fit &fit)
{
	QStringList list;
	list << "a" << "b";
	fit.setVariable(list);
	fit.setParameterFile(false);
	if (fit.isParameterFile()) {
		exit(1);
	}
	if (fit.command() != "f(x)=a*x+b\nfit f(x) 'foo' via a,b") {
		exit(1);
	}
}

static void test_function_name(Fit &fit)
{
	fit.setFunctionName("g(x)");
	if (fit.functionName() != "g(x)") {
		exit(1);
	}
	if (fit.command() != "g(x)=a*x+b\nfit g(x) 'foo' via a,b") {
		exit(1);
	}
}

int fittest(int, char *[])
{
	Fit fit;
	test_init(fit);
	test_range(fit);
	test_modifier(fit);
	test_variable(fit);
	test_function_name(fit);
	return 0;
}
