/*
 * Cueplot: a GUI front-end to gnuplot
 * Copyright (C) 2009 Muneyuki Noguchi <nogu@users.sourceforge.jp>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation, 
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
#include <cstdlib> // exit()

#include <QtCore/QString>

#include "plot.h"

static void test_init(Plot &plot)
{
	if (!plot.isFunction()) {
		exit(1);
	}
	plot.setFunction(QString("x"));
	if (plot.command() != "x") {
		exit(1);
	}
}

static void test_range(Plot &plot)
{
	PlotRange plotRange;
	if (plotRange.isParametric()) {
		exit(1);
	}
	plotRange.setMinimum(0, "0");
	plotRange.setMaximum(0, "10");
	plot.setRange(plotRange);
	if (plot.command() != "x") {
		exit(1);
	}
	plotRange.setMinimum(1, "-pi");
	plot.setRange(plotRange);
	if (plot.command() != "[-pi:] x") {
		exit(1);
	}
	plotRange.setMaximum(1, "pi");
	plot.setRange(plotRange);
	if (plot.command() != "[-pi:pi] x") {
		exit(1);
	}
	plotRange.setParametric(true);
	plot.setRange(plotRange);
	if (plot.command() != "[0:10] [-pi:pi] x") {
		exit(1);
	}
	plotRange.setMinimum(0, "");
	plotRange.setMaximum(0, "");
	plot.setRange(plotRange);
	if (plot.command() != "[ ] [-pi:pi] x") {
		exit(1);
	}
	plotRange.setParametric(false);
	plotRange.setDummyVariable(1, "y");
	plot.setRange(plotRange);
	if (plot.command() != "[y=-pi:pi] x") {
		exit(1);
	}
	plotRange.setDummyVariable(2, "z");
	plot.setRange(plotRange);
	if (plot.command() != "[y=-pi:pi] x") {
		exit(1);
	}
	plot.setRange(PlotRange());
}

static void test_data_file(Plot &plot)
{
	plot.setFunction(false);
	plot.setDataFile("foo");
	if (plot.isFunction()) {
		exit(1);
	}
	if (plot.command() != "'foo'") {
		exit(1);
	}
}

static void test_modifier(Plot &plot)
{
	PlotModifier plotModifier;
	plotModifier.setN(2);
	plot.setDataFileModifier(plotModifier);
	if (plot.command() != "'foo'") {
		exit(1);
	}
	plotModifier.setM(1);
	plot.setDataFileModifier(plotModifier);
	if (plot.command() != "'foo' index 1:2") {
		exit(1);
	}
	plotModifier.setP(1);
	plot.setDataFileModifier(plotModifier);
	if (plot.command() != "'foo' index 1:2:1") {
		exit(1);
	}
	plotModifier.setM(-1);
	plotModifier.setN(-1);
	plotModifier.setP(-1);
	plot.setDataFileModifier(plotModifier);
	if (plot.command() != "'foo'") {
		exit(1);
	}
	plotModifier.setStartPoint(3);
	plot.setDataFileModifier(plotModifier);
	if (plot.command() != "'foo' every ::3") {
		exit(1);
	}
	plotModifier.setPointIncrement(2);
	plot.setDataFileModifier(plotModifier);
	if (plot.command() != "'foo' every 2::3") {
		exit(1);
	}
	plotModifier.setEndPoint(10);
	plot.setDataFileModifier(plotModifier);
	if (plot.command() != "'foo' every 2::3::10") {
		exit(1);
	}
	plotModifier.setEndBlock(12);
	plot.setDataFileModifier(plotModifier);
	if (plot.command() != "'foo' every 2::3::10:12") {
		exit(1);
	}
	plotModifier.setStartBlock(4);
	plot.setDataFileModifier(plotModifier);
	if (plot.command() != "'foo' every 2::3:4:10:12") {
		exit(1);
	}
	plotModifier.setBlockIncrement(1);
	plot.setDataFileModifier(plotModifier);
	if (plot.command() != "'foo' every 2:1:3:4:10:12") {
		exit(1);
	}
	plotModifier.setPointIncrement(-1);
	plotModifier.setBlockIncrement(-1);
	plotModifier.setStartPoint(-1);
	plotModifier.setStartBlock(-1);
	plotModifier.setEndPoint(-1);
	plotModifier.setEndBlock(-1);
	plot.setDataFileModifier(plotModifier);
	if (plot.command() != "'foo'") {
		exit(1);
	}
	plotModifier.setEntry(1, "($2*3)");
	plot.setDataFileModifier(plotModifier);
	if (plot.command() != "'foo'") {
		exit(1);
	}
	plotModifier.setEntry(0, "($1/3)");
	plot.setDataFileModifier(plotModifier);
	if (plot.command() != "'foo' using ($1/3):($2*3)") {
		exit(1);
	}
	plotModifier.setEntry(2, "3");
	plot.setDataFileModifier(plotModifier);
	if (plot.command() != "'foo' using ($1/3):($2*3):3") {
		exit(1);
	}
	plotModifier.setFormat("%lf");
	plot.setDataFileModifier(plotModifier);
	if (plot.command() != "'foo' using ($1/3):($2*3):3 '%lf'") {
		exit(1);
	}
	plotModifier.setEntry(0, "");
	plotModifier.setFormat("");
	plot.setDataFileModifier(plotModifier);
	if (plot.command() != "'foo'") {
		exit(1);
	}
	plotModifier.setSmooth(0);
	plot.setDataFileModifier(plotModifier);
	if (plot.command() != "'foo' smooth unique") {
		exit(1);
	}
	plotModifier.setSmooth(4);
	plot.setDataFileModifier(plotModifier);
	if (plot.command() != "'foo' smooth bezier") {
		exit(1);
	}
	plot.setDataFileModifier(PlotModifier());
}

static void test_axes(Plot &plot)
{
	plot.setAxis(2);
	if (plot.command() != "'foo' axes x1y2") {
		exit(1);
	}
	plot.setFunction(true);
	if (plot.command() != "x axes x1y2") {
		exit(1);
	}
	plot.setFunction(false);

	plot.setAxis(-1);
	if (plot.command() != "'foo'") {
		exit(1);
	}
	plot.setFunction(true);
	if (plot.command() != "x") {
		exit(1);
	}
	plot.setFunction(false);
}

static void test_title(Plot &plot)
{
	PlotTitle plotTitle;
	if (plotTitle.isNoTitle()) {
		exit(1);
	}
	plot.setTitleSpecification(plotTitle);
	if (plot.command() != "'foo'") {
		exit(1);
	}

	plotTitle.setTitle("bar");
	plot.setTitleSpecification(plotTitle);
	if (plot.command() != "'foo' title \"bar\"") {
		exit(1);
	}

	plot.setFunction(true);
	if (plot.command() != "x title \"bar\"") {
		exit(1);
	}
	plot.setFunction(false);

	plotTitle.setNoTitle(true);
	if (!plotTitle.isNoTitle()) {
		exit(1);
	}
	plot.setTitleSpecification(plotTitle);
	if (plot.command() != "'foo' notitle") {
		exit(1);
	}

	plot.setFunction(true);
	if (plot.command() != "x notitle") {
		exit(1);
	}
	plot.setFunction(false);

	plot.setTitleSpecification(PlotTitle());
}

static void test_style(Plot &plot)
{
	PlotStyle plotStyle;
	plotStyle.setStyle(2);
	plot.setStyle(plotStyle);
	if (plot.command() != "'foo' with linespoints") {
		exit(1);
	}

	plot.setFunction(true);
	if (plot.command() != "x with linespoints") {
		exit(1);
	}
	plot.setFunction(false);

	plotStyle.setLineType(3);
	plotStyle.setLineWidth(1);
	plotStyle.setPointType(4);
	plotStyle.setPointSize(2);
	plot.setStyle(plotStyle);
	if (plot.command() != "'foo' with linespoints linetype 3 "
		"linewidth 1 pointtype 4 pointsize 2") {
		exit(1);
	}
	plot.setFunction(true);
	if (plot.command() != "x with linespoints linetype 3 "
		"linewidth 1 pointtype 4 pointsize 2") {
		exit(1);
	}
	plot.setFunction(false);

	plot.setStyle(PlotStyle());
}

int plottest(int, char *[])
{
	Plot plot;
	test_init(plot);
	test_range(plot);
	test_data_file(plot);
	test_modifier(plot);
	test_axes(plot);
	test_title(plot);
	test_style(plot);
	return 0;
}
