/*
 * NichExecuter.java
 *
 * Copyright 2014 the curewallet.org.
 * http://www.curewallet.org/
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.curewallet.nagesen.nich;

import java.util.ArrayList;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.curewallet.nagesen.AddressData;
import org.curewallet.nagesen.NagesenException;
import org.curewallet.nagesen.ececuter.AbstractExecuter;

/**
 * 
 * 
 * @author Nezumi Kozo
 * @since 1.0
 */
public class NichExecuter extends AbstractExecuter {
	
	private GetNichData getNichData = null;

	private String url = null;
	private String lasyModified = null;
	private long range = 0;
	private int statusCode = 0;

	private String type = null;
	private int startResNo = 0;
	private int endResNo = 0;
	private boolean includeMessageAddress = false;
	private String nameAddressPattern = null;
	private int[] nameAddressIndex = null;
	private String messageAddressPattern = null;
	private int[] messageAddressIndex = null;
	
	private int lastResNo = 0;
	
	
	/**
	 * 
	 * @since 1.0
	 */
	public NichExecuter() {
	}

	/* (非 Javadoc)
	 * @see org.curewallet.nagesen.AbstractExecuter#getData()
	 * @since 1.0
	 */
	@Override
	protected List<AddressData> getData() throws NagesenException  {
		
		GetNichDataResult result = getGetNichData().get(getUrl(), getLasyModified(), getRange());
		setStatusCode(result.getStatusCode());
		if (getStatusCode() == GetNichData.STATUS_NOT_FOUND) {
			return null;
		}

		List<AddressData> ret = new ArrayList<AddressData>();
		if (getStatusCode() == GetNichData.STATUS_NOT_MODIFIED) {
			return ret;
		}
		setLasyModified(result.getLasyModified());
		setRange(range + result.getRange());

		for (NichData data : result.getNichDataList()) {
			addLastResNo();
			if (getStartResNo() > getLastResNo() || getEndResNo() < getLastResNo()) {
				continue;
			}

			// 名前欄でデータ生成
			AddressData data2 = createAddressData(data);
			if (data2 != null) {
				ret.add(data2);
			}
			
			// メッセージのアドレスで生成
			if (isIncludeMessageAddress()) {
				ret.addAll(createMessageAddressData(data));
			}
		}
		
		return ret;
	}

	protected AddressData createAddressData (NichData data) {
		
		AddressData data2 = new AddressData();
		if (getNameAddressPattern() == null) {
			return null;
		}
		Pattern pattern = Pattern.compile(getNameAddressPattern());
		Matcher match = pattern.matcher(data.getName());
		if (match.find()) {
			int[] index = getNameAddressIndex();
			data2.setAddress(match.group().substring(index[0], index[1]));
			data2.setCoin(getType());
		} else {
			return null;
		}
		
		data2.setId(data.getTimestamp().substring(29));
		data2.setMessage(data.getMessage());
		return data2;
	}

	protected List<AddressData> createMessageAddressData (NichData data) {
		List<AddressData> ret = new ArrayList<AddressData>();
		if (getMessageAddressPattern() == null) {
			return ret;
		}
		
		Pattern pattern = Pattern.compile(getMessageAddressPattern());
		String message = data.getMessage();
		Matcher match = pattern.matcher(message);
		while (match.find()) {
			AddressData data2 = new AddressData();
			int[] index = getMessageAddressIndex();
			data2.setAddress(match.group().substring(index[0], index[1]));
			data2.setCoin(getType());
			data2.setId(data.getTimestamp().substring(29));
			data2.setMessage(data.getMessage());
			ret.add(data2);
			message = message.substring(match.end());
			match = pattern.matcher(message);
		}

		return ret;
	}
	
	
	/* (非 Javadoc)
	 * @see org.curewallet.nagesen.AbstractExecuter#isContinue(org.curewallet.nagesen.AddressData)
	 * @since 1.0
	 */
	@Override
	protected boolean isContinue(AddressData data) throws NagesenException  {
		if (getStatusCode() == GetNichData.STATUS_NOT_FOUND) {
			return false;
		}
		if (getEndResNo() < getLastResNo()) {
			return false;
		}
		return true;
	}

	
	
	/**
	 * @return url
	 * @since 1.0
	 */
	public String getUrl() {
		return url;
	}

	/**
	 * @param url セットする url
	 * @since 1.0
	 */
	public void setUrl(String url) {
		this.url = url;
	}

	/**
	 * @return getNichData
	 * @since 1.0
	 */
	public GetNichData getGetNichData() {
		return getNichData;
	}

	/**
	 * @param getNichData セットする getNichData
	 * @since 1.0
	 */
	public void setGetNichData(GetNichData getNichData) {
		this.getNichData = getNichData;
	}

	/**
	 * @return lasyModified
	 * @since 1.0
	 */
	public String getLasyModified() {
		return lasyModified;
	}

	/**
	 * @param lasyModified セットする lasyModified
	 * @since 1.0
	 */
	public void setLasyModified(String lasyModified) {
		this.lasyModified = lasyModified;
	}

	/**
	 * @return range
	 * @since 1.0
	 */
	public long getRange() {
		return range;
	}

	/**
	 * @param range セットする range
	 * @since 1.0
	 */
	public void setRange(long range) {
		this.range = range;
	}

	/**
	 * @return statusCode
	 * @since 1.0
	 */
	public int getStatusCode() {
		return statusCode;
	}

	/**
	 * @param statusCode セットする statusCode
	 * @since 1.0
	 */
	public void setStatusCode(int statusCode) {
		this.statusCode = statusCode;
	}

	/**
	 * @return lastResNo
	 * @since 1.0
	 */
	public int getLastResNo() {
		return lastResNo;
	}

	/**
	 * @param lastResNo セットする lastResNo
	 * @since 1.0
	 */
	public void setLastResNo(int lastResNo) {
		this.lastResNo = lastResNo;
	}

	public void addLastResNo() {
		setLastResNo(getLastResNo() + 1);
	}

	/**
	 * @return startResNo
	 * @since 1.0
	 */
	public int getStartResNo() {
		return startResNo;
	}

	/**
	 * @param startResNo セットする startResNo
	 * @since 1.0
	 */
	public void setStartResNo(int startResNo) {
		this.startResNo = startResNo;
	}

	/**
	 * @return endResNo
	 * @since 1.0
	 */
	public int getEndResNo() {
		return endResNo;
	}

	/**
	 * @param endResNo セットする endResNo
	 * @since 1.0
	 */
	public void setEndResNo(int endResNo) {
		this.endResNo = endResNo;
	}

	/**
	 * @return includeMessageAddress
	 * @since 
	 */
	public boolean isIncludeMessageAddress() {
		return includeMessageAddress;
	}

	/**
	 * @param includeMessageAddress セットする includeMessageAddress
	 * @since 
	 */
	public void setIncludeMessageAddress(boolean includeMessageAddress) {
		this.includeMessageAddress = includeMessageAddress;
	}

	/**
	 * @return nameAddressPattern
	 * @since 
	 */
	public String getNameAddressPattern() {
		return nameAddressPattern;
	}

	/**
	 * @param nameAddressPattern セットする nameAddressPattern
	 * @since 
	 */
	public void setNameAddressPattern(String nameAddressPattern) {
		this.nameAddressPattern = nameAddressPattern;
	}

	/**
	 * @return messageAddressPattern
	 * @since 
	 */
	public String getMessageAddressPattern() {
		return messageAddressPattern;
	}

	/**
	 * @param messageAddressPattern セットする messageAddressPattern
	 * @since 
	 */
	public void setMessageAddressPattern(String messageAddressPattern) {
		this.messageAddressPattern = messageAddressPattern;
	}

	/**
	 * @return nameAddressIndex
	 * @since 
	 */
	public int[] getNameAddressIndex() {
		return nameAddressIndex;
	}

	/**
	 * @param nameAddressIndex セットする nameAddressIndex
	 * @since 
	 */
	public void setNameAddressIndex(int[] nameAddressIndex) {
		this.nameAddressIndex = nameAddressIndex;
	}

	/**
	 * @return messageAddressIndex
	 * @since 
	 */
	public int[] getMessageAddressIndex() {
		return messageAddressIndex;
	}

	/**
	 * @param messageAddressIndex セットする messageAddressIndex
	 * @since 
	 */
	public void setMessageAddressIndex(int[] messageAddressIndex) {
		this.messageAddressIndex = messageAddressIndex;
	}

	/**
	 * @return type
	 * @since 
	 */
	public String getType() {
		return type;
	}

	/**
	 * @param type セットする type
	 * @since 
	 */
	public void setType(String type) {
		this.type = type;
	}

	

	
}
