/*
 * Copyright 2000-2001,2004 The Apache Software Foundation.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.jetspeed.services.security;

import java.util.Iterator;
import java.util.HashMap;

// Junit imports
import junit.framework.Test;
import junit.framework.TestSuite;

import org.apache.turbine.services.TurbineServices;
import org.apache.turbine.util.TurbineConfig;
import org.apache.turbine.util.StringUtils;

// Jetspeed imports
import org.apache.jetspeed.test.JetspeedTestCase;
import org.apache.jetspeed.om.security.Permission;
import org.apache.jetspeed.om.security.JetspeedPermissionFactory;

import org.apache.jetspeed.services.security.JetspeedSecurityCache;

/**
 * Unit test for PermissionManagement interface
 * 
 * @author <a href="mailto:david@bluesunrise.com">David Sean Taylor</a>
 * @version $Id: TestPermissionManagement.java,v 1.1 2004/04/07 22:02:43 jford Exp $
 */

public class TestPermissionManagement extends JetspeedTestCase {    

    /**
     * Defines the testcase name for JUnit.
     *
     * @param name the testcase's name.
     */
    public TestPermissionManagement( String name ) {
        super( name );
    }
    
    /**
     * Start the tests.
     *
     * @param args the arguments. Not used
     */
    public static void main(String args[]) 
    {
        junit.awtui.TestRunner.main( new String[] { TestPermissionManagement.class.getName() } );
    }
 
    public void setup() 
    {
        //System.out.println("Setup: Testing Turbine Permission Management");         
    }

    /**
     * Creates the test suite.
     *
     * @return a test suite (<code>TestSuite</code>) that includes all methods
     *         starting with "test"
     */
    public static Test suite() 
    {
        // All methods starting with "test" will be executed in the test suite.
        return new TestSuite( TestPermissionManagement.class );
    }

    /**
     * Tests getPermissions method
     * @throws Exception
     */

    public void testGetPermissions() throws Exception 
    {
        PermissionManagement service = getService();
        Permission permission = null;
        HashMap map = new HashMap();
        JetspeedSecurityCache.loadRolePermissions();

        try
        {
            Iterator permissions = service.getPermissions();
            while (permissions.hasNext())
            {
                permission = (Permission)permissions.next();
                map.put(permission.getName(), permission);
                //System.out.println("permission = " + permission.getName());
                //System.out.println("id = " + permission.getId());
            }
            assertTrue(map.get("view") != null);
            assertTrue(map.get("customize") != null);
        }
        catch (Exception e)
        {
            fail(StringUtils.stackTrace(e));
        }

        System.out.println("Completed getPermissions Test OK ");

    }

    /**
     * Tests getPermissions method
     * @throws Exception
     */

    public void testGetPermissionsForUser() throws Exception 
    {
        PermissionManagement service = getService();
        Permission permission = null;
        HashMap map = new HashMap();
        JetspeedSecurityCache.loadRolePermissions();

        try
        {
            Iterator permissions = service.getPermissions("user");
            while (permissions.hasNext())
            {
                permission = (Permission)permissions.next();
                map.put(permission.getName(), permission);
            }
            assertTrue(map.get("view") != null);
            assertTrue(map.get("customize") != null);
            assertTrue(map.get("close") == null);

            map.clear();
            permissions = service.getPermissions("admin");            
            while (permissions.hasNext())
            {
                permission = (Permission)permissions.next();
                map.put(permission.getName(), permission);
            }
            assertTrue(map.get("view") != null);
            assertTrue(map.get("customize") != null);
            assertTrue(map.get("close") != null);

        }
        catch (Exception e)
        {
            fail(StringUtils.stackTrace(e));
        }

        System.out.println("Completed getPermissions Test OK ");

    }

    /**
     * Tests addPermission method 
     * @throws Exception
     */

    public void testAddPermission() throws Exception 
    {
        PermissionManagement service = getService();
        Permission permission = null;
        JetspeedSecurityCache.loadRolePermissions();

        try
        {
            permission = JetspeedPermissionFactory.getInstance();
            permission.setName("bogus");
            service.addPermission(permission);
            System.out.println("new permission id = " + permission.getId());
            assertTrue(permission.getId() != null);
        }
        catch(Exception e)
        {
            fail(StringUtils.stackTrace(e));
        }
        try
        {
            permission = JetspeedPermissionFactory.getInstance();
            permission.setName("bogus");
            service.addPermission(permission);
            fail("Should've thrown a dup key exception on permission");
        }
        catch(Exception e)
        {
            assertTrue(e instanceof PermissionException);           
        }

        System.out.println("Completed addPermission Test OK ");

    }

    /**
     * Tests getRemovePermission method 
     * @throws Exception
     */

    public void testRemovePermission() throws Exception 
    {
        PermissionManagement service = getService();
        Permission permission = null;
        JetspeedSecurityCache.loadRolePermissions();

        try
        {
            service.removePermission("bogus");
        }
        catch(Exception e)
        {
            fail(StringUtils.stackTrace(e));
        }
        try
        {
            service.removePermission("catchmeifyoucan");
            fail("Should've thrown a not found exception on permission");
        }
        catch(Exception e)
        {
            assertTrue(e instanceof PermissionException);
        }

        System.out.println("Completed addPermission Test OK ");

    }

    /**
     * Tests getPermission method
     * @throws Exception
     */

    public void testGetPermission() throws Exception 
    {
        PermissionManagement service = getService();
        JetspeedSecurityCache.loadRolePermissions();

        try
        {
            Permission permission = service.getPermission("view");
            System.out.println("*** permission id = " + permission.getId());
            assertTrue(permission.getName().equals("view"));
        }
        catch (Exception e)
        {
            fail(StringUtils.stackTrace(e));
        }

        System.out.println("Completed getPermission Test OK ");

    }

    /**
     * Tests savePermission method 
     * @throws Exception
     */

    public void testSavePermission() throws Exception 
    {
        PermissionManagement service = getService();
        JetspeedSecurityCache.loadRolePermissions();

        try
        {
            Permission permission = service.getPermission("customize");
            service.savePermission(permission);
        }
        catch(Exception e)
        {
            fail(StringUtils.stackTrace(e));
        }

        System.out.println("Completed savePermission Test OK ");

    }

    /**
     * Tests grantPermission method 
     * @throws Exception
     */
    public void testGrantPermission() throws Exception 
    {
        PermissionManagement service = getService();
        Permission permission = null;
        JetspeedSecurityCache.loadRolePermissions();

        try
        {
            service.grantPermission("user", "close");
        }
        catch(Exception e)
        {
            fail(StringUtils.stackTrace(e));
        }
        try
        {
            service.grantPermission("badrole", "close");
            fail("Should've thrown a bad role exception on grant");
        }
        catch(Exception e)
        {
            assertTrue(e instanceof PermissionException);           
        }
        try
        {
            service.grantPermission("user", "badpermission");
            fail("Should've thrown a bad permission exception on grant");
        }
        catch(Exception e)
        {
            assertTrue(e instanceof PermissionException);           
        }

        System.out.println("Completed grantPermission Test OK ");

    }

    /**
     * Tests revokePermission method 
     * @throws Exception
     */
    public void testRevokePermission() throws Exception 
    {
        PermissionManagement service = getService();
        Permission permission = null;
        JetspeedSecurityCache.loadRolePermissions();

        try
        {
            service.revokePermission("user", "close");
        }
        catch(Exception e)
        {
            fail(StringUtils.stackTrace(e));
        }
        try
        {
            service.revokePermission("badrole", "close");
            fail("Should've thrown a bad user exception on revoke");
        }
        catch(Exception e)
        {
            assertTrue(e instanceof PermissionException);           
        }

        System.out.println("Completed revokePermission Test OK ");

    }

    /**
     * Tests hasPermission method 
     * @throws Exception
     */
    public void testHasPermission() throws Exception 
    {
        PermissionManagement service = getService();
        Permission permission = null;
        JetspeedSecurityCache.loadRolePermissions();

        try
        {
            boolean has = service.hasPermission("user", "view");
            assertTrue(true == has);
        }
        catch(Exception e)
        {
            fail(StringUtils.stackTrace(e));
        }
        try
        {
            boolean has = service.hasPermission("user", "close");
            assertTrue(false == has);
        }
        catch(Exception e)
        {
            fail(StringUtils.stackTrace(e));
        }

        System.out.println("Completed hasPermission Test OK ");

    }

  /*
    Configuration object to run Turbine outside a servlet container
    ( uses turbine.properties )
    */
    private static TurbineConfig config = null;
    
    /**
    Sets up TurbineConfig using the system property:
    <pre>turbine.properties</pre>
    */
    static
    {
        try
        {
            config = new TurbineConfig( "webapp", "/WEB-INF/conf/TurbineResources.properties");
            config.init();
        }
        catch (Exception e)
        {
            fail(StringUtils.stackTrace(e));
        }
    }

    private static PermissionManagement getService()
    {
        return (PermissionManagement)TurbineServices
                .getInstance()
                .getService(PermissionManagement.SERVICE_NAME);
    }

}






