/*
 * Aipo is a groupware program developed by Aimluck,Inc.
 * Copyright (C) 2004-2008 Aimluck,Inc.
 * http://aipostyle.com/
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */
package com.aimluck.eip.exttimecard;

import java.math.BigDecimal;
import java.text.DecimalFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.jar.Attributes;

import org.apache.cayenne.access.DataContext;
import org.apache.cayenne.exp.Expression;
import org.apache.cayenne.exp.ExpressionFactory;
import org.apache.cayenne.query.SelectQuery;
import org.apache.jetspeed.portal.portlets.VelocityPortlet;
import org.apache.jetspeed.services.logging.JetspeedLogFactoryService;
import org.apache.jetspeed.services.logging.JetspeedLogger;
import org.apache.turbine.services.TurbineServices;
import org.apache.turbine.util.RunData;
import org.apache.velocity.context.Context;

import com.aimluck.commons.field.ALDateTimeField;
import com.aimluck.commons.field.ALNumberField;
import com.aimluck.commons.field.ALStringField;
import com.aimluck.eip.cayenne.om.portlet.EipTExtTimecard;
import com.aimluck.eip.cayenne.om.portlet.EipTExtTimecardSystem;
import com.aimluck.eip.common.ALAbstractSelectData;
import com.aimluck.eip.common.ALDBErrorException;
import com.aimluck.eip.common.ALData;
import com.aimluck.eip.common.ALEipConstants;
import com.aimluck.eip.common.ALEipManager;
import com.aimluck.eip.common.ALEipUser;
import com.aimluck.eip.common.ALPageNotFoundException;
import com.aimluck.eip.common.ALPermissionException;
import com.aimluck.eip.exttimecard.util.ExtTimecardUtils;
import com.aimluck.eip.modules.actions.common.ALAction;
import com.aimluck.eip.orm.DatabaseOrmService;
import com.aimluck.eip.services.accessctl.ALAccessControlConstants;
import com.aimluck.eip.services.accessctl.ALAccessControlFactoryService;
import com.aimluck.eip.services.accessctl.ALAccessControlHandler;
import com.aimluck.eip.util.ALEipUtils;

/**
 * タイムカード集計の一覧を処理するクラスです。 <br />
 * 
 * 
 */

public class ExtTimecardSummaryListSelectData extends ALAbstractSelectData
    implements ALData {

  /** logger */
  private static final JetspeedLogger logger = JetspeedLogFactoryService
      .getLogger(ExtTimecardSummaryListSelectData.class.getName());

  /** <code>target_group_name</code> 表示対象の部署名 */
  private String target_group_name;

  /** <code>target_user_id</code> 表示対象のユーザ ID */
  private String target_user_id;

  /** <code>myGroupList</code> グループリスト（My グループと部署） */
  private List myGroupList = null;

  /** <code>userList</code> 表示切り替え用のユーザリスト */
  private List userList = null;

  /** 一覧データ */
  private ArrayList list;

  /** <code>userid</code> ユーザーID */
  private String userid;

  /** <code>TARGET_GROUP_NAME</code> グループによる表示切り替え用変数の識別子 */
  private final String TARGET_GROUP_NAME = "target_group_name";

  /** <code>TARGET_USER_ID</code> ユーザによる表示切り替え用変数の識別子 */
  private final String TARGET_USER_ID = "target_user_id";

  private String nowtime;

  /** ユーザーマップ */
  private Map usermap;

  /** 日付マップ */
  private Map datemap;

  private ALNumberField shugyoNissu;

  private ALNumberField shugyoJikan;

  private ALNumberField shugyoJikannaiNissu;

  private ALNumberField shugyoJikannaiJikan;

  private ALNumberField shugyoJikannai1Nissu;

  private ALNumberField shugyoJikannai1Jikan;

  private ALNumberField shugyoJikannai2Nissu;

  private ALNumberField shugyoJikannai2Jikan;

  private ALNumberField zangyoNissu;

  private ALNumberField zangyoJikan;

  private ALNumberField zangyoJikannaiNissu;

  private ALNumberField zangyoJikannaiJikan;

  private ALNumberField zangyoJikannai1Nissu;

  private ALNumberField zangyoJikannai1Jikan;

  private ALNumberField zangyoJikannai2Nissu;

  private ALNumberField zangyoJikannai2Jikan;

  private ALNumberField chikoku;

  private ALNumberField sotai;

  /** アクセス権限の機能名 */
  private String aclPortletFeature = null;

  /** 閲覧権限の有無 */
  private boolean hasAclSummaryOther;

  /** 他ユーザーのxlsエクスポート権限 */
  private boolean hasAclXlsExport;

  /** <code>viewMonth</code> 現在の月 */
  private ALDateTimeField viewMonth;

  /** <code>prevMonth</code> 前の月 */
  private ALDateTimeField prevMonth;

  /** <code>nextMonth</code> 次の月 */
  private ALDateTimeField nextMonth;

  /** <code>currentMonth</code> 今月 */
  private ALDateTimeField currentMonth;

  /** <code>today</code> 今日 */
  private ALDateTimeField today;

  /** <code>viewStart</code> 表示開始日時 */
  private ALDateTimeField viewStart;

  /** <code>viewEnd</code> 表示終了日時 */
  private ALDateTimeField viewEnd;

  /** <code>viewEndCrt</code> 表示終了日時 (Criteria) */
  private ALDateTimeField viewEndCrt;

  /** <code>viewtype</code> 表示タイプ */
  private String viewtype;

  /** <code>viewTodo</code> ToDo 表示設定 */
  protected int viewTodo;

  /** ポートレット ID */
  private String portletId;

  private DataContext dataContext;

  /** <code>facilityList</code> 表示切り替え用の施設リスト */
  private List facilityList;

  /** 閲覧権限の有無 */
  private boolean hasAclviewOther;

  /** <code>hasAuthoritySelfInsert</code> アクセス権限 */
  private boolean hasAuthoritySelfInsert = false;

  /** <code>hasAuthorityFacilityInsert</code> アクセス権限 */
  private boolean hasAuthorityFacilityInsert = false;

  /** <code>target_user_id</code> 表示対象のユーザ ログイン名 */
  private String target_user_name;

  /**
   *
   */
  public void initField() {

    shugyoNissu = new ALNumberField(0);
    shugyoJikan = new ALNumberField(0);
    shugyoJikannaiNissu = new ALNumberField(0);
    shugyoJikannaiJikan = new ALNumberField(0);
    shugyoJikannai1Nissu = new ALNumberField(0);
    shugyoJikannai1Jikan = new ALNumberField(0);
    shugyoJikannai2Nissu = new ALNumberField(0);
    shugyoJikannai2Jikan = new ALNumberField(0);

    zangyoNissu = new ALNumberField(0);
    zangyoJikan = new ALNumberField(0);
    zangyoJikannaiNissu = new ALNumberField(0);
    zangyoJikannaiJikan = new ALNumberField(0);
    zangyoJikannai1Nissu = new ALNumberField(0);
    zangyoJikannai1Jikan = new ALNumberField(0);
    zangyoJikannai2Nissu = new ALNumberField(0);
    zangyoJikannai2Jikan = new ALNumberField(0);

    chikoku = new ALNumberField(0);
    sotai = new ALNumberField(0);

  }

  /*
   * (非 Javadoc)
   * 
   * @see
   * com.aimluck.eip.common.ALAbstractSelectData#init(com.aimluck.eip.modules
   * .actions.common.ALAction, org.apache.turbine.util.RunData,
   * org.apache.velocity.context.Context)
   */
  public void init(ALAction action, RunData rundata, Context context)
      throws ALPageNotFoundException, ALDBErrorException {
    // TODO 自動生成されたメソッド・スタブ
    super.init(action, rundata, context);

    this.initField();

    dataContext = DatabaseOrmService.getInstance().getDataContext();

    // POST/GET から yyyy-MM の形式で受け渡される。
    // 現在の月
    viewMonth = new ALDateTimeField("yyyy-MM");
    viewMonth.setNotNull(true);
    // 前の月
    prevMonth = new ALDateTimeField("yyyy-MM");
    // 次の月
    nextMonth = new ALDateTimeField("yyyy-MM");
    // 今月
    currentMonth = new ALDateTimeField("yyyy-MM");
    // 表示開始日時
    viewStart = new ALDateTimeField("yyyy-MM-dd");
    // 表示終了日時
    viewEnd = new ALDateTimeField("yyyy-MM-dd");
    // 表示終了日時 (Criteria)
    viewEndCrt = new ALDateTimeField("yyyy-MM-dd");
    // 今日
    today = new ALDateTimeField("yyyy-MM-dd");
    Calendar to = Calendar.getInstance();
    to.set(Calendar.HOUR_OF_DAY, 0);
    to.set(Calendar.MINUTE, 0);
    today.setValue(to.getTime());
    currentMonth.setValue(to.getTime());

    // 自ポートレットからのリクエストであれば、パラメータを展開しセッションに保存する。
    if (ALEipUtils.isMatch(rundata, context)) {
      // スケジュールの表示開始日時
      // e.g. 2004-3-14
      if (rundata.getParameters().containsKey("view_month")) {
        ALEipUtils.setTemp(rundata, context, "view_month", rundata
            .getParameters().getString("view_month"));
      }
    }

    // 現在の月
    String tmpViewMonth = ALEipUtils.getTemp(rundata, context, "view_month");
    if (tmpViewMonth == null || tmpViewMonth.equals("")) {
      Calendar cal = Calendar.getInstance();
      cal.set(Calendar.DATE, 1);
      cal.set(Calendar.HOUR_OF_DAY, 0);
      cal.set(Calendar.MINUTE, 0);
      viewMonth.setValue(cal.getTime());
    } else {
      viewMonth.setValue(tmpViewMonth);
      if (!viewMonth.validate(new ArrayList())) {
        ALEipUtils.removeTemp(rundata, context, "view_month");
        throw new ALPageNotFoundException();
      }
    }

    // 表示開始日時
    Calendar cal = Calendar.getInstance();
    Calendar tmpCal = Calendar.getInstance();
    cal.setTime(viewMonth.getValue());
    tmpCal.setTime(viewMonth.getValue());
    int dayofweek = cal.get(Calendar.DAY_OF_WEEK);
    cal.add(Calendar.DATE, -dayofweek + 1);
    viewStart.setValue(cal.getTime());

    Calendar cal4 = Calendar.getInstance();
    cal4.setTime(cal.getTime());
    Calendar tmpCal4 = Calendar.getInstance();
    tmpCal4.setTime(tmpCal.getTime());

    Calendar cal5 = Calendar.getInstance();
    cal5.setTime(cal.getTime());
    Calendar tmpCal5 = Calendar.getInstance();
    tmpCal5.setTime(tmpCal.getTime());

    // 表示終了日時
    viewEndCrt.setValue(cal.getTime());
    cal.add(Calendar.DATE, -1);
    viewEnd.setValue(cal.getTime());
    // 次の月、前の月
    Calendar cal2 = Calendar.getInstance();
    cal2.setTime(viewMonth.getValue());
    cal2.add(Calendar.MONTH, 1);
    nextMonth.setValue(cal2.getTime());
    cal2.add(Calendar.MONTH, -2);
    prevMonth.setValue(cal2.getTime());

    ALEipUtils.setTemp(rundata, context, "tmpStart", viewStart.toString()
        + "-00-00");
    ALEipUtils.setTemp(rundata, context, "tmpEnd", viewStart.toString()
        + "-00-00");

    // ログインユーザの ID を設定する．
    userid = Integer.toString(ALEipUtils.getUserId(rundata));

    // My グループの一覧を取得する．
    List myGroups = ALEipUtils.getMyGroups(rundata);
    myGroupList = new ArrayList();
    int length = myGroups.size();
    for (int i = 0; i < length; i++) {
      myGroupList.add(myGroups.get(i));
    }

    try {
      // スケジュールを表示するユーザ ID をセッションに設定する．
      String userFilter = ALEipUtils.getTemp(rundata, context, TARGET_USER_ID);
      if (userFilter == null || userFilter.equals("")) {
        VelocityPortlet portlet = ALEipUtils.getPortlet(rundata, context);
        userFilter = portlet.getPortletConfig().getInitParameter("p3a-user");
      }

      if (userFilter != null && (!userFilter.equals(""))) {
        int paramId = -1;
        try {
          paramId = Integer.parseInt(userFilter);
          if (paramId > 3) {
            ALEipUser user = ALEipUtils.getALEipUser(paramId);
            if (user != null) {
              // 指定したユーザが存在する場合，セッションに保存する．
              ALEipUtils.setTemp(rundata, context, TARGET_USER_ID, userFilter);
            } else {
              ALEipUtils.removeTemp(rundata, context, TARGET_USER_ID);
            }
          }
        } catch (NumberFormatException e) {
        }
      } else {
        ALEipUtils.setTemp(rundata, context, TARGET_USER_ID, userid);
      }
    } catch (Exception ex) {
      logger.error("Exception", ex);
    }

    // アクセス権
    if (target_user_id == null || "".equals(target_user_id)
        || userid.equals(target_user_id)) {
      aclPortletFeature = ALAccessControlConstants.POERTLET_FEATURE_TIMECARD_TIMECARD_SELF;
    } else {
      aclPortletFeature = ALAccessControlConstants.POERTLET_FEATURE_TIMECARD_TIMECARD_OTHER;
    }
    ALAccessControlFactoryService aclservice = (ALAccessControlFactoryService) ((TurbineServices) TurbineServices
        .getInstance()).getService(ALAccessControlFactoryService.SERVICE_NAME);
    ALAccessControlHandler aclhandler = aclservice.getAccessControlHandler();
    hasAclSummaryOther = aclhandler.hasAuthority(ALEipUtils.getUserId(rundata),
        ALAccessControlConstants.POERTLET_FEATURE_TIMECARD_TIMECARD_OTHER,
        ALAccessControlConstants.VALUE_ACL_LIST);
    hasAclXlsExport = aclhandler.hasAuthority(ALEipUtils.getUserId(rundata),
        ALAccessControlConstants.POERTLET_FEATURE_TIMECARD_TIMECARD_OTHER,
        ALAccessControlConstants.VALUE_ACL_EXPORT);

    if (!hasAclSummaryOther) {
      // 他ユーザーの閲覧権限がないときには、グループを未選択にする。
      target_group_name = "only";
      aclPortletFeature = ALAccessControlConstants.POERTLET_FEATURE_TIMECARD_TIMECARD_SELF;
      hasAclXlsExport = aclhandler.hasAuthority(ALEipUtils.getUserId(rundata),
          ALAccessControlConstants.POERTLET_FEATURE_TIMECARD_TIMECARD_SELF,
          ALAccessControlConstants.VALUE_ACL_EXPORT);
    }

    datemap = new LinkedHashMap();

    usermap = new LinkedHashMap();

    setupLists(rundata, context);
  }

  /**
   * 一覧表示します。
   * 
   * @param action
   * @param rundata
   * @param context
   * @return TRUE 成功 FASLE 失敗
   */
  public boolean doViewList(ALAction action, RunData rundata, Context context) {
    try {
      init(action, rundata, context);
      doCheckAclPermission(rundata, context,
          ALAccessControlConstants.VALUE_ACL_LIST);
      action.setMode(ALEipConstants.MODE_LIST);
      for (int i = 0; i < userList.size(); i++) {
        ALEipUser eipUser = (ALEipUser) userList.get(i);
        List aList = selectList(rundata, context, eipUser.getUserId()
            .getValueAsString());
        if (aList != null) {
          list = new ArrayList();
          Object obj = null;
          int size = aList.size();
          for (int j = 0; j < size; j++) {
            obj = getResultData(aList.get(j));
            if (obj != null) {
              list.add(obj);
            }
          }
        }
      }
      action.setResultData(this);
      action.putData(rundata, context);
      ALEipUtils.removeTemp(rundata, context, ALEipConstants.ENTITY_ID);
      return (list != null);
    } catch (ALPermissionException e) {
      ALEipUtils.redirectPermissionError(rundata);
      return false;
    } catch (ALPageNotFoundException e) {
      ALEipUtils.redirectPageNotFound(rundata);
      return false;
    } catch (ALDBErrorException e) {
      ALEipUtils.redirectDBError(rundata);
      return false;
    }

  }

  /*
   * (非 Javadoc)
   * 
   * @see
   * com.aimluck.eip.common.ALAbstractSelectData#selectList(org.apache.turbine
   * .util.RunData, org.apache.velocity.context.Context)
   */
  protected List selectList(RunData rundata, Context context,
      String target_user_id) throws ALPageNotFoundException, ALDBErrorException {
    try {
      // 指定グループや指定ユーザをセッションに設定する．
      setupLists(rundata, context);

      if (!"".equals(target_user_id)) {
        DataContext dataContext = DatabaseOrmService.getInstance()
            .getDataContext();

        SelectQuery query = getSelectQuery(rundata, context, target_user_id);
        buildSelectQueryForListView(query);
        query.addOrdering(EipTExtTimecard.PUNCH_DATE_PROPERTY, true);

        List list = dataContext.performQuery(query);
        return buildPaginatedList(list);
      } else {
        return null;
      }
    } catch (Exception ex) {
      logger.error("Exception", ex);
      return null;
    }
  }

  /*
   * (非 Javadoc)
   * 
   * @see
   * com.aimluck.eip.common.ALAbstractSelectData#selectList(org.apache.turbine
   * .util.RunData, org.apache.velocity.context.Context)
   */
  protected List selectList(RunData rundata, Context context)
      throws ALPageNotFoundException, ALDBErrorException {
    try {
      // 指定グループや指定ユーザをセッションに設定する．
      setupLists(rundata, context);

      if (!"".equals(target_user_id)) {
        DataContext dataContext = DatabaseOrmService.getInstance()
            .getDataContext();

        SelectQuery query = getSelectQuery(rundata, context, target_user_id);
        buildSelectQueryForListView(query);
        query.addOrdering(EipTExtTimecard.PUNCH_DATE_PROPERTY, true);

        List list = dataContext.performQuery(query);
        return buildPaginatedList(list);
      } else {
        return null;
      }
    } catch (Exception ex) {
      logger.error("Exception", ex);
      return null;
    }
  }

  /*
   * (非 Javadoc)
   * 
   * @see
   * com.aimluck.eip.common.ALAbstractSelectData#selectDetail(org.apache.turbine
   * .util.RunData, org.apache.velocity.context.Context)
   */
  protected Object selectDetail(RunData rundata, Context context)
      throws ALPageNotFoundException, ALDBErrorException {
    // TODO 自動生成されたメソッド・スタブ
    return null;
  }

  /*
   * (非 Javadoc)
   * 
   * @see
   * com.aimluck.eip.common.ALAbstractSelectData#getResultData(java.lang.Object)
   */

  /**
   * ResultData に値を格納して返します。（一覧データ） <BR>
   * 
   * @param obj
   * @return
   * @see com.aimluck.eip.common.ALAbstractSelectData#getListData(java.lang.Object)
   */
  protected Object getResultData(Object obj) throws ALPageNotFoundException,
      ALDBErrorException {
    try {
      EipTExtTimecard record = (EipTExtTimecard) obj;
      int user_id = record.getUserId();

      ExtTimecardResultData rd = new ExtTimecardResultData();
      rd.initField();
      rd.setPunchDate(record.getPunchDate());
      rd.setRefixFlag(record.getCreateDate(), record.getUpdateDate());
      rd.setClockInTime(record.getClockInTime());
      rd.setClockOutTime(record.getClockOutTime());
      for (int i = 1; i <= EipTExtTimecard.OUTGOING_COMEBACK_PER_DAY; i++) {
        rd.setOutgoingTime(record.getOutgoingTime(i), i);
        rd.setComebackTime(record.getComebackTime(i), i);
      }
      rd.setType(record.getType());

      ArrayList list;
      if (usermap.containsKey(user_id)) {
        list = (ArrayList) usermap.get(user_id);
        list.add(rd);
        usermap.put(user_id, list);
      } else {
        list = new ArrayList();
        list.add(rd);
        usermap.put(user_id, list);
      }
    } catch (Exception ex) {
      logger.error("Exception", ex);
      return null;
    }
    return null;
  }

  /*
   * (非 Javadoc)
   * 
   * @see
   * com.aimluck.eip.common.ALAbstractSelectData#getResultDataDetail(java.lang
   * .Object)
   */
  protected Object getResultDataDetail(Object obj)
      throws ALPageNotFoundException, ALDBErrorException {
    // TODO 自動生成されたメソッド・スタブ
    return null;
  }

  /*
   * (非 Javadoc)
   * 
   * @see com.aimluck.eip.common.ALAbstractSelectData#getColumnMap()
   */
  protected Attributes getColumnMap() {
    // TODO 自動生成されたメソッド・スタブ
    return null;
  }

  /**
   * ログイン中のユーザー情報を取得
   * 
   */

  private List getUserList(int userid) {
    List list = new ArrayList();
    ALEipUser user = new ALEipUser();
    try {
      user = ALEipUtils.getALEipUser(userid);
    } catch (NumberFormatException e1) {
      // TODO 自動生成された catch ブロック
      e1.printStackTrace();
    } catch (ALDBErrorException e1) {
      // TODO 自動生成された catch ブロック
      e1.printStackTrace();
    }
    list.add(user);
    return list;
  }

  /**
   * 指定グループや指定ユーザをセッションに設定する．
   * 
   * @param rundata
   * @param context
   * @throws ALDBErrorException
   */
  private void setupLists(RunData rundata, Context context) {
    target_group_name = getTargetGroupName(rundata, context);
    if ((target_group_name != null) && (!target_group_name.equals(""))
        && (!target_group_name.equals("all"))
        && (!target_group_name.equals("only"))) {
      userList = ALEipUtils.getUsers(target_group_name);
    } else if (target_group_name.equals("all")
        || target_group_name.equals("only")) {

      userList = getUserList(Integer.parseInt(userid));
    } else {
      userList = ALEipUtils.getUsers("LoginUser");
    }

    if (userList == null || userList.size() == 0) {
      target_user_id = "";
      ALEipUtils.removeTemp(rundata, context, TARGET_USER_ID);
      return;
    }

    target_user_id = getTargetUserId(rundata, context);
  }

  /**
   * 表示切り替えで指定したグループ ID を取得する．
   * 
   * @param rundata
   * @param context
   * @return
   */
  private String getTargetGroupName(RunData rundata, Context context) {
    String target_group_name = null;
    String idParam = null;
    if (ALEipUtils.isMatch(rundata, context)) {
      // 自ポートレットへのリクエストの場合に，グループ名を取得する．
      idParam = rundata.getParameters().getString(TARGET_GROUP_NAME);
    }
    target_group_name = ALEipUtils.getTemp(rundata, context, TARGET_GROUP_NAME);

    if (idParam == null && target_group_name == null) {
      ALEipUtils.setTemp(rundata, context, TARGET_GROUP_NAME, "only");
      target_group_name = "only";
    } else if (idParam != null) {
      ALEipUtils.setTemp(rundata, context, TARGET_GROUP_NAME, idParam);
      target_group_name = idParam;
    }
    return target_group_name;
  }

  /**
   * 表示切り替えで指定したユーザ ID を取得する．
   * 
   * @param rundata
   * @param context
   * @return
   */
  private String getTargetUserId(RunData rundata, Context context) {
    String target_user_id = null;
    String idParam = null;
    if (ALEipUtils.isMatch(rundata, context)) {
      // 自ポートレットへのリクエストの場合に，ユーザ ID を取得する．
      idParam = rundata.getParameters().getString(TARGET_USER_ID);
    }
    target_user_id = ALEipUtils.getTemp(rundata, context, TARGET_USER_ID);

    if (idParam == null && (target_user_id == null)) {
      // ログインユーザのスケジュールを表示するため，ログイン ID を設定する．
      ALEipUtils.setTemp(rundata, context, TARGET_USER_ID, userid);
      target_user_id = userid;
    } else if (idParam != null) {
      if (idParam.equals("none")) {
        // グループで表示を切り替えた場合，
        // ログインユーザもしくはユーザリストの一番初めのユーザを
        // 表示するため，ユーザ ID を設定する．
        ALEipUser eipUser = null;
        boolean found = false;
        int length = userList.size();
        for (int i = 0; i < length; i++) {
          eipUser = (ALEipUser) userList.get(i);
          String eipUserId = eipUser.getUserId().getValueAsString();
          if (userid.equals(eipUserId)) {
            ALEipUtils.setTemp(rundata, context, TARGET_USER_ID, userid);
            target_user_id = userid;
            found = true;
            break;
          }
        }
        if (!found) {
          eipUser = (ALEipUser) userList.get(0);
          String userId = eipUser.getUserId().getValueAsString();
          ALEipUtils.setTemp(rundata, context, TARGET_USER_ID, userId);
          target_user_id = userId;
        }
      } else {
        // ユーザで表示を切り替えた場合，指定したユーザの ID を設定する．
        ALEipUtils.setTemp(rundata, context, TARGET_USER_ID, idParam);
        target_user_id = idParam;
      }
    }
    return target_user_id;
  }

  /**
   * 検索条件を設定した SelectQuery を返します。 <BR>
   * 
   * @param rundata
   * @param context
   * @return
   */
  private SelectQuery getSelectQuery(RunData rundata, Context context,
      String target_user_id) {
    SelectQuery query = new SelectQuery(EipTExtTimecard.class);

    Expression exp1 = ExpressionFactory.matchExp(
        EipTExtTimecard.USER_ID_PROPERTY, new Integer(target_user_id));
    query.setQualifier(exp1);

    Calendar cal = Calendar.getInstance();
    cal.setTime(viewMonth.getValue());
    cal.set(Calendar.DAY_OF_MONTH, 1);
    cal.set(Calendar.HOUR_OF_DAY, 0);
    cal.set(Calendar.MINUTE, 0);
    cal.set(Calendar.SECOND, 0);
    cal.set(Calendar.MILLISECOND, 0);
    Expression exp11 = ExpressionFactory.greaterOrEqualExp(
        EipTExtTimecard.PUNCH_DATE_PROPERTY, cal.getTime());

    cal.add(Calendar.MONTH, +1);
    cal.add(Calendar.MILLISECOND, -1);
    Expression exp12 = ExpressionFactory.lessOrEqualExp(
        EipTExtTimecard.PUNCH_DATE_PROPERTY, cal.getTime());
    query.andQualifier(exp11.andExp(exp12));

    return buildSelectQueryForFilter(query, rundata, context);
  }

  /**
   * ユーザー毎のタイムカード一覧を取得する。
   * 
   * @return
   */
  public List getUserExtTimecards() {
    ArrayList list = new ArrayList();
    Set userset = usermap.keySet();

    /** 以下、ユーザーごとの処理 */
    for (Object user_id : userset) {
      ArrayList userlist = (ArrayList) usermap.get(user_id);
      ExtTimecardSummaryResultData summary_rd = new ExtTimecardSummaryResultData();
      int work_day = 0, overtime_day = 0, off_day = 0;
      /** 就業、残業、休出日数 */
      float work_hour = 0, overtime_hour = 0, off_hour = 0;
      /** 就業、残業、休出時間 */
      int late_coming_day = 0, early_leaving_day = 0, absent_day = 0;
      /** 遅刻、早退、欠勤 */
      int paid_holiday = 0, compensatory_holiday = 0;
      /** 有休、代休 */
      int other_day = 0, no_input = 0;
      /** その他、未入力 */
      summary_rd.initField();

      /** タイムカード設定を取得 */
      EipTExtTimecardSystem timecard_system = ExtTimecardUtils
          .getEipTExtTimecardSystemByUserId((Integer) user_id);

      /**
       * userlistにはユーザー日ごとのタイムカードのResultDataがリストで入っているため、
       * ListResultDataに代入して各日数・時間を計算させる。
       */

      for (Object obj : userlist) {
        ExtTimecardResultData rd = (ExtTimecardResultData) obj;
        ExtTimecardListResultData lrd = new ExtTimecardListResultData();
        lrd.setRd(rd);
        lrd.setTimecardSystem(timecard_system);
        String type = rd.getType().getValue();
        if (type.equals(EipTExtTimecard.TYPE_WORK)) {
          /** 出勤 */
          if (lrd.getWorkHour() != ExtTimecardListResultData.NO_DATA) {
            work_day++;
            work_hour += lrd.getWorkHourWithoutRestHour();
          }
          if (lrd.getOvertimeHour() != ExtTimecardListResultData.NO_DATA) {
            overtime_day++;
            overtime_hour += lrd.getOvertimeHourWithoutRestHour();
          }
          if (lrd.getOffHour() != ExtTimecardListResultData.NO_DATA) {
            off_day++;
            off_hour += lrd.getOffHour();
          }
          if (lrd.isLateComing()) {
            late_coming_day++;
          }
          if (lrd.isEarlyLeaving()) {
            early_leaving_day++;
          }
        } else if (type.equals(EipTExtTimecard.TYPE_ABSENT)) {
          /** 欠勤 */
          absent_day++;
        } else if (type.equals(EipTExtTimecard.TYPE_HOLIDAY)) {
          /** 有休 */
          paid_holiday++;
        } else if (type.equals(EipTExtTimecard.TYPE_COMPENSATORY)) {
          /** 代休 */
          compensatory_holiday++;
        } else if (type.equals(EipTExtTimecard.TYPE_ETC)) {
          /** その他 */
          other_day++;
        }
      }

      /** ユーザーごとの合計をSummaryResultDataに代入する */
      work_hour = ExtTimecardUtils.roundHour(work_hour);
      overtime_hour = ExtTimecardUtils.roundHour(overtime_hour);
      off_hour = ExtTimecardUtils.roundHour(off_hour);

      summary_rd.setWorkDayHour(work_day, work_hour);
      summary_rd.setOvertimeDayHour(overtime_day, overtime_hour);
      summary_rd.setOffDayHour(off_day, off_hour);
      summary_rd.setLateComingDay(late_coming_day);
      summary_rd.setEarlyLeavingDay(early_leaving_day);
      summary_rd.setAbsentDay(absent_day);
      summary_rd.setPaidHoliday(paid_holiday);
      summary_rd.setCompensatoryHoliday(compensatory_holiday);
      summary_rd.setOtherDay(other_day);
      summary_rd.setNoInput(no_input);

      list.add(summary_rd);
    }
    return list;
  }

  /**
   * グループ毎のタイムカード一覧を取得する。
   * 
   * @return
   */

  public List getGroupExtTimecards() {
    ArrayList list = new ArrayList();
    Set userset = usermap.keySet();

    /** 以下、ユーザーごとの処理 */
    for (int i = 0; i < userList.size(); i++) {
      ALEipUser eipUser = (ALEipUser) userList.get(i);
      int user_id = Integer.parseInt(eipUser.getUserId().getValueAsString());
      ArrayList userlist = (ArrayList) usermap.get(user_id);
      ExtTimecardSummaryResultData summary_rd = new ExtTimecardSummaryResultData();
      int work_day = 0, overtime_day = 0, off_day = 0;
      /** 就業、残業、休出日数 */
      float work_hour = 0, overtime_hour = 0, off_hour = 0;
      /** 就業、残業、休出時間 */
      int late_coming_day = 0, early_leaving_day = 0, absent_day = 0;
      /** 遅刻、早退、欠勤 */
      int paid_holiday = 0, compensatory_holiday = 0;
      /** 有休、代休 */
      int other_day = 0, no_input = 0;
      /** その他、未入力 */
      summary_rd.initField();

      /** タイムカード設定を取得 */
      EipTExtTimecardSystem timecard_system = ExtTimecardUtils
          .getEipTExtTimecardSystemByUserId((Integer) user_id);

      /**
       * userlistにはユーザー日ごとのタイムカードのResultDataがリストで入っているため、
       * ListResultDataに代入して各日数・時間を計算させる。
       */
      if (userlist != null) {
        for (Object obj : userlist) {
          ExtTimecardResultData rd = (ExtTimecardResultData) obj;
          ExtTimecardListResultData lrd = new ExtTimecardListResultData();
          lrd.initField();
          lrd.setDate(rd.getPunchDate().getValue());
          lrd.setRd(rd);
          lrd.setTimecardSystem(timecard_system);
          String type = rd.getType().getValue();
          if (type.equals(EipTExtTimecard.TYPE_WORK)) {
            /** 出勤 */
            if (lrd.getWorkHour() != ExtTimecardListResultData.NO_DATA) {
              work_day++;
              work_hour += lrd.getWorkHourWithoutRestHour();
            }
            if (lrd.getOvertimeHour() > 0) {
              overtime_day++;
              overtime_hour += lrd.getOvertimeHourWithoutRestHour();
            }
            if (lrd.getOffHour() != ExtTimecardListResultData.NO_DATA) {
              off_day++;
              off_hour += lrd.getOffHour();
            }
            if (lrd.isLateComing()) {
              late_coming_day++;
            }
            if (lrd.isEarlyLeaving()) {
              early_leaving_day++;
            }
          } else if (type.equals(EipTExtTimecard.TYPE_ABSENT)) {
            /** 欠勤 */
            absent_day++;
          } else if (type.equals(EipTExtTimecard.TYPE_HOLIDAY)) {
            /** 有休 */
            paid_holiday++;
          } else if (type.equals(EipTExtTimecard.TYPE_COMPENSATORY)) {
            /** 代休 */
            compensatory_holiday++;
          } else if (type.equals(EipTExtTimecard.TYPE_ETC)) {
            /** その他 */
            other_day++;
          }
        }
      }

      /** ユーザーごとの合計をSummaryResultDataに代入する */
      work_hour = ExtTimecardUtils.roundHour(work_hour);
      overtime_hour = ExtTimecardUtils.roundHour(overtime_hour);
      off_hour = ExtTimecardUtils.roundHour(off_hour);

      summary_rd.setUserName(eipUser.getAliasName().getValue());
      summary_rd.setSystemName(timecard_system.getSystemName());
      summary_rd.setWorkDayHour(work_day, work_hour);
      summary_rd.setOvertimeDayHour(overtime_day, overtime_hour);
      summary_rd.setOffDayHour(off_day, off_hour);
      summary_rd.setLateComingDay(late_coming_day);
      summary_rd.setEarlyLeavingDay(early_leaving_day);
      summary_rd.setAbsentDay(absent_day);
      summary_rd.setPaidHoliday(paid_holiday);
      summary_rd.setCompensatoryHoliday(compensatory_holiday);
      summary_rd.setOtherDay(other_day);
      summary_rd.setNoInput(no_input);

      list.add(summary_rd);
    }
    return list;
  }

  /**
   * @return
   */
  public List getDateListKeys() {
    try {
      List list = new ArrayList();
      Set set = datemap.keySet();
      Iterator iter = set.iterator();
      while (iter.hasNext()) {
        list.add(iter.next());
      }

      if (list.size() > 1) {
        for (int i = 0; i < list.size() - 1; i++) {
          ExtTimecardSummaryResultData listrd1 = (ExtTimecardSummaryResultData) datemap
              .get(list.get(i));
          ExtTimecardSummaryResultData listrd2 = (ExtTimecardSummaryResultData) datemap
              .get(list.get(i + 1));
          int listrd1_size = listrd1.getList().size();
          if (listrd1_size > 0) {
            ExtTimecardResultData listrd1_lastrd = (ExtTimecardResultData) listrd1
                .getList().get(listrd1_size - 1);

            ExtTimecardResultData listrd2_firstrd = (ExtTimecardResultData) listrd2
                .getList().get(0);
            if (ExtTimecardUtils.WORK_FLG_OFF.equals(listrd2_firstrd
                .getWorkFlag().getValue())
                && ExtTimecardUtils.WORK_FLG_ON.equals(listrd1_lastrd
                    .getWorkFlag().getValue())
                && !sameDay(listrd1_lastrd.getWorkDate().getValue(),
                    listrd2_firstrd.getWorkDate().getValue())) {

              Date d = listrd2_firstrd.getWorkDate().getValue();
              Calendar cal = Calendar.getInstance();
              cal.setTime(d);
              cal.set(Calendar.HOUR_OF_DAY, 0);
              cal.set(Calendar.MINUTE, 0);

              ExtTimecardResultData dummyrd = new ExtTimecardResultData();
              dummyrd.initField();
              dummyrd.setWorkFlag(ExtTimecardUtils.WORK_FLG_DUMMY);
              dummyrd.setWorkDate(cal.getTime());

              listrd1.addExtTimecardResultData(dummyrd);
            }
          }
        }
      }
      return list;
    } catch (Exception e) {
      e.printStackTrace();
      return null;
    }
  }

  /**
   * @param date_str
   * @return
   */
  public ExtTimecardSummaryResultData getDateListValue(String date_str) {
    return (ExtTimecardSummaryResultData) datemap.get(date_str);
  }

  /**
   * 指定した2つの日付を比較する．
   * 
   * @param date1
   * @param date2
   * @param checkTime
   *          時間まで比較する場合，true．
   * @return 等しい場合，0. date1>date2の場合, 1. date1 <date2の場合, 2.
   */
  private boolean sameDay(Date date1, Date date2) {
    Calendar cal1 = Calendar.getInstance();
    Calendar cal2 = Calendar.getInstance();
    cal1.setTime(date1);
    cal2.setTime(date2);

    int date1Year = cal1.get(Calendar.YEAR);
    int date1Month = cal1.get(Calendar.MONTH) + 1;
    int date1Day = cal1.get(Calendar.DATE);
    int date2Year = cal2.get(Calendar.YEAR);
    int date2Month = cal2.get(Calendar.MONTH) + 1;
    int date2Day = cal2.get(Calendar.DATE);

    if (date1Year == date2Year && date1Month == date2Month
        && date1Day == date2Day) {
      return true;
    }
    return false;
  }

  /**
   *
   *
   */
  public void calc() {
    if (datemap != null) {
      int shugyoNissu_temp = 0;
      int shugyoJikan_temp = 0;
      int shugyoJikannaiNissu_temp = 0;
      int shugyoJikannaiJikan_temp = 0;
      int shugyoJikannai1Nissu_temp = 0;
      int shugyoJikannai1Jikan_temp = 0;
      int shugyoJikannai2Nissu_temp = 0;
      int shugyoJikannai2Jikan_temp = 0;

      int zangyoNissu_temp = 0;
      int zangyoJikan_temp = 0;
      int zangyoJikannaiNissu_temp = 0;
      int zangyoJikannaiJikan_temp = 0;
      int zangyoJikannai1Nissu_temp = 0;
      int zangyoJikannai1Jikan_temp = 0;
      int zangyoJikannai2Nissu_temp = 0;
      int zangyoJikannai2Jikan_temp = 0;

      int chikoku_temp = 0;
      int sotai_temp = 0;

      for (Object element : datemap.values()) {
        ExtTimecardSummaryResultData rd = (ExtTimecardSummaryResultData) element;
        rd.calc();

        //
        if (rd.getShugyo().getValue() > 0) {
          shugyoNissu_temp++;
          shugyoJikan_temp += rd.getShugyo().getValue();
        }
        if (rd.getJikannai().getValue() > 0) {
          shugyoJikannaiNissu_temp++;
          shugyoJikannaiJikan_temp += rd.getJikannai().getValue();
        }
        if (rd.getJikannai1().getValue() > 0) {
          shugyoJikannai1Nissu_temp++;
          shugyoJikannai1Jikan_temp += rd.getJikannai1().getValue();
        }
        if (rd.getJikannai2().getValue() > 0) {
          shugyoJikannai2Nissu_temp++;
          shugyoJikannai2Jikan_temp += rd.getJikannai2().getValue();
        }

        if (rd.getZangyo().getValue() > 0) {
          zangyoNissu_temp++;
          zangyoJikan_temp += rd.getZangyo().getValue();
        }
        if (rd.getZangyo1().getValue() > 0) {
          zangyoJikannai1Nissu_temp++;
          zangyoJikannai1Jikan_temp += rd.getZangyo1().getValue();
        }
        if (rd.getZangyo2().getValue() > 0) {
          zangyoJikannai2Nissu_temp++;
          zangyoJikannai2Jikan_temp += rd.getZangyo2().getValue();
        }

        if ("○".equals(rd.getChikoku().getValue())) {
          chikoku_temp++;
        }
        if ("○".equals(rd.getSotai().getValue())) {
          sotai_temp++;
        }
      }

      shugyoNissu.setValue(shugyoNissu_temp);
      shugyoJikan.setValue(shugyoJikan_temp);
      shugyoJikannaiNissu.setValue(shugyoJikannaiNissu_temp);
      shugyoJikannaiJikan.setValue(shugyoJikannaiJikan_temp);
      shugyoJikannai1Nissu.setValue(shugyoJikannai1Nissu_temp);
      shugyoJikannai1Jikan.setValue(shugyoJikannai1Jikan_temp);
      shugyoJikannai2Nissu.setValue(shugyoJikannai2Nissu_temp);
      shugyoJikannai2Jikan.setValue(shugyoJikannai2Jikan_temp);

      zangyoNissu.setValue(zangyoNissu_temp);
      zangyoJikan.setValue(zangyoJikan_temp);
      zangyoJikannaiNissu.setValue(zangyoJikannaiNissu_temp);
      zangyoJikannaiJikan.setValue(zangyoJikannaiJikan_temp);
      zangyoJikannai1Nissu.setValue(zangyoJikannai1Nissu_temp);
      zangyoJikannai1Jikan.setValue(zangyoJikannai1Jikan_temp);
      zangyoJikannai2Nissu.setValue(zangyoJikannai2Nissu_temp);
      zangyoJikannai2Jikan.setValue(zangyoJikannai2Jikan_temp);

      chikoku.setValue(chikoku_temp);
      sotai.setValue(sotai_temp);
    }
  }

  /**
   * userListを設定する．
   * 
   */
  public void setuserList(String target_group_name) {
    this.target_group_name = target_group_name;
    if ((this.target_group_name != null)
        && (!this.target_group_name.equals(""))
        && (!this.target_group_name.equals("all"))
        && (!this.target_group_name.equals("only"))) {
      userList = ALEipUtils.getUsers(target_group_name);
    } else if (this.target_group_name.equals("all")
        || this.target_group_name.equals("only")) {
      userList = getUserList(Integer.parseInt(userid));
    } else {
      userList = ALEipUtils.getUsers("LoginUser");
    }
  }

  /**
   * 表示切り替え時に指定するグループ名
   * 
   * @return
   */
  public String getTargetGroupName() {
    return target_group_name;
  }

  /**
   * 表示切り替え時に指定するユーザ ID
   * 
   * @return
   */
  public String getTargetUserId() {
    return target_user_id;
  }

  /**
   * 指定グループに属するユーザの一覧を取得する．
   * 
   * @param groupname
   * @return
   */
  public List getUsers() {
    if (hasAclSummaryOther) {
      return userList;
    } else {
      try {
        List users = new ArrayList();
        users.add(ALEipUtils.getALEipUser(Integer.parseInt(userid)));
        return users;
      } catch (Exception e) {
        return null;
      }
    }
  }

  /**
   * 部署の一覧を取得する．
   * 
   * @return
   */
  public Map getPostMap() {
    if (hasAclSummaryOther) {
      return ALEipManager.getInstance().getPostMap();
    } else {
      return null;
    }
  }

  /**
   * My グループの一覧を取得する．
   * 
   * @return
   */
  public List getMyGroupList() {
    if (hasAclSummaryOther) {
      return myGroupList;
    } else {
      return null;
    }
  }

  /**
   * ログインユーザの ID を取得する．
   * 
   * @return
   */
  public String getUserId() {
    return userid;
  }

  /**
   * 
   * @param id
   * @return
   */
  public boolean isMatch(int id1, long id2) {
    return id1 == (int) id2;
  }

  /**
   * @return
   */

  public String getNowTime() {
    return nowtime;
  }

  public ALNumberField getShugyoNissu() {
    return shugyoNissu;
  }

  public ALNumberField getShugyoJikan() {
    return shugyoJikan;
  }

  public ALStringField getShugyoJikanStr() {
    return new ALStringField(minuteToHour(shugyoJikan.getValue()));
  }

  public ALNumberField getShugyoJikannaiNissu() {
    return shugyoJikannaiNissu;
  }

  public ALNumberField getShugyoJikannaiJikan() {
    return shugyoJikannaiJikan;
  }

  public ALStringField getShugyoJikannaiJikanStr() {
    return new ALStringField(minuteToHour(shugyoJikannaiJikan.getValue()));
  }

  public ALNumberField getShugyoJikannai1Nissu() {
    return shugyoJikannai1Nissu;
  }

  public ALNumberField getShugyoJikannai1Jikan() {
    return shugyoJikannai1Jikan;
  }

  public ALStringField getShugyoJikannai1JikanStr() {
    return new ALStringField(minuteToHour(shugyoJikannai1Jikan.getValue()));
  }

  public ALNumberField getShugyoJikannai2Nissu() {
    return shugyoJikannai2Nissu;
  }

  public ALNumberField getShugyoJikannai2Jikan() {
    return shugyoJikannai2Jikan;
  }

  public ALStringField getShugyoJikannai2JikanStr() {
    return new ALStringField(minuteToHour(shugyoJikannai2Jikan.getValue()));
  }

  public ALNumberField getZangyoNissu() {
    return zangyoNissu;
  }

  public ALNumberField getZangyoJikan() {
    return zangyoJikan;
  }

  public ALStringField getZangyoJikanStr() {
    return new ALStringField(minuteToHour(zangyoJikan.getValue()));
  }

  public ALNumberField getZangyoJikannaiNissu() {
    return zangyoJikannaiNissu;
  }

  public ALNumberField getZangyoJikannaiJikan() {
    return zangyoJikannaiJikan;
  }

  public ALStringField getZangyoJikannaiJikanStr() {
    return new ALStringField(minuteToHour(zangyoJikannaiJikan.getValue()));
  }

  public ALNumberField getZangyoJikannai1Nissu() {
    return zangyoJikannai1Nissu;
  }

  public ALNumberField getZangyoJikannai1Jikan() {
    return zangyoJikannai1Jikan;
  }

  public ALStringField getZangyoJikannai1JikanStr() {
    return new ALStringField(minuteToHour(zangyoJikannai1Jikan.getValue()));
  }

  public ALNumberField getZangyoJikannai2Nissu() {
    return zangyoJikannai2Nissu;
  }

  public ALNumberField getZangyoJikannai2Jikan() {
    return zangyoJikannai2Jikan;
  }

  public ALStringField getZangyoJikannai2JikanStr() {
    return new ALStringField(minuteToHour(zangyoJikannai2Jikan.getValue()));
  }

  public ALNumberField getChikoku() {
    return chikoku;
  }

  public ALNumberField getSotai() {
    return sotai;
  }

  /**
   * 
   * @param minute
   * @return
   */
  private String minuteToHour(long minute) {
    BigDecimal decimal = new BigDecimal(minute / 60.0);
    DecimalFormat dformat = new DecimalFormat("##.#");
    String str = dformat.format(decimal.setScale(1, BigDecimal.ROUND_FLOOR)
        .doubleValue());
    return str;
  }

  /**
   * アクセス権限チェック用メソッド。<br />
   * アクセス権限の機能名を返します。
   * 
   * @return
   */
  public String getAclPortletFeature() {
    return aclPortletFeature;
  }

  /**
   * 表示開始日時を取得します。
   * 
   * @return
   */
  public ALDateTimeField getViewStart() {
    return viewStart;
  }

  /**
   * 表示終了日時を取得します。
   * 
   * @return
   */
  public ALDateTimeField getViewEnd() {
    return viewEnd;
  }

  /**
   * 表示終了日時 (Criteria) を取得します。
   * 
   * @return
   */
  public ALDateTimeField getViewEndCrt() {
    return viewEndCrt;
  }

  /**
   * 前の月を取得します。
   * 
   * @return
   */
  public ALDateTimeField getPrevMonth() {
    return prevMonth;
  }

  /**
   * 次の月を取得します。
   * 
   * @return
   */
  public ALDateTimeField getNextMonth() {
    return nextMonth;
  }

  /**
   * 今月を取得します。
   * 
   * @return
   */
  public ALDateTimeField getCurrentMonth() {
    return currentMonth;
  }

  /**
   * 現在の月を取得します。
   * 
   * @return
   */
  public ALDateTimeField getViewMonth() {
    return viewMonth;
  }

  /**
   * 今日を取得します。
   * 
   * @return
   */
  public ALDateTimeField getToday() {
    return today;
  }

  public boolean hasAclXlsExport() {
    return hasAclXlsExport;
  }

}
