/*
 * Aipo is a groupware program developed by Aimluck,Inc.
 * Copyright (C) 2004-2008 Aimluck,Inc.
 * http://aipostyle.com/
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */
package com.aimluck.eip.exttimecard;

import java.math.BigDecimal;
import java.text.DecimalFormat;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.List;

import org.apache.cayenne.access.DataContext;
import org.apache.cayenne.query.SelectQuery;

import com.aimluck.commons.field.ALDateField;
import com.aimluck.commons.field.ALNumberField;
import com.aimluck.commons.field.ALStringField;
import com.aimluck.eip.cayenne.om.portlet.EipTTimecardSettings;
import com.aimluck.eip.common.ALData;
import com.aimluck.eip.exttimecard.util.ExtTimecardUtils;
import com.aimluck.eip.orm.DatabaseOrmService;

/**
 * 一日分のタイムカード(出勤・退勤の履歴)を保持する。<br>
 * 一日ごとの勤務時間・残業時間などを計算し、その結果を保持する。
 *
 *
 */
public class ExtTimecardSummaryResultData implements ALData {

  private ALDateField date = null;

  private List<ExtTimecardResultData> list = null;

  /** 就業日数 */
  private ALNumberField work_day = null;

  /** 就業時間 */
  private ALNumberField work_hour = null;

  /** 残業日数 */
  private ALNumberField overtime_day = null;

  /** 残業時間 */
  private ALNumberField overtime_hour = null;

  /** 休出日数 */
  private ALNumberField off_day = null;

  /** 休出時間 */
  private ALNumberField off_hour = null;

  /** 遅刻日数 */
  private ALNumberField late_coming_day = null;

  /** 早退日数 */
  private ALNumberField early_leaving_day = null;

  /** 欠勤日数 */
  private ALNumberField absent_day = null;

  /** 有休日数 */
  private ALNumberField paid_holiday = null;

  /** 代休日数 */
  private ALNumberField compensatory_holiday = null;

  /** その他 */
  private ALNumberField other_day = null;

  /** 未入力 */
  private ALNumberField no_input = null;

  private ALNumberField shugyo = null;

  private ALNumberField jikannai = null;

  private ALNumberField jikannai1 = null;

  private ALNumberField jikannai2 = null;

  private ALNumberField zangyo = null;

  private ALNumberField zangyo1 = null;

  private ALNumberField zangyo2 = null;

  private ALNumberField kyushutsu = null;

  private ALStringField chikoku = null;

  private ALStringField sotai = null;

  private DataContext dataContext;

  private String user_name = null;

  private String systemName = null;

  /**
   *
   * @see com.aimluck.eip.common.ALData#initField()
   */
  public void initField() {
    work_day = new ALNumberField(0);
    work_hour = new ALNumberField(0);

    overtime_day = new ALNumberField(0);
    overtime_hour = new ALNumberField(0);

    off_day = new ALNumberField(0);
    off_hour = new ALNumberField(0);

    late_coming_day = new ALNumberField(0);
    early_leaving_day = new ALNumberField(0);
    absent_day = new ALNumberField(0);
    paid_holiday = new ALNumberField(0);
    compensatory_holiday = new ALNumberField(0);
    other_day = new ALNumberField(0);
    no_input = new ALNumberField(0);

    date = new ALDateField();
    date.setValue(new Date());
    list = new ArrayList<ExtTimecardResultData>();

    shugyo = new ALNumberField(0);

    jikannai = new ALNumberField(0);
    jikannai1 = new ALNumberField(0);
    jikannai2 = new ALNumberField(0);

    zangyo = new ALNumberField(0);
    zangyo1 = new ALNumberField(0);
    zangyo2 = new ALNumberField(0);

    kyushutsu = new ALNumberField(0);
    chikoku = new ALStringField();
    sotai = new ALStringField();

    dataContext = DatabaseOrmService.getInstance().getDataContext();
  }

  /**
   * 出勤日数、および時間を設定します。
   *
   * @param days
   * @param hour
   */
  public void setWorkDayHour(int days, float hour) {
    work_day.setValue(days);
    work_hour.setValue(String.valueOf(hour));
  }

  /**
   * 残業日数、および時間を設定します。
   *
   * @param days
   * @param hour
   */
  public void setOvertimeDayHour(int days, float hour) {
    overtime_day.setValue(days);
    overtime_hour.setValue(String.valueOf(hour));
  }

  /**
   * 休出日数、および時間を設定します。
   *
   * @param days
   * @param hour
   */
  public void setOffDayHour(int days, float hour) {
    off_day.setValue(days);
    off_hour.setValue(String.valueOf(hour));
  }

  /**
   * 遅刻日数を設定します。
   *
   * @param days
   */
  public void setLateComingDay(int days) {
    late_coming_day.setValue(days);
  }

  /**
   * 早退日数を設定します。
   *
   * @param days
   */
  public void setEarlyLeavingDay(int days) {
    early_leaving_day.setValue(days);
  }

  /**
   * 欠勤日数を設定します。
   *
   * @param days
   */
  public void setAbsentDay(int days) {
    absent_day.setValue(days);
  }

  /**
   * 有休日数を設定します。
   *
   * @param days
   */
  public void setPaidHoliday(int days) {
    paid_holiday.setValue(days);
  }

  /**
   * 代休日数を設定します。
   *
   * @param days
   */
  public void setCompensatoryHoliday(int days) {
    compensatory_holiday.setValue(days);
  }

  /**
   * その他日数を設定します。
   *
   * @param days
   */
  public void setOtherDay(int days) {
    other_day.setValue(days);
  }

  /**
   * 未入力日数を設定します。
   *
   * @param days
   */
  public void setNoInput(int days) {
    no_input.setValue(days);
  }

  /**
   * 就業日数を取得します。
   *
   * @return
   */
  public ALNumberField getWorkDay() {
    return work_day;
  }

  /**
   * 就業時間を取得します。
   *
   * @return
   */
  public ALNumberField getWorkHour() {
    return work_hour;
  }

  /**
   * 残業日数を取得します。
   *
   * @return
   */
  public ALNumberField getOvertimeDay() {
    return overtime_day;
  }

  /**
   * 残業時間を取得します。
   *
   * @return
   */
  public ALNumberField getOvertimeHour() {
    return overtime_hour;
  }

  /**
   * 休出日数を取得します。
   *
   * @return
   */
  public ALNumberField getOffDay() {
    return off_day;
  }

  /**
   * 休出時間を取得します。
   *
   * @return
   */
  public ALNumberField getOffHour() {
    return off_hour;
  }

  /**
   * 遅刻日数を取得します。
   *
   * @return
   */
  public ALNumberField getLateComingDay() {
    return late_coming_day;
  }

  /**
   * 早退日数を取得します。
   *
   * @return
   */
  public ALNumberField getEarlyLeavingDay() {
    return early_leaving_day;
  }

  /**
   * 欠勤日数を取得します。
   *
   * @return
   */
  public ALNumberField getAbsentDay() {
    return absent_day;
  }

  /**
   * 有休日数を取得します。
   *
   * @return
   */
  public ALNumberField getPaidHoliday() {
    return paid_holiday;
  }

  /**
   * 代休日数を取得します。
   *
   * @return
   */
  public ALNumberField getCompensatoryHoliday() {
    return compensatory_holiday;
  }

  /**
   * その他日数を取得します。
   *
   * @return
   */
  public ALNumberField getOtherDay() {
    return other_day;
  }

  /**
   * 未入力数を取得します。
   *
   * @return
   */
  public ALNumberField getNoInput() {
    return no_input;
  }

  // --------------------------------------------------------------------------------------
  /**
   *
   * @param date
   */
  public void setDate(Date date) {
    this.date.setValue(date);
  }

  /**
   * @return date
   */
  public ALDateField getDate() {
    return date;
  }

  /**
   *
   * @return
   */
  public ALNumberField getKyushutsu() {
    return kyushutsu;
  }

  /**
   *
   * @param kyushutsu
   */
  public void setKyushutsu(ALNumberField kyushutsu) {
    this.kyushutsu = kyushutsu;
  }

  /**
   *
   * @return
   */
  public ALNumberField getShugyo() {
    return shugyo;
  }

  /**
   *
   * @return
   */
  public ALStringField getShugyoStr() {
    return new ALStringField(minuteToHour(shugyo.getValue()));
  }

  /**
   *
   * @param shugyo
   */
  public void setShugyo(ALNumberField shugyo) {
    this.shugyo = shugyo;
  }

  /**
   *
   * @return
   */
  public ALNumberField getJikannai() {
    return jikannai;
  }

  /**
   *
   * @return
   */
  public ALStringField getJikannaiStr() {
    return new ALStringField(minuteToHour(jikannai.getValue()));
  }

  /**
   *
   * @return
   */
  public ALNumberField getJikannai1() {
    return jikannai1;
  }

  /**
   *
   * @return
   */
  public ALStringField getJikannai1Str() {
    return new ALStringField(minuteToHour(jikannai1.getValue()));
  }

  /**
   *
   * @return
   */
  public ALNumberField getJikannai2() {
    return jikannai2;
  }

  /**
   *
   * @return
   */
  public ALStringField getJikannai2Str() {
    return new ALStringField(minuteToHour(jikannai2.getValue()));
  }

  /**
   *
   * @return
   */
  public ALNumberField getZangyo() {
    return zangyo;
  }

  /**
   *
   * @return
   */
  public ALStringField getZangyoStr() {
    return new ALStringField(minuteToHour(zangyo.getValue()));
  }

  /**
   *
   * @param zangyo
   */
  public void setZangyo(ALNumberField zangyo) {
    this.zangyo = zangyo;
  }

  /**
   *
   * @return
   */
  public ALNumberField getZangyo1() {
    return zangyo1;
  }

  /**
   *
   * @return
   */
  public ALStringField getZangyo1Str() {
    return new ALStringField(minuteToHour(zangyo1.getValue()));
  }

  /**
   *
   * @return
   */
  public ALNumberField getZangyo2() {
    return zangyo2;
  }

  /**
   *
   * @return
   */
  public ALStringField getZangyo2Str() {
    return new ALStringField(minuteToHour(zangyo2.getValue()));
  }

  /**
   *
   * @return
   */
  public ALStringField getChikoku() {
    return chikoku;
  }

  /**
   *
   * @return
   */
  public ALStringField getSotai() {
    return sotai;
  }

  /**
   *
   * @param minute
   * @return
   */
  private String minuteToHour(long minute) {
    BigDecimal decimal = new BigDecimal(minute / 60.0);
    DecimalFormat dformat = new DecimalFormat("##.#");
    String str = dformat.format(decimal.setScale(1, BigDecimal.ROUND_FLOOR)
        .doubleValue());
    return str;
  }

  /**
   *
   * @return
   */
  public String getDateStr() {
    try {
      SimpleDateFormat sdf = new SimpleDateFormat("yyyy年MM月dd日（EE）");
      return sdf.format(date.getValue().getDate());
    } catch (Exception e) {
      return "";
    }
  }

  /**
   *
   * @return
   */
  public List getList() {
    return list;
  }

  public void setUserName(String user_name) {
      this.user_name = user_name;
  }

  public String getUserName() {
    return user_name;
  }

  public void setSystemName(String systemName) {
    this.systemName = systemName;
  }

  public String getSystemName() {
    return systemName;
  }



  /**
   *
   * @return
   */
  public List getViewList() {
    ArrayList viewlist = new ArrayList();

    ExtTimecardResultData rd = null;

    int size = list.size();
    for (int i = 0; i < size; i++) {
      rd = (ExtTimecardResultData) list.get(i);
      if (!ExtTimecardUtils.WORK_FLG_DUMMY.equals(rd.getWorkFlag().getValue())) {
        viewlist.add(rd);
      }
    }
    return viewlist;
  }

  /**
   *
   * @param rd
   */
  public void addExtTimecardResultData(ExtTimecardResultData rd) {
    list.add(rd);
  }

  /**
   * 就業時間、残業時間などを計算する。
   *
   */
  public void calc() {
    try {

      // 勤務時間設定データを取得
      EipTTimecardSettings settings = loadEipTTimecardSettings();

      // カレンダーオブジェクトを初期化
      Calendar cal = Calendar.getInstance();
      cal.set(Calendar.YEAR, Integer.parseInt(date.getYear()));
      cal.set(Calendar.MONTH, Integer.parseInt(date.getMonth()) - 1);
      cal.set(Calendar.DAY_OF_MONTH, Integer.parseInt(date.getDay()));
      cal.set(Calendar.HOUR_OF_DAY, 0);
      cal.set(Calendar.MINUTE, 0);
      cal.set(Calendar.SECOND, 0);
      cal.set(Calendar.MILLISECOND, 0);
      Date startDate = cal.getTime();
      Date endDate = null;

      cal.set(Calendar.HOUR_OF_DAY, settings.getStartHour());
      cal.set(Calendar.MINUTE, settings.getStartMinute());
      Date kinmuStartDate = cal.getTime();
      cal.set(Calendar.HOUR_OF_DAY, settings.getEndHour());
      cal.set(Calendar.MINUTE, settings.getEndMinute());
      Date kinmuEndDate = cal.getTime();

      ArrayList<ExtTimecardResultData> tempList = new ArrayList<ExtTimecardResultData>();

      boolean startflg = false;
      for (ExtTimecardResultData rd : this.list) {
        if (ExtTimecardUtils.WORK_FLG_ON.equals(rd.getWorkFlag().getValue())) {
          startDate = rd.getWorkDate().getValue();
          // 出勤データが連続するときは、退勤データを挿入
          if (startflg == true) {
            tempList.add(createExtTimecardResultData(startDate,
                ExtTimecardUtils.WORK_FLG_OFF));
          }

          tempList.add(rd);
          startflg = true;

        } else if (ExtTimecardUtils.WORK_FLG_OFF.equals(rd.getWorkFlag()
            .getValue())) {
          // 日の最初が出勤ではないときは、0:00のデータを追加する
          // 退勤データが連続するときは、出勤データを挿入
          if (startflg == false) {
            tempList.add(createExtTimecardResultData(startDate,
                ExtTimecardUtils.WORK_FLG_ON));
          }

          endDate = rd.getWorkDate().getValue();

          // 勤務時間内外の区切るを挿入
          if (kinmuStartDate.after(startDate) && kinmuStartDate.before(endDate)) {
            tempList.add(createExtTimecardResultData(kinmuStartDate,
                ExtTimecardUtils.WORK_FLG_OFF));
            tempList.add(createExtTimecardResultData(kinmuStartDate,
                ExtTimecardUtils.WORK_FLG_ON));
          }
          if (kinmuEndDate.after(startDate) && kinmuEndDate.before(endDate)) {
            tempList.add(createExtTimecardResultData(kinmuEndDate,
                ExtTimecardUtils.WORK_FLG_OFF));
            tempList.add(createExtTimecardResultData(kinmuEndDate,
                ExtTimecardUtils.WORK_FLG_ON));
          }

          tempList.add(rd);

          startDate = endDate;
          startflg = false;
        }
      }
      // 最後に退勤していない場合は、0:00退勤のデータを追加する。
      if (startflg) {

        cal.set(Calendar.DAY_OF_MONTH, cal.get(Calendar.DAY_OF_MONTH) + 1);
        cal.set(Calendar.HOUR_OF_DAY, 0);
        cal.set(Calendar.MINUTE, 0);
        endDate = cal.getTime();

        // 勤務時間内外の区別をつける
        if (kinmuStartDate.after(startDate) && kinmuStartDate.before(endDate)) {
          tempList.add(createExtTimecardResultData(kinmuStartDate,
              ExtTimecardUtils.WORK_FLG_OFF));
          tempList.add(createExtTimecardResultData(kinmuStartDate,
              ExtTimecardUtils.WORK_FLG_ON));
        }
        if (kinmuEndDate.after(startDate) && kinmuEndDate.before(endDate)) {
          tempList.add(createExtTimecardResultData(kinmuEndDate,
              ExtTimecardUtils.WORK_FLG_OFF));
          tempList.add(createExtTimecardResultData(kinmuEndDate,
              ExtTimecardUtils.WORK_FLG_ON));
        }

        ExtTimecardResultData rd = new ExtTimecardResultData();
        rd.initField();
        rd.setWorkDate(endDate);
        rd.setWorkFlag(ExtTimecardUtils.WORK_FLG_OFF);
        tempList.add(rd);
      }

      // 整列されたデータを順に見て、就業時間などを数える
      long shugyo_temp = 0;
      long jikannai_temp = 0;
      long zangyo_temp = 0;
      boolean iszangyo = false;
      for (ExtTimecardResultData rd : tempList) {
        if (ExtTimecardUtils.WORK_FLG_ON.equals(rd.getWorkFlag().getValue())) {
          startDate = rd.getWorkDate().getValue();
          if (kinmuStartDate.equals(startDate)
              || (kinmuStartDate.before(startDate) && kinmuEndDate
                  .after(startDate))) {
            iszangyo = false;
          } else {
            iszangyo = true;
          }
        } else if (ExtTimecardUtils.WORK_FLG_OFF.equals(rd.getWorkFlag()
            .getValue())) {
          endDate = rd.getWorkDate().getValue();
          long millisecond = endDate.getTime() - startDate.getTime();
          shugyo_temp += millisecond / 1000 / 60;
          if (iszangyo) {
            zangyo_temp += millisecond / 1000 / 60;
          } else {
            jikannai_temp += millisecond / 1000 / 60;
          }
        }
      }

      // 集計した結果を各変数にセットする。
      // 勤務時間内
      this.jikannai1.setValue(jikannai_temp);
      // if (settings != null && jikannai_temp >= settings.getWorktimeIn()) {
      if (jikannai_temp >= settings.getWorktimeIn()) {
        this.jikannai.setValue(jikannai_temp - settings.getResttimeIn());
        this.jikannai2.setValue(settings.getResttimeIn());
      } else {
        this.jikannai.setValue(jikannai_temp);
        this.jikannai2.setValue(0);
      }

      // 勤務時間外
      long nnn = zangyo_temp / settings.getWorktimeOut().intValue();
      this.zangyo.setValue(zangyo_temp - settings.getResttimeOut() * nnn);
      this.zangyo1.setValue(zangyo_temp);
      this.zangyo2.setValue(settings.getResttimeOut() * nnn);

      // 勤務時間内・外
      this.shugyo.setValue(this.jikannai.getValue() + this.zangyo.getValue());

      // 遅刻
      if (tempList.size() > 0) {
        if (kinmuStartDate.before(tempList.get(0).getWorkDate().getValue())) {
          this.chikoku.setValue("○");
        }
      }
      // 相対
      if (tempList.size() > 0) {
        if (kinmuEndDate.after(tempList.get(tempList.size() - 1).getWorkDate()
            .getValue())) {
          this.sotai.setValue("○");
        }
      }

    } catch (RuntimeException ex) {
      ex.printStackTrace();
      throw ex;
    }
  }

  /**
   * ExtTimecardResultDataオブジェクトのインスタンスを作る
   *
   * @param date
   * @param workflag
   * @return
   */
  private ExtTimecardResultData createExtTimecardResultData(Date date,
      String workflag) {
    ExtTimecardResultData rd = new ExtTimecardResultData();
    rd.initField();
    rd.setWorkDate(date);
    rd.setWorkFlag(workflag);
    return rd;
  }

  /**
   * 勤務時間設定をDBから取得する
   *
   * @return
   */
  private EipTTimecardSettings loadEipTTimecardSettings() {

    SelectQuery query = new SelectQuery(EipTTimecardSettings.class);
    List aList = dataContext.performQuery(query);
    if (aList != null && aList.size() > 0) {
      EipTTimecardSettings record = (EipTTimecardSettings) aList.get(0);
      return record;
    } else {
      return null;
    }
  }

}
