/* ====================================================================
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 2000-2001 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The names "Apache" and "Apache Software Foundation" and
 *     "Apache Jetspeed" must not be used to endorse or promote products
 *    derived from this software without prior written permission. For
 *    written permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache" or
 *    "Apache Jetspeed", nor may "Apache" appear in their name, without
 *    prior written permission of the Apache Software Foundation.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 */

package org.apache.jetspeed.portlet;

import java.util.Locale;
import java.util.Map;
import java.util.Enumeration;

/**
 ** The <CODE>PortletRequest</CODE> encapsulates the request sent by the
 ** client to the portlet.
 **
 ** @see   PortletResponse
 **
 ** @author   <A HREF="mailto:tboehme@us.ibm.com">Thomas F. Boehme</A>
 **/

public interface PortletRequest
{
    /**
     ** Returns the mode that the portlet is running in.
     **
     ** @return   the portlet mode
     **/

    public Portlet.Mode getMode ();

    /**
     ** Returns the HTTP method of this request. The most commonly
     ** used request methods are GET and POST.
     **
     ** @return   the method
     **/

    public String getMethod ();

    /**
     ** Returns the locale of the preferred language. The preference
     ** is based on the user's choice of language(s) and/or the
     ** client's Accept-Language header.
     ** <P>
     ** If more than one language is preferred, the locale returned by
     ** this method is the one with the highest preference.
     **
     ** @return   the locale of the preferred language
     **/

    public Locale getLocale ();

    /**
     ** Returns an object representing the client device that the
     ** user connects to the portal with.
     **
     ** @return   the client
     **/

    public Client getClient ();

    /**
     ** Returns the value of the parameter with the given name,
     ** or <CODE>null</CODE> if no such parameter exists.
     **
     ** <P>
     ** You should only use this method when you are sure the
     ** parameter has only one value. If not, use
     ** <CODE>getParameterValues(String)</CODE>
     **
     ** @param   name
     **          the parameter name
     **
     ** @return   the parameter value
     **
     ** @see   #getParameterMap()
     ** @see   #getParameterValues(String)
     **/

    public String getParameter (String name);

    /**
     ** Returns a map of the parameters of this request.
     **
     ** @return   a map of parameters
     **
     ** @see   #getParameter(String)
     ** @see   #getParameterValues(String)
     **/

    public Map getParameterMap ();

    /**
     ** Returns an enumeration of all parameter names. If this
     ** request
     **
     ** @return   the enumeration
     **/

    public Enumeration getParameterNames ();

    /**
     ** Returns the values of all parameters with the
     ** given name.
     ** <P>
     ** A request can carry more than one parameter with a
     ** certain name. This method returns these parameters
     ** in the order of appearance.
     **
     ** @param   name
     **          the parameter name
     **
     ** @return   the array of parameter values
     **
     ** @see   #getParameterMap()
     ** @see   #getParameter(String)
     **/

    public String [] getParameterValues (String name);

    /**
     ** Associates an attribute with the given name and value
     ** with this request. If a portlet needs to communicate
     ** information to embedded servlets or JSP, this methods
     ** can used carry the information along.
     **
     ** <P>
     ** The portlet provider should take care that the
     ** the namespace of attribute names is not unnecessarily
     ** polluted. It is recommended to prefix all attributes
     ** the package and class name of the portlet that
     ** makes use of this method.
     **
     ** @param   name
     **          the attribute name
     ** @param   value
     **          the attribute value
     **/

    public void setAttribute (String name, Object value);

    /**
     ** Removes the attribute with the given name.
     **
     ** @param   name
     **          the name of attribute to be removed
     **/

    public void removeAttribute (String name);

   /**
     ** Returns the value of the attribute with the given name,
     ** or <CODE>null</CODE> if no attribute with the given name exists.
     **
     ** @param   name
     **          the attribute name
     **
     ** @return   the attribute value
     **/

    public Object getAttribute (String name);

    /**
     ** Returns an enumeration of names of all attributes available to
     ** this request. This method returns an empty enumeration if the
     ** request has no attributes available to it.
     **
     ** @return   an enumeration of attribute names
     **/

    public Enumeration getAttributeNames ();

    /**
     ** Returns whether this request was made using a secure channel, such
     ** as HTTPS.
     **
     ** @return   <CODE>true</CODE> if channel is secure, <BR>
     **           <CODE>false</CODE> otherwise
     **/

    public boolean isSecure ();

    /**
     ** Returns the window that the portlet is running in.
     **
     ** @return   the portlet window
     **/

    public PortletWindow getWindow ();

    /**
     ** Returns the data of the virtual instance of this portlet
     ** <P>
     ** If the portlet is run in CONFIGURE mode, the
     ** portlet data is not accessible and this method will
     ** return <CODE>null</CODE>.
     ** </P>
     **
     ** @return   the portlet data
     **/

    public PortletData getData ();

    /**
     ** Returns the current session or, if there is no current session,
     ** it creates one and returns it.
     **
     ** @return   the portlet session
     **/

    public PortletSession getSession ();

    /**
     ** Returns the current session or, if there is no current session
     ** and the given flag is <CODE>true</CODE>, it creates one and returns
     ** it.
     **
     ** <P>
     ** If the given flag is <CODE>false</CODE> and there is no current
     ** portlet session, this method returns <CODE>null</CODE>.
     **
     ** @param   create
     **          <CODE>true</CODE> to create a news session, <BR>
     **          <CODE>false</CODE> to return <CODE>null</CODE> of there
     **          is no current session
     **
     ** @return   the portlet session
     **/

    public PortletSession getSession (boolean create);
}
