/*
 * Aipo is a groupware program developed by Aimluck,Inc.
 * Copyright (C) 2004-2008 Aimluck,Inc.
 * http://aipostyle.com/
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */
package com.aimluck.eip.whereaboutsinformation;

import java.util.Calendar;
import java.util.List;

import org.apache.cayenne.exp.Expression;
import org.apache.cayenne.exp.ExpressionFactory;
import org.apache.commons.lang.StringUtils;
import org.apache.jetspeed.services.logging.JetspeedLogFactoryService;
import org.apache.jetspeed.services.logging.JetspeedLogger;
import org.apache.turbine.util.RunData;
import org.apache.velocity.context.Context;

import com.aimluck.commons.field.ALDateTimeField;
import com.aimluck.commons.field.ALStringField;
import com.aimluck.eip.cayenne.om.portlet.AvzTWhereaboutsinformation;
import com.aimluck.eip.common.ALAbstractFormData;
import com.aimluck.eip.common.ALDBErrorException;
import com.aimluck.eip.common.ALPageNotFoundException;
import com.aimluck.eip.modules.actions.common.ALAction;
import com.aimluck.eip.orm.Database;
import com.aimluck.eip.orm.query.SelectQuery;
import com.aimluck.eip.util.ALEipUtils;
import com.aimluck.eip.whereaboutsinformation.util.WhereaboutsInformationUtils;

/**
 * WhereaboutsInformationのフォームデータを管理するクラスです。 <BR>
 * 
 */
public class WhereaboutsInformationFormData extends ALAbstractFormData {

  /** logger */
  private static final JetspeedLogger logger =
    JetspeedLogFactoryService.getLogger(WhereaboutsInformationFormData.class
      .getName());

  /** 離席理由 */
  private ALStringField reason;

  /** <code>return_date</code> 戻り日時 */
  private ALDateTimeField return_date;

  private ALDateTimeField now_date;

  /** ログインユーザーのID * */
  private int user_id;

  /** vmから渡ってきたユーザーID */
  private int vmUserId;

  /**
   * init
   * 
   * @param action
   * @param rundata
   * @param context
   * @see com.aimluck.eip.common.ALAbstractFormData#init(com.aimluck.eip.modules.actions.common.ALAction,
   *      org.apache.turbine.util.RunData, org.apache.velocity.context.Context)
   */
  @Override
  public void init(ALAction action, RunData rundata, Context context)
      throws ALPageNotFoundException, ALDBErrorException {
    super.init(action, rundata, context);

    String vmUserIdStr = rundata.getParameters().getString("vmUserId");
    if (StringUtils.isNotEmpty(vmUserIdStr)) {
      setVmUserId(Integer.parseInt(vmUserIdStr));
    }

    user_id = ALEipUtils.getUserId(rundata);
  }

  /**
   * 各フィールドを初期化します。 <BR>
   * 
   * @see com.aimluck.eip.common.ALData#initField()
   */
  public void initField() {
    // WhereaboutsInformation名
    reason = new ALStringField();
    reason.setFieldName("離席理由");
    reason.setTrim(true);

    // 戻り日時
    return_date = new ALDateTimeField("yyyy-MM-dd-HH-mm");
    // 55~59分の場合時間（時）繰上げ
    Calendar cal = Calendar.getInstance();
    cal.set(Calendar.YEAR, cal.get(Calendar.YEAR));
    cal.set(Calendar.MONTH, cal.get(Calendar.MONTH));
    cal.set(Calendar.DATE, cal.get(Calendar.DATE));
    cal.set(Calendar.DAY_OF_WEEK, cal.get(Calendar.DAY_OF_WEEK));
    cal.set(Calendar.MINUTE, cal.get(Calendar.MINUTE));
    // 現在時刻（分）
    int min = cal.get(Calendar.MINUTE);
    // 現在時刻（時）
    int hour = cal.get(Calendar.HOUR_OF_DAY);
    if (55 <= min && min <= 59) {
      hour = hour + 1;
    }
    cal.set(Calendar.HOUR_OF_DAY, hour);
    return_date.setValue(cal.getTime());
    return_date.setFieldName("戻り日時");
  }

  /**
   * WhereaboutsInformationの各フィールドに対する制約条件を設定します。 <BR>
   * 
   * @see com.aimluck.eip.common.ALAbstractFormData#setValidator()
   */
  @Override
  protected void setValidator() {
    // WhereaboutsInformation名の文字数制限
    reason.limitMaxLength(99);
  }

  /**
   * WhereaboutsInformationのフォームに入力されたデータの妥当性検証を行います。 <BR>
   * 
   * @param msgList
   * @return TRUE 成功 FALSE 失敗
   * @see com.aimluck.eip.common.ALAbstractFormData#validate(java.util.ArrayList)
   */
  @Override
  protected boolean validate(List<String> msgList) {

    try {
      // ノート
      reason.validate(msgList);
      // 戻り時間
      return_date.validate(msgList);
      // 現在時刻
      now_date = new ALDateTimeField("yyyy-MM-dd-HH-mm");
      Calendar cal = Calendar.getInstance();
      cal.set(Calendar.YEAR, cal.get(Calendar.YEAR));
      cal.set(Calendar.MONTH, cal.get(Calendar.MONTH));
      cal.set(Calendar.DATE, cal.get(Calendar.DATE));
      cal.set(Calendar.DAY_OF_WEEK, cal.get(Calendar.DAY_OF_WEEK));
      cal.set(Calendar.MINUTE, cal.get(Calendar.MINUTE));
      cal.set(Calendar.HOUR_OF_DAY, cal.get(Calendar.HOUR_OF_DAY));
      now_date.setValue(cal.getTime());
      if (return_date.getValue().before(now_date.getValue())) {
        msgList.add("『 <span class='em'>戻り時間</span> 』は現在日時以降の日時を指定してください。");
      }

    } catch (Exception e) {
      logger.error("Exception", e);
    }
    return (msgList.size() == 0);
  }

  public ALStringField getReason() {
    return reason;
  }

  /**
   * 戻り日時を取得します。
   * 
   * @return
   */
  public ALDateTimeField getReturnDate() {
    return return_date;
  }

  public int getUserId() {
    return user_id;
  }

  public void setUserId(int user_id) {
    this.user_id = user_id;
  }

  public int getVmUserId() {
    return vmUserId;
  }

  public void setVmUserId(int vmUserId) {
    this.vmUserId = vmUserId;
  }

  /**
   * WhereaboutsInformationをデータベースから読み出します。 <BR>
   * 
   * @param rundata
   * @param context
   * @param msgList
   * @return TRUE 成功 FALSE 失敗
   * @see com.aimluck.eip.common.ALAbstractFormData#loadFormData(org.apache.turbine.util.RunData,
   *      org.apache.velocity.context.Context)
   */
  @Override
  protected boolean loadFormData(RunData rundata, Context context,
      List<String> msgList) {
    try {
      // ノート
      reason.setValue("");

      // 戻り日時
      return_date.setValue(return_date.getValue());
    } catch (Exception e) {
      logger.error("Exception", e);
      return false;
    }
    return true;
  }

  /**
   * WhereaboutsInformationをデータベースから削除します。 <BR>
   * 
   * @param rundata
   * @param context
   * @param msgList
   * @return TRUE 成功 FALSE 失敗
   * @see com.aimluck.eip.common.ALAbstractFormData#deleteFormData(org.apache.turbine.util.RunData,
   *      org.apache.velocity.context.Context)
   */
  @Override
  protected boolean deleteFormData(RunData rundata, Context context,
      List<String> msgList) {
    return true;
  }

  /**
   * WhereaboutsInformationをデータベースに格納します。 <BR>
   * 
   * @param rundata
   * @param context
   * @param msgList
   * @return TRUE 成功 FALSE 失敗
   * @see com.aimluck.eip.common.ALAbstractFormData#insertFormData(org.apache.turbine.util.RunData,
   *      org.apache.velocity.context.Context, java.util.ArrayList)
   */
  @Override
  protected boolean insertFormData(RunData rundata, Context context,
      List<String> msgList) {
    return true;
  }

  /**
   * データベースに格納されているWhereaboutsInformationを更新します。 <BR>
   * 
   * @param rundata
   * @param context
   * @param msgList
   * @return TRUE 成功 FALSE 失敗
   * @see com.aimluck.eip.common.ALAbstractFormData#updateFormData(org.apache.turbine.util.RunData,
   *      org.apache.velocity.context.Context, java.util.ArrayList)
   */
  @Override
  protected boolean updateFormData(RunData rundata, Context context,
      List<String> msgList) {
    try {
      // TODO レコードが無ければ追加、有れば更新を行なう。
      // レコードはユーザーIDに引っ掛けましょう。
      SelectQuery<AvzTWhereaboutsinformation> query =
        Database.query(AvzTWhereaboutsinformation.class);
      Expression exp =
        ExpressionFactory.matchExp(
          AvzTWhereaboutsinformation.USER_ID_PROPERTY,
          vmUserId);
      query.setQualifier(exp);
      List<AvzTWhereaboutsinformation> list = query.getResultList();

      // user_idに紐づくレコードが存在するかどうか
      if (!(list.size() == 0)) {

        // 更新
        AvzTWhereaboutsinformation info =
          WhereaboutsInformationUtils.getAvzTwhereaboutsinformationById(String
            .valueOf(vmUserId));
        if (info == null) {
          return false;
        }

        // ログインユーザーID = info.getUserId()かどうか
        if (user_id == info.getUserId()) {
          info.setNote(reason.getValue());
          info.setUpdateDate(Calendar.getInstance().getTime());
          info.setReturnDate(return_date.getValue());
          // 在席・離席変更
          info.setStatus(WhereaboutsInformationUtils.STATUS_ABSENCE);
        } else {
          // 他者変更可になっているかどうか
          if (info.getPermission().equals("P")) {
            info.setNote(reason.getValue());
            info.setUpdateDate(Calendar.getInstance().getTime());
            info.setReturnDate(return_date.getValue());
            // 在席・離席変更
            info.setStatus(WhereaboutsInformationUtils.STATUS_ABSENCE);
          }
        }

      } else {

        // 追加
        AvzTWhereaboutsinformation info =
          Database.create(AvzTWhereaboutsinformation.class);

        // 自動採取→info.setWhereaboutsinformationId(id);
        info.setUserId(ALEipUtils.getUserId(rundata));
        info.setCreateDate(Calendar.getInstance().getTime());
        info.setNote(reason.getValue());
        info.setPermission(WhereaboutsInformationUtils.PERMISSION_OFF);
        info.setStatus(WhereaboutsInformationUtils.STATUS_ABSENCE);
        info.setUpdateDate(Calendar.getInstance().getTime());
        info.setReturnDate(return_date.getValue());
      }

      Database.commit();

    } catch (Exception e) {
      Database.rollback();
      logger.error("Exception", e);
      return false;
    }
    return true;
  }
}
