/*
 * Copyright(C) 2010 avanza Co.,Ltd. All rights reserved.
 * http://www.avnz.co.jp/
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

package com.aimluck.eip.enquete;

import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.List;

import org.apache.cayenne.access.DataContext;
import org.apache.cayenne.exp.Expression;
import org.apache.cayenne.exp.ExpressionFactory;
import org.apache.jetspeed.services.logging.JetspeedLogFactoryService;
import org.apache.jetspeed.services.logging.JetspeedLogger;
import org.apache.turbine.util.RunData;
import org.apache.velocity.context.Context;

import com.aimluck.commons.field.ALDateTimeField;
import com.aimluck.commons.field.ALNumberField;
import com.aimluck.commons.field.ALStringField;
import com.aimluck.eip.cayenne.om.portlet.AvzTEnquete;
import com.aimluck.eip.cayenne.om.portlet.AvzTEnqueteChoiceItem;
import com.aimluck.eip.cayenne.om.portlet.AvzTEnqueteItem;
import com.aimluck.eip.cayenne.om.security.TurbineUser;
import com.aimluck.eip.common.ALAbstractFormData;
import com.aimluck.eip.common.ALDBErrorException;
import com.aimluck.eip.common.ALEipConstants;
import com.aimluck.eip.common.ALEipUser;
import com.aimluck.eip.common.ALPageNotFoundException;
import com.aimluck.eip.enquete.util.EnqueteUtils;
import com.aimluck.eip.modules.actions.common.ALAction;
import com.aimluck.eip.orm.Database;
import com.aimluck.eip.orm.DatabaseOrmService;
import com.aimluck.eip.orm.query.SelectQuery;
import com.aimluck.eip.services.eventlog.ALEventlogConstants;
import com.aimluck.eip.services.eventlog.ALEventlogFactoryService;
import com.aimluck.eip.util.ALEipUtils;

/**
 * Enqueteのフォームデータを管理するクラスです。 <BR>
 * 
 */
public class EnqueteFormData extends ALAbstractFormData {

  /** logger */
  private static final JetspeedLogger logger =
    JetspeedLogFactoryService.getLogger(EnqueteFormData.class.getName());

  private DataContext dataContext;

  /** アンケートID */
  private ALNumberField enquete_id;

  /** アンケート名 */
  private ALStringField name;

  /** コメント */
  private ALStringField comment;

  /** 公開方法（記名・無記名） */
  private ALStringField method;

  /** 締め切り日 */
  private ALDateTimeField deadline_date;

  /** 締め切り日指定フラグ */
  private ALStringField deadline_method;

  /** 公開・締切状態 */
  private ALStringField status;

  /** 項目数 */
  private int itemSize;

  private ALStringField[] item_title = new ALStringField[itemSize];

  private ALStringField[] choice_item_select = new ALStringField[itemSize];

  private ALStringField[] choice_item_title = new ALStringField[itemSize];

  /** アンケート項目・選択肢取得リスト */
  private List<EnqueteItemRecordData> results;

  /** メンバーリスト */
  private List<ALEipUser> memberList;

  /** <code>currentYear</code> 現在の年 */
  private int currentYear;

  /** 項目タイトル */
  private List item_title_check;

  /** 状態（単一・複数・フリーテキスト） */
  private List choice_item_select_check;

  /** 選択肢タイトル */
  private List choice_item_title_check;

  /** 締切日 */
  private String deadline_flag;

  /**
   * init
   * 
   * @param action
   * @param rundata
   * @param context
   * @see com.aimluck.eip.common.ALAbstractFormData#init(com.aimluck.eip.modules.actions.common.ALAction,
   *      org.apache.turbine.util.RunData, org.apache.velocity.context.Context)
   */
  @Override
  public void init(ALAction action, RunData rundata, Context context)
      throws ALPageNotFoundException, ALDBErrorException {

    super.init(action, rundata, context);

    dataContext = DatabaseOrmService.getInstance().getDataContext();

    String enqueteId =
      rundata.getParameters().getString(ALEipConstants.ENTITY_ID);
    if (enqueteId == null || "new".equals(enqueteId)) {
      String str[] = rundata.getParameters().getStrings("member_to");
      if (str != null && str.length > 0) {
      }
    }

    item_title_check = new ArrayList();
    choice_item_select_check = new ArrayList();
    choice_item_title_check = new ArrayList();

    for (int i = 1; i < itemSize; i++) {
      item_title_check.add(i - 1, rundata.getParameters().getString(
        "item_title[" + i + "]"));
      choice_item_select_check.add(i - 1, rundata.getParameters().getString(
        "choice_item_select[" + i + "]"));
      choice_item_title_check.add(i - 1, rundata.getParameters().getString(
        "choice_item_title[" + i + "]"));

    }

    deadline_flag = rundata.getParameters().getString("deadline_date_check");
    if (deadline_flag == null) {
    }

    /**
     * 下記のコメントアウトのようにアクセスコントロールを追加する場合は
     * com.aimluck.eip.services.accessctl.ALAccessControlConstantsに
     * アクセスコントロール用の定数を追加してください。 (例) public static final String
     * POERTLET_FEATURE_ENQUETE_ENQUETE_OTHER = "enquete_enquete_other";
     */

    // aclPortletFeature =
    // ALAccessControlConstants.POERTLET_FEATURE_ENQUETE_ENQUETE_SELF;
  }

  /**
   * 各フィールドを初期化します。 <BR>
   * 
   * @see com.aimluck.eip.common.ALData#initField()
   */
  public void initField(RunData rundata) {

    String itemSizes[] = rundata.getParameters().getStrings("itemSum[]");
    if (itemSizes == null) {
      itemSize = 0;
    } else {
      itemSize = itemSizes.length;
    }

    item_title = new ALStringField[itemSize];
    choice_item_select = new ALStringField[itemSize];
    choice_item_title = new ALStringField[itemSize];

    Date now = new Date();

    Calendar cal = Calendar.getInstance();
    int min = cal.get(Calendar.MINUTE);
    if (min <= 15) {
      cal.set(Calendar.MINUTE, 15);
    } else if (min <= 30) {
      cal.set(Calendar.MINUTE, 30);
    } else if (min <= 45) {
      cal.set(Calendar.MINUTE, 45);
    } else {
      cal.set(Calendar.MINUTE, 60);
    }

    // アンケート名
    name = new ALStringField();
    name.setFieldName("アンケート名");
    name.setTrim(true);

    // メモ
    comment = new ALStringField();
    comment.setFieldName("コメント");
    comment.setTrim(false);

    // 公開方法（記名・無記名）
    method = new ALStringField();
    method.setFieldName("記名回答/無記名回答");
    method.setTrim(true);

    status = new ALStringField();
    status.setFieldName("公開・締切状態");

    // メンバーリスト
    memberList = new ArrayList();

    // 締め切り日指定フラグ
    deadline_method = new ALStringField();
    deadline_method.setFieldName("指定しない");

    // 締切日
    deadline_date = new ALDateTimeField("yyyy-MM-dd-HH-mm");
    deadline_date.setFieldName("終了日時");
    deadline_date.setValue(new Date());

    // 現在の年（年を選択するリストボックスに利用）
    currentYear = Calendar.getInstance().get(Calendar.YEAR);

    int length = getItemSize(rundata);

    for (int i = 0; i < length; i++) {
      item_title[i] = new ALStringField();
      item_title[i].setFieldName(i + "番目のアンケート項目名");
      item_title[i].setTrim(true);

      choice_item_title[i] = new ALStringField();
      choice_item_title[i].setFieldName(i + "番目のアンケート項目の選択肢");
      choice_item_title[i].setTrim(true);
    }
  }

  public void initField() {

  }

  /**
   * アンケートの各フィールドに対する制約条件を設定します。 <BR>
   * 
   * @see com.aimluck.eip.common.ALAbstractFormData#setValidator()
   */
  @Override
  protected void setValidator() {
    // アンケート名必須項目
    name.setNotNull(true);
    // アンケート名の文字数制限
    name.limitMaxLength(99);
    // コメントの文字数制限
    comment.limitMaxLength(1000);

    if (itemSize == 1) {
    } else {

      for (int i = 1; i < itemSize; i++) {

        item_title[i].setNotNull(true);
        item_title[i].limitMaxLength(35);

        choice_item_title[i].setNotNull(true);
        choice_item_title[i].limitMaxLength(40);
      }
    }
  }

  /**
   * 
   * @param rundata
   * @param context
   * @param msgList
   * @return
   * @see com.aimluck.eip.common.ALAbstractFormData#setFormData(org.apache.turbine.util.RunData,
   *      org.apache.velocity.context.Context, java.util.ArrayList)
   */
  @Override
  protected boolean setFormData(RunData rundata, Context context,
      List<String> msgList) throws ALPageNotFoundException, ALDBErrorException {
    boolean res = super.setFormData(rundata, context, msgList);
    try {
      if (res) {

        String[] str = rundata.getParameters().getStrings("member_to");
        if (str != null && str.length > 0) {
          SelectQuery query = new SelectQuery(TurbineUser.class);
          Expression exp =
            ExpressionFactory.inExp(TurbineUser.LOGIN_NAME_PROPERTY, str);
          query.setQualifier(exp);
          memberList.addAll(ALEipUtils.getUsersFromSelectQuery(query));
        }
      }
    } catch (Exception ex) {
      logger.error("Exception", ex);
      res = false;
    }
    return res;
  }

  /**
   * アンケートのフォームに入力されたデータの妥当性検証を行います。 <BR>
   * 
   * @param msgList
   * @return TRUE 成功 FALSE 失敗
   * @see com.aimluck.eip.common.ALAbstractFormData#validate(java.util.ArrayList)
   */
  @Override
  protected boolean validate(List<String> msgList) {

    try {

      // アンケート名
      name.validate(msgList);
      // コメント
      comment.validate(msgList);
      // 締切日
      deadline_date.validate(msgList);

      // 締め切り日指定フラグが設定されている場合は締め切り日入力フォームチェックを行いません。
      Date now = Calendar.getInstance().getTime();

      // 設定した締切日が現在時刻より前の場合エラーメッセージを返す
      if (deadline_flag == null || "".equals(deadline_flag)) {
        if (deadline_date.getValue().before(now)) {
          msgList
            .add("『 <span class='em'>締め切り日</span> 』は『 <span class='em'>現在</span> 』以降の日付で指定してください。");
        }
      } else if ("TRUE".equals(deadline_flag)) {
      }

      // 回答メンバーが選択されていない場合エラーメッセージを返す
      if (memberList.size() == 0) {
        msgList.add("『 <span class='em'>回答メンバー</span> 』は必ず指定してください。");
      }

      // アンケート項目・選択肢の必須入力チェック
      if (itemSize == 1) {
        msgList.add("『 <span class='em'>アンケート項目</span> 』は必ず入力してください。");
      }
      for (int i = 1; i < itemSize; i++) {

        if ("".equals(item_title_check.get(i - 1))) {
          item_title[i].validate(msgList);
        } else {
          if (item_title_check.get(i - 1).toString().length() > 35) {
            msgList.add("『 <span class='em'>アンケート項目</span> 』は35文字以内で入力してください。");
          }
        }

        // フリーテキスト以外の時、選択肢項目チェック

        if ("one_select".equals(choice_item_select_check.get(i - 1))
          || "multi_select".equals(choice_item_select_check.get(i - 1))) {
          String[] choiceItemToken = null;
          String cItem = choice_item_title_check.get(i - 1).toString();
          choiceItemToken = cItem.split(",");

          if ("".equals(choice_item_title_check.get(i - 1))) {
            choice_item_title[i].validate(msgList);
          } else if (choiceItemToken.length == 0) {

            msgList.add("『<span class='em'> "
              + i
              + " 番目の アンケート項目の選択肢</span> 』に空白が含まれています。");

          } else {

            for (int j = 0; j < choiceItemToken.length; j++) {
              ALStringField choiceItem = new ALStringField();
              choiceItem.setValue(choiceItemToken[j]);
              if ("".equals(choiceItem.getValue())) {
                msgList.add("『<span class='em'> "
                  + i
                  + " 番目の  アンケート項目の選択肢</span> 』に空白が含まれています。");
                break;
              } else if (choiceItem.getValue().length() > 40) {
                msgList.add("『 <span class='em'> "
                  + i
                  + " 番目の アンケート項目の選択肢</span> 』は40文字以内で入力してください。");
              }
            }
          }
        }
      }

    } catch (Exception e) {
      logger.error("Exception", e);

    }
    return (msgList.size() == 0);

  }

  /**
   * アンケートをデータベースから読み出します。 <BR>
   * 
   * @param rundata
   * @param context
   * @param msgList
   * @return TRUE 成功 FALSE 失敗
   * @see com.aimluck.eip.common.ALAbstractFormData#loadFormData(org.apache.turbine.util.RunData,
   *      org.apache.velocity.context.Context)
   */
  @Override
  protected boolean loadFormData(RunData rundata, Context context,
      List<String> msgList) {
    String date1 = null;
    try {
      // // オブジェクトモデルを取得
      AvzTEnquete enquete = EnqueteUtils.getAvzTEnquete(rundata, context);
      if (enquete == null) {
        return false;
      }

      // アンケート名
      name.setValue(enquete.getName());
      // コメント
      comment.setValue(enquete.getComment());
      // 公開方法（記名・無記名）
      method.setValue(enquete.getPublicMethod());
      // 公開・締切状態
      status.setValue(enquete.getPublicStatus());

      // 締め切り日
      if (EnqueteUtils.isEmptyDate(enquete.getDeadlineDate())) {
        deadline_method.setValue(EnqueteUtils.DEADLINE_METHOD_HANDOPERATED);
        deadline_date.setValue(date1);
      } else {
        deadline_date.setValue(enquete.getDeadlineDate());
      }

      // このアンケートを共有しているメンバーを取得
      SelectQuery<AvzTEnqueteChoiceItem> choiceitemquery2 =
        Database.query(AvzTEnqueteChoiceItem.class);
      Expression choiceitemexp2 =
        ExpressionFactory.matchExp(
          AvzTEnqueteChoiceItem.ENQUETE_ID_PROPERTY,
          enquete.getEnqueteId());

      choiceitemquery2.setQualifier(choiceitemexp2);
      choiceitemquery2.distinct(true);
      List<AvzTEnqueteChoiceItem> choicelist = choiceitemquery2.getResultList();

      // 回答メンバーのリスト
      List<Integer> users = new ArrayList<Integer>();

      for (int l = 0; l < choicelist.size(); l++) {
        AvzTEnqueteChoiceItem map = choicelist.get(l);

        users.add(map.getAnswerId().intValue());

      }

      if (users.size() > 0) {
        SelectQuery<TurbineUser> query = Database.query(TurbineUser.class);
        Expression exp =
          ExpressionFactory.inDbExp(TurbineUser.USER_ID_PK_COLUMN, users);
        query.setQualifier(exp);
        memberList.addAll(ALEipUtils.getUsersFromSelectQuery(query));
      }

      // アンケート項目入力フォーム内のデータの取得
      SelectQuery<AvzTEnqueteItem> itemquery =
        Database.query(AvzTEnqueteItem.class);
      Expression itemexp =
        ExpressionFactory
          .matchDbExp(AvzTEnqueteItem.AVZ_TENQUETE_PROPERTY
            + "."
            + AvzTEnquete.ENQUETE_ID_PK_COLUMN, enquete
            .getEnqueteId()
            .toString());
      itemquery.setQualifier(itemexp);
      itemquery.distinct(true);
      itemquery.orderAscending(AvzTEnqueteItem.ITEM_NUMBER_PROPERTY);

      List<AvzTEnqueteItem> itemlist = itemquery.getResultList();

      results = new ArrayList<EnqueteItemRecordData>();

      // アンケート項目に順ずる選択肢情報を取得
      for (Object r : itemlist) {
        AvzTEnqueteItem item = (AvzTEnqueteItem) r;
        EnqueteItemRecordData d = new EnqueteItemRecordData();
        ALStringField record = new ALStringField();

        d.setItemId(item.getItemId());
        int item_id = item.getItemId();
        d.setItemNumber(item.getItemNumber());
        record.setValue(item.getItemTitle());
        d.setItemName(record);
        d.setStatus(item.getStatus());
        if (EnqueteUtils.STATUS_FREE_TEXT.equals(item.getStatus())) {
          d.setChoiceItem(null);
        } else {
          d.setChoiceItem(EnqueteUtils.getChoiceItem(
            rundata,
            context,
            item_id,
            users));
        }
        results.add(d);
      }

    } catch (Exception ex) {
      dataContext.rollbackChanges();
      logger.error("Exception", ex);
      return false;
    }
    return true;
  }

  /**
   * Enqueteをデータベースに格納します。 <BR>
   * 
   * @param rundata
   * @param context
   * @param msgList
   * @return TRUE 成功 FALSE 失敗
   * @see com.aimluck.eip.common.ALAbstractFormData#insertFormData(org.apache.turbine.util.RunData,
   *      org.apache.velocity.context.Context, java.util.ArrayList)
   */
  @Override
  protected boolean insertFormData(RunData rundata, Context context,
      List<String> msgList) {
    try {
      // 新規オブジェクトモデル
      AvzTEnquete enquete = Database.create(AvzTEnquete.class);

      // 現在時刻を取得
      Date now = Calendar.getInstance().getTime();

      // アンケート名
      enquete.setName(name.getValue());

      // 作成ユーザーID
      enquete.setCreaterId(ALEipUtils.getUserId(rundata));

      // コメント
      enquete.setComment(comment.getValue());

      // 作成日
      enquete.setCreateDate(now);

      // 更新日
      enquete.setUpdateDate(now);

      // 締め切り日
      String deadline_flag =
        rundata.getParameters().getString("deadline_date_check");
      if (deadline_flag == null || "".equals(deadline_flag)) {
        enquete.setDeadlineDate(deadline_date.getValue());
        enquete.setDeadlineMethod(EnqueteUtils.DEADLINE_METHOD_AUTO);
      } else if ("TRUE".equals(deadline_flag)) {
        enquete.setDeadlineDate(EnqueteUtils.getEmptyDate());
        enquete.setDeadlineMethod(EnqueteUtils.DEADLINE_METHOD_HANDOPERATED);

      } else {
        return false;
      }

      // 項目数
      int length = getItemSize(rundata);
      enquete.setItemNum(length - 1);

      // 公開方法 (記名・無記名)
      if (EnqueteUtils.PUBLIC_METHOD_SECRET.equals(method.getValue())) {
        enquete.setPublicMethod(EnqueteUtils.PUBLIC_METHOD_SECRET);
      } else if (EnqueteUtils.PUBLIC_METHOD_SIGNED.equals(method.getValue())) {
        enquete.setPublicMethod(EnqueteUtils.PUBLIC_METHOD_SIGNED);
      }

      // 公開締切状態 ※現時点では作成後は未公開で設定
      enquete.setPublicStatus(EnqueteUtils.PUBLIC_STATUS_BEFORE);

      int length2 = getItemSize(rundata);
      for (int j = 1; j < length2; j++) {

        // 新規オブジェクトモデル
        AvzTEnqueteItem enqueteitem = Database.create(AvzTEnqueteItem.class);

        enqueteitem.setAvzTEnquete(enquete);
        // 項目番号
        enqueteitem.setItemNumber(j);

        // 項目タイトル
        String item_title =
          rundata.getParameters().getString("item_title[" + j + "]");
        ALStringField item = new ALStringField();
        item.setValue(item_title);
        enqueteitem.setItemTitle(item.getValue());

        // 状態(複数選択可不可・フリーテキスト)
        String status =
          rundata.getParameters().getString("choice_item_select[" + j + "]");
        if ("one_select".equals(status)) {
          enqueteitem.setStatus(EnqueteUtils.STATUS_ONE_SELECT);
        } else if ("multi_select".equals(status)) {
          enqueteitem.setStatus(EnqueteUtils.STATUS_MULTI_SELECT);
        } else if ("free_text".equals(status)) {
          enqueteitem.setStatus(EnqueteUtils.STATUS_FREE_TEXT);
        }

        // 選択肢情報を登録
        if ("free_text".equals(status)) {
          // フリーテキストの場合は選択肢名なしで1ユーザー1レコード
          EnqueteUtils.insertFreeTextChoiceItem(
            rundata,
            context,
            msgList,
            dataContext,
            choice_item_title,
            j,
            enquete,
            enqueteitem,
            memberList);
        } else {
          // 単一選択・複数選択可の場合
          EnqueteUtils.insertChoiceItem(
            rundata,
            context,
            msgList,
            dataContext,
            choice_item_title,
            j,
            enquete,
            enqueteitem,
            memberList);
        }

      }

      Database.commit();

      // イベントログに保存
      ALEventlogFactoryService.getInstance().getEventlogHandler().log(
        enquete.getEnqueteId(),
        ALEventlogConstants.PORTLET_TYPE_NONE,
        name.getValue());

    } catch (Exception ex) {
      Database.rollback();
      logger.error("Exception", ex);
      return false;
    }
    return true;
  }

  /**
   * データベースに格納されているEnqueteを更新します。 <BR>
   * 
   * @param rundata
   * @param context
   * @param msgList
   * @return TRUE 成功 FALSE 失敗
   * @see com.aimluck.eip.common.ALAbstractFormData#updateFormData(org.apache.turbine.util.RunData,
   *      org.apache.velocity.context.Context, java.util.ArrayList)
   */
  @Override
  protected boolean updateFormData(RunData rundata, Context context,
      List<String> msgList) {
    try {
      // オブジェクトモデルを取得
      AvzTEnquete enquete = EnqueteUtils.getAvzTEnquete(rundata, context);
      if (enquete == null) {
        return false;
      }

      String update_flag = enquete.getPublicStatus();

      // 現在時刻を取得
      Date now = Calendar.getInstance().getTime();

      if (update_flag.equals(EnqueteUtils.PUBLIC_STATUS_BEFORE)) {

        // アンケート作成日の取得
        Date create = enquete.getCreateDate();

        // Enqueteを削除
        dataContext.deleteObject(enquete);

        // 新規オブジェクトモデル
        AvzTEnquete updateEnquete = Database.create(AvzTEnquete.class);

        // アンケート名
        updateEnquete.setName(name.getValue());

        // 作成ユーザーID
        updateEnquete.setCreaterId(ALEipUtils.getUserId(rundata));

        // コメント
        updateEnquete.setComment(comment.getValue());

        // 作成日
        updateEnquete.setCreateDate(create);

        // 更新日
        updateEnquete.setUpdateDate(now);

        // 締め切り日
        String deadline_flag =
          rundata.getParameters().getString("deadline_date_check");
        if (deadline_flag == null || "".equals(deadline_flag)) {
          updateEnquete.setDeadlineDate(deadline_date.getValue());
          updateEnquete.setDeadlineMethod(EnqueteUtils.DEADLINE_METHOD_AUTO);
        } else if ("TRUE".equals(deadline_flag)) {
          updateEnquete.setDeadlineDate(EnqueteUtils.getEmptyDate());
          updateEnquete
            .setDeadlineMethod(EnqueteUtils.DEADLINE_METHOD_HANDOPERATED);
        } else {
          return false;
        }

        // 項目数
        int length = getItemSize(rundata);
        updateEnquete.setItemNum(length - 1);

        // 公開方法 (記名・無記名)
        if (EnqueteUtils.PUBLIC_METHOD_SECRET.equals(method.getValue())) {
          updateEnquete.setPublicMethod(EnqueteUtils.PUBLIC_METHOD_SECRET);
        } else if (EnqueteUtils.PUBLIC_METHOD_SIGNED.equals(method.getValue())) {
          updateEnquete.setPublicMethod(EnqueteUtils.PUBLIC_METHOD_SIGNED);
        }

        // 公開締切状態 ※現時点では作成後は「未公開」で設定
        updateEnquete.setPublicStatus(EnqueteUtils.PUBLIC_STATUS_BEFORE);

        int length2 = getItemSize(rundata);
        for (int j = 1; j < length2; j++) {

          // 新規オブジェクトモデル
          AvzTEnqueteItem enqueteitem = Database.create(AvzTEnqueteItem.class);

          enqueteitem.setAvzTEnquete(updateEnquete);
          // 項目番号
          enqueteitem.setItemNumber(j);

          // 項目タイトル
          String item_title =
            rundata.getParameters().getString("item_title[" + j + "]");
          ALStringField item = new ALStringField();
          item.setValue(item_title);
          enqueteitem.setItemTitle(item.getValue());

          // 状態(複数選択可不可・フリーテキスト)
          String status =
            rundata.getParameters().getString("choice_item_select[" + j + "]");
          if ("one_select".equals(status)) {
            enqueteitem.setStatus(EnqueteUtils.STATUS_ONE_SELECT);
          } else if ("multi_select".equals(status)) {
            enqueteitem.setStatus(EnqueteUtils.STATUS_MULTI_SELECT);
          } else if ("free_text".equals(status)) {
            enqueteitem.setStatus(EnqueteUtils.STATUS_FREE_TEXT);
          }

          // 選択肢情報を更新
          if ("free_text".equals(status)) {
            // フリーテキストの場合は選択肢名なしで1ユーザー1レコード
            EnqueteUtils.insertFreeTextChoiceItem(
              rundata,
              context,
              msgList,
              dataContext,
              choice_item_title,
              j,
              updateEnquete,
              enqueteitem,
              memberList);
          } else {
            // 単一選択・複数選択可の場合
            EnqueteUtils.insertChoiceItem(
              rundata,
              context,
              msgList,
              dataContext,
              choice_item_title,
              j,
              updateEnquete,
              enqueteitem,
              memberList);
          }
        }

      } else if (EnqueteUtils.PUBLIC_STATUS_PRACTICE.equals(update_flag)) {

        // アンケート名
        enquete.setName(name.getValue());
        // コメント
        enquete.setComment(comment.getValue());
        // 更新日
        enquete.setUpdateDate(now);
        // 締め切り日
        String deadline_flag =
          rundata.getParameters().getString("deadline_date_check");
        if (deadline_flag == null || "".equals(deadline_flag)) {
          enquete.setDeadlineDate(deadline_date.getValue());
          enquete.setDeadlineMethod(EnqueteUtils.DEADLINE_METHOD_AUTO);
        } else if ("TRUE".equals(deadline_flag)) {
          enquete.setDeadlineDate(EnqueteUtils.getEmptyDate());
          enquete.setDeadlineMethod(EnqueteUtils.DEADLINE_METHOD_HANDOPERATED);
        } else {
          return false;
        }

        // 公開方法 (記名・無記名)
        if (EnqueteUtils.PUBLIC_METHOD_SECRET.equals(method.getValue())) {
          enquete.setPublicMethod(EnqueteUtils.PUBLIC_METHOD_SECRET);
        } else if (EnqueteUtils.PUBLIC_METHOD_SIGNED.equals(method.getValue())) {
          enquete.setPublicMethod(EnqueteUtils.PUBLIC_METHOD_SIGNED);
        }

        List<ALEipUser> selectMember = new ArrayList<ALEipUser>();
        selectMember.addAll(memberList);

        List<Integer> new_member = new ArrayList<Integer>();

        for (int z = 0; z < selectMember.size(); z++) {
          int memberId = (int) (selectMember.get(z)).getUserId().getValue();

          new_member.add(memberId);
        }

        // 更新後
        List<AvzTEnqueteChoiceItem> citem2 = enquete.getAvzTEnqueteChoiceItem();
        for (int n = 0; n < citem2.size(); n++) {
          AvzTEnqueteChoiceItem choiceitem = citem2.get(n);
          int answer = choiceitem.getAnswerId();

          if (new_member.indexOf(answer) == -1) {
            dataContext.deleteObject(choiceitem);
          }
        }

        List count = new ArrayList();

        for (int a = 0; a < selectMember.size(); a++) {

          int memberId = (int) (selectMember.get(a)).getUserId().getValue();

          SelectQuery<AvzTEnqueteChoiceItem> query =
            Database.query(AvzTEnqueteChoiceItem.class);
          Expression exp =
            ExpressionFactory.matchDbExp(
              AvzTEnqueteChoiceItem.AVZ_TENQUETE_PROPERTY
                + "."
                + AvzTEnquete.ENQUETE_ID_PK_COLUMN,
              enquete.getEnqueteId().toString());
          Expression exp2 =
            ExpressionFactory.matchExp(
              AvzTEnqueteChoiceItem.ANSWER_ID_PROPERTY,
              memberId);
          query.setQualifier(exp.andExp(exp2));

          List<AvzTEnqueteChoiceItem> itemlist = query.getResultList();

          if (itemlist.size() != 0) {

            count.add(selectMember.get(a));

          }

        }

        memberList.removeAll(count);

        if (memberList.size() != 0) {

          int itemsize = enquete.getAvzTEnqueteItem().size();
          for (int m = 0; m < itemsize; m++) {
            AvzTEnqueteItem enqueteitem =
              (AvzTEnqueteItem) enquete.getAvzTEnqueteItem().get(m);

            // 選択肢情報を更新
            if ("free_text".equals(status)) {
              // フリーテキストの場合は選択肢名なしで1ユーザー1レコード
              EnqueteUtils.updateFreeTextChoiceItem(
                rundata,
                context,
                msgList,
                dataContext,
                choice_item_title,
                m,
                enquete,
                enqueteitem,
                memberList);
            } else {
              // 単一選択・複数選択可の場合
              EnqueteUtils.updateChoiceItem(
                rundata,
                context,
                msgList,
                dataContext,
                choice_item_title,
                m,
                enquete,
                enqueteitem,
                memberList);
            }
          }
        }

      }

      dataContext.commitChanges();

      // イベントログに保存
      ALEventlogFactoryService.getInstance().getEventlogHandler().log(
        enquete.getEnqueteId(),
        ALEventlogConstants.PORTLET_TYPE_NONE,
        name.getValue());

    } catch (Exception ex) {
      dataContext.rollbackChanges();
      logger.error("Exception", ex);
      return false;
    }
    return true;
  }

  /**
   * Enqueteをデータベースから削除します。 <BR>
   * 
   * @param rundata
   * @param context
   * @param msgList
   * @return TRUE 成功 FALSE 失敗
   * @see com.aimluck.eip.common.ALAbstractFormData#deleteFormData(org.apache.turbine.util.RunData,
   *      org.apache.velocity.context.Context)
   */
  @Override
  protected boolean deleteFormData(RunData rundata, Context context,
      List<String> msgList) {
    try {
      // オブジェクトモデルを取得
      AvzTEnquete enquete = EnqueteUtils.getAvzTEnquete(rundata, context);
      if (enquete == null) {
        return false;
      }

      // entityIdの取得
      int entityId = enquete.getEnqueteId();
      // Enquete名の取得
      String name = enquete.getName();

      // Enqueteを削除
      dataContext.deleteObject(enquete);
      dataContext.commitChanges();

      // イベントログに保存
      ALEventlogFactoryService.getInstance().getEventlogHandler().log(
        entityId,
        ALEventlogConstants.PORTLET_TYPE_NONE,
        name);

    } catch (Exception ex) {
      dataContext.rollbackChanges();
      logger.error("Exception", ex);
      return false;
    }
    return true;
  }

  public static JetspeedLogger getLogger() {
    return logger;
  }

  public DataContext getDataContext() {
    return dataContext;
  }

  /**
   * アンケートIDを取得します。 <BR>
   * 
   * @return
   */
  public ALNumberField getEnqueteId() {
    return enquete_id;
  }

  /**
   * アンケート名を取得します。 <BR>
   * 
   * @return
   */
  public ALStringField getName() {
    return name;
  }

  /**
   * コメントを取得します。 <BR>
   * 
   * @return
   */
  public ALStringField getComment() {
    return comment;
  }

  /**
   * 公開方法を取得します。 <BR>
   * 
   * @return
   */
  public ALStringField getMethod() {
    return method;
  }

  /**
   * 公開・締切状態を取得します。 <BR>
   * 
   * @return
   */
  public ALStringField getStatus() {
    return status;
  }

  /**
   * 締め切り日を取得します。 <BR>
   * 
   * @return
   */
  public ALDateTimeField getDeadlineDate() {
    return deadline_date;
  }

  /**
   * 終了日時を取得します。
   * 
   * @return
   */
  public ALDateTimeField getDeadlineDateSub() {
    ALDateTimeField tmpDate = new ALDateTimeField("yyyy/MM/dd");
    tmpDate.setValue(deadline_date.getValue());
    return tmpDate;
  }

  /**
   * 終了日時を取得します。
   * 
   * @return
   */
  public ALDateTimeField getDeadLineDateTime() {
    ALDateTimeField tmpDate = new ALDateTimeField("HH:mm");
    tmpDate.setValue(deadline_date.getValue());
    return tmpDate;
  }

  /**
   * 締め切り日指定フラグを取得します。 <BR>
   * 
   * @return
   */
  public ALStringField getDeadlineMethod() {
    return deadline_method;
  }

  /**
   * グループメンバーを取得します。
   * 
   * @return
   */
  public List getMemberList() {
    return memberList;
  }

  public List getResults() {
    return results;
  }

  /**
   * 現在の年数を取得します。
   * 
   * @return
   */
  public int getCurrentYear() {
    return currentYear;
  }

  /**
   * フォーム画面における項目数を取得します。
   * 
   * @return
   */
  public int getItemSize(RunData rundata) {
    String itemSizes[] = rundata.getParameters().getStrings("itemSum[]");
    if (itemSizes == null) {
      return 0;
    }

    int itemSize = itemSizes.length;
    return itemSize;
  }

  public ALStringField[] getItemTitle() {
    return item_title;
  }

  public ALStringField[] getChoiceItemSelect() {
    return choice_item_select;
  }

  public ALStringField[] getChoiceItemTitle() {
    return choice_item_title;
  }
}
